//////////////////////////////////////////////////////////////////////////////
// ParDEE - Parallel Dag Execution Engine
// heap.H
// Robert Blumofe
// $Id: heap.H,v 1.4 1998/08/04 20:59:02 emery Exp $
//
// Copyright (c) 1997, Robert Blumofe.  All rights reserved.
// Permission is given to use, copy, and modify this software for any
// non-commercial purpose as long as this copyright notice is not
// removed.  All other uses, including redistribution in whole or in
// part, are forbidden without prior written permission.  This software
// is provided with absolutely no warranty and no support.
//////////////////////////////////////////////////////////////////////////////

#ifndef _PARDEE_HEAP_INCLUDED_
#define _PARDEE_HEAP_INCLUDED_

#include <assert.h>

#include "atomic.H"


// A heap.
class ParHeap {
private:

  enum {
    ALIGN = 8                              // Alignment for heap objects.
  };

  // A heap node.
  union ParHeapNode {
    ParHeapNode* next;  // Next in free list (object is free).
    size_t size;        // Size of object (object is in use).
  };

  typedef ParHeapNode * pParHeapNode;

  // Return size of node rounded up to nearest alignment.
  static size_t nodeSize (void) {
    return (((sizeof(ParHeapNode) + ALIGN - 1) / ALIGN) * ALIGN);
  }

  char *heap, *heapEnd;               // Current heap.
  ParHeapNode* heapList;              // List of allocated heaps.

  unsigned int	       heapSize;	      // Size of a heap.
  unsigned int	       maxSize;	      	      // Max object size.
  unsigned int	       freeLists;	      // The number of free lists.
  ParHeapNode** freeList;  	      // Array of free lists.

  double _pad[PAD_DBL];
  Mutex	_lock;

  // Initialize heap.
  void initHeap (void);

  // Disable copy and assignment.
  ParHeap (const ParHeap&);
  ParHeap& operator= (const ParHeap&);

  void _free (void* p) {
    // Get size.
    ParHeapNode* c = (ParHeapNode*)((char*)p - nodeSize());
    size_t sz = c->size;
    assert (sz < freeLists);
    assert (heap);

    // Add to free list.
    c->next = freeList[sz];
    freeList[sz] = c;
  }

public:

  // Constructor and destructor.
  ParHeap (unsigned int heapsize = 1024 * 1024,
	   unsigned int maxsize = 512 * 1024);

  ~ParHeap (void);

  // Return maximum object size.
  size_t maxObjectSize (void) { return (maxSize - ALIGN); }

  // Allocate.
  void* alloc (size_t size) {
    Protect<Mutex> m (_lock);
    const size_t sz = (size + ALIGN - 1) / ALIGN;
    assert (sz <= maxSize);
    assert (heap);

    // Get node off free list.
    ParHeapNode* c;
    if ((c = freeList[sz]))
      freeList[sz] = c->next;

    // Make new node from heap.
    else {
      const int s = sz * ALIGN + nodeSize();
      if (heap + s > heapEnd)
	initHeap();
      c = (ParHeapNode*)heap;
      heap += s;
      assert (heap <= heapEnd);
    }

    // Record size and return storage.
    c->size = sz;
    return ((char*)c + nodeSize());
  }

  // Free.
  void free (void* p) {
    Protect<Mutex> m (_lock);
    _free (p);
  }
};

#endif // _PARDEE_HEAP_INCLUDED_

//////////////////////////////////////////////////////////////////////////////
// $Log: heap.H,v $
// Revision 1.4  1998/08/04 20:59:02  emery
// Oops - bug fix.
//
// Revision 1.3  1998/07/17 22:10:02  emery
// Upped maxsize of objects.
//
// Revision 1.2  1998/03/18 04:54:14  emery
// Yet more contention reduction (it's working!).
//
// Revision 1.1  1998/03/17 17:29:14  emery
// Somewhat more flexible version of Bobby's heap code.
//
// Revision 1.3  1997/12/03 17:46:10  rdb
// Change name to ParHeap.  Get rid of defines.
//
// Revision 1.2  1997/11/19 01:03:23  rdb
// Fix bugs so it will compile.
//
// Revision 1.1.1.1  1997/11/19 00:41:46  rdb
// Create ParDEE from old PCM
//
//////////////////////////////////////////////////////////////////////////////
