/*
  machine.C
  ------------------------------------------------------------------------
  The execution engine for a CODE program.
  ------------------------------------------------------------------------
  @(#) $Id: machine.C,v 1.69 1999/03/24 21:16:36 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>


#include "c2_addrmap.h"
#include "heap.H"
#include "machine.H"
#include "sermach.H"
#include "messenger.H"
#include "node.H"
#include "nsrel.H"
#include "parms.H"
#include "timer.H"

static char const rcsid[] = "$Id: machine.C,v 1.69 1999/03/24 21:16:36 emery Exp $";

const char * CODE_Machine::MessageTagString[] = {
  "ZERO",
  "REMOTE_ARC_DATA_TRANSFER",
  "REMOTE_RETURN_FROM_GRAPH",
  // Name-sharing relation lock management.
  "REMOTE_LOCK_REQUEST",
  "REMOTE_LOCK_GRANTED",
  "REMOTE_LOCK_RELEASE",
  "REMOTE_LOCK_DENIED",
  "ENQUEUE_NODE",
  // Work-stealing.
  "WORK_STEAL_REQUEST",
  "WORK_STEAL_GRANTED",
  "WORK_STEAL_DENIED",
  "WORK_STEAL_COMPLETE",
  "UPDATE_NODE_STATISTICS",
  "TERMINATE_ALL_PROCESSES",
  "PROGRAM_TERMINATED"
#ifdef _C2_DBGR_
  ,"TERMINATION_INSTANCE_ID",
  "INSTANCE",
  "EVENT",
  "NEWS_EVENT",
  "SLAVE_DEBUGGER_COMMAND"
#endif
};


typedef padclass<ParHeap> PadParHeap;

PadParHeap * Heap;

typedef padclass<ReadyDeque> PadReadyDeque;
typedef pad<u_long> padu_long;

// _c2_CopyArc is provided by CODE's backend.

extern void _c2_CopyArc (SerialMachine * SerMach,
			 _c2_MessageType mt,
			 int arc,
			 Graph * graph,
			 Index& graphIndex,
			 Index& nodeIndex,
			 Index& portIndex,
			 CompNode * node,
			 _c2_Value& value,
			 Messenger * messenger);


CODE_Machine::CODE_Machine (Messenger* messenger[],
			    int numworkers,
			    bool timenodes)
  : _lock ("CODE_Machine::Lock"),
    _NumWorkers (numworkers),
    _InitNode (NULL),
    _AwaitingStealResponse (0),
    _TimeNodes (timenodes),
    _Worker (new BoundThread[numworkers]),
    _WhichProcessor (messenger[0]->self()),
    _NumProcessors (messenger[0]->numProcessors()),
    _RemainingReports (messenger[0]->numProcessors() - 1),
    _RandVal (162347)
{
  //Heap = new PadParHeap[_NumWorkers];
  typedef SerialMachine * pserialmachine;
  _SerialMachine = new pserialmachine[_NumWorkers];
  int i;
  for (i = 0; i < _NumWorkers; i++) {
    _SerialMachine[i] = new SerialMachine (this, _NumWorkers, _NumProcessors,
					   _WhichProcessor, i, messenger[i], timenodes);
  }
  // Start every processor at a different point
  // in the random number generator series.
  for (i = 0; i < _WhichProcessor; i++) {
    lrand16 ();
  }
}


CODE_Machine::~CODE_Machine (void) {
  if (_InitNode != NULL) {
    delete _InitNode->MyGraph();
  }
  //delete [] Heap;
  delete [] _SerialMachine;
}


u_long CODE_Machine::lrand16 (void)
{
  // (from Blumofe's 'Hood' code)

  // Update random value.
  u_long m = _RandVal;
  m = m * 1103515245 + 12345;
  _RandVal = m;

  // Return 16 bits.
  m = m >> 16;
  return m;
}

// We need non-object functions as wrappers
// for the threads package.

static void CODE_Machine_MessageChecker (void * m) {
  CODE_Machine * machine = dynamic_cast(CODE_Machine *, m);
  while (1) {
    machine->MessageChecker ();
  }
}


void CODE_Machine::Schedule (CompNode * node, int stolen)
{
  int v = lrand16 () % GetNumWorkers();
  // cout << "v = " << v << endl;
  GetSerialMachine(v).Schedule (node, stolen);
}


void CODE_Machine::Run (CompNode * InitNode)
{
  int i;

  _InitNode = InitNode;

  _TotalRuntime.start ();

  // Start the workers.

  // Spawn threads.

  CODE_Machine::WorkerInfo * worker;

  if (GetNumWorkers() == 1) {
    worker = new WorkerInfo;
    worker->machine = _SerialMachine[0];
    worker->initnode = _InitNode;
    SerialMachine_Worker ((void *) worker);
  } else {
    for (i = 0; i < GetNumWorkers(); i++) {
      worker = new WorkerInfo;
      worker->machine = _SerialMachine[i];
      worker->initnode = _InitNode;
      _Worker[i].start (SerialMachine_Worker, (void *) worker);
    }
  }


  // Spawn a message checker
  // if there is more than
  // one processor.

#define USE_MESSAGE_CHECKER 0

#if USE_MESSAGE_CHECKER
  BoundThread	_Checker;	// The message checker thread
  				// (see MessageChecker()).

  if (GetNumProcessors() > 1) {
    _Checker.start (CODE_Machine_MessageChecker, (void *) this);
  }
#endif

  // Wait for everyone to terminate.

  if (GetNumWorkers() > 1) {
    for (i = 0; i < GetNumWorkers(); i++) {
      _Worker[i].join ();
    }
  }

#if USE_MESSAGE_CHECKER
  if (GetNumProcessors() > 1) {
    _Checker.cancel ();
  }
#endif

  delete [] _Worker;
}


void CODE_Machine::Stop (Messenger& m)
{
  // Guard m (_lock);

  // Execution has ended; stop the clock.

  _TotalRuntime.stop ();

#if DEBUG_PRINT
  cout << "Stop: processor = " << WhichProcessor() << endl << flush;
#endif

  if (GetNumProcessors() == 1) {
    assert (WhichProcessor() == 0);
    // Guard m2 (_lock);
#if DEBUG_PRINT
    cout << "Stop: Only one processor -- stop. processor = " << WhichProcessor() << endl << flush;
#endif

    // Tell the SerialMachines to stop.
    for (int i = 0; i < GetNumWorkers(); i++) {
      GetSerialMachine(i).Stop();
    }
  } else {
    if (WhichProcessor() != 0) {
      assert (GetNumProcessors() > 1);
      // Guard m2 (_lock);
#if DEBUG_PRINT
      cout << "Stop: Transmitting statistics. processor = " << WhichProcessor() << endl << flush;
#endif
      
      TransmitStatistics (m);
      
#if DEBUG_PRINT
      cout << "Stop: Transmitted statistics. processor = " << WhichProcessor() << endl << flush;
#endif
      
      // Tell the SerialMachines to stop.
      for (int i = 0; i < GetNumWorkers(); i++) {
	GetSerialMachine(i).Stop();
      }
    }
  }
}


// Check for messages from other processors.
void CODE_Machine::MessageChecker (void)
{
  const int nw = GetNumWorkers();
  int i;

  // Cycle through all the workers
  // in round-robin order and try
  // to send all their messages.

  for (i = 0; i < nw; i++) {
    SerialMachine & sm = GetSerialMachine (i);
    Messenger * m = sm.GetMessenger();
    while (m->flush ())
      ;
    //pthread_testcancel ();
  }
}


SerialMachine& CODE_Machine::GetSerialMachine (int index)
{
  return *_SerialMachine[index];
}


void CODE_Machine::HandleRemoteArcDataTransfer (Messenger& m, SerialMachine& sm)
{
  char Path[Pathname::PATH_ID_LENGTH];
  Index gi;
  Index ni;
  Index PortIndex;
  int arc;
  Graph * TmpGraph;
  // CompNode * TmpNode;
  _c2_Value value;

  m >> Path >> gi >> ni >> PortIndex >> arc;

  TmpGraph = (Graph *) (Pathname (Path)).decode (&sm);
#if DEBUG_PRINT
  cout << "Received arc " << arc << " for " << *TmpGraph << endl << flush;
#endif
  _c2_CopyArc (&sm, RECEIVE, arc, TmpGraph, gi, ni, PortIndex, (CompNode *) NULL, value, &m);
}


// Handle a request for a lock on a name-sharing relation.
void CODE_Machine::HandleRemoteLockRequest (Messenger& m, SerialMachine& sm)
{
  char Path[Pathname::PATH_ID_LENGTH];
  Index PortIndex;
  Index ind;
  Graph * TmpGraph;
  CompNode * TmpNode;
  NameSharingRelation * TmpNSRel;
  int NSRUID;
  int UID;
  int i;
  int ReqType;

  // Message data:
  //  Graph path, Name-sharing relation UID, N.S. relation index,
  //  comp. node path, request type, shared variable UID, and the node's processor.
  m >> Path >> NSRUID >> ind;
  TmpGraph = (Graph *) (Pathname (Path)).decode (&sm);

  m >> Path >> ReqType >> UID >> i;
  TmpNSRel = (NameSharingRelation *) _c2_GetAddr(TmpGraph, NSRUID, &ind, &sm);
  TmpNode = (CompNode *) (Pathname (Path)).decode (&sm);
  
  if (TmpNSRel->GetSharedVarInfo(UID).acquire (TmpNode, (_c2_RequestType) ReqType)) {
    // We acquired the lock:
    // transmit the shared variable's data to the remote processor.
    m << Path << NSRUID << UID;
    NameSharingRelation::Copy (NSRUID, UID, TmpNSRel->GetSharedVarInfo (UID).GetAddress(), TRANSMIT, m);
    m.send (i, CODE_Machine::REMOTE_LOCK_GRANTED);
  } else {
    m << Path << UID;
    m.send (i, CODE_Machine::REMOTE_LOCK_DENIED);
  }
}


void CODE_Machine::HandleRemoteLockGranted (Messenger& m, SerialMachine& sm)
{
  char Path[Pathname::PATH_ID_LENGTH];
  CompNode * TmpNode;
  int NSRUID;
  int UID;

  // I've been granted the lock.
  // Set the lock as acquired, receive the shared variable data,
  // and enqueue the node requesting the lock.
  m >> Path >> NSRUID >> UID;

  TmpNode = (CompNode *) (Pathname (Path)).decode(&sm);
  NameSharingRelation::Copy (NSRUID, UID, (TmpNode->GetLockSet ()[UID])->LocalAddr(), RECEIVE, m);
  ((RemoteNSLock *) TmpNode->GetLockSet ()[UID])->SetAcquired ();
  sm.Schedule (TmpNode);
}


void CODE_Machine::HandleRemoteLockRelease (Messenger& m, SerialMachine& sm)
{ 
  char Path[Pathname::PATH_ID_LENGTH];
  Index ind;
  int ReqType;
  int NSRUID;
  int UID;
  Graph * TmpGraph;
  NameSharingRelation * TmpNSRel;

  m >> Path >> NSRUID >> ind >> ReqType >> UID;

  TmpGraph = (Graph *) (Pathname (Path)).decode(&sm);
  TmpNSRel = (NameSharingRelation *) _c2_GetAddr(TmpGraph, NSRUID, &ind, &sm);
  if (ReqType == _c2_WRITER) {
    // The lock was held by someone with read/write access.
    // Copy back the entire shared variable.
    NameSharingRelation::Copy (NSRUID, UID, TmpNSRel->GetSharedVarInfo (UID).GetAddress(), RECEIVE, m);
  }
  TmpNSRel->GetSharedVarInfo(UID).release ((_c2_RequestType) ReqType);
}


void CODE_Machine::HandleRemoteLockDenied (Messenger& m, SerialMachine&)
{ 
  char Path[Pathname::PATH_ID_LENGTH];
  int UID;
  // CompNode * TmpNode;

  m >> Path >> UID;
  // TmpNode = (CompNode *) (Pathname (Path)).decode(&sm);
}



// Return 0 if I am denying a steal;
// otherwise sends info to thief.
int CODE_Machine::HandleStealRequest (Messenger& m, SerialMachine& sm)
{
  int from;
  CompNode * stolen_node;

  m >> from;

#if DEBUG_PRINT
  printf ("[%d] received a steal request from %d.\n", sm.WhichProcessor(), from);
  fflush (stdout);
#endif

  assert (from >= 0);
  assert (from < sm.GetNumProcessors());

#if !(ALLOW_REMOTE_WORK_STEALING)
  // Deny all steals.

  m << 0;
  m.send (from, WORK_STEAL_DENIED);
  return 0;
#endif

  // Pick a local queue to check.

  int localvictim = sm.lrand16() % sm.GetNumWorkers();

  if (!(sm.GetParallelMachine().GetSerialMachine(localvictim).GetReadyQ().TryStealNode (stolen_node))) {
    // No work -- deny steal.

    m << 0;
    m.send (from, WORK_STEAL_DENIED);
    return 0;

  } else {

    stolen_node->Lock ();

    // Send information to thief.

    // First, ensure that we can obtain all locks. Then, attempt a
    // "pre-firing" (to see if any firing rules are satisfied). Iff all
    // of these succeed, we grant the steal.

    // Set up for pre-firing.


    stolen_node->SetPhase (CompNode::PRE_THEFT_FIRING);

    if (stolen_node->Execute (&m)
	!= CompNode::SUCCESSFUL_PREFIRING) {

      // No firing rule succeeded -- deny steal.

      stolen_node->SetPhase (CompNode::NORMAL);
      stolen_node->Unlock ();
      sm.Schedule (stolen_node);

      m << 0;
      m.send (from, WORK_STEAL_DENIED);
      return 0;

    } else {

      // If this node has state, we can't let it get executed
      // until the current execution has completed.

      // Prevent this node from executing until the theft is complete.
      if (stolen_node->GetStatic ()) {
	stolen_node->Block ();
	stolen_node->SetPhase (CompNode::BLOCKED_AWAITING_THIEF);
      } else {
	stolen_node->SetPhase (CompNode::NORMAL);
      }

#if DEBUG_PRINT
      cout << "(" << sm.WhichProcessor() << ") sending stolen node (" << *stolen_node << ") to (" << from << ")" << endl;
#endif

      // Package up the state and grant the work-steal request.

      m << sm.WhichProcessor()
	<< (stolen_node->GetPathName()).str();
      // Transmit the node state.
      stolen_node->Copy (stolen_node, TRANSMIT, m);
      // Transmit creation parameters.
      Graph * mygraph = stolen_node->MyGraph();

      stolen_node->Unlock();

      // We could have missed an arc firing while doing all this processing,
      // and there's no need to wait on executing this node since it's
      // non-static, so let's schedule it again just in case.
      if (!stolen_node->GetStatic ()) {
	sm.Schedule (stolen_node);
      }

      mygraph->TransmitState (m, from);
      m.send (from, WORK_STEAL_GRANTED);
    }
  }
  return 1;
}



// Someone denied my steal request; deal with it.
void CODE_Machine::HandleStealDenied (Messenger&, SerialMachine& sm)
{
#if DEBUG_PRINT
  printf ("[%d] received a steal denial.\n", sm.WhichProcessor());
  fflush (stdout);
#endif
  --sm.GetParallelMachine().GetAwaitingStealResponse();
}


// This is executed when the thief receives a
// WORK_STEAL_GRANTED message from the victim.
void CODE_Machine::HandleStealGranted (Messenger& m, SerialMachine& sm)
{
#if DEBUG_PRINT
  printf ("[%d] received a steal grant.\n", sm.WhichProcessor());
  fflush (stdout);
#endif
  --sm.GetParallelMachine().GetAwaitingStealResponse();

  // Find out which node we've got.
  char path_string[Pathname::PATH_ID_LENGTH];
  int from;
  m >> from
    >> path_string;

  CompNode * node = dynamic_cast(CompNode *, (Pathname (path_string)).decode(&sm));

#if DEBUG_PRINT
  cout << "(" << sm.WhichProcessor() << ") received stolen node (" << *node << ") from (" << from << ")" << endl;
#endif

  node->Lock ();

  node->Initialize(); // This is a no-op if it was already initialized.
  
  node->SetStolenFrom (from);
  node->SetPhase (CompNode::STOLEN_EXECUTION);

  // Receive node state.
  node->Copy (node, RECEIVE, m);
  // Receive creation parameters.
  Graph * mygraph = node->MyGraph();
  mygraph->ReceiveState (m);
  node->Unlock ();

  // It's ready to go -- run it.

  sm.Execute (node);
}


// The thief has finished with work it stole from me;
// deal with the updated state of the stolen node.
void CODE_Machine::HandleStealComplete (Messenger& m, SerialMachine& sm)
{
  char path_string[Pathname::PATH_ID_LENGTH];

  m >> path_string;

  CompNode * node = dynamic_cast(CompNode *, (Pathname (path_string)).decode(&sm));

  node->Lock ();

  if (node->GetStatic()) {
    // Receive the node's state.
    node->Copy (node, RECEIVE_FROM_THIEF, m);

    // Allow it to execute normally.
    node->SetPhase (CompNode::NORMAL);
  }

  if (node->GetShared()) {
    // Release any locks this node may have held.
    node->RelAllLocks ();
  }

  if (node->GetStatic ()) {
    node->Unblock ();
  }

  node->Unlock ();

  sm.Schedule (node);
}


void CODE_Machine::HandleUpdateNodeStatistics (Messenger& m, SerialMachine& sm)
{
  char Path[Pathname::PATH_ID_LENGTH];
  CompNode * TmpNode;

  // Receive statistics for a node.
  m >> Path;
  TmpNode = (CompNode *) (Pathname (Path)).decode (&sm);
  double time;
  m >> time;

  // cout << "Received " << time << " from " << *TmpNode << endl;

  TmpNode->AddRuntime (time);

  // cout << "Time for " << *TmpNode << " is now " << TmpNode->GetRuntime() << endl;

  // Add targets.
  int targets;
  m >> targets;

  while (targets > 0) {
    CompNode * TargetNode;
    m >> Path;
    TargetNode = (CompNode *) (Pathname (Path)).decode(&sm);
    TmpNode->AddTarget (TargetNode);
    --targets;
  }

}


void CODE_Machine::HandleProgramTerminated (Messenger& m)
{
  assert (WhichProcessor() == 0);
  int dummy;
  m >> dummy;

#if DEBUG_PRINT
  cout << "HandleProgramTerminated: Received a report. processor = " << WhichProcessor() << endl << flush;
#endif

  --_RemainingReports;
  if (_RemainingReports <= 0) {

#if DEBUG_PRINT
    cout << "HandleProgramTerminated: Received all reports. processor = " << WhichProcessor() << endl << flush;
#endif

    if (GetNumProcessors() > 1) {
      WriteProfile ();

      // We're done:
      // tell the SerialMachines to stop.
      for (int i = 0; i < GetNumWorkers(); i++) {
	GetSerialMachine(i).Stop();
      }
    }
  }
}


int CODE_Machine::messageHandler (Messenger& messenger, SerialMachine& sm)
{
  char Path[Pathname::PATH_ID_LENGTH];
  CompNode * TmpNode;
  int MessageReceivedFrom = -1;

  // If a message has been received, process it.

  if (messenger.receive())
    {

      // Get the message length, tag, and sender.
      int nbytes = messenger.messageLength();
      MessageTag tag = (CODE_Machine::MessageTag) messenger.messageTag();
      int sender_tid = messenger.messageSender();

      MessageReceivedFrom = sender_tid;
      
#if DEBUG_PRINT
      cout << "[" << sm.WhichProcessor() << "] message received: tag = " << MessageTagString[tag] << ",  bytes= " << nbytes << endl;
#endif

      switch (tag) {
      case CODE_Machine::REMOTE_ARC_DATA_TRANSFER:
      case CODE_Machine::REMOTE_RETURN_FROM_GRAPH:
	HandleRemoteArcDataTransfer (messenger, sm);
	break;
    
      case CODE_Machine::REMOTE_LOCK_REQUEST:
	HandleRemoteLockRequest (messenger, sm);
	break;

      case CODE_Machine::REMOTE_LOCK_GRANTED:
	HandleRemoteLockGranted (messenger, sm);
	break;

      case CODE_Machine::REMOTE_LOCK_RELEASE:
	HandleRemoteLockRelease (messenger, sm);
	break;

      case CODE_Machine::REMOTE_LOCK_DENIED:
	HandleRemoteLockDenied (messenger, sm);
	break;

      case CODE_Machine::ENQUEUE_NODE:
	messenger >> Path;
	TmpNode = (CompNode *) (Pathname (Path)).decode (&sm);
	sm.Schedule (TmpNode);
	break;

      case CODE_Machine::WORK_STEAL_REQUEST:
	HandleStealRequest (messenger, sm);
	break;

      case CODE_Machine::WORK_STEAL_GRANTED:
	HandleStealGranted (messenger, sm);
	break;

      case CODE_Machine::WORK_STEAL_DENIED:
	HandleStealDenied (messenger, sm);
	break;

      case CODE_Machine::WORK_STEAL_COMPLETE:
	HandleStealComplete (messenger, sm);
	break;

	// Receive statistics for a node.
      case CODE_Machine::UPDATE_NODE_STATISTICS:
	HandleUpdateNodeStatistics (messenger, sm);
	break;

      case CODE_Machine::TERMINATE_ALL_PROCESSES:
	sm.GetParallelMachine().Stop (messenger);
	break;

      case CODE_Machine::PROGRAM_TERMINATED:
	sm.GetParallelMachine().HandleProgramTerminated (messenger);
	break;

      default:
	assert (0); // An unknown message tag has been received. This should never happen.
	break;
      }

      messenger.objectlock().release();
    }

  return MessageReceivedFrom;
}


// Try to steal some work remotely.

void CODE_Machine::AttemptSteal (Messenger& m)
{
  // Guard m1 (_lock);
  if (!_AwaitingStealResponse) {
    ++_AwaitingStealResponse;
    
    // Pick another processor at random (not me).
    int victim = lrand16() % (GetNumProcessors() - 1);
    victim += (victim >= WhichProcessor());
    
    Guard m1 (m.objectlock());
    m << WhichProcessor();
#if DEBUG_PRINT
    printf ("[%d] sending a steal request to %d.\n", WhichProcessor(), victim);
    fflush (stdout);
    //cout << "Send WORK_STEAL_REQUEST from " << WhichProcessor() << " to " << victim << endl;
#endif
    m.send (victim, WORK_STEAL_REQUEST);
  }
}


void CODE_Machine::TransmitStatistics (Messenger& m)
{
  // Guard m (_lock);

  const int id = 0;
  
  assert (WhichProcessor() != 0);

#if DEBUG_PRINT
  cout << "(" << WhichProcessor() << ") transmitting statistics." << endl << flush;
#endif

  // Send all statistics to processor 0.
  Guard m2 (m.objectlock ());
  //_InitNode->MyGraph()->TransmitStats (0, m);

  //cout << "Sending program terminated." << endl << flush;
  m << 0;
  m.send (0, PROGRAM_TERMINATED);

#if DEBUG_PRINT
  cout << "(" << WhichProcessor() << ") done transmitting statistics." << endl << flush;
#endif
}


void CODE_Machine::WriteProfile (void)
{

  const int id = 0;

  assert (WhichProcessor() == 0);

#if DEBUG_PRINT
  cout << "(" << WhichProcessor() << ") gathering statistics." << endl << flush;
#endif

  ofstream out ("profile");
  // out << "# Node\tWall clock\n";
  // _InitNode->MyGraph()->ReportStats (out);
  double t1 = _InitNode->MyGraph()->T1();

#if DEBUG_PRINT
  cout << "(" << WhichProcessor() << ") calculated T_1." << endl << flush;
#endif

  double tinfinity = _InitNode->Tinfinity();

#if DEBUG_PRINT
  cout << "(" << WhichProcessor() << ") calculated T_infty." << endl << flush;
#endif
    
  // Calculate the average load imbalance.
    
#if DEBUG_PRINT
  cout << "(" << WhichProcessor() << ") done calculating statistics." << endl << flush;
#endif

  out << "Total runtime:\t" << (double) _TotalRuntime << endl;
  out << "Total work:\t" << t1 << endl;
  out << "Critical path:\t" << tinfinity << endl;
  out << "Utilization:\t" << t1 / (GetNumProcessors() * GetNumWorkers() * (double) _TotalRuntime) << endl;
  // out << "Average load imbalance:\t" << imbalance / (GetNumProcessors() * GetNumWorkers() * (double) _TotalRuntime) << endl;
  if (tinfinity == 0) {
    out << "Average parallelism:\tInfinity" << endl;
  } else {
    out << "Average parallelism:\t" << t1 / tinfinity << endl;
  }
}
