/*
  machine.H
  ------------------------------------------------------------------------
  The execution engine for a CODE program.
  ------------------------------------------------------------------------
  @(#) $Id: machine.H,v 1.40 1998/11/06 05:08:15 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#ifndef _MACHINE_H_
#define _MACHINE_H_

#include "header.H"
#include "mybool.h"
#include "messenger.H"
#include "atomic.H"
#include "readyq.H"
#include "timer.H"
#include "unique.H"
#include "pad.h"

typedef unsigned long u_long;
typedef pad<u_long> padu_long;

// forward declarations
class CompNode;
class ReadyDeque;
class ParHeap;
class SerialMachine;


// The CODE execution engine.

class CODE_Machine : public Unique {
public:

  CODE_Machine (Messenger* messenger[],		// An array of messengers (one per worker).
		int numworkers,			// The number of worker threads to create.
		bool timenodes = false);	// Do we time the execution of every node?

  ~CODE_Machine (void);

  // The maximum number of worker threads that can be created.
  enum { MAX_THREADS = 64 };

  // The below are "tags" attached to messages sent via the messengers.
  // They indicate the kind of message sent.
  typedef
  enum { REMOTE_ARC_DATA_TRANSFER = 1,		// An arc between nodes.
	 REMOTE_RETURN_FROM_GRAPH,		// An arc from a graph to a node.
	 // Name-sharing relation lock management.
	 REMOTE_LOCK_REQUEST,			// A request of a lock on a remote shared-variable.
	 REMOTE_LOCK_GRANTED,			// A reply to the above (the lock was granted).
	 REMOTE_LOCK_RELEASE,			// A notification that the lock can be released.
	 REMOTE_LOCK_DENIED,			// A negative reply to a REMOTE_LOCK_REQUEST.
	 ENQUEUE_NODE,				// Enqueue a particular node.
	 // Work-stealing.
	 WORK_STEAL_REQUEST,			// I want work.
	 WORK_STEAL_GRANTED,			// You got it.
	 WORK_STEAL_DENIED,			// I ain't got none.
	 WORK_STEAL_COMPLETE,			// I finished your work.
	 // Statistics collection.
	 UPDATE_NODE_STATISTICS,		// Add my runtime for a node to yours.
	 // Stop computation.
	 TERMINATE_ALL_PROCESSES,		// We're done. Stop the presses.
	 PROGRAM_TERMINATED			// I've finished transmitting my statistics.
#ifdef _C2_DBGR_
	 TERMINATION_INSTANCE_ID,
	 INSTANCE,
	 EVENT,
	 NEWS_EVENT,
	 SLAVE_DEBUGGER_COMMAND
#endif
  } MessageTag;

  // An array of strings containing messages.
  // These correspond to the enum names
  // (i.e., MessageTagString[1] = "REMOTE_ARC_DATA_TRANSFER").
  static const char * MessageTagString[];

  // When we spawn workers, we pass in this information as the argument.
  typedef struct {
    SerialMachine * machine;
    CompNode * initnode;
  } WorkerInfo;


  // set the number of threads.
  void SetNumWorkers (const int np);

  // Schedule a particular node for execution.
  void Schedule (CompNode * node, int stolen = FALSE);

  // Schedule a stolen node.
  void StolenSchedule (CompNode * node) { Schedule (node, TRUE); }

  // Initialize CODE and begin running.
  void Run (CompNode * InitNode);

  // Stop CODE execution.
  void Stop (Messenger& m);

  // Check for messages from other processors.
  void MessageChecker (void);

  // Write the profile.
  void WriteProfile (void);

  // Try to steal some work (remotely).
  void AttemptSteal (Messenger& m);

  // ACCESSORS.


  // get the number of threads.
  const int GetNumWorkers (void) const { return _NumWorkers; }

  // get the number of processors (= machines).
  const int GetNumProcessors (void) const { return _NumProcessors; }

  // which processor am I?
  const int WhichProcessor (void) const { return _WhichProcessor; }

  SerialMachine& GetSerialMachine (int index);

  // handle incoming messages.
  // Returns -1 iff no message was received.
  // Otherwise, returns the message sender.
  static int messageHandler (Messenger& messenger, SerialMachine& sm);

  int& GetAwaitingStealResponse (void) { return _AwaitingStealResponse; }

private:

  // An internal random number generator.
  u_long lrand16 (void);

  //
  // MessageHandler subroutines:
  //

  static void HandleRemoteArcDataTransfer (Messenger& m, SerialMachine& sm);

  static void HandleRemoteLockRequest (Messenger& m, SerialMachine& sm);
  static void HandleRemoteLockGranted (Messenger& m, SerialMachine& sm);
  static void HandleRemoteLockRelease (Messenger& m, SerialMachine& sm);
  static void HandleRemoteLockDenied (Messenger& m, SerialMachine& sm);

  static int  HandleStealRequest (Messenger& m, SerialMachine& sm);
  // Return 0 if I am denying a steal;
  // otherwise sends info to thief.

  static void HandleStealDenied (Messenger& m, SerialMachine& sm);
  // Someone denied my steal request; deal with it.

  static void HandleStealGranted (Messenger& m, SerialMachine& sm);
  // Someone granted my steal request; deal with it.

  static void HandleStealComplete (Messenger& m, SerialMachine& sm);
  // The thief has finished with work it stole from me;
  // deal with the updated state of the stolen node.

  static void HandleUpdateNodeStatistics (Messenger& m, SerialMachine& sm);

  void HandleProgramTerminated (Messenger& m);



  // Transmit all timing statistics.
  void TransmitStatistics (Messenger& m);

  mutable MutexRec _lock;	// The lock for these data structures.

  double _pad[PAD_DBL]; // padding to avoid false sharing.

  u_long	_RandVal;	// Random values.
  const int	_NumWorkers;	// Number of workers (= number of CPUs).

  int	_AwaitingStealResponse;	// True iff we are awaiting a response from a work-steal victim.

  SerialMachine ** _SerialMachine; // All of the worker machines.

  Atomic<int>	_RemainingReports; // How many reporting processors remain (during statistics gathering after program termination).

  // Timing statistics.

  Timer		_TotalRuntime;	// Total runtime.
  bool		_TimeNodes;	// If true, we collect timings per node.

  //

  BoundThread *	_Worker;	// The array of worker threads (see SerialMachine::Worker()).
  CompNode 	*_InitNode;	// The start node.

  Condition	_MessengerCond;

  const int	_WhichProcessor;	// Which processor are we?
  const int	_NumProcessors;		// How many processors are there?
};


#endif /* _MACHINE_H_ */
