/*
  messenger.H
  ------------------------------------------------------------------------
  A Messenger handles sending and receipt of messages.
  This class hides the differences between PVM and MPI
  (or any similar point-to-point message-passing
  interface).
  ------------------------------------------------------------------------
  @(#) $Id: messenger.H,v 1.21 1998/09/15 21:42:28 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#ifndef _MESSENGER_H_
#define _MESSENGER_H_

#include <stdlib.h>

#include "atomic.H"
#include "unique.H"

class Packet {
public:
  Packet (int tag, void * buf, int length)
    : _tag (tag),
      _buf (new char[length]),
      _length (length)
    {
      memcpy ((void *) _buf, buf, length);
    }

  ~Packet (void) { delete _buf; }

  void * getBuffer (void) { return _buf; }
  const int getLength (void) { return _length; }
  const int getTag (void) { return _tag; }

private:
  const int _tag;
  void * _buf;
  const int _length;
};


class InPacket : public Packet {
public:

  InPacket (int src, int tag, void * buf, int length)
    : Packet (tag, buf, length),
      _src (src)
    {}

  ~InPacket (void) {}

  int getSource (void) { return _src; }

private:

  int _src;
};


class OutPacket : public Packet {
public:

  OutPacket (int dest, int tag, void * buf, int length)
    : Packet (tag, buf, length),
      _dest (dest)
    {}

  ~OutPacket (void) {}

  int getDestination (void) { return _dest; }

private:

  int _dest;
};


/*
  A Messenger handles sending and receipt of messages.
  This class hides the differences between PVM and MPI
  (or any similar point-to-point message-passing
  interface).

  All messages are packed into byte buffers and sent en bloc.
  */

class Messenger : public Unique {

  friend Messenger& operator>>(Messenger&m, char& v) { m.unpack (v); return m; }
  friend Messenger& operator>>(Messenger&m, int& v) { m.unpack (v); return m; }
  friend Messenger& operator>>(Messenger&m, long& v) { m.unpack (v); return m; }
  friend Messenger& operator>>(Messenger&m, double& v) { m.unpack (v); return m; }
  friend Messenger& operator>>(Messenger&m, float& v) { m.unpack (v); return m; }
  friend Messenger& operator>>(Messenger&m, char * v) { m.unpack (v); return m; }

  friend Messenger& operator<<(Messenger&m, const char v) { m.pack (v); return m; }
  friend Messenger& operator<<(Messenger&m, const int v) { m.pack (v); return m; }
  friend Messenger& operator<<(Messenger&m, const char * v) { m.pack (v); return m; }
  friend Messenger& operator<<(Messenger&m, char * v) { m.pack (v); return m; }
  friend Messenger& operator<<(Messenger&m, const long v) { m.pack (v); return m; }
  friend Messenger& operator<<(Messenger&m, const double v) { m.pack (v); return m; }
  friend Messenger& operator<<(Messenger&m, const float v) { m.pack (v); return m; }

public:

  // Maximum number of processors, also an arbitrary upper limit.
  // NB: Make this a multiple of 8 so that the MAX_PROCESSOR_BYTES
  // enum below works properly.
  enum { MAX_PROCESSORS   = 64 };

  // The number of bytes needed to represent MAX_PROCESSORS.
  enum { MAX_PROCESSOR_BYTES = MAX_PROCESSORS / 8 };

  // "Protocols" for sending and receiving.
  typedef enum { BLOCKING, NONBLOCKING } Protocol;

  // Matches a message with any tag.
  enum { ANY_TAG = -1 };

  // Matches a message from any source.
  enum { ANY_SOURCE = -1 };


  Messenger (void)
    // : _objectlock ("Messenger::ObjectLock")
    { }

  virtual ~Messenger (void) { }

  // How many processors are there in my cluster?
  virtual int numProcessors (void) = 0;

  // What is my rank among the processors in my cluster? (0..n-1)
  virtual int self (void) = 0;

  // OPERATIONS.

  // do any initializations from command-line parameters
  virtual void initialize (int& argc, char **& argv) = 0;

  // do any final cleanups required.
  virtual void finalize (void) = 0;

  // send to a processor. -1 = any processor.
  virtual void send (const int processor,
		     const int tag) = 0;

  // receive from anyone.
  virtual int receive (void) = 0;
  
  // broadcast to a range of processors.
  virtual void broadcast (const int firstProcessor,
			  const int lastProcessor,
			  const int tag) = 0;

  // pack some data into the send buffer.
  virtual void pack (const char datum) = 0;
  virtual void pack (const int datum) = 0;

  // array packing (note that the first argument must be a pointer).
  virtual void pack (const int* data, const int length) = 0;
  virtual void pack (const float * data, const int length) = 0;
  virtual void pack (const double * data, const int length) = 0;
  virtual void pack (const char* datum, const int length) = 0;

  virtual void pack (const char * string) = 0;
  virtual void pack (char * string) = 0;

  virtual void pack (const long datum) = 0;
  virtual void pack (const double datum) = 0;
  virtual void pack (const float datum) = 0;

  // unpack data from the receive buffer.
  virtual void unpack (char& datum) = 0;
  virtual void unpack (int& datum) = 0;

  // array unpacking (note that the first argument must be a pointer).
  virtual void unpack (int* data, const int length) = 0;
  virtual void unpack (double * data, const int length) = 0;
  virtual void unpack (float * data, const int length) = 0;
  virtual void unpack (char* datum, const int length = 0) = 0;

  virtual void unpack (long& datum) = 0;
  virtual void unpack (double& datum) = 0;
  virtual void unpack (float& datum) = 0;

  MutexRec& objectlock (void) {
    return _objectlock;
  }

  static MutexRec& classlock (void) {
    return _classlock;
  }

  virtual int messageLength (void) = 0;
  virtual int messageSender (void) = 0;
  virtual int messageTag (void) = 0;

  virtual int flush (void) = 0;

private:

  MutexRec	_objectlock;	// A per-object lock.

  double _pad1[PAD_DBL]; // padding to avoid false sharing.

  static MutexRec	 _classlock;	// A per-class lock, to let us use non-thread safe
  					// message passing systems (like MPI or PVM).

  double _pad2[PAD_DBL]; // padding to avoid false sharing.
};

#endif /* _MESSENGER_H_ */
