/*
  node.C
  ------------------------------------------------------------------------
  Computation node base class implementation.
  ------------------------------------------------------------------------
  @(#) $Id: node.C,v 1.55 1998/11/25 06:12:07 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#include <assert.h>
#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>

#include "c2_misc.h"

#include "header.H"
#include "machine.H"
#include "sermach.H"
#include "node.H"
#include "readyq.H"
#include "nsrel.H"
#include "nslocks.H"
#include "parms.H"

static char const rcsid[] = "$Id: node.C,v 1.55 1998/11/25 06:12:07 emery Exp $";


CompNode::CompNode (int uid,
		    Index * index,
		    Graph * graph,
		    CodeDataBase * localdata,
		    CompNode::ExecutionResult (*comp) (CompNode *),
		    void (*init) (CompNode *),
		    SerialMachine * machine,
		    bool isstatic = TRUE,
		    bool writesshared = FALSE)
  : CodeObject	(uid, index, localdata, graph, machine, CodeObject::UC),
    _phaselock ("CompNode::_phaselock"),
    _queuelock ("CompNode::_queuelock"),
    _NSState	(UNINITIALIZED),
    _Static	(isstatic),
    _Shared	(writesshared),
    _initialized (FALSE),  // Haven't run InitProc yet.
    _CompProc	(comp),
    _InitProc	(init),
    _Phase	(NORMAL),
    _NQueued (0),
    _QueueStatus (OFFQUEUE),
    _Processor (GetProcessor ()),
    _StolenFrom (-1)
{
  ASSERT (_CompProc != NULL);
  ASSERT (_InitProc != NULL);
  ASSERT (MyGraph()->GetCrepsToGo() >= 0);
  if (_Shared) {
    _LockSet = new NSLockSet;
  }
}


void CompNode::StartTimer (void) {
  _Timer.start();
}


void CompNode::StopTimer (void) {
  _Timer.stop();
}


void CompNode::Initialize (void) 
{
  if (!_initialized) {
    _InitProc (this);
    _initialized = 1;
  }
}


void CompNode::Block (void)
{
  Guard m (_queuelock);
  assert (_QueueStatus == OFFQUEUE);
  _QueueStatus = STOLEN;
}


void CompNode::Unblock (void) {
  Guard m (_queuelock);
  assert (_QueueStatus == STOLEN);
  _QueueStatus = OFFQUEUE;
}


void CompNode::Lock (void) 
{
  Guard m (_lock);
  _lock.acquire ();
}


void CompNode::Unlock (void) 
{
  Guard m (_lock);
  _lock.release ();
}


CompNode::ExecutionBehavior CompNode::GetPhase (void)
{
  Guard m (_phaselock);
  return _Phase;
}


void CompNode::SetPhase (CompNode::ExecutionBehavior phase)
{
  Guard m (_phaselock);
  _Phase = phase;
}


int CompNode::NSInitialized (void)
{
  Guard m (_lock);
  return _NSState != UNINITIALIZED; 
}


void CompNode::NSInitialize (void) 
{
  Guard m (_lock);
  assert (_NSState == UNINITIALIZED);
  _NSState = ACQUIRED;
}


void CompNode::Enqueue (void)
{
  GetMachine().GetReadyQ().Enqueue (this);
}


int CompNode::NSBusy (void)
{
  Guard m (_lock);
  return (_NSState == BUSY);
}


void CompNode::MakeOffqueue (void)
{
  Guard m (_queuelock);
  assert (_QueueStatus == ONQUEUE);
  _QueueStatus = OFFQUEUE;
}


void CompNode::MakeOnqueue (void)
{
  Guard m (_queuelock);
  assert (_QueueStatus == OFFQUEUE);
  _QueueStatus = ONQUEUE;
}


void CompNode::PopFront (deq<CompNode *>& dq) 
{
  Guard m (_queuelock);
  assert (_QueueStatus == ONQUEUE);
  dq.pop_front ();
  MakeOffqueue ();
}


void CompNode::PopBack (deq<CompNode *>& dq) 
{
  Guard m (_queuelock);
  assert (_QueueStatus == ONQUEUE);
  dq.pop_back ();
  MakeOffqueue ();
}


void CompNode::PushFront (deq<CompNode *>& dq) 
{
  Guard m (_queuelock);
  if (_QueueStatus == OFFQUEUE) {
    dq.push_front (this);
    MakeOnqueue ();
  }
}


void CompNode::PushBack (deq<CompNode *>& dq) 
{
  Guard m (_queuelock);
  if (_QueueStatus == OFFQUEUE) {
    dq.push_back (this);
    MakeOnqueue ();
  }
}


CompNode::ExecutionResult CompNode::Execute (Messenger * messenger)
{
  // Guard m (_lock);
  _Messenger = messenger;

  ExecutionResult result;

  if (!_initialized) {
    if (MyGraph()->GetCrepsToGo() == 0) {
      Initialize();
    } else {
      // The graph has not yet received all of its creation parameters;
      // when it does, this node will be requeued.
      return NOT_CREPPED;
    }
  }

  ASSERT (_initialized);
  ASSERT (GetPhase() != CompNode::BLOCKED_AWAITING_THIEF);

  result = _CompProc (this);
  
  if (result == SUCCESSFUL_STOLEN)
    {
      if (GetStatic() || GetShared()) {
	// Send back information to the victim.
	Guard m (GetMessenger()->objectlock ());
	*GetMessenger() << GetPathName().str();
	// Send back node state.
	if (GetStatic()) {
	  Copy (this, TRANSMIT_FROM_THIEF, *GetMessenger());
	}
	GetMessenger()->send (GetStolenFrom(), CODE_Machine::WORK_STEAL_COMPLETE);
      }
      SetPhase (NORMAL);
    }

  return result;
}


int CompNode::RemoveFromInputPort (Port * Q,
				   Index& I,
				   _c2_Value& V) 
{
  assert (Q);
  if (Q->Remove (I, V)) {
    return 1;
  } else {
    return 0;
  }
}


void CompNode::RelAllLocks (void)
{
  if (GetShared()) {
    Guard m (_lock);
    GetLockSet().release();
    _NSState = ACQUIRED;
  }
}


int CompNode::GetAllLocks (void)
{
  if (GetShared()) {
    Guard m (_lock);
    assert (_NSState == BUSY || _NSState == ACQUIRED);
    int acquired = GetLockSet().acquire (this);
    if (acquired) {
      _NSState = ACQUIRED;
    } else {
      _NSState = BUSY;
    }
    return acquired;
  } else {
    return 1; // Act like we acquired all our locks.
  }
}


ostream& operator<<(ostream& os,
		    CompNode& node)
{
  extern _c2_NameTable NameTable;
  _c2_NameTable * Table = &NameTable;

  int i;

  for (i = 0; Table[i].UID > 0; i++) {
    if (node.ID() == Table[i].UID)
      break;
  }

  assert (node.ID() == Table[i].UID);

  os << *(node.MyGraph()) << "/";

  for (int index = 0; index < node.MyIndex().length(); index++)
    os << "[" << node.MyIndex()[index] << "]";

  os << Table[i].Name;

  return os;
}


void CompNode::ReportStats (ostream& out)
{
  out << "\t" << (double) _Timer << endl;
}


void CompNode::TransmitStats (int processor, Messenger& messenger)
{
  if ((double) _Timer == 0.0) {
    return;
  }

  messenger << GetPathName().str()
	    << (double) _Timer;
  
#if !(USE_TARGETS)
  messenger << (int) 0;
#else
  messenger << (int) _Targets.size ();

  set<CompNode *, less<CompNode *> >::iterator i;

  for (i = _Targets.begin ();
       i != _Targets.end ();
       ++i)
    {
      messenger << (*i)->GetPathName().str();
    }

  messenger.send (processor, CODE_Machine::UPDATE_NODE_STATISTICS);
#endif
}



int CompNode::GetProcessor (void) {
  int processor;
  if (GetMachine().GetNumProcessors() == 1) {
    processor = 0;
  } else {

    // Guard m (_lock);

#if ONE_PROCESSOR_PLACEMENT
    // One-processor placement.
    processor = 0;
    return processor;
#endif
    
#if PER_GRAPH_PLACEMENT
    // Per-graph placement.
    processor = MyGraph()->Processor();
    return processor;
#endif
    
#if MAP_ACROSS_ALL
    // Old-style placement.
    processor = GetPathName().map (GetMachine().GetNumProcessors());
    return processor;
#endif
    
#if MAP_NON_SHARED_ACROSS_ALL
    if (GetShared()) {
      
      // If the node uses a shared variable,
      // keep it with its graph.
      
      processor = MyGraph()->Processor();
    } else {
      if (GetStatic()) {
	processor = GetPathName().map (GetMachine().GetNumProcessors());
      } else {
#if MAP_NONSTATIC_LOCALLY
	// If the node is not static, it can be anywhere,
	// so to avoid communication, it might as well be "here".
	processor = GetMachine().WhichProcessor();
#else
	  processor = GetPathName().map (GetMachine().GetNumProcessors());
#endif
      }
    }
#endif
    
  }
    
  // cout << "Mapped " << GetPathName().str() << " to " << processor << "." << endl;
  return processor;
}


double CompNode::Visit (CompNode * root, set <CompNode *, less<CompNode *> >& visited)
{
  set <CompNode *, less<CompNode *> >::iterator i;
  double max = 0.0;
  
  visited.insert (root);
#if USE_TARGETS
  for (i = root->_Targets.begin();
       i != root->_Targets.end();
       ++i) {

    // As long as this target is not in the visited set,
    // visit the target node.

#if DEBUG_PRINT
    cout << "Visiting " << *(*i) << endl;
#endif

    if (visited.find (*i) == visited.end()) {
      double tmp = Visit (*i, visited);
      max = tmp > max ? tmp : max;
    }
  }
  visited.erase (root);
#endif
  
  return max + (double) root->_Timer;
}


double CompNode::Tinfinity (void)
{
  set <CompNode *, less<CompNode *> > visited;
  return Visit (this, visited);
}
