/*
  node.H
  ------------------------------------------------------------------------
  Computation node base class.
  ------------------------------------------------------------------------
  @(#) $Id: node.H,v 1.48 1998/11/06 05:17:34 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/



#ifndef _NODE_H_
#define _NODE_H_

#include <stdio.h>

#define _PTHREADS

#define USE_TARGETS 1

#if USE_TARGETS
#define times times_
#include <set.h> // STL
#undef times
#endif


#include "codeobject.H"
#include "graph.H"
#include "nodedeque.H"
#include "nslocks.H"
#include "port.H"
#include "timer.H"
#include "sermach.H"
#include "deq.h"

class ReadyDeque; // forward declaration


class CompNode : public CodeObject {
  friend ostream& operator<<(ostream&, CompNode&);

  friend int operator<(CompNode& a, CompNode& b) {
    return a.ID() < b.ID();
  }

  // friend class ReadyDeque;

public:

  // These define the behavior of a node when it is executed.
  typedef
  enum {
    NORMAL = 0,			// Normal (runs to completion).
    PRE_THEFT_FIRING,		// Only runs the firing rules.
    STOLEN_EXECUTION,		// Skips the firing rules and sends back state.
    BLOCKED_AWAITING_THIEF	// Won't execute at all.
  }
  ExecutionBehavior;
  
  
  // These define the node's state with respect to "its" name-sharing relations.
  typedef
  enum {
    UNINITIALIZED,		// The node has not initialized its name-sharing relations.
    ACQUIRED,			// The node has acquired all locks on all of its n.s. relations.
    BUSY			// Some other node may hold some locks on this node's n.s. relations.
  }
  NameSharingRelationState;


  typedef
  enum {
    SUCCESSFUL,			// The node ran to completion.
    SUCCESSFUL_PREFIRING,	// The node fired (in prep for stealing).
    SUCCESSFUL_STOLEN,		// The (stolen) node completed.
    NSREL_BUSY,			// The node couldn't lock an nsrel.
    NOT_CREPPED,		// The node's graph isn't crepped.
    NO_FIRING_RULES_TRUE
  }
  ExecutionResult;

public:

  CompNode (int uid,
	    Index * index,
	    Graph * graph,
	    CodeDataBase * localdata,
	    CompNode::ExecutionResult (*comp) (CompNode *),
	    void (*init) (CompNode *),
	    SerialMachine * machine,
	    bool isstatic,		// Is this node "static"?
	    bool writesshared);	// Does this node write to a shared variable?

  ~CompNode (void) {
    if (_Shared) {
      delete _LockSet;
    }
  }

  inline void * operator new (size_t sz, SerialMachine * sm);
  void * operator new (size_t sz) { return (void *) ::new char[sz]; }

  void operator delete (void *) {}

  // Tell the machine to schedule this node.
  void	Enqueue (void);

  // Block the node from execution or queueing.
  void	Block (void);

  // Unblock the node (reverse of above).
  void	Unblock (void);

  // Is this node a Static node? (does it have state?)
  bool  GetStatic (void) const	{ return _Static; }

  // Is this node a Shared node? (does it write a shared variable?)
  bool  GetShared (void) const	{ return _Shared; }



  // Run the computation procedure.
  // We pass in a messenger so it can send
  // messages to remote nodes.
  ExecutionResult Execute (Messenger*);

  // Acquire a lock on the node.
  void	Lock (void);

  // Release the lock on the node.
  void	Unlock (void);


  // ACCESSORS AND MUTATORS.


  ExecutionBehavior GetPhase (void);

  void	SetPhase (ExecutionBehavior phase);

  void	SetStolenFrom (int processor) {
    Guard m (_lock);
    _StolenFrom = processor;
  }

  int	GetStolenFrom (void) {
    Guard m (_lock);
    return _StolenFrom;
  }

  // Add the value to one of my input ports of a given index
  static void AddToInputPort (Port * Q,
			      const Index * I,
			      const _c2_Value& V)
    {
      Q->Add (*I, V); 
    }

  // Add the value to one of my input ports of a given index
  static void AddToInputPort (Port * Q,
			      const Index * I,
			      const _c2_Value& V,
			      SerialMachine * sm)
    {
      Q->Add (*I, V, sm);
    }

  // Return 1 iff there is something on a given input port and index.
  static int CheckInputPort (Port * Q,
			     Index * I)
    {
      return Q->Find (*I);
    }


  // Remove the value (storing into the _c2_Value parameter) from an input port and index.
  static int RemoveFromInputPort (Port *,
				  Index&,
				  _c2_Value&);


  // Acquire all locks in list from NextLock to end.  If fail to get a lock,
  // leave NextLock pointing at it to try again next time. Return 1 if reach
  // end of list (all locks acquired), return 0 if fail to get one-- hence more
  // locks needed.
  int GetAllLocks (void);

  // Release all locks.
  void RelAllLocks (void);

  // Report all statistics for this node.
  void ReportStats (ostream& out);

  // Send all statistics for this node to a processor.
  void TransmitStats (int processor, Messenger& messenger);

  // Return the processor for this node.
  int Processor (void) { return _Processor; }

  int NSInitialized (void);

  void NSInitialize (void);

  int NSBusy (void);

  NSLockSet& GetLockSet (void) { return *_LockSet; }

  //// The following routine is provided
  //// by the CODE backend.

  void Copy (CompNode *, int, Messenger&);

  // MEMBERS.

  Messenger * GetMessenger (void) { return _Messenger; }


  void Push (NodeDeque& ndq);

  void PushFront (deq<CompNode *>& dq);
  void PushBack (deq<CompNode *>& dq);

  void PopFront (deq<CompNode *>& dq);
  void PopBack (deq<CompNode *>& dq);

  // Add the given time to the node's total elapsed runtime.
  void AddRuntime (double time) {
    Guard m (_lock);
    _Timer.set ((double) _Timer + time);
  }

  double GetRuntime (void) {
    Guard m (_lock);
    return (double) _Timer;
  }

  void SetRuntime (double time) {
    Guard m (_lock);
    _Timer.set (time);
  }

  void StartTimer (void);

  void StopTimer (void);


  // This node sent a message to the CompNode target;
  // add it to our set of targets (used in constructing
  // the critical path).
#if USE_TARGETS
  void AddTarget (CompNode * target) {
    Guard m (_lock);
    _Targets.insert (target);
  }
#else
  void AddTarget (CompNode *) {}
#endif


  // Return the critical path (in seconds of computation time)
  // starting from this node.
  double Tinfinity (void);

  // Execute the InitProc and do some housekeeping.
  void Initialize (void);

private:

  // These are all of the possible queue states of a node.
  typedef enum
    { OFFQUEUE,	// Not on any queue.
      STOLEN,	// Not on any queue (stolen).
      ONQUEUE	// On the ready queue.
    } QueueStatus;


  void	MakeOffqueue (void);
  void	MakeOnqueue (void);

  int	GetProcessor (void);

  // Visit the subgraph implied by the dag of computations and
  // return the critical path sum of execution time.
  static double Visit (CompNode *, set <CompNode *, less<CompNode *> >&);

  bool		_initialized;		// Equals 1 iff the InitProc has run.

  ExecutionResult (*_CompProc) (CompNode *);	// Every computation procedure. 
  void (*_InitProc) (CompNode *);	// Initial computation procedure. 

  Timer		_Timer;			// Keeps track of computation time.

  MutexRec	_phaselock;		// This lock protects the member below.
  ExecutionBehavior _Phase;     	// See above (ExecutionBehavior).

  int		_StolenFrom;		// The processor this node was last stolen from (if any).

  const bool	_Static;		// A flag - if true, the node requires persistent state. 
  const bool	_Shared;		// Equals 1 iff this node accesses any shared variables.

  int		_Processor;		// The processor this node is mapped to (see GetProcessor()).

#ifdef _C2_DBGR_
  _c2_DbgUc 	_dbg;
#endif

  NameSharingRelationState	_NSState;
  NSLockSet *			_LockSet;

  Messenger *	_Messenger;


  QueueStatus	_QueueStatus;		// See above (definition of Status).
  int		_NQueued;		// How many instances of this node are virtually on the queue.


  double _pad[PAD_DBL]; // padding to avoid false sharing.

#if USE_TARGETS
  set<CompNode *, less<CompNode *> > _Targets;		// The set of all nodes that this node has sent data to.
#endif

  MutexRec	_queuelock;		// This lock protects the queue-related members above.

};


inline void * CompNode::operator new (size_t sz, SerialMachine * sm)
{
  assert (sm);
  return sm->alloc (sz);
}


#endif /* _NODE_H_ */
