/*
  nodedeque.H
  ------------------------------------------------------------------------
  A deque of computation nodes, with some extra functionality.
  ------------------------------------------------------------------------
  @(#) $Id: nodedeque.H,v 1.18 1998/03/18 04:54:18 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#ifndef _NODEDEQUE_H_
#define _NODEDEQUE_H_

#define _PTHREADS

#include <iostream.h>

#include "atomic.H"
#include "header.H"
#include "unique.H"
#include "deq.h"


class SerialMachine; // forward declaration
class CompNode;  // forward declaration


class NodeDeque : public Unique {

  friend ostream& operator<<(ostream&, NodeDeque&);

public:

  NodeDeque (SerialMachine * sermach)
    : _deque (sermach),
      _Stop (FALSE)
    {}

  ~NodeDeque () {}

  // Pop everything from the deque.
  void Clear (void) {
    Protect<Mutex> m (_lock);
    while (!_deque.empty()) {
      _deque.pop_front();
    }
  }


  // Disable the queue.
  void Stop (void) {
    Protect<Mutex> m (_lock);
    _Stop = TRUE;
    // Wake up all waiters.
    _cond.broadcast ();
  }

 
  // Push a node on the front of the deque.
  void PushFront (CompNode * n);

  // Push a node on the back of the deque.
  void PushBack (CompNode * n);

  // Pop a node from the front of the deque.
  // Returns NULL if the deque is empty.
  CompNode * PopFront (void);

  // Pop a node from the front of the deque.
  // Returns NULL if the deque is empty.
  CompNode * PopBack (void);

  // Return the front element of the deque.
  CompNode * Front (void) const {
    Protect<Mutex> m (_lock);
    return _deque.front ();
  }

  // Return the back element of the deque.
  CompNode * Back (void) const {
    Protect<Mutex> m (_lock);
    return _deque.back ();
  }


  // Is the deque empty?
  bool Empty (void) const {
    Protect<Mutex> m (_lock);
    return (_deque.empty());
  }


  // Wait for something to be added to the queue.
  void Wait (void) {
    Protect<Mutex> m (_lock);
    if (!_Stop) {
      if (_deque.empty()) {
	_lock.wait (_cond);
      }
    }
  }


  int Size (void) const {
    Protect<Mutex> m (_lock);
    return (_deque.size());
  }

private:

  Mutex		_lock;		// The lock for this data structure.

  double _pad[PAD_DBL]; // padding to avoid false sharing.

  deq<CompNode *>	_deque;	// The deque itself.
  Condition	_cond;		// Is the queue empty?
  bool		_Stop;		// If true, disable the deque.
};


#endif // _NODEDEQUE_H_
