/*
  nslocks.C
  ------------------------------------------------------------------------
  Lock management for name-sharing relations.
  ------------------------------------------------------------------------
  @(#) $Id: nslocks.C,v 1.11 1998/03/17 00:57:22 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#include "sermach.H"
#include "messenger.H"
#include "node.H"
#include "nslocks.H"

#define _PTHREADS
#include <deque.h> // STL


int operator< (const NSLock& lhs, const NSLock& rhs)
{
  return (lhs.GetUID() < rhs.GetUID());
}


int operator==(const NSLock& lhs, const NSLock& rhs)
{
  return (lhs.GetUID() == rhs.GetUID());
}


// Copy the shared value into the local address.
int LocalNSLock::acquire (CompNode * node)
{
  Guard m1 (_lock);
  Guard m2 (_SharedVarInfo.GetLock());

  if (!_SharedVarInfo.acquire (node, _ReqType)) {
    return 0;
  }

  assert (_SharedAddr != NULL);

  switch (_TypeTag) {
  case _c2_IsAnInt:
    *((int *) (_LocalAddr)) = *((int *) (_SharedAddr));
    break;
  case _c2_IsADouble:
    *((double *) (_LocalAddr)) = *((double *) (_SharedAddr));
    break;
  case _c2_IsAChar:
    *((char *) (_LocalAddr)) = *((char *) (_SharedAddr));
    break;
  case _c2_IsAnArrayPtr:
  case _c2_IsAStructPtr:
    *_LocalAddr = *((void **) _SharedAddr);
    break;
  }

  assert (_LocalAddr != NULL);

  return 1;
}


// Copy the local value into the shared address.
void LocalNSLock::release (void) {

  assert (_LocalAddr != NULL);
  assert (_SharedAddr != NULL);

  Guard m1 (_lock);
  Guard m2 (_SharedVarInfo.GetLock());

  switch (_TypeTag) {
  case _c2_IsAnInt:
    *((int *) (_SharedAddr)) = *((int *) (_LocalAddr));
    break;
  case _c2_IsADouble:
    *((double *) (_SharedAddr)) = *((double *) (_LocalAddr));
    break;
  case _c2_IsAChar:
    *((char *) (_SharedAddr)) = *((char *) (_LocalAddr));
    break;
  case _c2_IsAnArrayPtr:
  case _c2_IsAStructPtr:
    *((void **) _SharedAddr) = *_LocalAddr;
    break;
  }

  _SharedVarInfo.release (_ReqType);
}


int RemoteNSLock::acquire (CompNode * node) {
  Guard m (_lock);

  if (_Acquired) {
    // cout << "Tried to re-acquire an acquired lock." << endl;
    return 1;
  }

  // Send a message to the remote NSRel's processor
  // to attempt to obtain a lock.

  int processor = ((Graph *) (Pathname (_NSRelGraphPath)).decode(&node->GetMachine()))->Processor();

  Guard m2 ((node->GetMessenger())->objectlock());

  *(node->GetMessenger()) << _NSRelGraphPath
			  << _NSRelUID
			  << _NSRelIndex
			  << node->GetPathName().str()
			  << _ReqType
			  << GetUID()
			  << node->Processor();
  node->GetMessenger()->send (processor, CODE_Machine::REMOTE_LOCK_REQUEST);

  return 0;
}


void RemoteNSLock::release (void) {

  Guard m (_lock);

  assert (_Acquired);

  // Send a message to the remote NSRel's processor
  // to release this lock.

  Graph * gr = (Graph *) (Pathname (_NSRelGraphPath)).decode(NULL); // FIX ME!!!!
  Messenger * messenger = gr->GetMachine().GetMessenger ();

  int processor = gr->Processor();
  
  Guard m2 (messenger->objectlock());
  *messenger << _NSRelGraphPath
		<< _NSRelUID
		<< _NSRelIndex
		<< _ReqType
		<< GetUID();
  if (_ReqType == _c2_WRITER) {
    // I had write access to this variable.
    // Assume that I modified it -- send the whole thing back.
    NameSharingRelation::Copy (_NSRelUID, GetUID(), _LocalAddr, TRANSMIT, *messenger);
  }
  messenger->send (processor, CODE_Machine::REMOTE_LOCK_RELEASE);

  _Acquired = 0;
}


int NSLockSet::acquire (CompNode * node) {
  Guard m (_lock);
  if (_AwaitingLocks.empty ()) {
    return 1;
  }
  // Attempt to acquire the locks I'm waiting for.
  set<NSLock *, less<NSLock *> >::iterator i;
  for (i = _AwaitingLocks.begin();
       i != _AwaitingLocks.end();
       ++i)
    {
      if ((*i)->acquire (node)) {
	// Move the lock from Awaiting to Acquired.
	_AcquiredLocks.insert (*i);
	_AwaitingLocks.erase (i);
      } else {
	// I couldn't acquire one.
	// cout << "NSLockSet could not acquire a lock." << endl;
	return 0;
      }
    }
  assert (_AwaitingLocks.empty ());
  return 1;
}


void NSLockSet::release (void) {
  Guard m (_lock);
  set<NSLock *, less<NSLock *> >::iterator i;

  assert (_AwaitingLocks.empty ());

  for (i = _AcquiredLocks.begin();
       i != _AcquiredLocks.end();
       ++i)
    {
      (*i)->release ();
      _AwaitingLocks.insert (*i);
    }
  _AcquiredLocks.erase (_AcquiredLocks.begin(), _AcquiredLocks.end());
}
