/*
  nslocks.H
  ------------------------------------------------------------------------
  Lock management for name-sharing relations.
  ------------------------------------------------------------------------
  @(#) $Id: nslocks.H,v 1.15 1998/03/17 00:57:22 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#ifndef _NSLOCKS_H_
#define _NSLOCKS_H_

#include "atomic.H"
#include "header.H"
#include "index.H"
#include "nsrel.H"
#include "sharedvariable.H"
#include "unique.H"

#define _PTHREADS

#include <deque.h> // STL
#include <algo.h> // STL
#include <function.h> // STL
#include <set.h> // STL


class CompNode; // forward declaration
class NameSharingRelation; // forward declaration

// A lock on a name-sharing relation variable.

class NSLock : public Unique {

  // Since we can only hold one lock at a time
  // on any shared variable, we can distinguish
  // locks uniquely by the UID of their shared variable.
  // The operators below operate on those UIDs.

  friend int operator< (const NSLock& lhs, const NSLock& rhs);
  friend int operator==(const NSLock& lhs, const NSLock& rhs);

public:

  NSLock (_c2_ValueType TypeTag,	// I want access to this type of variable,
	  _c2_RequestType ReqType,	// and this kind of access.
	  void ** LocalAddr,		// Here's the address of the pointer to the local variable.
	  SharedVariableInfo& SVInfo)	// And here's the shared variable info struct.
    : _lock ("NSLock::lock"),
      _TypeTag (TypeTag),
      _ReqType (ReqType),
      _LocalAddr (LocalAddr),
      _SharedVarInfo (SVInfo)
  {}

  virtual ~NSLock (void) {}

  // Acquire this lock (for access by a particular node).
  virtual int acquire (CompNode *) = 0;

  // Release the lock.
  virtual void release (void) = 0;

  // Is the shared variable remote (i.e., not on this processor)?
  virtual int isRemote (void) = 0;

  int GetUID (void) const { return _SharedVarInfo.GetUID(); }

  _c2_ValueType GetTypeTag (void)  { Guard m (_lock); return _TypeTag; }
  void SetTypeTag (_c2_ValueType vt) 	{ Guard m (_lock); _TypeTag = vt; }

  void ** LocalAddr (void) { return _LocalAddr; }

protected:

  MutexRec	_lock;		// A lock for this object.

  SharedVariableInfo&	_SharedVarInfo;

  _c2_ValueType		_TypeTag;	// The kind of value this lock refers to.
  _c2_RequestType	_ReqType;	// The access I'm seeking to the variable (read-only or read-write).
  void **		_LocalAddr;	// The address the shared variable will be stored in when access is granted.
};



// The name-sharing relation is local (it resides on this processor).
class LocalNSLock : public NSLock {
public:

  LocalNSLock (NameSharingRelation * NSRelAddr,
	       void * SharedAddr,
       	       _c2_RequestType ReqType,
	       void ** LocalAddr,
	       int NSRelUID,
	       _c2_ValueType TypeTag,
	       SharedVariableInfo& SharedVarInfo)
    : NSLock (TypeTag, ReqType, LocalAddr, SharedVarInfo),
      _NSRelAddr (NSRelAddr),
      _SharedAddr (SharedAddr),
      _NSRelUID (NSRelUID)
  {}

  ~LocalNSLock (void) {}

  int isRemote (void) { return 0; }
  int acquire (CompNode *);
  void release (void);

  // Local NSLock specific methods.

  void *& SharedAddr (void) { return _SharedAddr; }

private:

  NameSharingRelation * const	_NSRelAddr;
  int			_NSRelUID;
  void *		_SharedAddr;
};


// The name-sharing relation is remote (it resides on another processor).
class RemoteNSLock : public NSLock {
public:

  RemoteNSLock (const int NSRelUID,		// The UID of the name-sharing relation. 
		const char *NSRelGraphPath,	// The pathname of the n.s. relation's graph. 
		const Index NSRelIndex,	// The index of the n.s. relation. 
		_c2_RequestType ReqType,
		void ** LocalAddr,
		_c2_ValueType TypeTag,
		SharedVariableInfo& SharedVarInfo)
    : NSLock (TypeTag, ReqType, LocalAddr, SharedVarInfo),
      _NSRelUID (NSRelUID),
      _NSRelGraphPath (new char[strlen(NSRelGraphPath) + 1]),
      _NSRelIndex (NSRelIndex),
      _Acquired (0)
  {
    strncpy (_NSRelGraphPath, NSRelGraphPath, Pathname::PATH_ID_LENGTH);
  }

  ~RemoteNSLock (void) {
    delete _NSRelGraphPath;
  }

  int acquire (CompNode * node);
  void release (void);
  int isRemote (void) { return 1; }

  // RemoteNSLock specific methods.

  void SetAcquired (void) { Guard m (_lock); _Acquired = 1; }

private:

  int		_NSRelUID;		// The UID of the name-sharing relation. 
  char *	_NSRelGraphPath;	// The pathname of the n.s. relation's graph. 
  Index		_NSRelIndex;		// The index of the n.s. relation. 
  
  int		_Acquired;		// Have I acquired this lock?
};



// A function object used for finding a matching entry.
class UIDEquals {
public:

  UIDEquals (int uid)
    : _UID (uid)
  {}

  int operator() (const NSLock * nslock) const
  {
    return (nslock->GetUID() == _UID);
  }

private:
  int	_UID;
};



class NSLockSet {
public:

  NSLockSet (void) {}
  ~NSLockSet (void) {}

  // Return 1 if I've acquired all of the locks I was waiting for.
  int acquire (CompNode *);

  // Release all of my locks.
  void release (void);

  // Insert a lock (in UID order).
  void insert (NSLock * newlock) {
    Guard m (_lock);
    assert (_AcquiredLocks.empty());
    _AwaitingLocks.insert (newlock);
  }

  NSLock * operator[](int UID) {
    Guard m (_lock);
    set<NSLock *, less<NSLock *> >::iterator i;

    i = find_if (_AwaitingLocks.begin (),
		 _AwaitingLocks.end (),
		 UIDEquals (UID));
    
    if (i != _AwaitingLocks.end()) {
      return (*i);
    } else {
      cerr << "***Tried to access an unknown UID (" << UID << ")!" << endl;
      return NULL;
    }
  }

private:

  MutexRec				_lock;
  set<NSLock *, less<NSLock *> >	_AcquiredLocks;
  set<NSLock *, less<NSLock *> >	_AwaitingLocks;
};


#endif // _NSLOCKS_H_ 
