/*
  path.C
  ------------------------------------------------------------------------
  CODE object pathname class.
  A pathname has an integer value (its "sum") and a string value.
  The string structure is given by the following regular expression:
    " UID (/ UID ( [ INDEX ] )* )* ".
  The sum equals the sum of all UID's and INDEX values.
  ------------------------------------------------------------------------
  @(#) $Id: path.C,v 1.21 1998/11/06 06:36:20 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#include "path.H"

#include <assert.h>
#include <ctype.h>
#include <iostream.h>
#include <string.h>

#include "c2_addrmap.h"
#include "codeobject.H"
#include "index.H"
#include "graph.H"
#include "node.H"


// Initialize the path from a string.
Pathname::Pathname (const char * text)
{

  // Assume that the path is a legal one.
  strncpy (_buf, text, Pathname::PATH_ID_LENGTH);

  // Calculate the sum by adding in every
  // number found.
  _sum = 0;
  int len = strlen (_buf);
  int temp = 0;
  for (int i = 0; i < len; ++i) {
    if (isdigit (_buf[i])) {
      temp = (temp * 10) + (_buf[i] - '0');
    } else {
      _sum += temp;
      temp = 0;
    }
  }
  _sum += temp;
}

 
// Initialize the path from an object.
Pathname::Pathname (const CodeObject& obj)
{
  if (obj.MyGraph() == NULL) {
    // This object is the main graph.
    sprintf (_buf, "%d", obj.ID());
    _sum = obj.ID();
  } else {
    // Append /ID and indices onto mygraph's path.

    const Pathname& mygraph_path = obj.MyGraph()->GetPathName();

    // Build up the index string first.

    char indexstr[PATH_ID_LENGTH];
    char * iptr = &indexstr[0];

    for (int i = 0; i < obj.MyIndex().length(); i++) {
      sprintf (iptr, "[%d]", obj.MyIndex()[i]);
      iptr += strlen (iptr);
    }
    *iptr = '\0';

    sprintf(_buf, "%s/%d%s", mygraph_path.str(), obj.ID(), indexstr);

    _sum = mygraph_path.sum() + obj.ID();

    int length = obj.MyIndex().length();
    for (int j = 0; j < length; ++j) {
	_sum += obj.MyIndex()[j];
    }
  }

  // cout << "path::Encoded " << str() << endl;
}


Pathname::~Pathname (void) {
}


// Return a string representation of the path.
const char * Pathname::str (void) const {
  return _buf;
}


// Return the sum of the path.
int Pathname::sum (void) const {
  return _sum;
}


// Turn a path into a CodeObject.
CodeObject * Pathname::decode (SerialMachine * SerMach) const
{
  extern Graph * MainGraph;
  Pathname * self = const_cast(Pathname *, this); // cast away const-ness.

  char * pathstr = const_cast(char *, self->str());

  Graph *_c2_gr;
  CodeObject * cobj;
  int NumInd = 0;
  int index_list[Index::MAXINDICES];
  int id = 0;
  int len = 0;
  int i = 0;
  char *Pos;

  char *s;
  if ((s = strchr (pathstr, '/')) == NULL) {
    return (CodeObject *) MainGraph;
  }

  len = strlen(pathstr);
  Pos = strchr(pathstr, '/');
  NumInd = 0;
  _c2_gr = MainGraph;
  while (i < len) {
    id = atoi(pathstr+i);
    while ((i < len) && isdigit(pathstr[i])) i++;
    
    while (pathstr[i] == '[') {
      i++;
      index_list[NumInd] = atoi(pathstr+i);
      NumInd++;
      while(pathstr[i] != ']') i++;
      i++;
    }
    
    if (pathstr[i] == '/') {
      if ((pathstr+i) != Pos) {
	Index tmpIndex (index_list, NumInd);
	_c2_gr = (Graph *) _c2_GetAddr( _c2_gr, id, &tmpIndex, SerMach);
      }
      NumInd = 0;
      i++;
    } else {
      Index tmpIndex (index_list, NumInd);
      cobj = (CodeObject *) _c2_GetAddr(_c2_gr, id, &tmpIndex, SerMach);
    }
  }
  
  assert (cobj->GetPathName() == *this);
  return cobj;
}

ostream& operator<<(ostream& os,
		    const Pathname& pathname)
{
  os << pathname._buf;
  return os;
}


int Pathname::map (int nprocs) const
{
  return sum() % nprocs;
}


int operator==(const Pathname& lhs, const Pathname& rhs)
{
  if (strcmp (lhs.str(), rhs.str()) == 0) {
    return 1;
  } else {
    return 0;
  }
}
