/*
  readyq.C
  ------------------------------------------------------------------------
  A queue (ok, a deque) of computation nodes ready for execution.
  ------------------------------------------------------------------------
  @(#) $Id: readyq.C,v 1.32 1998/03/17 00:57:28 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#include "readyq.H"
#include "node.H"
#include "machine.H"

#define DEBUG_PRINT 0

// IMPLEMENTATION OF ReadyDeque

#undef ALLOW_THEFT_OFF_HEAVY_QUEUE

// Pop a node off of the queue for a local request.
bool ReadyDeque::TryPopNode (CompNode *& node)
{
  return RemoveNode (node, LOCAL);
}


// Pop a node off of the queue for a thief.
bool ReadyDeque::TryLocalStealNode (CompNode *& node)
{
  return RemoveNode (node, LOCAL_THIEF);
}


// Pop a node off of the queue for a thief.
bool ReadyDeque::TryStealNode (CompNode *& node)
{
  return RemoveNode (node, REMOTE_THIEF);
}


// Erase the entire ready queue.
void ReadyDeque::Clear (void)
{
  Protect<Mutex> m (_lock);
  for (int i = 0; i < 2; ++i) {
    rq[i]->Clear();
  }
}


void ReadyDeque::Enqueue (CompNode *Node) 
{
  Protect<Mutex> m (_lock);

  int queue;
  if (Node->GetStatic() || Node->GetShared()) {
    queue = HEAVY_QUEUE;
  } else {
    queue = LIGHT_QUEUE;
  }

  rq[queue]->PushFront (Node);

#if DEBUG_PRINT
  cout << "(" << Node->GetMachine().WhichProcessor() << ") " << rq[0] << endl << rq[1] << endl;
#endif

}


void ReadyDeque::EnqueueStolen (CompNode *Node) 
{
  Protect<Mutex> m (_lock);

  int queue = HEAVY_QUEUE;
  rq[queue]->PushFront (Node);

#if DEBUG_PRINT
  cout << "(" << Node->GetMachine().WhichProcessor() << ") " << rq[0] << endl << rq[1] << endl;
#endif

}


// Return a node from the ready queue.
// Returns NULL if empty.
CompNode * ReadyDeque::Pop (int Thief)
{
  Protect<Mutex> m (_lock);
  CompNode * node;

  switch (Thief) {
  case LOCAL:
    if (rq[LIGHT_QUEUE]->Empty() && rq[HEAVY_QUEUE]->Empty()) {
      return NULL;
    } else {
      // Try to use the heavy queue if there is anything on it.
      int queue = HEAVY_QUEUE;

      if (rq[HEAVY_QUEUE]->Empty ()) {
	queue = LIGHT_QUEUE;
      }

      ASSERT (!(rq[queue]->Empty()));

      // Implement the normal FIFO policy of queue removal.
      node = rq[queue]->PopFront();

      ASSERT (node != NULL);
      return node;
    }
    break;
  case LOCAL_THIEF:
    if (rq[LIGHT_QUEUE]->Empty() && rq[HEAVY_QUEUE]->Empty()) {
      return NULL;
    } else {
      // Try to use the heavy queue if there is anything on it.
      int queue = HEAVY_QUEUE;

      if (rq[HEAVY_QUEUE]->Empty ()) {
	queue = LIGHT_QUEUE;
      }

      ASSERT (!(rq[queue]->Empty()));

      // Steal off of the top.
      node = rq[queue]->PopBack();

      ASSERT (node != NULL);
      return node;
    }
    break;
  case REMOTE_THIEF:
    if (rq[LIGHT_QUEUE]->Empty()) {
      return NULL;
    } else {
      // Try to use the light queue if there is anything on it.
      int queue = LIGHT_QUEUE;

      ASSERT (!(rq[queue]->Empty()));

      // Pop off the top.
      node = rq[queue]->PopBack ();
      
      ASSERT (node != NULL);
      return node;
    }
    break;
  }
}


void ReadyDeque::Stop (void)
{
  Protect<Mutex> m (_lock);
  _Stop = TRUE;
}


// FALSE if the removal was unsuccessful.
bool ReadyDeque::RemoveNode (CompNode *& node,
			    int Thief)
{
  node = Pop (Thief);
  return (node != NULL);
}
