/*
  sermach.C
  ------------------------------------------------------------------------
  The execution engine for a CODE program.
  ------------------------------------------------------------------------
  @(#) $Id: sermach.C,v 1.5 1998/11/06 05:06:21 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#include <fstream.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "c2_addrmap.h"
#include "heap.H"
#include "machine.H"
#include "sermach.H"
#include "messenger.H"
#include "node.H"
#include "nsrel.H"
#include "parms.H"
#include "timer.H"
#include "readyq.H"


static char const rcsid[] = "$Id: sermach.C,v 1.5 1998/11/06 05:06:21 emery Exp $";


SerialMachine::SerialMachine (CODE_Machine * parmach,
			      int numworkers,
			      int numprocessors,
			      int whichprocessor,
			      int whichworker,
			      Messenger * messenger,
			      bool timenodes)
  : _lock ("SerialMachine::Lock"),
    _NumWorkers (numworkers),
    _WhichWorker (whichworker),
    _NumProcessors (numprocessors),
    _WhichProcessor (whichprocessor),
    _Heap (new ParHeap),
    _ReadyQ (new ReadyDeque (this)),
    _ParallelMachine (parmach),
    _Messenger (messenger),
    _TimeNodes (timenodes),
    _RandVal (162347 + 2 * whichworker),
    _Stopped (FALSE)
{}


SerialMachine::~SerialMachine (void)
{
  delete _Heap;
  delete _ReadyQ;
}


void * SerialMachine::alloc (size_t sz)
{
  return _Heap->alloc (sz);
}


void SerialMachine::free (void * p)
{
  _Heap->free (p);
}


u_long SerialMachine::lrand16 (void)
{
  // (from Blumofe's 'Hood' code)

  // Update random value.
  u_long m = _RandVal;
  m = m * 1103515245 + 12345;
  _RandVal = m;

  // Return 16 bits.
  m = m >> 16;
  return m;
}


// We need non-object functions as wrappers
// for the threads package.

void SerialMachine_Worker (void * m) {
  CODE_Machine::WorkerInfo * worker = dynamic_cast(CODE_Machine::WorkerInfo *, m);
  worker->machine->Worker (worker->initnode);
  delete worker;
}


void SerialMachine::Execute (CompNode * node)
{
  Timer t;

  // Assertion:
  // We only execute nodes that live on this processor,
  // unless they are stolen nodes or not static.

#if DENY_EXECUTION_OF_NONLOCAL_NODES
  assert ((node->Processor () == WhichProcessor())
	  || (node->GetPhase () == CompNode::STOLEN_EXECUTION)
	  || !(node->GetStatic ()));
#endif
  
  int repeat = 1;
  CompNode::ExecutionResult result;
  node->SetMachine (*this);

  while (Running() && repeat) {
#if DEBUG_PRINT
    cout << "Executing " << *node << endl;
#endif

    if (TimeNodes()) {

      t.reset ();
      t.start ();
      result = node->Execute (_Messenger);
      t.stop ();
      node->AddRuntime ((double) t);
      _Runtime += t;

#if DEBUG_PRINT
      cout << "Executed " << *node << ": total runtime = " << node->GetRuntime() << endl;
#endif

    } else {
      result = node->Execute (_Messenger);
    }

    switch (result) {
      
    case CompNode::SUCCESSFUL:
#if DEBUG_PRINT
      cout << "Successfully executed " << *node << endl;
#endif
      if (node->GetPhase () == CompNode::NORMAL) {
	repeat = 1;
      }
      break;

    case CompNode::NO_FIRING_RULES_TRUE:
#if DEBUG_PRINT
      cout << "No firing rules true for " << *node << endl;
#endif
      repeat = 0;
      break;

    case CompNode::NSREL_BUSY:
    case CompNode::NOT_CREPPED:
      // Don't do anything.
      // These will be queued again when the nsrel
      // is unlocked or when the graph is crepped.
      repeat = 0;
      break;
    case CompNode::SUCCESSFUL_STOLEN:
      // Don't repeat if it was a stolen node.
      repeat = 0;
      break;
    case CompNode::SUCCESSFUL_PREFIRING:
      // This would only happen if we were
      // pre-firing a node to be stolen.
      repeat = 0;
      break;
    }
  }
}


void SerialMachine::LocalStop (void)
{
#if DEBUG_PRINT
  cout << "LocalStop: processor = " << WhichProcessor() << endl << flush;
#endif

  // Tell everyone else to stop.
  if (GetNumProcessors() > 1) {
    *GetMessenger() << 0;
    GetMessenger()->broadcast (0, GetNumProcessors() - 1, CODE_Machine::TERMINATE_ALL_PROCESSES);
  }
  GetParallelMachine().Stop (*GetMessenger());
}


void SerialMachine::Schedule (CompNode * node, int stolen)
{
  if (stolen) {
    // Give it to one of the workers.
    _ReadyQ->EnqueueStolen (node);
  } else {
 
#if DENY_EXECUTION_OF_NONLOCAL_NODES
    if (!((node->Processor() == WhichProcessor())
	  || (node->GetPhase () == CompNode::STOLEN_EXECUTION)
	  || !(node->GetStatic ()))) {
      cerr << "node = " << *node << endl;
      cerr << "node->Processor() = " << node->Processor() << endl;
      cerr << "WhichProcessor() = " << WhichProcessor() << endl;
      cerr << "node->GetPhase() = " << node->GetPhase() << endl;
      cerr << "node->GetStatic() = " << node->GetStatic() << endl;
    }
    assert ((node->Processor() == WhichProcessor())
	    || (node->GetPhase () == CompNode::STOLEN_EXECUTION)
	    || !(node->GetStatic ()));
#endif
    
    _ReadyQ->Enqueue (node);
  }
}


void SerialMachine::Worker (CompNode * initnode)
{
  CompNode * node;

  //cout << "Worker " << WhichWorker() << " starting." << endl;

  // Run the start node (if it is on our processor
  // and we are the first thread).
  if ((initnode->Processor() == WhichProcessor ())
      && (WhichWorker() == 0)) {
    Execute (initnode);
  }

  int waiting = 1;
  int from;

  // Loop until someone calls Stop().
  while (Running()) {

    // Handle any incoming messages.
    while (CODE_Machine::messageHandler (*GetMessenger(), *this) >= 0)
      {
	waiting = 1;
      }

    // Send all pending messages.
    while (GetMessenger()->flush ()) {
      waiting = 1;
    }

#if 0
    if (waiting % 100000 == 0) {
      printf ("[%d waiting]", _WhichProcessor);
      fflush (stdout);
    }
#endif

    // Check the ready queue.
    // Atomically pop and execute the node, if one is there.
    if (_ReadyQ->TryPopNode (node)) {

      // Got some work.

      waiting = 1;

#if DEBUG_PRINT
      cout << "running a node (" << node->GetPathName().str() << ") " << *node << " on processor " << WhichProcessor() << " (thread " << WhichWorker() << ")" << endl;
#endif
      
      Execute (node);
      
    } else {
      
      // There was nothing on the ready queue.

      waiting++;
      AttemptSteal ();
    }

    // Send all pending messages.
    while (GetMessenger()->flush ()) {
      waiting = 1;
    }
  }

  // If we are the root node, save a profile file.
  if (WhichWorker() == 0 && GetNumProcessors() == 1) {
    GetParallelMachine().WriteProfile ();
  }
  
}


// Try to steal some work.

void SerialMachine::AttemptSteal (void)
{
  // Try a local steal.

  if (GetNumWorkers() > 1) {

    // Pick a random local victim (another worker).
    // If it has work, execute it.

    CompNode * node;

    // Pick another worker (not me).
    int localvictim = lrand16() % (GetNumWorkers() - 1);
    localvictim += (localvictim >= WhichWorker());

    SerialMachine& victim_machine = GetParallelMachine().GetSerialMachine(localvictim);
    if (victim_machine.GetReadyQ().TryStealNode (node)) {
      //cout << "I (" << WhichWorker() << " stole work from " << localvictim << endl;
      Execute (node);
      return;
    } else {
      //cout << "I (" << WhichWorker() << " couldn't steal work from " << localvictim << endl;
    }
  }

  // Try a remote steal.

  if (GetNumProcessors() > 1) {
    // Poll for any incoming messages.
    // (possibly work for me!)
    // CODE_Machine::messageHandler (*GetMessenger(), *this);
    GetParallelMachine().AttemptSteal (*GetMessenger());
  }
}
