/*
  timer.H
  ------------------------------------------------------------------------
  A universal timer class.
  ------------------------------------------------------------------------
  @(#) $Id: timer.H,v 1.9 1998/09/15 21:42:30 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/


#ifndef _TIMER_H_
#define _TIMER_H_

#ifdef __SVR4 // Solaris
#include <sys/time.h>
#endif
#define times times_
#include <time.h>
#undef times


class Timer {

public:

  Timer (void)
    : _starttime (0),
      _elapsedtime (0)
    {}

  // Start the timer.
  void start (void) { _starttime = _time(); }

  // Stop the timer.
  void stop (void) { _elapsedtime += _time() - _starttime; }

  // Reset the timer.
  void reset (void) { _starttime = _elapsedtime = 0; }

  // Set the timer.
  void set (double secs) { _starttime = 0; _elapsedtime = _sectotime (secs);}

  // Return the number of seconds elapsed.
  operator double (void) { return _timetosec (_elapsedtime); }

  // Prevents copying and assignment.

  Timer (const Timer& t) {
    _starttime = t._starttime;
    _elapsedtime = t._elapsedtime;
  }

  const Timer& operator= (const Timer& t) {
    _starttime = t._starttime;
    _elapsedtime = t._elapsedtime;
    return *this;
  }

  Timer& operator+= (const Timer& t) {
    _elapsedtime += t._elapsedtime;
    return *this;
  }

private:


  // The _timer variable will be different depending on the OS.
  // We try to use the best timer available.

#ifdef __SVR4 // Solaris

#define TIMER_FOUND
  hrtime_t	_starttime, _elapsedtime;

  hrtime_t _time (void) {
    return gethrtime();
  }

  hrtime_t _sectotime (double sec) { return (hrtime_t) (sec * 1.0e9); }

  double _timetosec (hrtime_t t) { return ((double) (t) / 1.0e9); }
  double _timetosec (hrtime_t t) const { return ((double) (t) / 1.0e9); }
#endif // __SVR4


#ifdef __xlC__ // IBM SP2

#include <sys/time.h>

#define TIMER_FOUND

  double _starttime, _elapsedtime;
  
  double _time (void) {
    timebasestruct_t start;
    int secs;
    int n_secs;

    read_real_time(&start, TIMEBASE_SZ);
    /*
     * Call the conversion routines unconditionally, to ensure
     * that both values are in seconds and nanoseconds regardless
     * of the hardware platform. 
     */
    time_base_to_time(&start, TIMEBASE_SZ);
    secs = start.tb_high;
    n_secs = start.tb_low;

    double t = (double) secs + ((double) n_secs / 1.0e9);
    return t;
  }

  double _sectotime (double sec) { return sec; }

  double _timetosec (double t) { return t; }
  double _timetosec (double t) const { return t; }
#else
#endif

#if defined(MAC) || defined(macintosh)
#define TIMER_FOUND
  double		_starttime, _elapsedtime;

  double _time (void) {
    return get_Mac_microseconds();
  }

  double _timetosec (hrtime_t& t) {
    return t;
  }

  double _timetosec (hrtime_t& t) const {
    return t;
  }
#endif // defined(MAC) || defined(macintosh)

#ifdef __WIN32__
#define TIMER_FOUND
  unsigned long	_starttime, _elapsedtime;
  extern unsigned long GetPerfTime(void);
  extern double GetPerfSec(double ticks);

  unsigned long _time (void) {
    return GetPerfTime();
  }

  double _timetosec (unsigned long& t) {
    return GetPerfSec(t);
  }

  double _timetosec (unsigned long& t) const {
    return GetPerfSec(t);
  }
#endif // __WIN32__


#ifndef TIMER_FOUND
#define TIMER_FOUND
  clock_t	_starttime, _elapsedtime;

#ifndef CLOCKS_PER_SEC

#ifdef __sun
#define CLOCKS_PER_SEC 1000000
  extern long clock(void);
#else // not (__sun)
#error Please define CLOCKS_PER_SEC
#endif // __sun

#endif // not (CLOCKS_PER_SEC)

  clock_t _time (void) {
    return clock();
  }

  clock_t _sectotime (double secs) {
    return ((clock_t) secs * CLOCKS_PER_SEC);
  }

  double _timetosec (clock_t t) {
    return ((double) (t) / CLOCKS_PER_SEC);
  }

  double _timetosec (clock_t t) const {
    return ((double) (t) / CLOCKS_PER_SEC);
  }

#endif // not (TIMER_FOUND)


#undef TIMER_FOUND

};


#endif
