#
# CallNode.tcl
# ------------------------------------------------------------------------
# This node holds a subgraph, and can be expanded (by a double-click) to
# reveal its sub-canvas.
# ------------------------------------------------------------------------
# @(#) $Id: CallNode.tcl,v 1.24 1998/01/07 01:01:28 emery Exp $
#
# ------------------------------------------------------------------------
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================


class Code::CallNode {

    inherit Code::Node

    constructor {args} {
	if {!$_initialized} {
	    init
	}
	eval Code::Node::constructor $args
    } {}

    destructor {}

    public method attributes {{update 1}}
    public method bitmap {}
    public method translate {ProgramAST GraphAST}

    public method save {out}
    public method fill {}
    public method expand {}
    public method collapse {}
    public method print {}
    public method thisimage {}

    protected method update_name {args}
    private method occupied {}
    private method init {}
    private method destroy_contents {}

    private common _initialized 0
    common image
    common aliasimage  ;# the image to show I am an alias (if I am).
    
    # attributes

    public variable IsAliasToGraph 0  ;# am I an alias to another CallNode?
    public variable AliasGraph     "" {update_node} ;# the name of the other node (if any).

    protected variable attributes {Name TraceCall GraphFunctionSignatures \
				       GraphFunctionDefinitions GraphDocumentation \
				   IsAliasToGraph }

    public variable TraceCall 0
    public variable GraphFunctionSignatures ""
    public variable GraphFunctionDefinitions ""
    public variable GraphDocumentation ""


    public variable subgraph ""             ;# the canvas in the expanded window

    common Aliases                          ;# an assoc. array of all aliases

    #
    # default dimensions of expanded window
    #

    common width
    common height

    protected variable windowname ""  ;# the name of the expanded window
    protected variable expanded 0     ;# == window is expanded. initially zero
    protected variable graphid        ;# unique id number for subgraph

    common aliasimagefile   ;# array for the node image files (if an alias)
    common imagefile        ;# array for the node image files
    common bitmapfile       ;# a bitmap file for the image

}



body Code::CallNode::constructor {args} {
    addport InputPort top
    addport OutputPort bottom
    set Aliases($this) ""
    set graphid [$canvas new_id]   ;# maintain an extra identifier for subgraph
    itk_initialize
}


body Code::CallNode::destructor {} {

    if {$IsAliasToGraph} {

	# Delete the alias from the list of aliases held by
	# the aliased graph.

	set index [lsearch -exact $Aliases($AliasGraph) $this]
	lreplace $Aliases($AliasGraph) $index [expr $index+1]
	return
    }

    destroy_contents
}


body Code::CallNode::destroy_contents {} {

    if { $IsAliasToGraph } {
	return
    }

    # if there is a subwindow...
    if { ($subgraph != "") && [winfo exists $windowname] } {
	# delete everything in the subgraph
	$subgraph delete_items [winfo children $subgraph]

	# delete the window
	destroy $windowname

	# If the focus was on this window,
	# move it to the parent window.
	if { ![string compare [$canvas info variable CurrentWindow -value] $subgraph] } {
	    ::focus -force [info namespace tail $canvas]
	    Code::CodeCanvas::SetCurrentWindow $canvas
	}

	set windowname ""
	set subgraph   ""

	foreach attr {GraphFunctionSignatures GraphFunctionDefinitions GraphDocumentation} {
	    set $attr ""
	}

	# Eliminate any aliases to this node.

	foreach alias $Aliases($name) {
	    delete object $alias
	}
	set Aliases($name) ""
    }
}


body Code::CallNode::attributes {{update 1}} {
    set OldIsAliasToGraph $IsAliasToGraph
    set OldAliasGraph $AliasGraph

    # Update the dialog box.
    global c2_tmp
    set c2_tmp(AliasGraph) $AliasGraph
    set c2_tmp(IsAliasToGraph) $IsAliasToGraph

    # Nasty hack here -- AliasGraph can't be in the attribute list
    # for saving, but it has to be here.

    set oldattributes $attributes
    set attributes "AliasGraph $attributes"
    Node::attributes $update
    set attributes $oldattributes

    update_node

    # We have to handle a few cases:
    #  (1) We were an alias to one graph, but now we're an alias to a different one.
    #  (2) We weren't an alias before, but we are now.
    #  (3) We were an alias before, but we aren't anymore.

    if {$IsAliasToGraph} {
	if {!$OldIsAliasToGraph} {
	    # Case (2).

	    # Destroy the contents of this graph (since it's now just an alias).
	    # (We should warn the user with a dialog here, and abort
	    #  the changes if cancelled.)  TO BE CHANGED!!! REMOVE!!!

	    # destroy_contents
	    lappend Aliases($AliasGraph) $this

	} elseif [string compare $OldAliasGraph $AliasGraph] {

	    # Case (1).
	    # Remove this object from the alias list of $OldAliasGraph
	    # and add it to $AliasGraph.

	    set index [lsearch -exact $Aliases($OldAliasGraph) $this]
	    set Aliases($OldAliasGraph) [lreplace $Aliases($OldAliasGraph) $index [expr $index+1]]
	    lappend Aliases($AliasGraph) $this

	}

    } else {
	if {$OldIsAliasToGraph} {

	    # Case (3).
	    # Remove this object from the alias list of $OldAliasGraph.

	    set index [lsearch -exact $Aliases($OldAliasGraph) $this]
	    set Aliases($OldAliasGraph) [lreplace $Aliases($OldAliasGraph) $index $index]

	}
	set AliasGraph ""
    }
}


body Code::CallNode::bitmap {} {

    # change the image to the appropriate call node picture
    # (depending on if it is open or has children)

    return "@[$canvas info variable ReleaseDirectory -value]/bitmaps/$bitmapfile($expanded,[occupied])"

}


body Code::CallNode::collapse {} {

    # Close up the subgraph window.

    if {$IsAliasToGraph} {
	$AliasGraph collapse
	return
    }
    wm withdraw $windowname
    set expanded 0
    update_node
}


# Create a subgraph for the Call node (if none exists).

body Code::CallNode::fill {} {

    # fill only if it hasn't been filled yet.
    if { ($subgraph == "") || (![winfo exists $windowname]) } {

	## create the window
	
	set windowname "._CallNode$id"
	uplevel #0 toplevel $windowname
	wm withdraw $windowname

	global ::c2_ReleaseDirectory
	global ::c2_MenuFont
	
	# wm minsize $windowname $width $height
	wm geometry $windowname ${width}x$height
	wm title $windowname "CODE - $Name"
	wm protocol $windowname WM_DELETE_WINDOW {c2_QuitProgram}

	# wm iconbitmap $windowname "@$c2_ReleaseDirectory/bitmaps/ICON.bit"

	uplevel #0 frame $windowname.parent
	uplevel #0 frame $windowname.parent.fone  ;# canvas window must not contain a number
	frame $windowname.parent.menubar -borderwidth 2 -relief raised
	frame $windowname.parent.iconbar -borderwidth 2 -relief raised

	label $windowname.parent.statusbar -text "" -height 1 -relief sunken -borderwidth 2 -font $c2_MenuFont

	build_iconbar $windowname.parent.iconbar $windowname.parent.statusbar

	uplevel #0 "scrollbar $windowname.parent.fone.scrollbar1 \
		-command \"$windowname.parent.fone.canvas xview\" \
		-orient {horizontal} \
		-relief {raised}"
	
	uplevel #0 "scrollbar $windowname.parent.fone.scrollbar2 \
		-command \"$windowname.parent.fone.canvas yview\" \
		-relief {raised}"
	
	set subgraph [ \
		uplevel #0 "CodeCanvas $windowname.parent.fone.canvas \
		-closeenough {0.2} \
		-height [expr $height/10] \
		-insertofftime {600} \
		-relief {sunken} \
		-scrollregion {0 0 1000 1000} \
		-width [expr $width/10] \
		-xscrollcommand \"$windowname.parent.fone.scrollbar1 set\" \
		-yscrollcommand \"$windowname.parent.fone.scrollbar2 set\" " \
		]


	build_menubar $windowname.parent.menubar $subgraph


	# Tell the canvas that its parent is this call node.

	$subgraph configure -parentnode $this
	
	pack append $windowname.parent.fone \
		$windowname.parent.fone.scrollbar2 {right frame center filly} \
		$windowname.parent.fone.canvas {top frame center expand fill} \
		$windowname.parent.fone.scrollbar1 {top frame center fillx} \
	        $windowname.parent.statusbar {bottom frame s fillx}
	
	pack append $windowname.parent \
		$windowname.parent.menubar {top frame nw fillx} \
                $windowname.parent.iconbar {top frame nw fillx} \
		$windowname.parent.fone {right frame ne expand fill} \
	
	pack append $windowname \
		$windowname.parent {top frame center expand fill}

        update_node
    }
}


# Zoom open the subgraph (invoked by a double-click).

body Code::CallNode::expand {} {

    #
    # Change the node to the "open" state.
    #

    set expanded 1
    update_node

    # if we're an alias, expand the actual graph and return.

    if {$IsAliasToGraph} {
	$AliasGraph expand
	return
    }

    # Make sure the node's subgraph is initialized
    # and display it.

    fill
    wm deiconify $windowname
    raise $windowname
}


body Code::CallNode::init {} {

    # Initialize all of the file arrays.

    global c2_ReleaseDirectory
    set width [Code::CodeCanvas::window_width]
    set height [Code::CodeCanvas::window_height]

    #
    # first value == (1) expanded / (0) closed
    # second      == (1) occupied / (0) empty
    # third       == (1) selected / (0) not selected
    #

    set imagefile(0,0,0) "CallNode.gif"
    set imagefile(0,1,0) "CallNode.gif" ;# "CallNodeGraph.gif"
    set imagefile(1,0,0) "CallNode.gif" ;# "CallNodeOpen.gif"
    set imagefile(1,1,0) "CallNode.gif" ;# "CallNodeOpenGraph.gif"

    set imagefile(0,0,1) "CallNode.gif" ;# "CallNode-Selected.gif"
    set imagefile(0,1,1) "CallNode.gif" ;# "CallNodeGraph-Selected.gif"
    set imagefile(1,0,1) "CallNode.gif" ;# "CallNodeOpen-Selected.gif"
    set imagefile(1,1,1) "CallNode.gif" ;# "CallNodeOpenGraph-Selected.gif"

    set aliasimagefile(0) "CallNodeAlias.gif"
    set aliasimagefile(1) "CallNodeAlias-Selected.gif"

    set bitmapfile(0,0) "CallNode.bit"
    set bitmapfile(0,1) "CallNodeGraph.bit"
    set bitmapfile(1,0) "CallNodeOpen.bit"
    set bitmapfile(1,1) "CallNodeOpenGraph.bit"

    foreach i {0 1} {
	foreach j {0 1} {
	    foreach k {0 1} {
		set image($i,$j,$k) [uplevel #0 image create photo -palette 256x256x256 -format GIF \
	    -file "$c2_ReleaseDirectory/bitmaps/$imagefile($i,$j,$k)"]
	    }
	}
	set aliasimage($i) [uplevel #0 image create photo -palette 256x256x256 -format GIF -file "$c2_ReleaseDirectory/bitmaps/$aliasimagefile($i)"]
    }

    set _initialized 1
}


body Code::CallNode::occupied {} {
    # True iff there are contents in the subgraph.

    if {$IsAliasToGraph} {
	return [$AliasGraph occupied]
    }

    if {($subgraph != "") && ([winfo children $subgraph] != "")} {
	return 1
    } else {
	return 0
    }
}


body Code::CallNode::print {} {
    if {$IsAliasToGraph} {
	$AliasGraph print
	return
    }

    if { $subgraph != "" } {
	$subgraph print
    }
}


body Code::CallNode::save {out} {
    # recursively save subgraphs (which are sub-canvases)
    Node::save $out
    
    if {!$IsAliasToGraph} {
	puts $out "\$objindex($this) configure -AliasGraph {}"
	if { ($subgraph != "") && ([winfo children $subgraph] != "") } {
	    puts $out "\$objindex($this) fill"
	    puts $out "set oldcanvas($this) \$canvas"
	    puts $out "set canvas \[\$objindex($this) info variable subgraph -value\]"
	    $subgraph save $out
	    puts $out "set canvas \$oldcanvas($this)"
	}
    } else {
	puts $out "\$objindex($this) configure -AliasGraph \$objindex($AliasGraph)"
    }
}


body Code::CallNode::thisimage {} {
    if {$IsAliasToGraph} {
	return $aliasimage($selected)
    } else {
	return $image($expanded,[occupied],$selected)
	# change the image to the appropriate call node picture
	# (depending on if it is open or has children or is selected)
    }
}


body Code::CallNode::translate {ProgramAST GraphAST} {

    if {$IsAliasToGraph} {
	set graphname "[$AliasGraph info variable Name -value]"
	# set guid $id
	set guid "[$AliasGraph info variable id -value]"
    } else {
	set graphname $Name
	set guid $id
    }

    # Add the unique identifier to the graph name so that
    # we can have multiple graphs with the same name.
    # This is an unfortunate hack.

    set graphname "${graphname}_${guid}"

    set result [c2_CreateCallNodeAST $ProgramAST $GraphAST \
	    $Name $id $graphname $TraceCall]

    if { $result != "" } {
	if { ($subgraph != "") && ([winfo children $subgraph] != "") && !$IsAliasToGraph } {
	    $subgraph translate $ProgramAST $graphname $graphid
	}
    }
    return $result
}


body Code::CallNode::update_name {args} {
    if { [winfo exists $windowname] } {
	wm title $windowname "CODE - $Name"
    }
}
