#
# CodeObj.tcl
# ------------------------------------------------------------------------
# The base class for all Code objects to appear on the CodeCanvas.
# ------------------------------------------------------------------------
# @(#) $Id: CodeObj.tcl,v 1.16 1997/08/06 17:07:35 emery Exp $
#
# ------------------------------------------------------------------------
# AUTHOR:
#
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================


class Code::CodeObj {

    inherit wigwam::itcl_frame

    constructor {w args} { 
	eval wigwam::itcl_frame::constructor $args 
    } {}
    destructor {}

    public method attributes {{update 1}}
    public method print_attributes {}
    public method canvas {}
    public method deselect {}
    public method deselect_group {}
    public method group_parent_name {}
    public method group_name {}
    public method id {}
    public method in_group {g}
    public method isa {type}
    public method move {dx dy}
    public method save {out}
    public method scale {}
    public method select {}
    public method selected {}
    public method select_group {}
    public method set_group {g}
    public method toggle {}
    public method toggle_group {}
    public method ungroup {}
    public method update_attributes {}

    public proc mangle_group {g}


    protected method setvar {vname val}


    public variable group ""
    public variable id 0       ;# a unique identifier for this object

    protected variable attributes {}
    protected variable canvas      ;# the canvas the object is in
    protected variable selected 0  ;# whether or not the object is selected

    protected variable name
}

#
#  METHODS
#
#


body Code::CodeObj::constructor {w args} {

    set name $this  ; # WORKAROUND for destructor losing $this.

    set canvas $w

    set id [$canvas new_id]
    $canvas add_object $id $this
    
    set_group ".$id"
    $canvas set_group $id "$this"

    # Make the object include all the classes in its heritage
    # in its binding tags, instead of just the one window name.

    bindtags [info namespace tail $this] \
	[eval linsert [list [bindtags [info namespace tail $this]]] 1 [$this info heritage]]

    eval itk_initialize $args
}



#
#  calls up the dialog for the class
#  to modify "attributes"
#  (dialog defined in "createDialogs.tcl")
#

body Code::CodeObj::attributes {{update 1}} {
    set windowname "._[info namespace tail [$this info class]]attributes"

    if {[winfo ismapped $windowname]} {
	raise $windowname
    } else {
	global ::c2_tmp

	if { $update } {
	    update_attributes
	}

	# Bring up the dialog box and check if the user hit "OK".
	if {[::$windowname activate]} {
	    # Accept any changes made to the c2_tmp array.
	    set attribs [$this info variable attributes -value]
	    foreach i $attribs {
		configure -$i "[set c2_tmp($i)]"
	    }
	} else {
	    # Restore the old values.
	    update_attributes
	}
    }
}


body Code::CodeObj::canvas {} { return $canvas }

#
#  deselect this object
#

body Code::CodeObj::deselect {} {
    $canvas dtag $this selected
    set selected 0
    $canvas unset_selected $this
}

#
#  deselect this object and everything grouped with it
#

body Code::CodeObj::deselect_group {} {
    set groupitems [$canvas items_in_group [group_parent_name]]
    map "apply deselect" $groupitems
    $canvas remove_bbox [group_parent_name]
}


body Code::CodeObj::id {} {
    return $id
}


#
#  returns a string in the form ".n"
#  where n is the object's most-general group
#

body Code::CodeObj::group_parent_name {} {
    return [file extension $group]
}

#
#  returns a string in the form ".n"
#  where n is the object's most-general group
#

body Code::CodeObj::group_name {} {
    return $group
}



#
#  returns a boolean value
#  == (is this object in group "g"?)
#

body Code::CodeObj::in_group {g} {
    if { ![string compare [group_parent_name] $g] } {
	return 1
    } else {
	return 0
    }
}

body Code::CodeObj::isa {type} {
    foreach c [$this info heritage] {
	if ![string compare $c $type] {
	    return 1
	}
    }
    return 0
}


#
# takes a string representing a group
# and changes all occurrences of any number n
# to [expr $id+n] -- this prevents id clashes
# if this object is pasted or inserted onto
# the canvas.
#

body Code::CodeObj::mangle_group {g} {

    # replace every instance of .n with .[expr $id+n]
    # -- we need this for object re-creation.

    regsub -all {([0-9]+)} $g {[expr $id+\1]} newg
    return $newg
}

#
#  move this object by (dx,dy) on the canvas
#

body Code::CodeObj::move {dx dy} {
    $canvas move $this $dx $dy
}


#
#  prints a list of all the attributes
#

body Code::CodeObj::print_attributes {} {
    global \
	    c2_PrintToPrinter \
	    c2_PrintFilename \
	    c2_PrintCommand \
	    c2_PrintFirstPageFirst \
	    c2_PrintPortraitMode \
	    c2_PrintGreyscale \
	    c2_PrintPaperSize

    if {$c2_PrintToPrinter} {
	set printbuf [::open "|$c2_PrintCommand -T[file tail [$canvas info variable FileName -value]]" w]
    } else {
	set printbuf [::open $c2_PrintFilename w]
    }

    set attribs [$this info variable attributes -value]
    
    puts $printbuf "-------------------------"
    puts $printbuf [$this info class]
    puts $printbuf "-------------------------"

    foreach attr $attribs {
	puts $printbuf [concat "---" $attr "---"]
	puts $printbuf [$this info variable $attr -value]
	puts $printbuf ""
    }

    close $printbuf
}


#
#  output the Tcl code which, when sourced,
#  will recreate this object on the canvas
#

body Code::CodeObj::save {out} {

    puts $out "set obj \[\$canvas create [info namespace tail [info class]] \
	    [lindex [$canvas coords $this] 0] \
	    [lindex [$canvas coords $this] 1] \]"
    puts $out "lappend objlist \$obj"
    puts $out "set objindex($this) \$obj"

    #
    # output all attribute values
    #

    set attribs [$this info variable attributes -value]
    foreach i $attribs {
	puts $out "\$obj configure -$i \"[massage [$this info variable $i -value]]\""
    }

}



body Code::CodeObj::scale {} {}


#
#  select this object
#

body Code::CodeObj::select {} {
    $canvas addtag selected withtag $this
    set selected 1
    $canvas set_selected $this
}


#
#  selects this object and all objects in the same group
#

body Code::CodeObj::select_group {} {
    return [$canvas select_group [group_parent_name]]
}


#
#  returns boolean value
#  == (is this object selected?)
#

body Code::CodeObj::selected {} {
    return $selected
}


body Code::CodeObj::set_group {g} {
    set group $g
}


body Code::CodeObj::setvar {vname val} {
    set $vname $val
}


#
#  toggle between selected and unselected
#

body Code::CodeObj::toggle {} {
    if { !$selected } {
	select
    } else {
	deselect
    }
}

#
#  toggle the entire group between selected and unselected
#

body Code::CodeObj::toggle_group {} {
    if { !$selected } {
	select_group
    } else {
	deselect_group
    }
}

#
#  break up a group into its constituent subgroups
#

body Code::CodeObj::ungroup {} {
    set newgroup [file rootname $group]
    if { [notempty $newgroup] } {
	$canvas delete_from_group [group_parent_name] $this
	set_group $newgroup
    }
}

#
# Fill the c2_tmp array with the current values of the attributes.
#

body Code::CodeObj::update_attributes {} {
    global ::c2_tmp

    set attribs [$this info variable attributes -value]
    foreach i $attribs {
	set c2_tmp($i) [$this info variable $i -value]
    }
}
