#
# CompNode.tcl
# ------------------------------------------------------------------------
# The node which holds sequential computations.
# ------------------------------------------------------------------------
# @(#) $Id: CompNode.tcl,v 1.22 1997/11/16 04:30:35 emery Exp $
# ------------------------------------------------------------------------
# AUTHOR:
#
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================


class Code::CompNode {

    inherit Code::Node

    constructor {args} { 
	eval Code::Node::constructor $args
    } {}

    public method attributes {{update 1}}
    public method bitmap {}
    public method thisimage {}
    public method translate {ProgramAST GraphAST}
    public method raise_culprit {line col}

    private method update_ports {}

    common image
    
    private variable oldstart 0
    private variable oldend  0

    public variable IsStartNode 0 {update_ports ; set oldstart $IsStartNode}
    public variable IsTerminationNode 0 {update_ports ; set oldend $IsTerminationNode}
    public variable IsStatic 0 {update_node}

    public variable InitComputation ""
    public variable Computation ""
    public variable LocalVariables ""
    public variable StaticVariables ""
    public variable SharedVariables ""
    public variable FiringRules ""
    public variable InputPorts ""
    public variable RoutingRules ""
    public variable OutputPorts ""

    public variable Documentation ""
    public variable TraceEnable 0
    public variable IsBreakPoint 0
    public variable TracePointBeforeNode 0
    public variable TraceWhen ""
    public variable TraceFrequency ""
    public variable VariablesToTrace ""

    protected variable attributes \
	{Name IsStartNode IsTerminationNode IsStatic \
	     InitComputation Computation LocalVariables StaticVariables \
	     SharedVariables FiringRules InputPorts RoutingRules OutputPorts \
	     Documentation TraceEnable IsBreakPoint TracePointBeforeNode \
	     TraceWhen TraceFrequency VariablesToTrace}

    set image(0) [uplevel #0 image create photo -palette 256x256x256 -format GIF -file "$c2_ReleaseDirectory/bitmaps/CompNode.gif"]
    set image(1) [uplevel #0 image create photo -palette 256x256x256 -format GIF -file "$c2_ReleaseDirectory/bitmaps/CompNode-Start.gif"]
    set image(2) [uplevel #0 image create photo -palette 256x256x256 -format GIF -file "$c2_ReleaseDirectory/bitmaps/CompNode-Stop.gif"]
    set image(3) [uplevel #0 image create photo -palette 256x256x256 -format GIF -file "$c2_ReleaseDirectory/bitmaps/CompNode-Static.gif"]
}


body Code::CompNode::constructor {args} {
    addport InputPort top
    addport OutputPort bottom
    itk_initialize
}


body Code::CompNode::attributes {{update 1}} {
    Node::attributes $update

    # A node is static if it has more than one binding in
    # a firing rule or if it has any static variable declarations.
    # Assume more than one binding if there are any replicated rules
    # (i.e., any curly braces).
    
    set HasMoreThanOneBinding [expr [regexp -- {->.*->} $FiringRules] || [regexp -- {\{} $FiringRules]]
			      
    # We just look for a semicolon in the static variables.

    set HasStaticVariableDeclaration [regexp -- {;} $StaticVariables]
    
    # Certain other conditions also make the node static
    # (namely, any of the below disjunctions).
    
    set IsStatic [expr $HasMoreThanOneBinding \
		      || $HasStaticVariableDeclaration \
		      || $IsStartNode \
		      || $IsTerminationNode]

    update_node
}


body Code::CompNode::update_ports {} {

    # Delete/add input and output ports as necessary.
    # (4 cases)

    # DISABLED the output port additions/deletions
    # -- we may need this for connection to a NameSharing Node
    # (even from the termination node).

    if {$oldstart && !$IsStartNode} {
	# We were a Start node, now we aren't.
	addport InputPort top
    } elseif {$oldend && !$IsTerminationNode} {
	# We were a Stop node, now we aren't.
	# addport OutputPort bottom
    } elseif {!$oldstart && $IsStartNode} {
	# Now we're a Start node.
	deleteport InputPort top
    } elseif {!$oldend && $IsTerminationNode} {
	# Now we're a Stop node.
	# deleteport OutputPort bottom
    }
    update_node
}


body Code::CompNode::bitmap {} {
    return "@[$canvas info variable ReleaseDirectory -value]/bitmaps/CompNode.bit"
}


body Code::CompNode::thisimage {} {
    if [expr $IsStartNode] {
	return $image(1)
    }
    if [expr $IsTerminationNode] {
	return $image(2)
    }
    if (0) {
    if [expr $IsStatic] {
	return $image(3)
    }
    }
    return $image(0)
}


# display the dialog and position the cursor
# at the given line and column (for debugging).

body Code::CompNode::raise_culprit {line col} {
    $canvas deselect_all
    select_group
    if 0 {
	set Specification [format "input_ports \{%s\n\}\noutput_ports \{%s\n\}\nshared_vars \{%s\n\}\nvars \{%s\n\}\ninit_comp \{%s\n\}\nfiring_rules \{%s\n\}\ncomp \{%s\n\}\nrouting_rules \{%s\n\}" $InputPorts $OutputPorts $SharedVariables [concat $LocalVariables $StaticVariables] $InitComputation $FiringRules $Computation $RoutingRules]
	puts "spec = "
	puts $Specification
	puts "line = $line, col = $col"
    }
    set stanzas {InputPorts OutputPorts SharedVariables LocalVariables InitComputation FiringRules Computation RoutingRules}
    set names {input_ports output_ports shared_vars vars init_comp firing_rules comp routing_rules}
    
    for {set i 0} {$i < [llength $stanzas]} {incr i} {
	set stanza_name([lindex $stanzas $i]) [lindex $names $i]
    }
    
    # Find out which stanza we should be in.
    set ln 0
    foreach s $stanzas {
	
	# puts "current ln = $ln"
	
	# Find out how many lines are in this stanza.
	
	if { [string compare $s LocalVariables] } {
	    set val [set $s]
	} else {
	    set val [concat $LocalVariables $StaticVariables]
	}

	# Add 2 to the number of carriage returns;
	# See Translate.
	set lines [expr 2+[regsub -all "\n" $val {} newval]]
	
	# puts "there are $lines in stanza $s."
	# puts "stanza $s = [set $s]"

	set mystanza $s
	
	# If the culprit line is in this stanza, stop.
	if {$line < [expr $ln + $lines]} {
	    if {$line == $ln} {
		# Move the cursor back the space of the name
		# of the stanza and two spaces (for the opening brace and space).
		set col [expr $col - [string length $stanza_name($s)] - 2]
	    }
	    break
	}
	incr ln $lines
    }
    
    # puts "Found in $mystanza"

    # Raise the appropriate notebook page(s) based on the stanza.
    
    switch $mystanza {
	InputPorts -
	OutputPorts -
	FiringRules -
	RoutingRules {
	    ._CompNodeattributes._Computation view "I/O"
	    set f [._CompNodeattributes._Computation childsite "I/O"]._IO
	    switch $mystanza {
		InputPorts -
		FiringRules {
		    $f view "Input"
		    set textwin [$f childsite "Input"]._$mystanza
		}
		OutputPorts -
		RoutingRules {
		    $f view "Output"
		    set textwin [$f childsite "Output"]._$mystanza
		}
	    }
	}
	InitComputation -
	Computation {
	    set f ._CompNodeattributes._Computation
	    $f view "Computation"
	    set textwin [$f childsite "Computation"]._$mystanza
	}
	SharedVariables -
	StaticVariables -
	LocalVariables {
	    set f ._CompNodeattributes._Computation
	    $f view "Variables"
	    set textwin [$f childsite "Variables"]._$mystanza
	}
    }

    set index "[expr $line - $ln].[expr $col - 1]"
    
    # puts "index = $index"
    
    # The cursor will appear in this window...
    focus [$textwin component text]
    
    # Make sure the text item has been initialized!
    update_attributes

    # Now put the cursor where it belongs and make it visible.
    $textwin mark set insert $index
    $textwin see $index
}


body Code::CompNode::translate {ProgramAST GraphAST} {
    set Specification [format "input_ports \{%s\n\}\noutput_ports \{%s\n\}\nshared_vars \{%s\n\}\nvars \{%s\n\}\ninit_comp \{%s\n\}\nfiring_rules \{%s\n\}\ncomp \{%s\n\}\nrouting_rules \{%s\n\}" $InputPorts $OutputPorts $SharedVariables [concat $LocalVariables $StaticVariables] $InitComputation $FiringRules $Computation $RoutingRules]

    # See if the node uses any shared variables.
    # (Just look for a semicolon, as in attributes.)

    set UsesSharedVariable [regexp -- {;} $SharedVariables]

    # This node is a writer if it declares itself
    # to be a writer to any shared variables.

    set IsWriter [regexp -- writer $SharedVariables]
    
    return [c2_CreateCompNodeAST $ProgramAST $GraphAST $Name $id \
		$IsStartNode $IsTerminationNode "$Specification" "$Documentation" \
		$TraceEnable $IsBreakPoint $TracePointBeforeNode "$TraceWhen" \
		"$TraceFrequency" "$VariablesToTrace" \
		$IsStatic $IsWriter]
    
    # $IsWriter was $UsesSharedVariable.

}


