#
# codeprocs.tcl
# ------------------------------------------------------------------------
# A variety of procedures used by CODE, all with some global effect
# (i.e., they didn't fit anywhere else).
# ------------------------------------------------------------------------
# @(#) $Id: codeprocs.tcl,v 1.30 1998/11/25 05:30:49 emery Exp $
#
# ------------------------------------------------------------------------
# AUTHOR:
#
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================


proc c2_SetReshapeMode { name e op } {
    global c2_ReshapeArcs

    if {$c2_ReshapeArcs == 0} {
	map "apply no-reshape" [itcl_info objects -isa Arc]
    } else {
	map "apply reshape" [itcl_info objects -isa Arc]
    }
}


proc c2_SetArcDisplayMode { name e op } {
    global c2_DisplayArcLabels

    map "apply update_rule" [itcl_info objects -isa Arc]
}

############################################


proc c2_GlobalAnnotations {{raise_culprit 0}} {
    global c2_MainCanvas
    if {$raise_culprit} {
	$c2_MainCanvas raise_culprit
    } else {
	$c2_MainCanvas attributes
    }
}


#
#  create a new file
#

proc c2_NewFile {} {
    global c2_MainCanvas
    if { [.newdialog activate] } {
	$c2_MainCanvas new_file
    }
}

proc c2_manage_dialog {dlog attrs} {
    global c2_tmp

    foreach attr $attrs {
	global $attr
	set c2_tmp($attr) "[set $attr]"
    }

    if {[$dlog activate]} {
	foreach attr $attrs {
	    global $attr
	    set $attr "$c2_tmp($attr)"
	}
    }
}


proc c2_Debugger {} {
    global ::c2_DebuggerCanvas

    Code::CodeCanvas::SetCurrentWindow $c2_DebuggerCanvas
    ._DebuggerWindow activate
}


proc c2_ExecutionSetup {} {
    set attrs {
	c2_SummaryStatistics
	c2_OptimizeProgram
	c2_DebuggerOn
	c2_BufferTraceEvents
	c2_TranslationTarget
	c2_SequentProcessors
	c2_SequentHost
	c2_PVMHosts
    }
    c2_manage_dialog ._MachineSetup $attrs
}

proc c2_PrintSetup {} {
    set attrs {
	c2_PrintToPrinter
	c2_PrintFilename
	c2_PrintCommand
	c2_PrintFirstPageFirst
	c2_PrintPortraitMode
	c2_PrintGreyscale
	c2_PrintPaperSize
    }
    c2_manage_dialog .printdialog $attrs
}


proc c2_QuitProgram {} {
    global c2_CutBuffer
    global c2_UndoBuffer
    set shell [open "|sh" w+]

    if { [.quitdialog activate] } {

	#
	#  before we exit, save the preferences
	#  and erase auxiliary files
	#

	c2_SavePrefs
	if { [file exists $c2_CutBuffer] } {
	    puts $shell "rm $c2_CutBuffer &"
	}
	if { [file exists $c2_UndoBuffer] } {
	    puts $shell "rm $c2_UndoBuffer &"
	}
	close $shell
	exit
    }
}


proc c2_SaveFile {} {
    global c2_MainCanvas
    $c2_MainCanvas save_file
}


#
#  call up an external text editor
#  to edit a given file
#

proc c2_EditFile {filename} {
    global c2_Editor
    if { [file exists $filename] } {
	set isemacs [string compare $c2_Editor "emacs"]
	set isvi [string compare $c2_Editor "vi"]
	if { $isemacs == 0 } {
	    exec $c2_Editor $filename
	} elseif { $isvi == 0 } {
	    exec xterm -e vi $filename 
	} else { 
	    catch { exec $c2_Editor $filename }
	}
    }
}


proc c2_LoadPrefs {} {
    global c2_PrefsFile c2_PreferenceVars

    foreach v $c2_PreferenceVars {
	global $v
    }

    #
    #  if we can't find the prefs file, save the defaults,
    #  thus creating a prefs file for the next time around.
    #

    if { [catch "source $c2_PrefsFile"] } {
	c2_SavePrefs
    }
}


proc c2_SavePrefs {} {

    global c2_PrefsFile c2_PreferenceVars

    set out [open $c2_PrefsFile w]
    puts $out "# CODE preference file"
    foreach v $c2_PreferenceVars {
	global $v
	puts $out "set $v \"[massage [set $v]]\""
    }
    close $out
}


# translates the program.
# if successful, returns 1, otherwise returns 0.


proc Translate {} {
    global \
	    c2_TranslationTarget \
	    c2_MainCanvas \
	    c2_SaveDirectory

    set filename \
	    [file root \
	    [file tail \
	    [$c2_MainCanvas info variable FileName -value]]]

    if { $filename == "" } {
	c2_SaveFile
    }

    set filename \
	    [file root \
	    [file tail \
	    [$c2_MainCanvas info variable FileName -value]]]

    if { $filename != "" } {

	set c2_SaveDirectory \
		[file dirname [$c2_MainCanvas info variable FileName -value]]

	set programAST [c2_MakeAST $filename]

	if { $programAST == "" } {
	    return 0
	}
	set result [c2_TranslateAST $programAST $c2_TranslationTarget]

	return $result
    } else {
	return 0
    }
}


proc c2_Translate {} {
    global c2_MainCanvas
    global c2_ErrorUID
    global c2_yylinenum c2_yycolumn

    if { [Translate] } {
	.goodcompiledialog activate
    } else {
	bell
	.badcompiledialog activate
	
	# if an error log file has been written, read it.
	
	catch { set f [open "/tmp/code_error_log[pid]" r] }
	set errortext ""
	catch { set errortext [read $f] }
	catch { [._Errorlog subwidget frame]._Text insert end $errortext }
	catch { close $f }
	
	# only display the log if we got some text from it.
	if [string compare $errortext ""] {
	    ._Errorlog activate
	    set obj [$c2_MainCanvas find_object $c2_ErrorUID]
	    if { [string compare $obj ""] } {
		# Pop up the culprit's window and its attributes.
		[$obj canvas] expand
		# culprit only implemented for Comp & NS nodes so far.
		if { ([$obj isa ::Code::CompNode] ||
		      [$obj isa ::Code::NameSharingNode]) } {
		    $obj raise_culprit $c2_yylinenum $c2_yycolumn
		    # Don't update attributes (that's what the zero is for).
		    # This is needed to keep the cursor at the right location.
		    $obj attributes 0
		} else {
		    $obj attributes
		}
	    }
	}
    }
}


proc prompt {} {
    return "(%|#|\\\$|>) $"
}

proc send_seq {host cmds} {
    set timeout -1
    foreach cmd $cmds {
	expect -i $host -re [prompt]
	exp_send -i $host $cmd
    }
}


proc c2_RunProgram {target} {
    global \
	    c2_TranslationTarget \
	    c2_MainCanvas \
	    c2_SaveDirectory \
	    c2_SequentHost \
	    c2_SequentProcessors \
	    c2_PVMHosts \
	    env

    set c2_TranslationTarget $target

    set filename \
	    [file root \
	    [file tail \
	    [$c2_MainCanvas info variable FileName -value]]]
    
    set c2_SaveDirectory \
	    [file dirname [$c2_MainCanvas info variable FileName -value]]
    

    # . configure -cursor "watch"
    .framesix.statusbar configure -text "Compiling..."
    # update

    set result [c2_Translate]

    # . configure -cursor ""
    .framesix.statusbar configure -text ""

    return  ;# since we don't want to actually run it these days...

    set debugger_prompt "(c2.udbg) "
    
    set timeout -1
    exp_internal 0
    log_user 0
    

    #
    #  prepare a pty to use with xterm and
    #  spawn xterm
    #
    
    spawn -pty -noecho
    set xterm $spawn_id
    
    stty raw -echo < $spawn_out(slave,name)
    regexp ".*(.)(.)" $spawn_out(slave,name) dummy c1 c2
    if {[string compare $c1 "/"] == 0} {
	set c1 "0"
    }
    
    set xterm_pid [exec xterm -sb -S$c1$c2$spawn_out(slave,fd) &]
    close -slave


    #
    #  wait for xterm's window id & discard it
    #
    
    expect {
	eof { wait ; return }
	-re (.*)\n
    }
    
    exp_send -i $xterm "Now compiling...\r\n"
    
    #
    #  spawn a shell, build the code, and
    #  hand control over to the user.
    #
    
    spawn $env(SHELL)
    set shell $spawn_id
    
    #
    #  send the appropriate sequence of commands
    #  to build and execute the program.
    #

    switch $c2_TranslationTarget {
	Sequent {
	    send_seq $shell [ list \
		    "rlogin $c2_SequentHost\r" \
		    "cd $c2_SaveDirectory/$filename.sequent\r" \
		    "make\r" \
		    "$filename -n$c2_SequentProcessors ; exit\r" ]
	}
	PVM {
	    send_seq $shell [list \
		    "cd $c2_SaveDirectory/$filename.pvm\r" \
		    "pvm\r" ]
	    foreach host $c2_PVMHosts {
		send_seq $shell [list "add $host\r"]
	    }
	    send_seq $shell [list "quit\r" "make\r" "$filename\r"]
	}
	Serial {
	    send_seq $shell [list \
		    "cd $c2_SaveDirectory/$filename.serial\r" \
		    "make\r" \
		    "$filename\r"]
	}
    }

    
    #
    #  connect the xterm to the shell process,
    #  and wait for the shell to issue a prompt.
    #  it then exits the shell (via a cleanup proc);
    #  once the shell exits, the xterm is killed and
    #  the interact ends.
    #
    
    interact {
	-u $xterm
	-i $shell eof {
	    exp_send -i $xterm "Press any key to return:"
	    expect -i $xterm "?" {
		exec kill $xterm_pid
		close -i $xterm
		wait
		. configure -cursor ""
		return
	    }
	}
	$debugger_prompt {
	    exp_send -i $xterm $debugger_prompt
	}
	-re [prompt] {

	    # exit from the shell

	    switch $c2_TranslationTarget {
		Serial -
		Sequent {
		    exp_send -i $shell "exit\r"
		}
		PVM {
		    exp_send -i $shell "pvm\r"
		    send_seq $shell { \
			    "halt\r" \
			    "exit\r" }
		}
	    }
	}
    }


}

proc c2_PrintAnnotations {} {
    c2_HierarchySelect printannotations .hierarchy
}

proc c2_PrintCanvases {} {
    c2_HierarchySelect printcanvases .hierarchy
}

proc c2_GotoCallNode {} {
    c2_HierarchySelect goto .hierarchy
}

proc c2_FindNode {} {
    c2_HierarchySelect findnode .hierarchy
}

proc c2_HierarchySelect {mode hierarchy {variable ""}} {
    foreach btn [[$hierarchy subwidget buttonbox] buttons] {
	catch {destroy $btn}
    }
 
    switch $mode {
	printannotations {
	    $hierarchy configure -title "CODE: Print Annotations..." -callnodesonly 0
	    [$hierarchy hlist] configure -selectmode extended
	    [$hierarchy subwidget buttonbox] add _SelectAll -text "Select All" -command "$hierarchy select_all"
	    [$hierarchy subwidget buttonbox] add _Print -text "Print" -command "set selection \[\[$hierarchy hlist\] info selection\] ; foreach i \$selection \{\[\[$hierarchy hlist\] info data \$i\]\ print_attributes\}"
	    [$hierarchy subwidget buttonbox] add _View -text "View" -command "set selection \[\[$hierarchy hlist\] info selection\] ; foreach i \$selection \{\[\[$hierarchy hlist\] info data \$i\] attributes\} "
	}
	printcanvases {
	    $hierarchy configure -title "CODE: Print Graphs..." -callnodesonly 1
	    [$hierarchy hlist] configure -selectmode extended
	    [$hierarchy subwidget buttonbox] add _SelectAll -text "Select All" -command "$hierarchy select_all"
	    [$hierarchy subwidget buttonbox] add _Print -text "Print" -command "set selection \[\[$hierarchy hlist\] info selection\] ; foreach i \$selection \{ \[\[$hierarchy hlist\] info data \$i\] print \}"
	    [$hierarchy subwidget buttonbox] add _Open -text "Open" -command "set selection \[\[$hierarchy hlist\] info selection\] ; if \{ \$selection != \"\" \} \{ foreach item \$selection \{ \[\[$hierarchy hlist\] info data \$item\] expand \} \}"
	}
	findnode {
	    $hierarchy configure -title "CODE: Find a Node..." -callnodesonly 0
	    [$hierarchy hlist] configure -selectmode browse
	    [$hierarchy subwidget buttonbox] add _Info -text "Info" -command "set selection \[\[$hierarchy hlist\] info selection\] ; if \{ \$selection != \"\" \} \{ foreach item \$selection \{ \[\[$hierarchy hlist\] info data \$item\] attributes \} \}"
	    set gotocommand {
		set selection \[[$hierarchy hlist] info selection\]
	        if { \$selection != "" } {
		    foreach item \$selection {
			set node \[[$hierarchy hlist] info data \$item\]
			\[\$node info variable canvas -value\] expand
			\[\$node info variable canvas -value\] deselect_all
			\$node select_group
		    }
		}
	    }
	    [$hierarchy subwidget buttonbox] add _Goto -text "Goto" -command [subst $gotocommand]
	}
	goto {
	    $hierarchy configure -title "CODE: Hierarchy Browser..." -callnodesonly 1
	    [$hierarchy hlist] configure -selectmode browse
	    [$hierarchy subwidget buttonbox] add _Open -text "Open" -command "set selection \[\[$hierarchy hlist\] info selection\] ; if \{ \$selection != \"\" \} \{ foreach item \$selection \{ \[\[$hierarchy hlist\] info data \$item\] expand \} \}"
	}
	alias {
	    $hierarchy configure -title "CODE: Choose a Node to Alias..." -callnodesonly 1
	    [$hierarchy hlist] configure -selectmode single
	    [$hierarchy subwidget buttonbox] add _Select -text "Select" -command "set selection \[\[$hierarchy hlist\] info selection\] ; global $variable ; set $variable \[\[$hierarchy hlist\] info data \$selection\] ; $hierarchy okcmd"
	}
    }
    [$hierarchy subwidget buttonbox] add _Dismiss -text "Done" -command "$hierarchy cancelcmd"

    $hierarchy activate
}

   

proc c2_MakeAST {filename} {
    global \
	    c2_MainCanvas                c2_TranslationTarget \
	    c2_GlobalTypes               c2_GlobalFunctionSignatures \
	    c2_GlobalFunctionDefinitions c2_GlobalDocumentation \
	    c2_SummaryStatistics         c2_OptimizeProgram \
	    c2_FilesToLink               c2_DebuggerOn \
	    c2_BufferTraceEvents         c2_CodeDirectory

    set include_file [open "${c2_CodeDirectory}/ui/includes.h" r]
    set includes [read $include_file]
    close $include_file

    set programAST [c2_CreateProgramAST \
	    $filename 0 "$c2_GlobalTypes" "$c2_GlobalFunctionSignatures\n$includes" "$c2_GlobalFunctionDefinitions" "$c2_GlobalDocumentation" "$c2_SummaryStatistics" "$c2_OptimizeProgram" "$c2_FilesToLink" "$c2_DebuggerOn" "$c2_BufferTraceEvents"]

    if { $programAST == "" } {
	.badcompiledialog activate
	c2_GlobalAnnotations 1 ;# raise the culprit.
	return ""
    }

    set graphId [$c2_MainCanvas info variable id -value]
    set graphname "main"
    set result [$c2_MainCanvas translate $programAST $graphname $graphId]
    if {$result} {
	return $programAST
    } else {
	return ""
    }
}


#
# The main program.
#

proc code2 {{option "display"}} {
    global \
	    env \
	    auto_path \
	    tcl_interactive \
	    c2_MenuFont \
	    c2_TextFont \
	    c2_MonoFont \
	    c2_Editor \
	    c2_CodeDirectory \
	    c2_ReleaseDirectory \
	    c2_SaveDirectory \
	    c2_FGColor \
	    c2_SelectedFGColor \
	    c2_UndoBuffer \
	    c2_CutBuffer \
	    c2_MainCanvas \
	    c2_PrefsFile \
	    c2_TranslationTarget \
	    c2_SummaryStatistics \
	    c2_OptimizeProgram \
	    c2_DebuggerOn \
	    c2_BufferTraceEvents \
	    c2_SequentHost \
	    c2_PVMHosts \
	    c2_SequentProcessors \
	    c2_PrintToPrinter \
	    c2_PrintFilename \
	    c2_PrintCommand \
	    c2_PrintFirstPageFirst \
	    c2_PrintPortraitMode \
	    c2_PrintGreyscale \
	    c2_PrintPaperSize \
	    c2_PreferenceVars \
	    c2_GlobalTypes \
	    c2_GlobalFunctionSignatures \
	    c2_GlobalFunctionDefinitions \
	    c2_GlobalDocumentation \
	    c2_FilesToLink \
	    c2_ReshapeArcs \
	    c2_DisplayArcLabels


    #  this is not to interact via stdin/out
    
    # set tcl_interactive 0
    
    #
    # fonts (for menu items, and for normal text items)
    #
    
    set c2_MenuFont -adobe-helvetica-medium-r-normal--14-*-*
    set c2_TextFont -Adobe-Helvetica-Bold-R-Normal--14-*-*
    set c2_MonoFont -adobe-courier-bold-r-normal--14-*-* 

    option add *Entry.Font $c2_MonoFont
    option add *text.Font $c2_MonoFont

    #
    # load up all environment variables
    #   (editor and release directory)
    #
    
    if { [catch { set c2_Editor $env(EDITOR) }] } {
	set c2_Editor "emacs"  ;# default editor is emacs
    }
    
    set c2_ReleaseDirectory ${c2_CodeDirectory}/ui
    set c2_SaveDirectory $c2_ReleaseDirectory
    
    # color values
    
    set c2_FGColor           black       ;# foreground color
    set c2_SelectedFGColor   blue        ;# foreground color of selected objects
    
    
    # these values will be assigned below (via the "init" procedure)
    # to common values in CodeCanvas.
    
    set tmp "/tmp"
    catch { set tmp $env(TMPDIR) }

    set c2_UndoBuffer "$tmp/code2-undobuf[pid]"
    set c2_CutBuffer  "$tmp/code2-cutbuf[pid]"
    set c2_MainCanvas ".framesix.framezero.canvastwo"  ;# as defined in "mainscreen.tcl"
    set c2_PrefsFile  "$env(HOME)/.CODErc"
    
    #
    #  These are really program-wide preferences
    #  and will be loaded from a ~/.CODErc file --
    #  they are declared here for documentation purposes and
    #  to provide default initial values.
    #
    
    set c2_TranslationTarget "Unified"
    set c2_SummaryStatistics 0
    set c2_OptimizeProgram   0
    set c2_DebuggerOn        0
    set c2_BufferTraceEvents 0
    set c2_SequentHost       ""
    set c2_PVMHosts          {}
    set c2_SequentProcessors 2
    
    #
    #  printing preferences
    #
    
    set c2_PrintToPrinter 1
    set c2_PrintFilename ""
    set c2_PrintCommand "lpr -h"
    set c2_PrintFirstPageFirst 1
    set c2_PrintPortraitMode 1
    set c2_PrintGreyscale 1
    set c2_PrintPaperSize 0
    
    
    #
    #  This is a list of all variables whose
    #  values will be saved to and restored from
    #  the preferences file.
    #
    
    set c2_PreferenceVars {
	c2_TranslationTarget 
	c2_PrintToPrinter
	c2_PrintFilename
	c2_PrintCommand
	c2_PrintFirstPageFirst
	c2_PrintPortraitMode
	c2_PrintGreyscale
	c2_PrintPaperSize
	c2_DisplayArcLabels
    }
    
    
    ### program values
    
    set c2_GlobalTypes ""
    set c2_GlobalFunctionSignatures ""
    set c2_GlobalFunctionDefinitions ""
    set c2_GlobalDocumentation ""
    set c2_FilesToLink ""
    
    set c2_ReshapeArcs 0  ;# initially we are not in reshape mode
    trace variable c2_ReshapeArcs w c2_SetReshapeMode

    set c2_DisplayArcLabels 1
    trace variable c2_DisplayArcLabels w c2_SetArcDisplayMode
    
    #
    # load up the preferences from the prefs file
    #
    
    c2_LoadPrefs

    #
    # initialize CodeCanvas
    #
    
    Code::CodeCanvas::init $c2_UndoBuffer $c2_CutBuffer $c2_MainCanvas \
	    $c2_SelectedFGColor $c2_FGColor $c2_ReleaseDirectory 760 522
    
    #
    # bring up the main window
    #

    if { $option == "display"} {
	ShowWindow
	# update
    }
}
