#
# utils.tcl
# ------------------------------------------------------------------------
# Generic utilities, most of them implementing LISP/Haskell-like functions
# (map, filter...)
# ------------------------------------------------------------------------
# @(#) $Id: utils.tcl,v 1.7 1997/03/16 04:05:31 emery Exp $
#
# ------------------------------------------------------------------------
# AUTHOR:
#
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================

#
#  like LISP, Haskell...
#    maps function fn onto every item in the list
#    and returns the result
#

proc map { fn list } {

    set newlist ""
    foreach item $list {
	lappend newlist [uplevel "$fn $item"]
    }
    return $newlist
}


#
#  reverses application order,
#  so that we can map methods onto lists of objects
#

proc apply {x y} {
    return [eval $y $x]
}


proc function {vars body arglist} {
    foreach va [zip $vars $arglist] {
	set [lindex $va 0] [lindex $va 1]
    }
    eval $body
}


proc zip {list1 list2} {
    set z ""
    set n [min [llength $list1] [llength $list2]]
    for {set i 0} {$i < $n} {incr i} {
	lappend z [list [lindex $list1 $i] [lindex $list2 $i]]
    }
    return $z
}


#
#  returns only those items in list for which [fn item] is true
#

proc filter { fn list } {

    set newlist ""
    foreach item $list {
	if {[eval "$fn {$item}"]} {
	    lappend newlist $item
	}
    }
    return $newlist
}

proc min { a b } {
    if { $a < $b } {
	return $a
    } else {
	return $b
    }
}


proc max { a b } {
    if { $a > $b } {
	return $a
    } else {
	return $b
    }
}


proc unique { list } {

    #
    #  Returns a list containing only one
    #    occurrence of each item.
    #  Takes advantage of Tcl's built-in
    #    associative array facility.
    #

    foreach item $list {
        set assoc($item) 1
    }
    return [array names assoc]
}


proc extract_arg {name list} {
    
    # destructively extracts a "command-line"
    # parameter from the argument list and
    # returns its corresponding value
    # (or "" if the parameter was not found)
    
    upvar $list l
    set found [lsearch -exact $l $name]
    if { $found == -1 } {
	return ""
    } else {
	set argval [lindex $l [expr $found+1]]
	set l [lreplace $l $found [expr $found+1]]
	return $argval
    }
}


#
#  returns a string with all Tcl special characters backquoted
#

proc massage {s} {
    regsub -all {(\[|\]|\"|\$|\\|\{|\})} $s {\\\1} new_s
    return $new_s
}


proc notempty {x} {
    return [string compare $x ""]
}

proc first {x} {
    return [lindex $x 0]
}

proc last {x} {
    return [lindex $x [expr [llength $x]-1]]
}

proc tail {x} {
    return [lrange $x 1 end]
}

# Returns 1 iff n is prime.
proc is_prime {n} {
    set sqrt_n [expr round(sqrt($n))+1]
    for {set i 3} {$i < $n} {incr i 2} {
	if {[expr 1.0*$n/$i] == [expr round($n/$i)]} {
	    return 0
	}
    }
    return 1
}

proc tcl_next_prime {n} {
    set next [expr $n+2]
    while {![is_prime $next]} {
	incr next 2
    }
    return $next
}

# returns 1 iff the string is a valid C identifier

proc c_identifier {string} {
    return [regexp -nocase {^[a-z]+[0-9a-z_]*$} $string]
}


# returns 1 iff the string is marginally acceptable as a valid CODE index expression
proc index_expression {string} {
    return [regexp -nocase {^(\[[0-9a-z_\+-\*/\ ]+\])*$} $string]
}


# on a UNIX box, returns a list of all available printers

proc printers {} {
  set f [open /etc/printcap r]
  set str [read $f]
  set printers {}
  foreach s $str {
    if {[regexp {^([^\|]*)\|.*$} $s foo match]} {
      lappend printers $match
    }
  }
  return $printers
}


proc lookbusy {w status} {
    switch $status {
	on {
	    $w configure -cursor "watch"
	}
	off {
	    $w configure -cursor ""
	}
    }
}
