#include <config.h>
// fileutil.C

//   Misc. routines for file and directory manipulation.

#include <stream.h>
#include <fstream.h>
#include <sysent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <pwd.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include <lib/MsgUI.h>
#include "general.h"
#include "fileutil.h"

#define CONFIGFILENAME "/.coderc"
#define AOK 0

static ifstream ConfigFile;
char *TestDirectory;

  // MakeSrcDir Creates a directory called dirname that has read, write,
  // and search permissions.  Returns 1 for success, 0 otherwise.  Does
  // nothing if directory is already there.

int MakeSrcDir (char *dirname)
{
   struct stat buf;
   int s;

   s = lstat(dirname, &buf);
   if (s != 0) s = errno;

   switch (s) {
    case AOK:  // dirname exists
      if (!S_ISDIR(buf.st_mode)) {
         YYUIMessage(0, ErrorLevel,
		     "%s is not a directory.  Did CODE make it?\n", dirname);
	 YYUIMessageDisplay();
	 return 0;
      }
      break;
    case ENOENT:
      if (mkdir(dirname, 511)) {
         YYUIMessage(0, ErrorLevel,
		     "Could not create directory %s\n", dirname);
	 YYUIMessageDisplay();
	 return 0;
      }
      break;
    default:
      YYUIMessage(0, ErrorLevel,
		  "Could not stat directory %s\n", dirname);
      YYUIMessageDisplay();
      return 0;
   }

   if (access(dirname, R_OK|W_OK|X_OK)) {
      YYUIMessage(0, ErrorLevel,
		  "Access denied to %s.  Did CODE make it?\n", dirname);
      YYUIMessageDisplay();
      return 0;
   }

   return 1;
}


int OpenConfigFile()
{
   char fname[256];
   struct passwd *pwent;
   char *tmp;

   // Get complete path to config file in user's home dir.

   fname[0] = 0;
   tmp = getenv("HOME");
   if (tmp != 0)
     strcpy(fname, tmp);
   else {
      pwent = getpwuid(getuid());
      if (pwent != 0)
	if (pwent->pw_dir != 0)
	  strcpy(fname, pwent->pw_dir);
   }

   if (fname == 0) {
      fprintf(stderr, "\n*** Warning: could not get home directory\n\n");
      return 0;
   }

   strcat(fname, CONFIGFILENAME);   

   ConfigFile.open(fname);
   if (!ConfigFile)
     return 0;      // Cound open file-- it's not there we assume
   else
     return 1;
}


void CloseConfigFile()
{
   ConfigFile.close();
}


#define KEYMAX 32
#define VALUEMAX 64

static char KeyBuf[KEYMAX];
static char ValueBuf[VALUEMAX];

static int is_simple_space(char ch)
{
   if (ch == ' ') return 1;
   if (ch == '\t') return 1;
   return 0;
}


   // GetKeyAndValue reads a string key and value from the file opened
   // by OpenConfigFile.  See fileutil.h.  It is implemented as a
   // state machine derived from a FSA I drew on a piece of paper.

int GetKeyAndValue(char **Key, char **Value)
{
   char ch;
   int i;
   int state;

   state = 0;

   while (1)
     switch (state) {
      case 0:                       // Eat leading white space
	ch = ConfigFile.get();
	if (ch == '#')
	  state = 7;
	else if (isspace(ch))
	  state = 0;
	else if (ch == EOF)
	  state = 100;
	else if (isalnum(ch)) {
	   i = 0;
	   KeyBuf[i] = ch;
	   state = 1;
	} else
	  state = 99;
	break;
      case 1:                     // Fill in KeyBuf with letters and digits
	ch = ConfigFile.get();
	if (isalnum(ch)) {
	   i++;
	   if (i >= KEYMAX) Die(" *** Config key too long\n");
	   KeyBuf[i] = ch;
	   state = 1;
	} else if (is_simple_space(ch)) {
	   i++;
	   if (i >= KEYMAX) Die(" *** Config key too long\n");
	   KeyBuf[i] = '\0';
	   state = 2;
	} else if (ch == '=') {
	   i++;
	   if (i >= KEYMAX) Die(" *** Config key too long\n");
	   KeyBuf[i] = '\0';
	   state = 3;
	} else
	  state = 99;
	break;
      case 2:                    // Eat white space before =
	ch = ConfigFile.get();
	if (is_simple_space(ch))
	  state = 2;
	else if (ch == '=')
	  state = 3;
	else
	  state = 99;
	break;
      case 3:                   // Eat white space after =
	ch = ConfigFile.get();
	if (is_simple_space(ch))
	  state = 3;
	else if (ch == '~')
	  state = 99;
	else if (isgraph(ch)) {
	   i = 0;
	   ValueBuf[i] = ch;
	   state = 4;
	} else
	  state = 99;
	break;
      case 4:                  // Fill ValueBuf with graphic chars
	ch = ConfigFile.get();
	if (ch == '~')
	  state = 99;
	else if (isgraph(ch)) {
	   i++;
	   if (i >= VALUEMAX) Die(" *** Config value too long\n");
	   ValueBuf[i] = ch;
	   state = 4;
	} else if (is_simple_space(ch)) {
	   i++;
	   if (i >= VALUEMAX) Die(" *** Config value too long\n");
	   ValueBuf[i] = '\0';
	   state = 5;
	} else if (ch == '\n') {
	   i++;
	   if (i >= VALUEMAX) Die(" *** Config value too long\n");
	   ValueBuf[i] = '\0';
	   state = 6;
	} else if (ch == EOF) {
	   i++;
	   if (i >= VALUEMAX) Die(" *** Config value too long\n");
	   ValueBuf[i] = '\0';
	   state = 6;
	} else
	  state = 99;
	break;
      case 5:                       // Eat trailing white space
	ch = ConfigFile.get();
	if (is_simple_space(ch))
	  state = 5;
	else if (ch == '\n')
	  state = 6;
	else if (ch == EOF)
	  state = 6;
	else
	  state = 99;
	break;
      case 6:                    // return a pair
	*Key = KeyBuf;
	*Value = ValueBuf;
	return 1;
      case 7:                   // Eat rest of line
	ch = ConfigFile.get();
	if (ch == '\n')
	  state = 0;
	else if (ch == EOF)
	  state = 100;
	else
	  state = 7;
	break;
      case 99:                  // A sytax error is found
	fprintf(stderr, "\n*** Warning: syntax error in config file\n\n");
        state = 7;
	break;
      case 100:                 // No more pairs
	return 0;
      default:
	fprintf(stderr, "\n** CODE internal error in GetKeyAndValue\n\n");
	return(0);
     }
}
