/*
  sfile.h
  ------------------------------------------------------------------------
  Structured file definitions.
  ------------------------------------------------------------------------
  @(#) $Id: sfile.h,v 1.12 1997/07/03 17:24:13 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

// sfile.h                                    Structured Files

// Structured files (sfile) are used to structure output.  An sfile
// consists of a tree of cBlocks.  Each cBlock contains text and
// a list of sub-blocks.  When an sfile is dumped to disk, its text
// is written first followed by each of its sub-blocks.

// One navigates through an sfile tree by means of class cCursor.  This
// class contains a static pointer to the root of the sfile tree.  Hence,
// a program may use only a single sfile tree.  Different branches can
// be output to separate disk files by explicit user control-- you
// write a procedure to do this.  Use SequentWriteFiles as a guide.
// This aspect of sfiles is quite a hack.  The WriteFile routines
// depend upon the internal structure of the sfile classes.

// There is a tricky aspect to structured files.  A cBlock contains a
// list of sub-blocks.  In what order should they be output to disk?
// You have to define this and hence cBlocks must have IDs.  In fact,
// they have an ID and an Index.  The ID is best thought of as a
// type ID and the Index as an instance ID.  (I am not talking about C++
// objects here.  I am talking about types of cBlocks, intended for
// different purposes.  The IDs should be defined using an enum that
// provide.  See the enum with eRoot in [architecture]_aux.h.

// The order in which the enums are defined is important since it
// determines the order in which sub-blocks will be output.

#ifndef _sfile_x
#define _sfile_x

#include <stream.h>
#include <fstream.h>
#include "estring.h"
#include "mstring.h"
#include "voidlist.h"


class cBlock {
 public:

   cBlock(cBlock *Parent, int InitID);
   cBlock(cBlock *Parent, int InitID, int InitIndex);
   ~cBlock();

   int ID;
   int UseIndex;
   int Index;

   cBlock *Parent;
   EString Text;
   VoidList SubBlocks;   // list of cBlock

   void Output(ofstream &os);  // dump block and its children

   void PrintPath();   // Print own and parent's ID and Index to root.
   void DumpID();      // A debugging tool.

};

   // Call InitCursor before using class cCursor

void InitCursor(int ID);


class cCursor {
   friend void InitCursor(int ID);
 public:

   cCursor();

   static cBlock *Root;
   cBlock *Current;

   cBlock *RootBlock();

   void cd();            // Set cursor to root block
   void cd(int ID);   
   void cd(int ID, int Index);
   void up();           // To parent

   cCursor &operator << (EString E)        { Current->Text << E; return *this; }
   cCursor &operator << (const MString& M) { Current->Text << M; return *this; }
   cCursor &operator << (const char *s)    { Current->Text << s; return *this; }
   cCursor &operator << (const char c)     { Current->Text << c; return *this; }
   cCursor &operator << (const int i)      { Current->Text << i; return *this; }

   void pwd();       // Print path to parent, for debugging.

};

#endif
