/*
  gram.y
  ------------------------------------------------------------------------
  YACC++ grammar for CODE.
  ------------------------------------------------------------------------
  @(#) $Id: gram.y,v 1.16 1996/09/12 17:41:46 emery Exp $
  ------------------------------------------------------------------------
  AUTHOR/CONTACT:
 
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
  Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

/* Grammar for CODE 2

   This grammar actually parses several different things including...

   o) lists of functions signatures
   o) arc topology specifications
   o) typedef lists
   o) creation parameter declarations
   o) UC node definitions
   o) statement lists

   A special marker symbol chooses what will be parsed.  See the rule
   that defines "code2_thing".

   NOTICE: Running yacc++ produces 1 shift/reduce conflict.  It is
   caused by the "If" statement ambiguity and may be ignored since
   the parser does the right thing.

   Nodes in the tree add themselve to a UID table when they are
   created.  Storage for them is freed when the UID table is freed.
   This is true even for failed parses.  Never delete storage for
   an AST node except via the UID table.

                                                                         */
%{
#include "../misc/general.h"
#include "../exmodel/exmodel.h"
#include "parser.h"
#include "scan.h"

int yyparse();

typedef cAbsTree *yyMyType;

#define YYSTYPE yyMyType
static YYSTYPE AbsTreeHead;

static int yyMakeTree;

%}

%token NEW
%token WRONG
%token IDENT
%token INTCONST
%token REALCONST
%token LE_OP
%token GE_OP
%token EQ_OP
%token NE_OP
%token AND_OP
%token OR_OP
%token TYPE
%token IS
%token ARRAY
%token OF
%token STRUCT
%token WHILE
%token VOID
%token IF
%token ELSE
%token VAR
%token INPUTRULES
%token OUTPUTRULES
%token COMP
%token INITCOMP
%token INPUTPORTS
%token OUTPUTPORTS
%token PROD
%token CONS
%token SHARED_VAR
%token READER
%token WRITER
%token QQNSICOMP
%token QQFUNCSIG
%token QQARCTOPO
%token QQUC
%token QQTYPEDEF
%token QQEXPR
%token PARTITION
%token MERGE
%token EXPAND
%token CONTRACT
%token ALIGN
%token STRIP_UPDATE
%token BLOCK
%token CYCLIC
%token BLOCK_CYCLIC
%token OVERLAP
%token SUMMARY


%start code2_thing

%%

code2_thing
	: QQFUNCSIG func_sig_list
		{ if (yyMakeTree) AbsTreeHead = $2; }
	| QQARCTOPO arc_topo_spec
		{ if (yyMakeTree) AbsTreeHead = $2; }
	| QQTYPEDEF type_spec_list
		{ if (yyMakeTree) AbsTreeHead = $2; }
	| QQUC uc
		{ if (yyMakeTree) AbsTreeHead = $2; }
	| QQNSICOMP ns_init_comp
		{ if (yyMakeTree) AbsTreeHead = $2; }
	| QQEXPR expr
		{ if (yyMakeTree) AbsTreeHead = $2; }
	;

/* Function signatures */

func_sig_list             /* Returns cGather*  */
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| func_sig_list func_sig
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($2);
		  $$ = $1; }}
	;

func_sig
	: IDENT IDENT '(' fc_arg_list ')' ';'
		{ if (yyMakeTree)
		     $$ = new cFuncSig((char *) $2, (char *) $1,
				       ((cGather *) $4)->L); }
	| IDENT IDENT '(' ')' ';'
		{ if (yyMakeTree) $$ = new cFuncSig((char *) $2,
						    (char *) $1); }
	| VOID IDENT '(' fc_arg_list ')' ';'
		{ if (yyMakeTree)
		     $$ = new cFuncSig((char *) $2, ((cGather *) $4)->L); }
	| VOID IDENT '(' ')' ';'
		{ if (yyMakeTree) $$ = new cFuncSig((char *) $2); }
	| IDENT IDENT '(' '.' '.' '.' ')' ';'
		{ if (yyMakeTree) {
                     $$ = new cFuncSig((char *) $2, (char *) $1);
                     ((cFuncSig *) $$)->DontCheckArgs = 1; } }
	| VOID IDENT '(' '.' '.' '.' ')' ';'
		{ if (yyMakeTree) {
		     $$ = new cFuncSig((char *) $2);
                     ((cFuncSig *) $$)->DontCheckArgs = 1; } }
	;

fc_arg_list
	: fc_arg_spec
		{ if (yyMakeTree) {
  		     $$ = new cGather;
		     ((cGather *) $$)->L.Insert($1); }}
	| fc_arg_list ',' fc_arg_spec
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($3);
		     $$ = $1; }}
	;

fc_arg_spec
	: IDENT IDENT
		{ if (yyMakeTree)
		     $$ = new cFuncArgSpec((char *) $1, (char *) $2, 0); }
	| IDENT '*' IDENT
		{ if (yyMakeTree)
		     $$ = new cFuncArgSpec((char *) $1, (char *) $3, 1); }
	;


/* arc topology specs */

arc_topo_spec
	: type_1_arc
		{ if (yyMakeTree) $$ = $1; }
	| type_2_arc
		{ if (yyMakeTree) $$ = $1; }
	| type_3_arc
		{ if (yyMakeTree) $$ = $1; }
	| type_4_arc
		{ if (yyMakeTree) $$ = $1; }
	| type_5_arc
		{ if (yyMakeTree) $$ = $1; }
	| type_6_arc
		{ if (yyMakeTree) $$ = $1; }
	| Part_arc
		{ if (yyMakeTree) $$ = $1; }
	| Merge_arc
		{ if (yyMakeTree) $$ = $1; }
	| Expand_arc
		{ if (yyMakeTree) $$ = $1; }
	| Contract_arc
		{ if (yyMakeTree) $$ = $1; }
	| Strip_Update_arc
		{ if (yyMakeTree) $$ = $1; }
	| Align_arc
		{ if (yyMakeTree) $$ = $1; }
	;

type_1_arc
	: IDENT
		{ cArc *tmp;
		  if (yyMakeTree) {
		     tmp = new cArc;
		     tmp->ArcType = 1;
		     tmp->FromPort = (char *) $1;
		     $$ = tmp; }}
	;

type_2_arc
	: varlist '.' IDENT varlist '=' '>'
	  exprlist '.' IDENT exprlist
		{ cArc *tmp;
		  if (yyMakeTree) {
		     tmp = new cArc;
		     tmp->ArcType = 2;
		     tmp->FromPort = (char *) $3;
		     tmp->NodeVarList = ((cGather *) $1)->L;
		     tmp->PortVarList = ((cGather *) $4)->L;
                     tmp->ToPort = (char *) $9;
		     tmp->NodeExprList = ((cGather *) $7)->L;
		     tmp->PortExprList = ((cGather *) $10)->L;
		     $$ = tmp; }}
	;

type_3_arc
	: varlist '.' IDENT varlist '=' '>'
	  exprlist '.' exprlist '.' IDENT exprlist
		{ cArc *tmp;
		  if (yyMakeTree) {
		     tmp = new cArc;
		     tmp->ArcType = 3;
		     tmp->FromPort = (char *) $3;
		     tmp->NodeVarList = ((cGather *) $1)->L;
		     tmp->PortVarList = ((cGather *) $4)->L;
                     tmp->ToPort = (char *) $11;
		     tmp->CallExprList = ((cGather *) $7)->L;
		     tmp->NodeExprList = ((cGather *) $9)->L;
		     tmp->PortExprList = ((cGather *) $12)->L;
		     $$ = tmp; }}
	;

type_4_arc
	: varlist '.' varlist '.' IDENT varlist '=' '>'
	  exprlist '.' IDENT exprlist
		{ cArc *tmp;
		  if (yyMakeTree) {
		     tmp = new cArc;
		     tmp->ArcType = 4;
		     tmp->FromPort = (char *) $5;
		     tmp->CallVarList = ((cGather *) $1)->L;
		     tmp->NodeVarList = ((cGather *) $3)->L;
		     tmp->PortVarList = ((cGather *) $6)->L;
                     tmp->ToPort = (char *) $11;
		     tmp->NodeExprList = ((cGather *) $9)->L;
		     tmp->PortExprList = ((cGather *) $12)->L;
		     $$ = tmp; }}
	;

type_5_arc
  	: varlist '.' varlist '.' IDENT varlist '=' '>'
	  exprlist '.' exprlist '.' IDENT exprlist
		{ cArc *tmp;
		  if (yyMakeTree) {
		     tmp = new cArc;
		     tmp->ArcType = 5;
		     tmp->FromPort = (char *) $5;
		     tmp->CallVarList = ((cGather *) $1)->L;
		     tmp->NodeVarList = ((cGather *) $3)->L;
		     tmp->PortVarList = ((cGather *) $6)->L;
                     tmp->ToPort = (char *) $13;
		     tmp->CallExprList = ((cGather *) $9)->L;
		     tmp->NodeExprList = ((cGather *) $11)->L;
		     tmp->PortExprList = ((cGather *) $14)->L;
		     $$ = tmp; }}
	;

type_6_arc
	: /* empty */
		{ cArc *tmp;
		  if (yyMakeTree) {
		     tmp = new cArc;
		     tmp->ArcType = 6;
		     $$ = tmp; }}
	;

Part_arc 
        : PARTITION '(' IDENT ',' IDENT '('part_spec ')' ',' 
          '('size_spec ')' ',' '(' overlap_spec ')' ')'
                { cPartArc *tmp;
                  if (yyMakeTree) {
                }}
        ; 

part_spec
        : dist
                {  if (yyMakeTree)  {
                 }}
        | dist ',' part_spec
                {  if (yyMakeTree) {
                }}
        ;

dist
        : BLOCK 
               { if (yyMakeTree)  {
               }}
        | CYCLIC
               { if (yyMakeTree)  {
               }}
        | BLOCK_CYCLIC '(' param ')'
               { if (yyMakeTree)  {
               }}
        | '*'
               { if (yyMakeTree)  {
               }}
        ;

param 
        : IDENT
               { if (yyMakeTree)  {
               }}
        | INTCONST
               { if (yyMakeTree)  {
               }}
        ;


size_spec
        : size
                {  if (yyMakeTree)  {
                 }}
        | size ',' size_spec
                {  if (yyMakeTree) {
                }}
        ;

size
        : IDENT 
               { if (yyMakeTree)  {
               }}
        | INTCONST 
               { if (yyMakeTree)  {
               }}
        | '*'
               { if (yyMakeTree)  {
               }}
        ;

overlap_spec
        :overlap 
                {  if (yyMakeTree)  {
                 }}
        | overlap ',' overlap_spec
                {  if (yyMakeTree) {
                }}
        ;

overlap
        : OVERLAP '=' IDENT 
               { if (yyMakeTree)  {
               }}
        | OVERLAP '=' INTCONST 
               { if (yyMakeTree)  {
               }}
        | '*'
               { if (yyMakeTree)  {
               }}
        ;

Merge_arc 
        : MERGE '(' IDENT ',' IDENT '(' merge_spec ')' ',' 
          '('size_spec ')' ',' '(' overlap_spec ')' ')'
                { cMergeArc *tmp;
                  if (yyMakeTree) {
                }}
        ; 

merge_spec
        : mrg 
                {  if (yyMakeTree)  {
                 }}
        | mrg ',' merge_spec
                {  if (yyMakeTree) {
                }}
        ;

mrg
        : BLOCK 
               { if (yyMakeTree)  {
               }}
        | CYCLIC
               { if (yyMakeTree)  {
               }}
        | BLOCK_CYCLIC '(' param ')'
               { if (yyMakeTree)  {
               }}
        | '*'
               { if (yyMakeTree)  {
               }}
        ;

Expand_arc 
        :  EXPAND '(' IDENT ',' IDENT ',' IDENT ',' IDENT ',' IDENT ')'
                { cExpandArc *tmp;
                  if (yyMakeTree) {
                }}
        ; 

Contract_arc 
        :  CONTRACT '(' IDENT ',' IDENT ',' IDENT ',' IDENT ',' IDENT ')'
                { cContractArc *tmp;
                  if (yyMakeTree) {
                }}
        ; 

Strip_Update_arc 
        : STRIP_UPDATE '(' IDENT ',' IDENT '(' update_spec ')' ',' 
          '('size_spec ')' ',' '(' overlap_spec ')' ')'
                { cStrip_UpdateArc *tmp;
                  if (yyMakeTree) {
                }}
        ; 

update_spec
        : updt 
                {  if (yyMakeTree)  {
                 }}
        | updt ',' update_spec
                {  if (yyMakeTree) {
                }}
        ;

updt
        : BLOCK 
               { if (yyMakeTree)  {
               }}
        | CYCLIC
               { if (yyMakeTree)  {
               }}
        | BLOCK_CYCLIC '(' param ')'
               { if (yyMakeTree)  {
               }}
        | '*'
               { if (yyMakeTree)  {
               }}
        ;

Align_arc 
        : SUMMARY '=' ALIGN '(' primary ',' secondary_list 
                { cAlignArc *tmp;
                  if (yyMakeTree) {
                }}
        ; 

primary
        : IDENT '(' ident_stride_spec ')'
        ;

ident_stride_spec
        : ident_stride 
                {  if (yyMakeTree)  {
                 }}
        | ident_stride ',' ident_stride_spec
                {  if (yyMakeTree) {
                }}
        ;

ident_stride
        : '(' expr ',' expr ')' 
               { if (yyMakeTree)  {
               }}
        ;

secondary_list
        : secondary 
                {  if (yyMakeTree)  {
                 }}
        | secondary ',' secondary_list 
                {  if (yyMakeTree) {
                }}
        ;

secondary
        : IDENT '(' ident_stride_spec ')'
                {  if (yyMakeTree) {
                }}
        ;


varlist    /* Returns cGather * */
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| varlist '[' IDENT ']'
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append(new cVarDecl((char *) $3));
		     $$ = $1; }}
	;

exprlist    /* Returns cGather * */
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| exprlist '[' additive_expr ']'
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($3);
		     $$ = $1; }}
	;


/* typedefs */

type_spec_list    /* Returns cGather * */
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| type_spec_list type_spec
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($2);
		     $$ = $1; }}
	;

type_spec
	: TYPE IDENT IS IDENT ';'
		{ if (yyMakeTree)             /* handles int, real, char... */
		     $$ = new cTypeDef((char *) $2,
				       new cNamedType((char *) $4)); }
	| TYPE IDENT IS array_type ';'
		{ if (yyMakeTree)
		     $$ = new cTypeDef((char *) $2, (cAbsTree *) $4); }
	| TYPE IDENT IS struct_type ';'
		{ if (yyMakeTree)
		     $$ = new cTypeDef((char *) $2, (cAbsTree *) $4); }
	;

struct_type
	: STRUCT '{' component_decl_list '}'
		{ if (yyMakeTree)
		     $$ = new cStructType(((cGather *) $3)->L); }
	;

component_decl_list
        : component_decl
		{ if (yyMakeTree) {
		     $$ = new cGather;
		     ((cGather *) $$)->L.Insert($1); }}
        | component_decl_list component_decl
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($2);
		     $$ = $1; }}
        ;

component_decl
	: IDENT IDENT ';'
		{ if (yyMakeTree)
		     $$ = new cStructMem((char *) $1, (char *) $2); }
        ;

array_type
	: ARRAY OF IDENT
		{ if (yyMakeTree) $$ = new cArrayType((char *) $3); }
	;


/* UC */
uc
	: input_port_part
	  output_port_part
	  name_sharing_part
	  var_specifier_part
	  init_comp_part
	  input_rule_part
	  comp_part
	  output_rule_part
		{ if (yyMakeTree) {
		     $$ = new cUC;
		     ((cUC *) $$)->InputPorts = ((cGather *) $1)->L;
		     ((cUC *) $$)->OutputPorts = ((cGather *) $2)->L;
		     ((cUC *) $$)->NSPorts = ((cGather *) $3)->L;
		     ((cUC *) $$)->LocalVars = ((cGather *) $4)->L;
		     ((cUC *) $$)->InitComp = (cStmt *) $5;
		     ((cUC *) $$)->InRules = ((cGather *) $6)->L;
		     ((cUC *) $$)->Comp = (cStmt *) $7;
		     ((cUC *) $$)->OutRules = ((cGather *) $8)->L; }}
 	;
	
input_port_part     /* Returns cGather * */
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| INPUTPORTS '{' port_decl_list '}'
		{ if (yyMakeTree) $$ = $3; }
	;

output_port_part     /* Returns cGather * */
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| OUTPUTPORTS '{' port_decl_list '}'
		{ if (yyMakeTree) $$ = $3; }
	;

port_decl_list         /* Returns cGather * */
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
        | port_decl_list port_decl
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($2);
		     $$ = $1; }}
        ;

port_decl
	: IDENT IDENT ';'
		{ if (yyMakeTree)
		     $$ = new cPortSpec((char *) $1, (char *) $2); }
        ;

name_sharing_part
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| SHARED_VAR '{' ns_list '}'
		{ if (yyMakeTree) $$ = $3; }
	;

ns_list
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	|  ns_list IDENT IDENT use_rule ';'
		{ if (yyMakeTree) {
		     ((cNSPortSpec *) $4)->TypeName = (char *) $2;
		     ((cNSPortSpec *) $4)->PortName = (char *) $3;
		     ((cGather *) $1)->L.Append($4);
		     $$ = $1; }}
	;

use_rule
	: READER
		{ if (yyMakeTree) $$ = new cNSPortSpec(NSReader); }
	| WRITER
		{ if (yyMakeTree) $$ = new cNSPortSpec(NSWriter); }
	;

var_specifier_part         /* Returns cGather * */
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| VAR '{' var_decl_list '}'
		{ if (yyMakeTree) $$ = $3; }
	;

var_decl_list         /* Returns cGather * */
        : /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
        | var_decl_list var_decl
		{ if (yyMakeTree) {
		      ((cGather *) $1)->L.Append($2);
		      $$ = $1; }}
        ;

var_decl
	: IDENT var_spec ';'
		{ if (yyMakeTree) {
		     ((cVarDecl *) $2)->TypeName = (char *) $1;
		     $$ = $2; }}
        ;

var_spec
	: IDENT
		{ if (yyMakeTree)
		     $$ = new cVarDecl((char *) $1); }
	| IDENT size_spec
		{ if (yyMakeTree)
		     $$ = new cVarDecl((char *) $1, (cSizeSpec *) $2); }
        ;

size_spec
	: array_size_list
		{ if (yyMakeTree) {
		     $$ = new cSizeSpec();
		     ((cSizeSpec *) $$)->ArrayBounds = ((cGather *) $1)->L; }}
	| '{' mem_size_list '}'
		{ if (yyMakeTree) {
		     $$ = new cSizeSpec();
		     ((cSizeSpec *) $$)->MemberSizes = ((cGather *) $2)->L; }}
	| array_size_list '{' mem_size_list '}'
		{ if (yyMakeTree) {
		     $$ = new cSizeSpec();
		     ((cSizeSpec *) $$)->ArrayBounds = ((cGather *) $1)->L; 
		     ((cSizeSpec *) $$)->MemberSizes = ((cGather *) $3)->L; }}
	;

mem_size_list
	: mem_size
		{ if (yyMakeTree) {
		      $$ = new cGather;
		      ((cGather *) $$)->L.Insert($1); }}
	| mem_size_list mem_size
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($2);
		     $$ = $1; }}
	;

mem_size
	: IDENT size_spec ';'
		{ if (yyMakeTree)
		     $$ = new cMemSize((char *) $1, (cSizeSpec *) $2); }
	;


array_size_list
	: array_size 
		{ if (yyMakeTree) {
		     $$ = new cGather;
		     ((cGather *) $$)->L.Insert($1); }}
	| array_size_list array_size
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($2);
		     $$ = $1; }}
	;

array_size
	: '[' additive_expr ']' 
		{ if (yyMakeTree) $$ = $2 ; }
	;

input_rule_part
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| INPUTRULES '{' in_rule_list '}'
		{ if (yyMakeTree) $$ = $3; }
	| INPUTRULES '{' '}'
		{ if (yyMakeTree) $$ = new cGather; }
	;

output_rule_part
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| OUTPUTRULES '{' out_rule_list '}'
		{ if (yyMakeTree) $$ = $3; }
	| OUTPUTRULES '{' '}'
		{ if (yyMakeTree) $$ = new cGather; }
	;

in_rule_list
	: in_rule_list OR_OP rule
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($3);
		     $$ = $1; }}
	| rule
		{ if (yyMakeTree) {
		     $$ = new cGather;
		     ((cGather *) $$)->L.Insert($1); }}
	;

out_rule_list
	: out_rule_list AND_OP rule
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($3);
		     $$ = $1; }}
	| rule
		{ if (yyMakeTree) {
		     $$ = new cGather;
		     ((cGather *) $$)->L.Insert($1); }}
	;

rule
	: guard_list '=' '>' bind_list
		{ if (yyMakeTree)
		     $$ = new cRule(((cGather *) $1)->L, ((cGather *) $4)->L);}
	| '{' guard_list '=' '>' bind_list ':' repl_list '}'
		{ if (yyMakeTree) {
		     $$ = new cRule(((cGather *) $2)->L, ((cGather *) $5)->L);
		     ((cRule *) $$)->Repls = ((cGather *) $7)->L; }}
	| '{' guard_list '=' '>' bind_list ':' repl_list ':' expr '}'
		{ if (yyMakeTree) {
		     $$ = new cRule(((cGather *) $2)->L, ((cGather *) $5)->L);
		     ((cRule *) $$)->Repls = ((cGather *) $7)->L;
		     ((cRule *) $$)->ReplCond = (cExpr *) $9; }}
	;

repl_list
	: repl_spec
		{ if (yyMakeTree) {
		     $$ = new cGather;
		     ((cGather *) $$)->L.Insert($1); }}
	| repl_list repl_spec
		{ if (yyMakeTree) {
		      ((cGather *) $1)->L.Append($2);
		      $$ = $1; }}
	;

repl_spec
	: '(' IDENT additive_expr ')'
		{ if (yyMakeTree)
		     $$ = new cReplSpec((char *) $2, (cExpr *) $3); }
	;

guard_list
	: guard
		{ if (yyMakeTree) {
		     $$ = new cGather;
		     ((cGather *) $$)->L.Insert($1); }}
	| guard_list ',' guard
		{ if (yyMakeTree) {
		      ((cGather *) $1)->L.Append($3);
		      $$ = $1; }}
	;

guard
	: guard_expr
		{ if (yyMakeTree) $$ = $1; }
	| '{' guard_expr ':' repl_list '}'
		{ if (yyMakeTree) {
		     ((cReplThing *) $2)->Repls = ((cGather *) $4)->L; 
		     $$ = $2; }}
	| '{' guard_expr ':' repl_list ':' expr '}'
		{ if (yyMakeTree) {
		     ((cReplThing *) $2)->Repls = ((cGather *) $4)->L;
		     ((cReplThing *) $2)->ReplCond = (cExpr *) $6; 
		     $$ = $2; }}
	;

guard_expr
	: expr CONS lhs_expr
		{ if (yyMakeTree)
		     $$ = new cArcInput((cExpr *) $1, (cExpr *) $3); }
	| expr
		{ if (yyMakeTree)
		     $$ = new cGuard((cExpr *) $1); }
	;

bind_list
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| bind_list binding
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($2);
		     $$ = $1; }}
	;

binding
	: bind_kind
		{ if (yyMakeTree) $$ = $1; }
	| '{' bind_kind ':' repl_list '}' ';'
		{ if (yyMakeTree) {
		     ((cReplThing *) $2)->Repls = ((cGather *) $4)->L;
		     $$ = $2; }}
	| '{' bind_kind ':' repl_list ':' expr '}' ';'
		{ if (yyMakeTree) {
		     ((cReplThing *) $2)->Repls = ((cGather *) $4)->L;
		     ((cReplThing *) $2)->ReplCond = (cExpr *) $6;
		     $$ = $2; }}
	;

bind_kind
	: assign_stmt
		{ if (yyMakeTree)
		     $$ = new cBinding((cAssignStmt *) $1); }
	| lhs_expr PROD expr ';'
		{ if (yyMakeTree)
		     $$ = new cArcOutput((cExpr *) $1, (cExpr *) $3); }
	;

init_comp_part    /* Returns a cBlockStmt * */
	: /* empty */
		{ if (yyMakeTree) $$ = 0; }
	| INITCOMP block
		{ if (yyMakeTree) $$ = $2; }
	;

comp_part    /* Returns a cBlockStmt * */
	: /* empty */
		{ if (yyMakeTree) $$ = 0; }
	| COMP block
		{ if (yyMakeTree) $$ = $2; }
	;

stmt_list
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| stmt_list stmt
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($2);
		     $$ = $1; }}
	;

stmt
	: call_stmt
		{ if (yyMakeTree) $$ = $1; }
	| assign_stmt
		{ if (yyMakeTree) $$ = $1; }
	| if_stmt
		{ if (yyMakeTree) $$ = $1; }
	| while_stmt
		{ if (yyMakeTree) $$ = $1; }
	| null_stmt
		{ if (yyMakeTree) $$ = $1; }
	;

null_stmt
	: ';'
		{ if (yyMakeTree) $$ = new cNullStmt; }
	;

lhs_expr
	: IDENT
		{ if (yyMakeTree) $$ = new cIdent((char *) $1); }
	| lhs_expr '.' IDENT
		{ if (yyMakeTree)
		     $$ = new cSelectOp((char *) $3, (cExpr *) $1); }
	| lhs_expr '[' additive_expr ']'
		{ if (yyMakeTree)
		     $$ = new cArrayIndOp((cExpr *) $1, (cExpr *) $3); }
	;

assign_stmt
	: lhs_expr '=' expr ';'
		{ if (yyMakeTree)
		     $$ = new cAssignStmt((cExpr *) $1, (cExpr *) $3); }
	;

if_stmt
	: IF '(' expr ')' block
		{ if (yyMakeTree)
		     $$ = new cIfStmt((cExpr *) $3, (cStmt *) $5); }
	| IF '(' expr ')' block ELSE block
		{ if (yyMakeTree)
		     $$ = new cIfStmt((cExpr *) $3, (cStmt *) $5,
		                      (cStmt *) $7); }
	;

block
	: stmt
		{ if (yyMakeTree) $$ = $1; }
	| '{' stmt_list '}'
		{ if (yyMakeTree) $$ = new cBlockStmt(((cGather *) $2)->L); }
	;

while_stmt
	: WHILE '(' expr ')' block
		{ if (yyMakeTree)
		     $$ = new cWhileStmt((cExpr *) $3, (cStmt *) $5); }
	;

call_stmt
	: IDENT '(' ')' ';'
		{ if (yyMakeTree) $$ = new cCallStmt((char *) $1); }
	| IDENT '(' argument_expr_list ')' ';'
		{ if (yyMakeTree)
		     $$ = new cCallStmt((char *) $1, ((cGather *) $3)->L); }
	;

argument_expr_list
	: expr
		{ if (yyMakeTree)  {
		     $$ = new cGather;
		     ((cGather *) $$)->L.Insert($1); }}
	| argument_expr_list ',' expr
		{ if (yyMakeTree) {
		     ((cGather *) $1)->L.Append($3);
		     $$ = $1; }}
	;

/* name sharing relation initial computations */

ns_init_comp
	: shared_var_part
	  var_specifier_part
	  init_comp_part
		{ if (yyMakeTree)
		     $$ = new cNSRel(((cGather *) $1)->L, ((cGather *) $2)->L,
				     (cStmt *) $3); }
	;

shared_var_part  /* Returns cGather * */
	: /* empty */
		{ if (yyMakeTree) $$ = new cGather; }
	| SHARED_VAR '{' var_decl_list '}'
		{ if (yyMakeTree) $$ = $3; }
	;

/* expressions */

numbers
	: INTCONST
		{ if (yyMakeTree) $$ = $1; }
	| REALCONST
		{ if (yyMakeTree) $$ = $1; }
	;

primary_expr
	: IDENT
		{ if (yyMakeTree) $$ = new cIdent((char *) $1); }
	| '(' expr ')'
		{ if (yyMakeTree) $$ = $2; } /* AbsTree knows precedence! */
	;

func_call
	: IDENT '(' ')'
		{ if (yyMakeTree) $$ = new cFuncCallOp((char *) $1); }
	| IDENT '(' argument_expr_list ')'
		{ if (yyMakeTree)
		     $$ = new cFuncCallOp((char *) $1,
					  ((cGather *) $3)->L); }
	;

postfix_expr
	: IDENT '.' IDENT
		{ if (yyMakeTree)
		     $$ = new cSelectOp((char *) $3,
					new cIdent((char *) $1)); }
	| postfix_expr '.' IDENT
		{ if (yyMakeTree)
		     $$ = new cSelectOp((char *) $3, (cExpr *) $1); }
	| IDENT '[' additive_expr ']'
		{ if (yyMakeTree) 
		     $$ = new cArrayIndOp(new cIdent ((char *) $1),
					  (cExpr *) $3); }
	| postfix_expr '[' additive_expr ']'
		{ if (yyMakeTree)
		     $$ = new cArrayIndOp((cExpr *) $1, (cExpr *) $3); }
	| func_call
		{ if (yyMakeTree) $$ = $1; }
	;

unary_expr
	: primary_expr
		{ if (yyMakeTree) $$ = $1; }
        | numbers
		{ if (yyMakeTree) $$ = $1; }
	| postfix_expr
		{ if (yyMakeTree) $$ = $1; }
	| '-' unary_expr
		{ if (yyMakeTree) $$ = new cUnaryMinusOp((cExpr *) $2); }
	| '!' unary_expr
		{ if (yyMakeTree) $$ = new cNotOp((cExpr *) $2); }
	;

multiplicative_expr
	: unary_expr
		{ if (yyMakeTree) $$ = $1; }
	| multiplicative_expr '*' unary_expr
		{ if (yyMakeTree)
		     $$ = new cMultOp((cExpr *) $1, (cExpr *) $3); }
	| multiplicative_expr '/' unary_expr
		{ if (yyMakeTree)
		     $$ = new cDivOp((cExpr *) $1, (cExpr *) $3); }
	| multiplicative_expr '%' unary_expr
		{ if (yyMakeTree)
		     $$ = new cModOp((cExpr *) $1, (cExpr *) $3); }
	;

additive_expr
	: multiplicative_expr
		{ if (yyMakeTree) $$ = $1; }
	| additive_expr '+' multiplicative_expr
		{ if (yyMakeTree)
		     $$ = new cPlusOp((cExpr *) $1, (cExpr *) $3); }
	| additive_expr '-' multiplicative_expr
		{ if (yyMakeTree)
		     $$ = new cMinusOp((cExpr *) $1, (cExpr *) $3); }
	;

relational_expr
	: additive_expr
		{ if (yyMakeTree) $$ = $1; }
	| additive_expr '<' additive_expr
		{ if (yyMakeTree)
		     $$ = new cLTOp((cExpr *) $1, (cExpr *) $3); }
	| additive_expr '>' additive_expr 
		{ if (yyMakeTree)
		     $$ = new cGTOp((cExpr *) $1, (cExpr *) $3); }
	| additive_expr LE_OP additive_expr 
		{ if (yyMakeTree)
		     $$ = new cLEOp((cExpr *) $1, (cExpr *) $3); }
	| additive_expr GE_OP additive_expr 
		{ if (yyMakeTree)
		     $$ = new cGEOp((cExpr *) $1, (cExpr *) $3); }
	;

equality_expr
	: relational_expr
		{ if (yyMakeTree) $$ = $1; }
	| equality_expr EQ_OP relational_expr
		{ if (yyMakeTree)
		     $$ = new cEqOp((cExpr *) $1, (cExpr *) $3); }
	| equality_expr NE_OP relational_expr
		{ if (yyMakeTree)
		     $$ = new cNEOp((cExpr *) $1, (cExpr *) $3); }
	;

logical_and_expr
	: equality_expr
		{ if (yyMakeTree) $$ = $1; }
	| logical_and_expr AND_OP equality_expr
		{ if (yyMakeTree)
		     $$ = new cAndOp((cExpr *) $1, (cExpr *) $3); }
	;

logical_or_expr
	: logical_and_expr
		{ if (yyMakeTree) $$ = $1; }
	| logical_or_expr OR_OP logical_and_expr
		{ if (yyMakeTree) 
		     $$ = new cOrOp((cExpr *) $1, (cExpr *) $3); }
	;

new_call
	: NEW IDENT
		{ if (yyMakeTree)
		     $$ = new cNewExpr((char *) $2); }
	| NEW IDENT size_spec
		{ if (yyMakeTree)
		     $$ = new cNewExpr((char *) $2, (cSizeSpec *) $3); }
	;

expr
	: logical_or_expr
		{ if (yyMakeTree) $$ = $1; }
	| new_call
		{ if (yyMakeTree) $$ = $1; }
	;

%%

int yyDoParse(char *Code, char *Buf, int DoListing, cAbsTree *&TreeRoot,
	      int MakeTree)
{
   int ParseOK;

   yyset_input(Code, Buf, DoListing);
   yyMakeTree = MakeTree;

   if (yyparse()) {
      ParseOK = 0;
      TreeRoot = 0;
   } else {
      ParseOK = 1;
      TreeRoot = AbsTreeHead;
   }

   if (!MakeTree) TreeRoot = 0;

   return ParseOK;
}

