#include <config.h>
#include <stdarg.h>
#include <stream.h>
#include "../misc/general.h"
#include "../exmodel/object.h"
#include "stype.h"

// Structural types are used to define types so that C declarations for
// them can be created and to allow automatic conversion of compatible
// types.

// A structural type is a linked structure that shows all of the types
// that make up a compound type.  For example, an array of integers is
// represented by an array class instance with a pointer to an int class
// instance.

// Storage management is complex since more than one symtab entry may
// point to the same structural type.  Consider the code
//    type foo is int; type bar is foo; type hoo is int;
// The symtab entries for both foo and bar will point to the same int
// instance, but hoo will not.  Because of this symtab entries should
// not delete STypes.  Instead they are kept on a linked list used
// for freeing storage.


  // Call this before creating any cStype instances
void InitSTypes()
{
   cSType::Head = 0;
}

  // Call this to free storage used by cStype instances
void FreeSTypes()
{
   cSType *tmp;

   while (cSType::Head != 0) {
      tmp = cSType::Head;
      cSType::Head = cSType::Head->Next;
      delete tmp;
   }
}


cSType::cSType()
{
   Next = Head;   // Add to list of things to free later
   Head = this;
}


cIntSType::cIntSType()
{

   if (DebugLevel >= 7) {
      cout << "cIntSType created\n";
   }
}


cIntSType::~cIntSType()
{

   if (DebugLevel >= 7) {
      cout << "cIntSType destroyed\n";
   }
}


void cIntSType::List()
{
  cout << "int";
}


cRealSType::cRealSType()
{

   if (DebugLevel >= 7) {
      cout << "cRealSType created\n";
   }
}


cRealSType::~cRealSType()
{

   if (DebugLevel >= 7) {
      cout << "cRealSType destroyed\n";
   }
}


void cRealSType::List()
{
   cout << "real";
}


cCharSType::cCharSType()
{

   if (DebugLevel >= 7) {
      cout << "cCharSType created\n";
   }
}


cCharSType::~cCharSType()
{

   if (DebugLevel >= 7) {
      cout << "cCharSType destroyed\n";
   }
}


void cCharSType::List()
{
   cout << "char";
}


cPortSType::cPortSType()
{
   BaseType = 0;

   if (DebugLevel >= 7) {
      cout << "cPortSType created\n";
   }
}


cPortSType::~cPortSType()
{

   if (DebugLevel >= 7) {
      cout << "cPortSType destroyed\n";
   }
}


void cPortSType::List()
{
   if (BaseType == 0) Die("BaseType in cPortSType is zero.");

   if (IsInputPort)
     cout << "input port for ";
   else
     cout << "output port for ";

   BaseType->List();
}


cArraySType::cArraySType()
{
   BaseType =0;

   if (DebugLevel >= 7) {
      cout << "cArraySType created\n";
   }
}


cArraySType::~cArraySType()
{

   if (DebugLevel >= 7) {
      cout << "cArraySType destroyed\n";
   }
}


void cArraySType::List()
{
   if (BaseType != 0) {
      cout << "array of ";
      BaseType->List();
   } else 
      Die("BaseType in cArraySType is zero.");
}


cStructMemSType::cStructMemSType()
{

   if (DebugLevel >= 7) {
      cout << "cStructMemSType created\n";
   }
}


cStructMemSType::~cStructMemSType()
{

   if (DebugLevel >= 7) {
      cout << "cStructMemSType destroyed\n";
   }
}


void cStructMemSType::List()
{
   if (Type != 0) {
      cout << "   " <<  Ident << " is a ";
      Type->List();
      cout << "\n";
   } else 
      Die("Type in cStructMemSType is zero.");
}


cStructSType::cStructSType()
{

   if (DebugLevel >= 7) {
      cout << "cStructSType created\n";
   }
}


cStructSType::~cStructSType()
{

   if (DebugLevel >= 7) {
      cout << "cStructSType destroyed\n";
   }
}


void cStructSType::List()
{
   cObject *p;
   cObIterator i;

   cout << "struct {\n";
   for (p = i.Init(Members); i.MoreLeft(); p = i.Next())
      p->List();
   cout << "} \n";
}


   // Get struct member called Name.  0 if not there.

cStructMemSType *cStructSType::GetMember(MString Name)
{
   cObject *p;
   cObIterator i;
   cStructMemSType *Mem;

   for (p = i.Init(Members); i.MoreLeft(); p = i.Next()) {
      Mem = (cStructMemSType *) p;
      if ((char *) Mem->Ident == Name) return Mem;
   }

   return 0;
}


  // Return 1 if two structural types are the same, 0 otherwise.

int AreSTypesEq(cSType *s1, cSType *s2)
{
   cObIterator i1, i2;
   cObject *p1, *p2;
   cStructMemSType *m1, *m2;

   if (s1 == 0 || s2 == 0) Die("Null pointer deref in AreSTypesEq");

   if (s1 == s2) return 1;    // They point to the same object.

   if (s1->IsA() == IntSType) {
      if (s2->IsA() == IntSType) return 1;
      else return 0;
   } else if (s1->IsA() == RealSType) {
      if (s2->IsA() == RealSType) return 1;
      else return 0;
   } else if (s1->IsA() == CharSType) {
      if (s2->IsA() == CharSType) return 1;
      else return 0;
   } else if (s1->IsA() == ArraySType) {
      if (s2->IsA() == ArraySType)
	return AreSTypesEq(((cArraySType *) s1)->BaseType,
			   ((cArraySType *) s2)->BaseType);
      else
	return 0;
   } else if (s1->IsA() == PortSType) {
      if (s2->IsA() == PortSType)
	return AreSTypesEq(((cPortSType *) s1)->BaseType,
			   ((cPortSType *) s2)->BaseType);
      else
	return 0;
   } else if (s1->IsA() == StructSType) {
      if (((cStructSType *) s1)->Members.Length() !=
	  ((cStructSType *) s2)->Members.Length()) return 0;
      p1 = i1.Init(((cStructSType *) s1)->Members);
      p2 = i2.Init(((cStructSType *) s2)->Members);
      while (i1.MoreLeft()) {
	 m1 = (cStructMemSType *) p1;
	 m2 = (cStructMemSType *) p2;
	 if (!AreSTypesEq(m1->Type, m2->Type)) return 0;
	 p1 = i1.Next();
	 p2 = i2.Next();
      }
      return 1;
   } else
     Die("Internal error: Unknown type is AreSTypesEq.\n");

   Die("Internal error: AreSTypesEq should not reach here.\n");
   return 0;  // Shutup, gcc.
}



  // return 1 if s is of one of the class types given by the ...
  // arguments which must be of type ClassId.  Return 0 otherwise.
  // The list must end with a zero.

  //   Example: (void) IsSTypeOneOf(s, IntSType, 0);

int IsSTypeOneOf(cSType *s, ...)
{
   va_list ap;
   ClassId c1, c2;

   c1 = s->IsA();
   va_start(ap, s);

   while ((c2 = va_arg(ap, ClassId)) != 0) 
     if (c1 == c2) {
	va_end(ap);
	return 1;
     }

   va_end(ap);
   return 0;

}



  // Return 1 if it is legal to assign FromSType to ToSType or to pass by value
  // FromSType as an actual parameter bound to ToSType as a formal.  Return
  // 0 otherwise.

int CanAssignSTypes(cSType *ToSType, cSType *FromSType)
{

   // We allow any assignment among scalars.

   if (IsSTypeOneOf(ToSType, IntSType, CharSType, RealSType, 0) &&
       IsSTypeOneOf(FromSType, IntSType, CharSType, RealSType, 0))
     return 1;

   // We allow assignment among identical arrays and structs

   if (IsSTypeOneOf(ToSType, ArraySType, StructSType, 0) &&
       IsSTypeOneOf(FromSType, ArraySType, StructSType, 0))
     return AreSTypesEq(FromSType, ToSType);

   // We allow no other assignments

   return 0;
}
