#include <config.h>

// pass2 performs the following tasks

//   Propagates type through expressions checking operator types.
//   Find all undeclared idents (outside of declarations).

#include <assert.h>
#include <setjmp.h>
#include <stdlib.h>
#include <stream.h>
#include <ctype.h>
#include <misc/general.h>
#include <exmodel/abstree.h>
#include <exmodel/exmodel.h>
#include <symtab/symtab.h>
#include <lib/UidTable.h>
#include <lib/MsgUI.h>
#include <lib/Syntax.h>
#include "pass.h"

static cSType *PtrToIntSType;  // Since all relops need it.

  // Order types as follows: real, int, char.  Return t1 or t2,
  // whichever is higher in this list.  Used for deciding that
  // the type of something like (2 + 4.5) is real, not int. 

static cSType *MixedMode(cSType *t1, cSType *t2)
{
   if (t1->IsA() == ::RealSType) return t1;
   if (t2->IsA() == ::RealSType) return t2;
   if (t1->IsA() == ::IntSType) return t1;
   if (t2->IsA() == ::IntSType) return t2;
   if (t1->IsA() == ::CharSType) return t1;
   if (t2->IsA() == ::CharSType) return t2;

   Die("Internal error: bad type in MixedMode");
   return 0;  // Quiets compiler warning.
}


   // Check Polymorphic with name FuncName.  Pass 1 for FromFuncOp
   // if checking as a functions call.  Pass 0 if checking as a
   // procedure call.  Provide actual parameters in Actuals.
   // Returns the function's return type in ReturnType and returns
   // 1 if call is OK; 0 otherwise.

static int CheckPoly(MString &FuncName, int FromFuncOp, cObList &Actuals,
	      cSType **ReturnType)
{
   cExpr *e;

   // Nasty case by case treatment of special builtin types that
   // have return types bases on parameter types, accept arguments
   // of multiple types, or varying numbers of arguments.

   if (FuncName == MString("size")) {         // Array size function

      // Must of one arg of type array; returns int
      if (FromFuncOp) {
         if (Actuals.Length() == 1) {
            e = (cExpr *) Actuals.First();
            if (IsSTypeOneOf(e->StructuralType, ArraySType, 0)) {
               *ReturnType = PtrToIntSType;
               return 1;
            } else 
               return 0;
         } else
            return 0;
      } else 
         return 0;

   } else if (FuncName == MString("copy")) {

      if (FromFuncOp) {
         if (Actuals.Length() == 1) {
            e = (cExpr *) Actuals.First();
            if (IsSTypeOneOf(e->StructuralType, ArraySType, StructSType, 0)) {
               *ReturnType = e->StructuralType;
               return 1;
            } else 
               return 0;
         } else
            return 0;
      } else 
         return 0;

   } else if (FuncName == MString("free")) {

      if (!FromFuncOp) {
         if (Actuals.Length() == 1) {
            e = (cExpr *) Actuals.First();
            if (IsSTypeOneOf(e->StructuralType, ArraySType, StructSType, 0)) {
               *ReturnType = 0;
               return 1;
            } else 
               return 0;
         } else
            return 0;
      } else 
         return 0;

   } else
     Die("Internal error:  Unknown polymorph in CheckPoly.\n");

  Die("Internal error: CheckPoly should not reach here.\n");
  return 0; // Shutup, gcc.
}      


   // Check if expr is something that can be assigned to-- like
   // an ident or structure dref or array ref

static int IsComplexExpr(cExpr *e)
{
   switch (e->IsA()) {

    case ::Ident:
    case ::SelectOp:
    case ::ArrayIndOp:
      return 0;

    default:
      return 1;
   }
}


   // Check if function parameter lists match in type and number.
   // Return 1 if yes and 0 if no.

         // Actuals - list of cExpr
         // Formals - list of cSEFuncArg

static int CheckArgs(cObList &Actuals, cObList &Formals)
{
   cObIterator iap;
   cObIterator ifp;
   cObject *pa, *pf;
   cExpr *ea;
   cSEFuncArg *ef;

   if (Actuals.Length() != Formals.Length()) return 0;

   pa = iap.Init(Actuals);
   ea = (cExpr *) pa;
   pf = ifp.Init(Formals);
   ef = (cSEFuncArg *) pf;

   while (iap.MoreLeft() && ifp.MoreLeft()) {

      if (!AreSTypesEq(ef->Type, ea->StructuralType)) return 0;

      if (!(ef->IsByValue))
	if (IsComplexExpr(ea)) return 0;   // got to be a legal LHS

      pa = iap.Next();
      ea = (cExpr *) pa;
      pf = ifp.Next();
      ef = (cSEFuncArg *) pf;
   }

   return 1;
}

static int LegalIdent(MString S)
{
   int i;

   if (S.length() > 0 && isalpha(S[0])) {
      for (i = 1; i < S.length(); i++) {
         if (!isalnum(S[i]) && S[i] != '_')
            return 0;
      }
      return 1;
   } else 
      return 0;
}


static MString FindPortVariableName(cExpr *Expr) {
   if      (Expr->IsA() == ::Ident) 
      return ((cIdent *)Expr)->PrintName;

   else if (Expr->IsA() == ::ArrayIndOp) 
      return FindPortVariableName( ((cBinOp *)Expr)->Left );

  Die("Internal error: FindPortVariableName should not reach here.\n");
  return ""; // Shutup, gcc.
}


static int PortIndicesCount(cExpr *Expr) {
   if      (Expr->IsA() == ::Ident) 
      return 0;

   else if (Expr->IsA() == ::ArrayIndOp) 
      return 1 + PortIndicesCount( ((cBinOp *)Expr)->Left );

  Die("Internal error: PortIndicesCount should not reach here.\n");
  return 0; // Shutup, gcc.
}



void cIdent::Pass2()
{
   cSymTabEntry *Entry;

   Entry = GetSymbol(PrintName, this, SEVariable, SEPortName,
		     SESharedVar, SECreP, 0);

   if (Entry == 0) {
      YYUIMessage(UID, ErrorLevel, "Ident \"%s\" not declared or wrong type",
		  (char *) PrintName);
      longjmp(PassEnv, 1);
   }

   switch (Entry->IsA()) {
    case ::SEVariable:
      StructuralType = ((cSEVariable *) Entry)->Type;
      break;
    case ::SEPortName:
      StructuralType = ((cSEPortName *) Entry)->Type;
      break;
    case ::SESharedVar:
      StructuralType = ((cSESharedVar *) Entry)->Type;
      break;
    case ::SECreP:
      StructuralType = ((cSECreP *) Entry)->Type;
      break;
    default:
      Die("Internal error: unexpected case in cIdent::pass2");
   }
   

}


void cIntConst::Pass2()
{

   StructuralType = PtrToIntSType;
}


void cRealConst::Pass2()
{
   cSETypeName *Entry;


   Entry = (cSETypeName *) GetSymbol("real", this, SETypeName, 0);
   if (Entry == 0)
     Die("Internal error:  real not declared in cIntConst::pass2");
   StructuralType = Entry->Type;
}


static void SemCheckSizeSpec(cSizeSpec *SizeSpec, cSType *Type);

void cNewExpr::Pass2()
{
   cSETypeName *SETypeName;

   SETypeName = (cSETypeName *) GetSymbol(TypeName, this, ::SETypeName, 0);

   if (SETypeName == 0) {
      YYUIMessage(UID, ErrorLevel, "Type \"%s\" in new expression is "
        "undefined.\n", (char *)TypeName);
      longjmp(PassEnv, 1);
   }

   if (Sizes != 0) {
      Sizes->Pass2();
      SemCheckSizeSpec(Sizes, SETypeName->Type);
   } else {
      // arrays and structs need an empty size spec for translation
      switch (SETypeName->Type->IsA()) {
      case ::CharSType:
      case ::IntSType:
      case ::RealSType:
         break;
      case ::ArraySType:
      case ::StructSType:
         Sizes = new cSizeSpec;
         Sizes->Parent = this;
         break;
      default:
         break;
      }
   }

   StructuralType = SETypeName->Type;
}


void cFuncCallOp::Pass2()
{
   cObIterator i;
   cObject *p;
   cSType *r;


   for (p = i.Init(Args); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }


   cSEFuncName *SEFuncName = 
     (cSEFuncName *)GetSymbol(FuncName, this, ::SEFuncName, 0);
   if (SEFuncName == 0) {
      YYUIMessage(UID, ErrorLevel, "Function \"%s\" not declared\n",
		  (char *) FuncName);
      longjmp(PassEnv, 1);
   }

   if (SEFuncName->IsPolyMorphic) {
      if (!CheckPoly(FuncName, 1, Args, &r)) {
	 YYUIMessage(UID, ErrorLevel, "Function \"%s\" misused\n",
		     (char *) FuncName);
	 longjmp(PassEnv, 1);
      }
      StructuralType = r;
   } else {
      if(!SEFuncName->DontCheckArgs) {
         if (!CheckArgs(Args, SEFuncName->ArgList)) {
	    YYUIMessage(UID, ErrorLevel,
              "function argument type or number mismatch\n");
            longjmp(PassEnv, 1);
         }
      }
      if (SEFuncName->ReturnType == 0) {
         YYUIMessage(UID, ErrorLevel, "Procedure call used where function call "
           "expected.\n");
         longjmp(PassEnv, 1);
      }
      StructuralType = SEFuncName->ReturnType;
   }
}


void cNotOp::Pass2()
{


   Left->Pass2();


   if (!IsSTypeOneOf(Left->StructuralType, IntSType, 0)) {
      YYUIMessage(UID, ErrorLevel, "Wrong type for NOT\n");
      longjmp(PassEnv, 1);
   }

   StructuralType = Left->StructuralType;
}


void cUnaryMinusOp::Pass2()
{


   Left->Pass2();


   if (!IsSTypeOneOf(Left->StructuralType, IntSType, RealSType, 0)) { 
      YYUIMessage(UID, ErrorLevel, "Wrong type for UNARY MINUS\n");
      longjmp(PassEnv, 1);
   }

   StructuralType = Left->StructuralType;
}


void cOrOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, IntSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, IntSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for OR\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = PtrToIntSType;
}


void cAndOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, IntSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, IntSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for AND\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = PtrToIntSType;
}


void cNEOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, RealSType, IntSType, CharSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, RealSType, IntSType, CharSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for NOT_EQUAL\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = PtrToIntSType;
}


void cEqOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, RealSType, IntSType, CharSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, RealSType, IntSType, CharSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for EQUAL\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = PtrToIntSType;
}


void cGEOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, RealSType, IntSType, CharSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, RealSType, IntSType, CharSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for GT_OR_EQUAL\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = PtrToIntSType;
}


void cLEOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, RealSType, IntSType, CharSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, RealSType, IntSType, CharSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for LT_OR_EQUAL\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = PtrToIntSType;
}


void cLTOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, RealSType, IntSType, CharSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, RealSType, IntSType, CharSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for LESS_THAN\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = PtrToIntSType;
}


void cGTOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, RealSType, IntSType, CharSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, RealSType, IntSType, CharSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for GREATER_THAN\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = PtrToIntSType;
}


void cMinusOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, RealSType, IntSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, RealSType, IntSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for MINUS\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = MixedMode(Left->StructuralType, Right->StructuralType);
}


void cPlusOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, RealSType, IntSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, RealSType, IntSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for PLUS\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = MixedMode(Left->StructuralType, Right->StructuralType);
}


void cModOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, IntSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, IntSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for MOD\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = PtrToIntSType;
}


void cDivOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, RealSType, IntSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, RealSType, IntSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for DIVIDE\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = MixedMode(Left->StructuralType, Right->StructuralType);
}


void cMultOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if(!IsSTypeOneOf(Left->StructuralType, RealSType, IntSType, 0) ||
      !IsSTypeOneOf(Right->StructuralType, RealSType, IntSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for TIMES\n");
	longjmp(PassEnv, 1);
     }
   
   StructuralType = MixedMode(Left->StructuralType, Right->StructuralType);
}


void cSelectOp::Pass2()
{
   cStructSType *t;
   cStructMemSType *Mem;

   Left->Pass2();

   if (!IsSTypeOneOf(Left->StructuralType, StructSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for STRUCT_SEL\n");
	longjmp(PassEnv, 1);
     }

   // Check to see if member is defined and get its SType.

   t = ((cStructSType *) (Left->StructuralType));
   Mem = t->GetMember(SelName);
   if (Mem ==0)
     {
	YYUIMessage(UID, ErrorLevel, "Struct Member \"%s\" undefined\n",
		    (char *) SelName);
	longjmp(PassEnv, 1);
     }

   StructuralType = Mem->Type;
}


void cArrayIndOp::Pass2()
{


   Left->Pass2();
   Right->Pass2();


   if (!IsSTypeOneOf(Right->StructuralType, IntSType, 0)) 
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for INDEX\n");
	longjmp(PassEnv, 1);
     }

   if(IsSTypeOneOf(Left->StructuralType, ArraySType, 0)) {
      StructuralType = ((cArraySType *) (Left->StructuralType))->BaseType;
      return;
   } else if(IsSTypeOneOf(Left->StructuralType, PortSType, 0)) {
      StructuralType = Left->StructuralType;
      return;
   } else {
      YYUIMessage(UID, ErrorLevel, "Wrong type for ARRAY_REF\n");
      longjmp(PassEnv, 1);
   }
   
}


void cFuncArgSpec::Pass2()
{
   cSETypeName *SETypeName = 
     (cSETypeName *)GetSymbol(TypeName, this, ::SETypeName, 0);
   if (SETypeName == 0) {
      YYUIMessage(UID, ErrorLevel, "Type \"%s\" for argument \"%s\" is undefined.\n",
        (char *)TypeName, (char *)ArgName);
      longjmp(PassEnv, 1);
   }
}


void cFuncSig::Pass2()
{
   cObIterator i, j;
   cObject *p;

   if (ReturnType != MString("")) {
      cSETypeName *SETypeName = 
        (cSETypeName *)GetSymbol(ReturnType, this, ::SETypeName, 0);
      if (SETypeName == 0) {
         YYUIMessage(UID, ErrorLevel, "Return type \"%s\" for Function \"%s\" is "
           "undefined.\n", (char *)ReturnType, (char *)FuncName);
         longjmp(PassEnv, 1);
      }
   }

   for (p = i.Init(Args); i.MoreLeft(); p = i.Next()) {
      ((cFuncArgSpec *) p)->Pass2();
   }

   for (cFuncArgSpec *FuncArgI = (cFuncArgSpec *)i.Init(Args); i.MoreLeft(); FuncArgI = (cFuncArgSpec *)i.Next())
      for (cFuncArgSpec *FuncArgJ = (cFuncArgSpec *)j.Init(Args); j.MoreLeft(); FuncArgJ = (cFuncArgSpec *)j.Next())
         if (FuncArgI != FuncArgJ && (char *) FuncArgI->ArgName == FuncArgJ->ArgName) {
            YYUIMessage(UID, ErrorLevel, "Function argument name \"%s\" is re-used "
              "in Function \"%s\".\n", (char *)FuncArgI->ArgName, (char *)FuncName);
            longjmp(PassEnv, 1);
         }
}


void cArc::Pass2()
{
   cObIterator i;
   cObject *p;


   for (p = i.Init(CallVarList); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(NodeVarList); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(PortVarList); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(CallExprList); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(NodeExprList); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(PortExprList); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   static const char * const ArcDescription[7] = {
     "<INTERNAL ERROR>",
     "<variable name>", 
     ". => .",  
     ". => ..",
     ".. => .",
     ".. => ..",    
     "<empty>"} ;
   cSType *FromSType, *ToSType;

   cAbsTree *FromNode = FindUidTable(FromUID);
   ClassId FromClassId = FromNode->IsA();
   cAbsTree *ToNode = FindUidTable(ToUID);
   ClassId ToClassId = ToNode->IsA();

   // check From Port Variable
   switch (FromClassId) {
   case ::UC:
   {
      cUC *FromUCNode = (cUC *)FromNode;
      cSymTabEntry *SymTabEntry = FromUCNode->SymbolTable->Get(FromPort);
      if (SymTabEntry != 0) {
         if (SymTabEntry->IsA() == ::SEPortName) {
            if (((cSEPortName *)SymTabEntry)->IsInputPort == 0) {
               cPortSType *FromPortSType = (cPortSType *)((cSEPortName *)SymTabEntry)->Type;
               FromSType = FromPortSType->BaseType;
            } else {
               YYUIMessage(UID, ErrorLevel, "Variable \"%s\" is "
                 "declared in UC Node \"%s\" but not as a output_ports variable.\n", 
                 (char *)FromPort, 
                 (FromUCNode->Name == MString("") ? "untitled" : (char *) FromUCNode->Name));
               longjmp(PassEnv, 1);
            }
         } else if (SymTabEntry->IsA() == ::SESharedVar) {
            FromSType = ((cSESharedVar *)SymTabEntry)->Type;
         } else {
            YYUIMessage(UID, ErrorLevel, "Variable \"%s\" is "
              "declared in UC Node \"%s\" but not as a output_ports variable "
              "or as a Shared Variable.\n", (char *)FromPort, 
              (FromUCNode->Name == MString("") ? "untitled" : (char *)FromUCNode->Name));
            longjmp(PassEnv, 1);
         }
      } else {
         YYUIMessage(UID, ErrorLevel, "Output Port Variable \"%s\" is "
           "not declared in UC Node \"%s\".\n", (char *)FromPort, 
           (FromUCNode->Name == MString("") ? "untitled" : (char *)FromUCNode->Name));
         longjmp(PassEnv, 1);
      }
    }  // prevents lame compiler bitch
      break;
   case ::CallNode:
   {
      cCallNode *FromCallNode = (cCallNode *)FromNode;
      cSEGraph *SEGraph = (cSEGraph *)
        GetSymbol(FromCallNode->CalledGraph, this, ::SEGraph, 0);
      if (SEGraph != 0) {
         cSymTabEntry *SymTabEntry = SEGraph->Graph->SymbolTable->Get(FromPort);
         if (SymTabEntry == 0) {
            YYUIMessage(UID, ErrorLevel, "Output Interface Node Variable \"%s\" is "
              "not declared in Graph \"%s\".\n", (char *)FromPort, 
              (char *)FromCallNode->CalledGraph);
            longjmp(PassEnv, 1);
         } else if (SymTabEntry->IsA() != ::SEIntNode) {  
            YYUIMessage(UID, ErrorLevel, "Variable \"%s\" is "
              "declared in Graph \"%s\" but not as a Output Interface Node.\n", 
              (char *)FromPort, 
              (char *)FromCallNode->CalledGraph);
            longjmp(PassEnv, 1);
         } else if (((cSEIntNode *)SymTabEntry)->IsInputNode == 1) {
            YYUIMessage(UID, ErrorLevel, "Interface Node Variable \"%s\" is "
              "declared in Graph \"%s\" but not as an output.\n", (char *)FromPort, 
              (char *)FromCallNode->CalledGraph);
            longjmp(PassEnv, 1);
         }
         FromSType = ((cSEIntNode *)SymTabEntry)->Type;
      } else {
         YYUIMessage(UID, ErrorLevel, "Graph \"%s\" is undefined.\n",
           (char *)FromCallNode->CalledGraph);
         longjmp(PassEnv, 1);
      }
   }
      break;
   case ::IntNode:
   {
      cSEIntNode *SEIntNode = (cSEIntNode *)
        GetSymbol(((cIntNode *)FromNode)->Name, this, ::SEIntNode, 0);
      // or could have called Parent->SymbolTable->Get(...);
      FromSType = SEIntNode->Type;
   }
      break; 
   default:
     assert (0); // We should never get here.
     break;
   }

   // check To Port Variable
   switch (ToClassId) {
   case ::UC:       
   {
      cSymTabEntry *SymTabEntry; 
      cUC *ToUCNode = (cUC *)ToNode;
      if (FromClassId == IntNode)
         SymTabEntry = ToUCNode->SymbolTable->Get(FromPort);
      else
         SymTabEntry = ToUCNode->SymbolTable->Get(ToPort);
      if (SymTabEntry != 0) {
         if (SymTabEntry->IsA() == ::SEPortName) {
            if (((cSEPortName *)SymTabEntry)->IsInputPort == 1) {
               cPortSType *ToPortSType = (cPortSType *)((cSEPortName *)SymTabEntry)->Type;
               ToSType = ToPortSType->BaseType;
            } else {
               YYUIMessage(UID, ErrorLevel, "Variable \"%s\" is "
                 "declared in UC Node \"%s\" but not as a input_ports variable.\n", 
                 (char *)ToPort, 
                 (ToUCNode->Name == MString("") ? "untitled" : (char *)ToUCNode->Name));
               longjmp(PassEnv, 1);
            }
         } else {
            YYUIMessage(UID, ErrorLevel, "Variable \"%s\" is "
              "declared in UC Node \"%s\" but not as a input_ports variable.\n", 
              (char *)ToPort, 
              (ToUCNode->Name == MString("") ? "untitled" : (char *)ToUCNode->Name));
            longjmp(PassEnv, 1);
         }
      } else {
         YYUIMessage(UID, ErrorLevel, "Input Port Variable \"%s\" is "
           "not declared in UC Node \"%s\".\n", (char *)ToPort, 
           (ToUCNode->Name == MString("") ? "untitled" : (char *)ToUCNode->Name));
         longjmp(PassEnv, 1);
      }
   }
      break;
   case ::CallNode: 
   {
      cSymTabEntry *SymTabEntry; 
      cCallNode *ToCallNode = (cCallNode *)ToNode;
      cSEGraph *SEGraph = (cSEGraph *)
        GetSymbol(ToCallNode->CalledGraph, this, ::SEGraph, 0);
      if (SEGraph != 0) {
         if (FromClassId == IntNode)
            SymTabEntry = SEGraph->Graph->SymbolTable->Get(FromPort);
         else
            SymTabEntry = SEGraph->Graph->SymbolTable->Get(ToPort);
         if (SymTabEntry != 0) {
            ClassId WhichClass = SymTabEntry->IsA();
            if (WhichClass == ::SEIntNode) {
               if (((cSEIntNode *)SymTabEntry)->IsInputNode == 0) {
                  YYUIMessage(UID, ErrorLevel, "Interface Node Variable \"%s\" is "
                    "declared in Graph \"%s\" but not as an input.\n", (char *)ToPort, 
                    (char *)ToCallNode->CalledGraph);
                  longjmp(PassEnv, 1);
               }
               ToSType = ((cSEIntNode *)SymTabEntry)->Type;
            } else if (WhichClass == ::SECreP) {
               ToSType = ((cSECreP *)SymTabEntry)->Type;
            } else {
               YYUIMessage(UID, ErrorLevel, "Variable \"%s\" is "
                 "declared in Graph \"%s\" but not as a Input Interface Node"
                 " or Creation Parameter.\n", (char *)ToPort, 
                 (char *)ToCallNode->CalledGraph);
               longjmp(PassEnv, 1);
            }
         } else  {
            YYUIMessage(UID, ErrorLevel, "Input Interface Node Variable \"%s\" is "
              "not declared in Graph \"%s\".\n", (char *)ToPort, 
              (char *)ToCallNode->CalledGraph);
            longjmp(PassEnv, 1);
         }
      } else {
         YYUIMessage(UID, ErrorLevel, "Graph \"%s\" is undefined.\n",
           (char *)ToCallNode->CalledGraph);
         longjmp(PassEnv, 1);
      }
   }
      break;
   case ::IntNode: 
   {
      cSEIntNode *SEIntNode = (cSEIntNode *)
        GetSymbol(((cIntNode *)ToNode)->Name, this, ::SEIntNode, 0);
      // or could have called Parent->SymbolTable->Get(...);
      ToSType = SEIntNode->Type;
   }
      break;
   case ::NSRel:
   {
      cSymTabEntry *SymTabEntry; 
      cNSRel *ToNSRelNode = (cNSRel *)ToNode;
      SymTabEntry = ToNSRelNode->SymbolTable->Get(ToPort);
      if (SymTabEntry != 0) {
         if (SymTabEntry->IsA() == ::SEVariable) {
            cSEVariable *SEVariable = (cSEVariable *)SymTabEntry;
            if (SEVariable->Shared == 1) {
               ToSType = ((cSEVariable *)SymTabEntry)->Type;
            } else {
               YYUIMessage(UID, ErrorLevel, "Variable \"%s\" is declared in Name "
                 "Sharing Relation Node \"%s\" but not as a Shared Variable (in shared_vars stanza.)",
                 (char *)ToPort, 
                 (ToNSRelNode->Name == MString("") ? "untitled" : (char *)ToNSRelNode->Name));
               longjmp(PassEnv, 1);
            }
         } else {
            YYUIMessage(UID, ErrorLevel, "Variable \"%s\" is not declared in Name "
              "Sharing Relation Node \"%s\" as a Shared Variable (in shared_vars stanza.)",
              (char *)ToPort, 
              (ToNSRelNode->Name == MString("") ? "untitled" : (char *)ToNSRelNode->Name));
            longjmp(PassEnv, 1);
         }
      } else {
         YYUIMessage(UID, ErrorLevel, "Variable \"%s\" is not declared in Name "
           "Sharing Relation Node \"%s\" as a Shared Variable (in shared_vars stanza.)",
           (char *)ToPort,
           (ToNSRelNode->Name == MString("") ? "untitled" : (char *)ToNSRelNode->Name));
         longjmp(PassEnv, 1);
      }
   }
      break;
   default:
     assert (0); // We should never get here.
     break;
   }

   // Check Arc Input and Output Variable Types
   if (AreSTypesEq(FromSType, ToSType) != 1) {
      YYUIMessage(UID, ErrorLevel, "The structural type for the input variable is "
        "different than the output variable."); // PrintID() is too lame to use
      longjmp(PassEnv, 1);
   }
   BaseType = FromSType;

   // Check Arc Topology 
   int ExpectedArcType = 0;
   switch (FromClassId) {
   case ::UC:
      switch (ToClassId) {
      case ::UC:        //  . => .
         ExpectedArcType = 2;
         break; 
      case ::CallNode:  //  . => ..
         ExpectedArcType = 3;
         break; 
      case ::IntNode:   //  var
         ExpectedArcType = 1;
         break; 
      case ::NSRel:     //  . => .
         ExpectedArcType = 2;
         break; 
      default:
	assert (0); // We should never get here.
	break;
      }
      break;
   case ::CallNode:
      switch (ToClassId) {
      case ::UC:        //  .. => .
         ExpectedArcType = 4;
         break; 
      case ::CallNode:  //  .. => ..
         ExpectedArcType = 5;
         break; 
      case ::IntNode:   //  var
         ExpectedArcType = 1;
         break; 
      case ::NSRel:     //  .. => .
         ExpectedArcType = 4;
         break; 
      default:
	assert (0); // We should never get here.
	break;
      }
      break;
   case ::IntNode:
      switch (ToClassId) {
      case ::UC:        // var
         ExpectedArcType = 1;
         break; 
      case ::CallNode:  // var
         ExpectedArcType = 1;
         break; 
      case ::IntNode:   // empty 
         ExpectedArcType = 6;
         break; 
      case ::NSRel:     //  var
         ExpectedArcType = 1;
         break; 
      default:
	assert (0); // We should never get here.
	break;
      }
      break; 
   default:
     assert (0); // We should never get here.
     break;
   }
   if (ArcType != ExpectedArcType) {
      YYUIMessage(UID, ErrorLevel, "Arc topology description is "
        "incorrect.\nExpected description is %s\n"
        "Received description is %s\n", 
        ArcDescription[ExpectedArcType], 
        ArcDescription[ArcType]);
      longjmp(PassEnv, 1);
   } 
}


void cPartArc::Pass2()
{
}


void cMergeArc::Pass2()
{
}


void cExpandArc::Pass2()
{
}


void cContractArc::Pass2()
{
}


void cStrip_UpdateArc::Pass2()
{
}


void cAlignArc::Pass2()
{
}


void cNamedType::Pass2()
{
}


void cArrayType::Pass2()
{
}


void cStructMem::Pass2()
{
}


void cStructType::Pass2()
{
   cObIterator i, j;
   cObject *p, *q;


   for (p = i.Init(Members); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(Members); i.MoreLeft(); p = i.Next()) {
      cStructMem *MemberI = (cStructMem *)p;
      for (q = j.Init(Members); j.MoreLeft(); q = j.Next()) {
         cStructMem *MemberJ = (cStructMem *)q;
         if (MemberI != MemberJ && (char *) MemberI->Name == MemberJ->Name) {
            YYUIMessage(UID, ErrorLevel, "In Struct \"%s\", member name \"%s\" is "
              "re-used.\n", 
              (char *)((cTypeDef *)Parent)->Name,
              (char *)MemberI->Name); 
            longjmp(PassEnv, 1);
         }
      }
   }
}


void cTypeDef::Pass2()
{
   Type->Pass2();
}


void cCallStmt::Pass2()
{
   cObIterator i;
   cObject *p;
   cSType *r;

   for (p = i.Init(Args); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   cSEFuncName *SEFuncName = 
     (cSEFuncName *)GetSymbol(FuncName, this, ::SEFuncName, 0);
   if (SEFuncName == 0) {
      YYUIMessage(UID, ErrorLevel, "Function \"%s\" not declared\n",
		  (char *) FuncName);
      longjmp(PassEnv, 1);
   }

   if (SEFuncName->IsPolyMorphic) {
      if (!CheckPoly(FuncName, 0, Args, &r)) {
	 YYUIMessage(UID, ErrorLevel, "Built-In Function \"%s\" is misused\n",
		     (char *) FuncName);
	 longjmp(PassEnv, 1);
      }

      return;
   }

   if(!SEFuncName->DontCheckArgs) {
      if (!CheckArgs(Args, SEFuncName->ArgList)) {
	 YYUIMessage(UID, ErrorLevel,
		     "Function argument type or number mismatch\n");
	 longjmp(PassEnv, 1);
      }
   }

   if (SEFuncName->ReturnType != 0) {
      YYUIMessage(UID, ErrorLevel, "Non-void return type for Function \"%s\".\n",
		  (char *) FuncName);
      longjmp(PassEnv, 1);
   }


}


void cWhileStmt::Pass2()
{


   Cond->Pass2();
   Body->Pass2();


   if (!IsSTypeOneOf(Cond->StructuralType, IntSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for WHILE COND\n");
	longjmp(PassEnv, 1);
     }
}


void cBlockStmt::Pass2()
{
   cObIterator i;
   cObject *p;


   for (p = i.Init(StmtList); i.MoreLeft(); p = i.Next()) {

      ((cAbsTree *) p)->Pass2();
   }
}


void cIfStmt::Pass2()
{


   Cond->Pass2();
   IfPart->Pass2();
   if (ElsePart != 0) {
      ElsePart->Pass2();
   }


   if (!IsSTypeOneOf(Cond->StructuralType, IntSType, 0))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for IF COND\n");
	longjmp(PassEnv, 1);
     }
}


void cAssignStmt::Pass2()
{


   LHS->Pass2();
   RHS->Pass2();


   if (!CanAssignSTypes(LHS->StructuralType, RHS->StructuralType))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for ASSIGNMENT\n");
	longjmp(PassEnv, 1);
     }
}


void cNullStmt::Pass2()
{
}


static void SemCheckSizeSpec(cSizeSpec *SizeSpec, cSType *Type) 
{
   cObIterator i;

   if (Type->IsA() == ::ArraySType) {
      int NumTypeArrayDimensions = 0;
      for (cSType *ThisType = Type; 
        ThisType->IsA() == ::ArraySType; 
        NumTypeArrayDimensions++, ThisType = ((cArraySType *)ThisType)->BaseType);
      cSType *BaseType = ThisType;
      int NumSpecArrayDimensions = SizeSpec->ArrayBounds.Length();

      if (NumSpecArrayDimensions <= NumTypeArrayDimensions) {
         for (cExpr *Expr = (cExpr *)i.Init(SizeSpec->ArrayBounds); 
           i.MoreLeft(); 
           Expr = (cExpr *)i.Next()) {
             if (IsSTypeOneOf(Expr->StructuralType, IntSType, 0) != 1) {
                YYUIMessage(SizeSpec->UID, ErrorLevel, "Dimension must be "
                  "specified with an integer expression.\n");
                longjmp(PassEnv, 1);
             }
         }

         if (BaseType->IsA() == ::StructSType) {
            cStructSType *StructSType = (cStructSType *)BaseType;
            for (cMemSize *MemSize = (cMemSize *)i.Init(SizeSpec->MemberSizes);
              i.MoreLeft();
              MemSize = (cMemSize *)i.Next()) {
                cStructMemSType *StructMemSType  = 
                  StructSType->GetMember(MemSize->MemberName);
                if (StructMemSType != 0) {
                   SemCheckSizeSpec(MemSize->Sizes, StructMemSType->Type);
                } else {
                   YYUIMessage(SizeSpec->UID, ErrorLevel, "Member \"%s\" is not a "
                     "valid structure member for the structure type indicated.\n", 
                     (char *)MemSize->MemberName);
                   longjmp(PassEnv, 1);
                }
            }
         } else if (SizeSpec->MemberSizes.Length() != 0) {
            YYUIMessage(SizeSpec->UID, ErrorLevel, "Structure member(s) used in "
              "size specification, but indicated type is not a structure.\n");
            longjmp(PassEnv, 1);
         }
      } else {
         YYUIMessage(SizeSpec->UID, ErrorLevel, "Too many dimensions "
           "specified.\n");
         longjmp(PassEnv, 1);
      }
   } else if (Type->IsA() == ::StructSType) {
      if (SizeSpec->ArrayBounds.Length() == 0) {
            cStructSType *StructSType = (cStructSType *)Type;
            for (cMemSize *MemSize = (cMemSize *)i.Init(SizeSpec->MemberSizes);
              i.MoreLeft();
              MemSize = (cMemSize *)i.Next()) {
                cStructMemSType *StructMemSType  = 
                  StructSType->GetMember(MemSize->MemberName);
                if (StructMemSType != 0) {
                   SemCheckSizeSpec(MemSize->Sizes, StructMemSType->Type);
                } else {
                   YYUIMessage(SizeSpec->UID, ErrorLevel, "Member \"%s\" is not a "
                     "valid structure member for the structure type indicated.\n", 
                     (char *)MemSize->MemberName);
                   longjmp(PassEnv, 1);
                }
            }
      } else {
         YYUIMessage(SizeSpec->UID, ErrorLevel, "Array bound(s) given for an "
           "un-arrayed structure.\n");
         longjmp(PassEnv, 1);
      }
   } else {
      YYUIMessage(SizeSpec->UID, ErrorLevel, "Arrayness specified but "
        "not needed. An array type or a structure type with a nested array "
        "was expected.\n");
      longjmp(PassEnv, 1);
   }
}


void cSizeSpec::Pass2()
{
   cObIterator i;
   cObject *p;


   for (p = i.Init(ArrayBounds); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(MemberSizes); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }
}


void cMemSize::Pass2()
{

   Sizes->Pass2();
}


void cVarDecl::Pass2()
{
   cSETypeName *SETypeName;

   SETypeName = (cSETypeName *)GetSymbol(TypeName, this, ::SETypeName, 0);
   if (SETypeName == 0) {
      YYUIMessage(UID, ErrorLevel, "Type \"%s\" in variable declaration is "
        "undefined.\n", (char *)TypeName);
      longjmp(PassEnv, 1);
   }

   if (Sizes != 0) {
      Sizes->Pass2();
      SemCheckSizeSpec(Sizes, SETypeName->Type);
   } else {
      // arrays and structs need an empty size spec for translation
      switch (SETypeName->Type->IsA()) {
      case ::CharSType:
      case ::IntSType:
      case ::RealSType:
         break;
      case ::ArraySType:
      case ::StructSType:
         Sizes = new cSizeSpec;
         Sizes->Parent = this;
         break;
      default:
         break;
      }
   }
}


void cNSRel::Pass2()
{
   cObIterator i;
   cObject *p;


   if (InitComp != 0) {
      ((cAbsTree *) InitComp)->Pass2();
   }

   for (p = i.Init(FuncSigs); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(SharedVars); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(LocalVars); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }
}


void cReplSpec::Pass2()
{


   Bound->Pass2();
   if (!IsSTypeOneOf(Bound->StructuralType, IntSType, 0)) {
      YYUIMessage(UID, ErrorLevel, "Wrong type for REPL_INDEX\n");
      longjmp(PassEnv, 1);
   }
}


void cGuard::Pass2()
{
   cObIterator i;
   cObject *p;


   Cond->Pass2();
   if (!IsSTypeOneOf(Cond->StructuralType, IntSType, 0)) {
      YYUIMessage(UID, ErrorLevel, "Wrong type for GUARD_COND\n");
      longjmp(PassEnv, 1);
   }

   if (ReplCond != 0) {
      ReplCond->Pass2();
      if (!IsSTypeOneOf(ReplCond->StructuralType, IntSType, 0)) {
	 YYUIMessage(UID, ErrorLevel, "Wrong type for REPL_COND\n");
	 longjmp(PassEnv, 1);
      }
   }

   for (p = i.Init(Repls); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }
}


void cArcInput::Pass2()
{
   cObIterator i;
   cObject *p;


   PortExpr->Parent = this;
   PortExpr->Pass2();

   if (!IsSTypeOneOf(PortExpr->StructuralType, PortSType, 0)) {
      YYUIMessage(UID, ErrorLevel, "Wrong type for PORT_EXPR\n");
      longjmp(PassEnv, 1);
   }

   if (Repls.Length() > 7) {
      YYUIMessage(UID, ErrorLevel, "The PORT_EXPR may not have more than "
        "7 indices (replication specifications.)\n");
      longjmp(PassEnv, 1);
   }

   VarExpr->Parent = this;
   VarExpr->Pass2();

   if (!AreSTypesEq(VarExpr->StructuralType,
		    ((cPortSType *) (PortExpr->StructuralType))->BaseType))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for VAR_EXPR\n");
	longjmp(PassEnv, 1);
     }

   if (ReplCond != 0) {
      ReplCond->Pass2();
      if (!IsSTypeOneOf(ReplCond->StructuralType, IntSType, 0)) {
	 YYUIMessage(UID, ErrorLevel, "Wrong type for REPL_COND\n");
	 longjmp(PassEnv, 1);
      }
   }

   for (p = i.Init(Repls); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Parent = this;
      ((cAbsTree *) p)->Pass2();
   }
}


void cBinding::Pass2()
{
   cObIterator i;
   cObject *p;


   Bind->Pass2();

   if (ReplCond != 0) {
      ReplCond->Pass2();
      if (!IsSTypeOneOf(ReplCond->StructuralType, IntSType, 0)) {
	 YYUIMessage(UID, ErrorLevel, "Wrong type for REPL_COND\n");
	 longjmp(PassEnv, 1);
      }
   }

   for (p = i.Init(Repls); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }
}


void cArcOutput::Pass2()
{
   cObIterator i;
   cObject *p;


   PortExpr->Parent = this;
   PortExpr->Pass2();

   if (!IsSTypeOneOf(PortExpr->StructuralType, PortSType, 0)) {
      YYUIMessage(UID, ErrorLevel, "Wrong type for PORT_EXPR\n");
      longjmp(PassEnv, 1);
   }

   if (Repls.Length() > 7) {
      YYUIMessage(UID, ErrorLevel, "The PORT_EXPR may not have more than "
        "7 indices.\n");
      longjmp(PassEnv, 1);
   }

   ValueExpr->Parent = this;
   ValueExpr->Pass2();

   if (!CanAssignSTypes(
	ValueExpr->StructuralType,
	((cPortSType *) (PortExpr->StructuralType))->BaseType))
     {
	YYUIMessage(UID, ErrorLevel, "Wrong type for VALUE\n");
	longjmp(PassEnv, 1);
     }

   if (ReplCond != 0) {
      ReplCond->Pass2();
      if (!IsSTypeOneOf(ReplCond->StructuralType, IntSType, 0)) {
	 YYUIMessage(UID, ErrorLevel, "Wrong type for REPL_COND\n");
	 longjmp(PassEnv, 1);
      }
   }

   for (p = i.Init(Repls); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Parent = this;
      ((cAbsTree *) p)->Pass2();
   }
}


void cRule::Pass2()
{
   cObIterator i;
   cObject *p;
   int ArcInputFound;

   ArcInputFound = 0;
   for (p = i.Init(Guards); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
      if (FiringRule == 1) {
         if (p->IsA() == ::ArcInput) {
            ArcInputFound = 1;
         } else if (p->IsA() != ::Guard) {
            YYUIMessage(UID, ErrorLevel, "The guard portion of a firing rule expects "
              "expressions that bind port variables to local variables of the UC.\n");
	    longjmp(PassEnv, 1);
         }
      } else {
         if (p->IsA() != ::Guard) {
            YYUIMessage(UID, ErrorLevel, "The guard portion of a routing rule"
              " expects integer expressions that evaluate to TRUE or FALSE.\n");
	    longjmp(PassEnv, 1);
         }
      }
   }

   if (FiringRule == 1 && ArcInputFound == 0) {
      YYUIMessage(UID, ErrorLevel, "Firing rules must have at least one expression "
        "that binds port variables to local variables of the UC.\n");
      longjmp(PassEnv, 1);
   }

   for (p = i.Init(Binds); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
      if (FiringRule == 1) {
         if (p->IsA() != ::Binding) {
            YYUIMessage(UID, ErrorLevel, "The bind portion of a firing rule expects "
              "assignment statements.\n");
	    longjmp(PassEnv, 1);
         }
      } else {
         // Binds can be of type cBinding or cArcOutput : guaranteed by syntax
      }
   }

   if (ReplCond != 0) {
      ReplCond->Pass2();
      if (!IsSTypeOneOf(ReplCond->StructuralType, IntSType, 0)) {
	 YYUIMessage(UID, ErrorLevel, "Wrong type for REPL_COND\n");
	 longjmp(PassEnv, 1);
      }
   }

   for (p = i.Init(Repls); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }
}


void cPortSpec::Pass2()
{
   cSETypeName *Entry;

   Entry = (cSETypeName *) GetSymbol(TypeName, this, SETypeName, 0);
   if (Entry == 0) {
      YYUIMessage(UID, ErrorLevel, "Type, \"%s\", for Port \"%s\" is undefined.\n",
        (char *)TypeName, (char *)PortName);
      longjmp(PassEnv, 1);
   }
}


void cNSPortSpec::Pass2()
{
   cSETypeName *Entry;

   Entry = (cSETypeName *) GetSymbol(TypeName, this, SETypeName, 0);
   if (Entry == 0) {
      YYUIMessage(UID, ErrorLevel, "Type, \"%s\", for Port \"%s\" is undefined.\n",
        (char *)TypeName, (char *)PortName);
      longjmp(PassEnv, 1);
   }
}


void cUC::Pass2()
{
   cObIterator i, j, k;
   cObject *p, *q, *r;
   MString PortName;
   int Found, FoundCond;


   if (InitComp != 0) {
      InitComp->Pass2();
   }

   if (Comp != 0) {
      Comp->Pass2();
   }

   for (p = i.Init(FuncSigs); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(InputPorts); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(OutputPorts); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(NSPorts); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(LocalVars); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   p = i.Init(InRules);
   if (StartNode != 1 && !i.MoreLeft()) {
      YYUIMessage(UID, ErrorLevel, "Only the start node can have no firing rules.\n");
      longjmp(PassEnv, 1);
   }
   for (; i.MoreLeft(); p = i.Next()) {
      ((cRule *) p)->FiringRule = 1;
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(OutRules); i.MoreLeft(); p = i.Next()) {
      ((cRule *) p)->FiringRule = 0;
      ((cAbsTree *) p)->Pass2();
   }

   // make sure all port variables are used in {routing,firing}_rules : warning
   for (p = i.Init(InputPorts); i.MoreLeft(); p = i.Next()) {
      cPortSpec *PortSpec = (cPortSpec *)p;
      Found = 0;
      for (q = j.Init(InRules); j.MoreLeft(); q = j.Next()) {
         cRule *Rule = (cRule *)q;
         for (r = k.Init(Rule->Guards); k.MoreLeft(); r = k.Next()) {
            if (r->IsA() ==  ::ArcInput) {
               cArcInput *ArcInput = (cArcInput *)r;
               PortName = FindPortVariableName(ArcInput->PortExpr);

               if ((char *) PortName == PortSpec->PortName) {
                   Found = 1;
                   break;
               }
            }
         }
         if (Found == 1) break;
      }
      if (Found == 0) {
         cGraph *Graph = (cGraph *)Parent;
         for (q = k.Init(Graph->Arcs); k.MoreLeft(); q = k.Next()) {
            cArc *Arc = (cArc *)q; 
            if (FindUidTable(Arc->FromUID)->IsA() == ::IntNode) 
               FoundCond = Arc->ToUID == UID && (char *) Arc->FromPort == PortSpec->PortName;
            else
               FoundCond = Arc->ToUID == UID && (char *) Arc->ToPort == PortSpec->PortName;
            if (FoundCond) {
               YYUIMessage(UID, ErrorLevel, "Input Port Variable \"%s\" has been declared "
                 "and not used in a Firing Rule but has been bound in incoming Arc \"%s\".",
                 (char * )PortSpec->PortName, (char *)Arc->Name);
               longjmp(PassEnv, 1);
            }
         }
         YYUIMessage(UID, ErrorLevel, "Input Port Variable \"%s\" was declared but not used.",
           (char *)PortSpec->PortName);
         longjmp(PassEnv, 1);
      }
   }
   for (p = i.Init(OutputPorts); i.MoreLeft(); p = i.Next()) {
      cPortSpec *PortSpec = (cPortSpec *)p;
      Found = 0;
      for (q = j.Init(OutRules); j.MoreLeft(); q = j.Next()) {
         cRule *Rule = (cRule *)q;
         for (r = k.Init(Rule->Binds); k.MoreLeft(); r = k.Next()) {
            if (r->IsA() == ::ArcOutput) {
               cArcOutput *ArcOutput = (cArcOutput *)r;
               PortName = FindPortVariableName(ArcOutput->PortExpr);

               if ((char *) PortName == PortSpec->PortName) {
                  Found = 1;
                  break;
               }
            }
         }
         if (Found == 1) break;
      }
      if (Found == 0) {
         cGraph *Graph = (cGraph *)Parent;
         for (q = k.Init(Graph->Arcs); k.MoreLeft(); q = k.Next()) {
            cArc *Arc = (cArc *)q;
            if (Arc->FromUID == UID && (char *) Arc->FromPort == PortSpec->PortName) {
               YYUIMessage(UID, ErrorLevel, "Output Port Variable \"%s\" has been declared "
                 "and not used in a Routing Rule but has been bound in outgoing Arc \"%s\".",
                 (char * )PortSpec->PortName, (char *)Arc->Name);
               longjmp(PassEnv, 1);
            }
         }
         YYUIMessage(UID, ErrorLevel, "Output Port Variable \"%s\" was declared but not "
           "used.", (char *)PortSpec->PortName);
         longjmp(PassEnv, 1);
      }
   }

   // make sure that all port variables that are declared and used are bound to only one arc
   for (p = i.Init(OutRules); i.MoreLeft(); p = i.Next()) {
      cRule *Rule = (cRule *)p;
      for (q = j.Init(Rule->Binds); j.MoreLeft(); q = j.Next()) {
         if (q->IsA() == ::ArcOutput) {
            cArcOutput *ArcOutput = (cArcOutput *)q;
            PortName = FindPortVariableName(ArcOutput->PortExpr);

            cGraph *Graph = (cGraph *)Parent;
            Found = 0;
            for (q = k.Init(Graph->Arcs); k.MoreLeft(); q = k.Next()) {
               cArc *Arc = (cArc *)q;
               if (Arc->FromUID == UID && (char *) Arc->FromPort == PortName) {
                  if (Found == 0) { 
                     Found = 1;
                     if (Arc->PortVarList.Length() != PortIndicesCount(ArcOutput->PortExpr)) {
                        YYUIMessage(UID, ErrorLevel, "The number of indices used by Output Port "
                          "Variable \"%s\" in a Routing Rule does not agree with the number of "
                          "indices in the arc topology in Arc \"%s\".\n",
                          (char *)PortName, (char *)Arc->Name);
                        longjmp(PassEnv, 1);
                     }
                  } else {
                     YYUIMessage(UID, ErrorLevel, "Output Port Variable \"%s\" has been declared "
                      "and used in a Routing Rule but has been bound more than once in the outgoing"
                      " arcs.\n", (char *)PortName);
                     longjmp(PassEnv, 1);
                  }
               }
            }
         }
         if (Found == 0) {
            YYUIMessage(UID, ErrorLevel, "Output Port Variable \"%s\" has been declared and used "
              "in a Routing Rule but is not bound in any outgoing arc.\n", (char *)PortName);
            longjmp(PassEnv, 1);
         }
      }
   }

   // make sure that all shared variables that are declared are bound to only one arc  NSPorts
   for (p = i.Init(NSPorts); i.MoreLeft(); p = i.Next()) {
      cNSPortSpec *NSPortSpec = (cNSPortSpec *)p;

      cGraph *Graph = (cGraph *)Parent;
      Found = 0;
      for (q = k.Init(Graph->Arcs); k.MoreLeft(); q = k.Next()) {
         cArc *Arc = (cArc *)q;

         if (Arc->FromUID == UID && (char *) Arc->FromPort == NSPortSpec->PortName) {
            if (Found == 0) { 
               Found = 1;
               if (Arc->PortVarList.Length() != 0) {
                  YYUIMessage(UID, ErrorLevel, "Port Variable \"%s\" which is bound to a Shared "
                    "Variable in UC Node \"%s\" can not have indices in the arc topology (\"%s\").\n", 
                    (char *)NSPortSpec->PortName, 
                    (Name == MString("") ? "untitled" : (char *)Name),
                    (char *)Arc->Name);                    

               }
               //  recursively track down path to see that it is connected to NSRel Node
            } else {
               YYUIMessage(UID, ErrorLevel, "Shared Variable \"%s\" has been bound more "
                 "than once in the outgoing arcs.\n", (char *)NSPortSpec->PortName);
               longjmp(PassEnv, 1);
            }
         }
      }
      if (Found == 0) {
            YYUIMessage(UID, ErrorLevel, "Shared Variable \"%s\" has been declared "
              "but is not bound in any outgoing arc.\n", (char *)NSPortSpec->PortName);
            longjmp(PassEnv, 1);
      }
   }
}


void cIntNode::Pass2()
{
   cSETypeName *Entry;
   cObIterator i;
   cGraph *Graph;
   cArc *Arc;
   int Found;
  
   Entry = (cSETypeName *) GetSymbol(Type, this, SETypeName, 0);
   if (Entry == 0) {
      YYUIMessage(UID, ErrorLevel, "Type, \"%s\", for Interface Node \"%s\" is undefined.\n",
        (char *)Type, (char *)Name);
      longjmp(PassEnv, 1);
   }

   if (InParam == 1) {
      Graph = (cGraph *)Parent;
      Found = 0;
      for (Arc = (cArc *)i.Init(Graph->Arcs); i.MoreLeft(); Arc = (cArc *)i.Next()) {
         if ((InParam == 1 ? Arc->FromUID : Arc->ToUID) == UID) {
            if (Found == 1) {
               YYUIMessage(UID, ErrorLevel, "There are more than one %s arcs from %s "
                 "Interface Node \"%s\".\n", 
                 (InParam == 1 ? "outgoing" : "incoming"),
                 (InParam == 1 ? "input"    : "output"), (char *)Name);
               longjmp(PassEnv, 1);
            } else {
               Found = 1;
            }
         }
      }
      if (Found == 0) {
         YYUIMessage(UID, ErrorLevel, "There are no %s arcs from %s "
           "Interface Node \"%s\".\n", (InParam == 1 ? "outgoing" : "incoming"),
           (InParam == 1 ? "input" : "output"), (char *)Name);
         longjmp(PassEnv, 1);
      }
   }
 }


void cCrepNode::Pass2()
{
   cSETypeName *Entry;

   Entry = (cSETypeName *) GetSymbol(Type, this, SETypeName, 0);
   if (Entry == 0) {
      YYUIMessage(UID, ErrorLevel, "Type, \"%s\", for Creation Parameter \"%s\" is undefined.\n",
        (char *)Type, (char *)Name);
      longjmp(PassEnv, 1);
   }
}


void cCallNode::Pass2()
{
   cObIterator i, j;
   cSEGraph *SEGraph;
   cGraph *CalledGraph, *CallingGraph;
   cIntNode *IntNode;
   cCrepNode *CrepNode; 
   cArc *Arc;
   MString PortName;
   int Found, MatchUID;

   SEGraph = (cSEGraph *) GetSymbol(this->CalledGraph, this, ::SEGraph, 0);
   if (SEGraph != 0) {
      CalledGraph = SEGraph->Graph;
      if (CalledGraph == 0) {
         YYUIMessage(UID, ErrorLevel, "The called Graph \"%s\" indicated by the Call Node \"%s\" "
           "is undefined.\n", (char *)this->CalledGraph, (char *)Name);
         longjmp(PassEnv, 1);
      }
   } else {
      YYUIMessage(UID, ErrorLevel, "The called Graph \"%s\" indicated by the Call Node \"%s\" "
        "is undefined.\n", (char *)this->CalledGraph, (char *)Name);
      longjmp(PassEnv, 1);
   }

   CallingGraph = (cGraph *)Parent;

   for (
    IntNode = (cIntNode *)i.Init(CalledGraph->IntNodes); 
    i.MoreLeft(); 
    IntNode = (cIntNode *)i.Next() ) 
   {
      Found = 0;
      for (Arc = (cArc *)j.Init(CallingGraph->Arcs); j.MoreLeft(); Arc = (cArc *)j.Next()) {
         PortName = (IntNode->InParam == 1 ? Arc->ToPort : Arc->FromPort);
         MatchUID = (IntNode->InParam == 1 ? Arc->ToUID : Arc->FromUID);
         if (UID == MatchUID && (char *) PortName == IntNode->Name) {
            if (Found == 0) 
               Found = 1;
            else {
               YYUIMessage(UID, ErrorLevel, "Interface Node \"%s\" has been bound more than"
                 " once on %s arcs to Calling Node \"%s\".\n", (char *)IntNode->Name,
                 (IntNode-> InParam == 1 ? "incoming" : "outgoing"),
                 (Name == MString("") ? "<No Name>" : (char *)Name)); 
               longjmp(PassEnv, 1);
            }
         }
      }
      if (!Found) {
         YYUIMessage(UID, ErrorLevel, "Interface Node \"%s\" has not been bound"
           " once on %s arcs to Calling Node \"%s\".\n", (char *)IntNode->Name, 
           (IntNode-> InParam == 1 ? "incoming" : "outgoing"),
           (Name == MString("") ? "<No Name>" : (char *)Name)); 
         longjmp(PassEnv, 1);
      }
   }

   for (
    CrepNode = (cCrepNode *)i.Init(CalledGraph->CrepNodes);
    i.MoreLeft();
    CrepNode = (cCrepNode *)i.Next() ) 
   {
      Found = 0;
      for (Arc = (cArc *)j.Init(CallingGraph->Arcs); j.MoreLeft(); Arc = (cArc *)j.Next()) {
         if (UID == Arc->ToUID && (char *) Arc->ToPort == CrepNode->Name) {
            if (Found == 0)
               Found = 1;
            else {
               YYUIMessage(UID, ErrorLevel, "Creation Parameter \"%s\" has been bound more than"
                 " once on incoming arcs to Calling Node \"%s\".\n", (char *)CrepNode->Name, 
                 (char *)Name); 
               longjmp(PassEnv, 1);
            }
         }
      }
      if (!Found) {
         YYUIMessage(UID, ErrorLevel, "Creation Parameter \"%s\" has not been bound"
           " once on incoming arcs to Calling Node \"%s\".\n", (char *)CrepNode->Name, 
           (Name == MString("") ? "<No Name>" : (char *)Name)); 
         longjmp(PassEnv, 1);
      }
   }
 
}


void cGraph::Pass2()
{
   cObIterator i;
   cObject *p;


   for (p = i.Init(FuncSigs); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(Types); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(IntNodes); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(CrepNodes); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(UCNodes); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(CallNodes); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(NSRelNodes); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(Arcs); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   if (!LegalIdent(Name)) {
      YYUIMessage(UID, ErrorLevel, "Graph name \"%s\" is not a legal identifier.\n",
        (char *)Name);
      longjmp(PassEnv, 1);
   }
}


void cProgram::Pass2()
{
   cObIterator i, j, k;
   cObject *p, *q, *r;
   cSETypeName *Entry;
   int StartFound, TerminateFound;

   Entry = (cSETypeName *) GetSymbol("int", this, SETypeName, 0);

   if (Entry == 0)
     Die("Internal error:  int not declared in cProgramConst::pass2");

   PtrToIntSType = Entry->Type;

   if (setjmp(PassEnv)) {
      PassOK = 0;
      return;
   }

   for (p = i.Init(FuncSigs); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(Types); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   for (p = i.Init(Graphs); i.MoreLeft(); p = i.Next()) {
      ((cAbsTree *) p)->Pass2();
   }

   StartFound = TerminateFound = 0;
   for (p = i.Init(Graphs); i.MoreLeft(); p = i.Next()) {
      cGraph *ThisGraph = (cGraph *)p;
      for (q = j.Init(ThisGraph->UCNodes); j.MoreLeft(); q = j.Next()) {
         cUC *ThisUCNode = (cUC *)q;
         if (ThisUCNode->StartNode == 1) {
            if (StartFound == 0) {
	       StartNodeUID = ThisUCNode->UID;
	       StartGraphUID = ThisUCNode->Parent->UID;
               StartFound = 1;
            } else {
               YYUIMessage(UID, ErrorLevel, "More than one start node has been "
                 "specified for program.");
               longjmp(PassEnv, 1);
             }
         }

         if (ThisUCNode->TermNode == 1) {
            if (TerminateFound == 0) {
               TerminateFound = 1;
               for (r = k.Init(ThisGraph->Arcs); k.MoreLeft(); r = k.Next()) {
                  cArc *ThisArc = (cArc *)r;
                  if (ThisArc->ToUID == ThisUCNode->UID) {
                     if (!ThisArc->NodeExprList.Empty()) {
                         YYUIMessage(UID, ErrorLevel, "Termination node for program "
                            "can not be indexed.");
                         longjmp(PassEnv, 1);
                     }
                  }
               }
            } else {
               YYUIMessage(UID, ErrorLevel, "More than one termination node has been "
                  "specified for program.");
               longjmp(PassEnv, 1);
            }
         }
      }
   }
   if (StartFound == 0) {
      YYUIMessage(UID, ErrorLevel, "Program does not have a start node.");
      longjmp(PassEnv, 1);
   }

   if (!LegalIdent(Name)) {
      YYUIMessage(UID, ErrorLevel, "Program name \"%s\" is not a legal identifier.\n",
        (char *)Name);
      longjmp(PassEnv, 1);
   }

}

