/*
 *   Convex Hull Library
 *
 *     Function definitions
 *
 */

#include <math.h>
#include <stdio.h>
#include "c2_globtype.h"

/* Interfaces:

   void PrintList(tConvInfo ConvInfo, IntAarry OutInds);
   void ReadList(tConvInfo ConvInfo);
   void FindHull(int Quad, tConvInfo ConvInfo, IntArray OutInds);
   void FindCorners(tConvInfo ConvInfo);
*/


  /* ftheta - finds angle for minimization between two points */

static double ftheta(x1, y1, x2, y2, GoUp)
   double x1, y1, x2, y2;
   int GoUp;
{
   double dx, dy, theta, ax, ay;

   dx = x2 - x1;
   dy = y2 - y1;
   ax = fabs(dx);
   ay = fabs(dy);

   if (dx == 0.0 && dy == 0.0)
     theta = 100000000.0;
   else
     theta = dy/(ax + ay);

   if (dx < 0.0)
     theta  = 2.0 - theta;
   else if (dy < 0.0)
     theta = 4.0 + theta;

   if (GoUp == 0) {
     theta = theta + 2.0;
     if (theta >= 4.0) theta = theta - 4.0;
  }

return theta;

}


#define BOTTOM 0
#define RIGHT 1
#define TOP 2
#define LEFT 3


  /* FindCorners - computes CornerInds in ConvInfo, assuming that
     Point and NumPoints are already present. */

void FindCorners(ConvInfo)
   tConvInfo ConvInfo;
{
   int i;
   double maxy, miny, maxx, minx;

   miny = 100000000.0;
   maxy = - 100000000.0;
   minx = 100000000.0;
   maxx = - 100000000.0;

   for (i = 0; i < 4; i++)
     ConvInfo->CornerInds[i] = -1;

   for (i = 0; i < ConvInfo->NumPoints; i++) {

      if (ConvInfo->y[i] > maxy) {
	 maxy = ConvInfo->y[i];
	 ConvInfo->CornerInds[TOP] = i;
      }

      if (ConvInfo->y[i] < miny) {
	 miny = ConvInfo->y[i];
	 ConvInfo->CornerInds[BOTTOM] = i;
      }

      if (ConvInfo->x[i] < minx) {
	 minx = ConvInfo->x[i];
	 ConvInfo->CornerInds[LEFT] = i;
      }

      if (ConvInfo->x[i] > maxx) {
	 maxx = ConvInfo->x[i];
	 ConvInfo->CornerInds[RIGHT] = i;
      }
   }

   return;

}


  /* FindHull - Finds indices of point on convex hull in the selected
     quadrant.  Quad is in the range 0..3.  Assumes ConvInfo is fully
     computed.  Indices are stored in OutInds. */

void FindHull(Quad, ConvInfo, OutInds)
   int Quad;
   tConvInfo ConvInfo;
   IntArray OutInds;
{
   int i, icur, iminth, GoUp, iout;
   double theta, minthe;

   iout = 0;
   icur = ConvInfo->CornerInds[Quad];
   if (Quad <= 1)
     GoUp = 1;
   else
     GoUp = 0;

   while (icur != ConvInfo->CornerInds[(Quad+1)%4]) {

      iout = iout + 1;      /* Add this point to the hull */
      OutInds[iout] =icur;

      minthe = 100000000.0;
      for (i = 0; i < ConvInfo->NumPoints; i++) {
	 theta = ftheta(ConvInfo->x[icur], ConvInfo->y[icur],
			ConvInfo->x[i], ConvInfo->y[i], GoUp);
	 if (theta < minthe) {
	    minthe = theta;
	    iminth = i;
	 }
      }

      icur = iminth;
   } /* while */

   OutInds[0] = iout;

   return;
}


  /* ReadList - Reads Points and NumPoints into ConvInfo.  Data file
     format is

          x0 y0
          x1 y1
           ...
  */

void ReadList(ConvInfo)
   tConvInfo ConvInfo;
{
   int i;
   int res;

   i = 0;

   res = scanf("%lf %lf", &ConvInfo->x[i], &ConvInfo->y[i]);
   while (res != 0 && res != EOF) {
      printf("move %f %f\n", ConvInfo->x[i], ConvInfo->y[i]);
      printf("draw %f %f\n", ConvInfo->x[i], ConvInfo->y[i]);
      i += 1;
      res = scanf("%lf %lf", &ConvInfo->x[i], &ConvInfo->y[i]);
   }

   if (res == 0)
     printf("term on 0\n");

   ConvInfo->NumPoints = i;

   printf("\n");

   return;
}


  /* PrintList - Prints points in ConvInfo whose indices are in
     OutInds */

void PrintList(ConvInfo, OutInds)
   tConvInfo ConvInfo;
   IntArray OutInds;
{
   int i;

   for (i = 1; i <= OutInds[0]; i++) {
      printf("%f %f\n", ConvInfo->x[OutInds[i]], ConvInfo->y[OutInds[i]]);
   }
   return;
}

#ifdef TEST

main ()
{
   tConvInfo ConvInfo = (tConvInfo) malloc (sizeof(* ConvInfo));
   int i;
   IntMat OutInds = (IntMat) malloc (4*sizeof(int *));

   ConvInfo->x  = (double *) malloc(40*sizeof(double));
   ConvInfo->y  = (double *) malloc(40*sizeof(double));
   ConvInfo->CornerInds  = (int *) malloc(40*sizeof(int));

   for (i = 0; i < 4; i++)
     OutInds[i] = (int *) malloc (40*sizeof(int));

   ReadList(ConvInfo);
   FindCorners(ConvInfo);

   for (i = 0; i < 4; i++)
     FindHull(i, ConvInfo, OutInds[i]);

   for (i = 0; i < 4; i++)
     PrintList(ConvInfo, OutInds[i]);
}

#endif


