/* Synchronization data structures and routines, available publicly */

#include "asm.h"

/* Condition variables are always protected by spin or blocking locks */
typedef struct condition {
	unsigned int type;
	Thread *waiting;
	struct synch_profile *p;
	} Condition;

typedef Condition SpinCondition;

typedef struct lock {
	unsigned int type;
	int lock;
	SpinLock slock; 
	Thread *waiting;
	struct synch_profile *p;
	} Lock;

/* Potentially fuzzy-barriers
 * and all those that check in do not have to wait (ie, their threads can die)
 * (but all that wait must check in)
 */

typedef struct barrier {
	unsigned int type;
	SpinLock l;
	int initial;
	int count;
	int iteration;
	SpinCondition c;
	} Barrier;

typedef Barrier SpinBarrier;

/* Exported */

void SpinLockInit(), SpinLockDispose();

#ifdef PROFILE
void SpinLockAcquire(), SpinLockRelease(), SpinLockRelBlock();
#else
#define SpinLockAcquire(s) 	MY_LOCK(s)
#define SpinLockRelease(s)  MY_UNLOCK(s)
#endif

#define SLNPInit(s)		((s)->l = 0)
#define SLNPAcquire(s)	MY_LOCK(s)
#define SLNPRelease(s)	MY_UNLOCK(s)
#define SLNPTestAndGet(s)	MY_CLOCK(s)
#define SpinLockTestAndGet(s)	MY_CLOCK(s)

void LockInit(), LockAcquire(), LockRelease(), LockDispose();

void ConditionInit(), ConditionSignal(), ConditionWait(), ConditionDispose();
#define ConditionWaitLock(c,l)  	(ConditionWait(c,l), LockAcquire(l))

void SpinConditionWait();

#define SpinConditionInit(c,n)		ConditionInit((Condition *)c,n)
#define SpinConditionSignal(c)		ConditionSignal((Condition *)c)
#define SpinConditionBroadcast(c)	ConditionBroadcast((Condition *)c)
#define SpinConditionDispose(c)		ConditionDispose((Condition *)c)
#define SpinConditionWaitLock(c,l)  (SpinConditionWait(c,l), SpinLockAcquire(l))

void SpinBarrierInit(), SpinBarrierWait(), BarrierInit(), BarrierWait();
int BarrierCheckIn();

#define SpinBarrierCheckIn(b)		BarrierCheckIn((Barrier *)b)
#define SpinBarrierHit(b)			SpinBarrierWait(b, SpinBarrierCheckIn(b))
#define SpinBarrierDispose(b)		BarrierDispose((Barrier *)b)

#define BarrierHit(b)				BarrierWait(b, BarrierCheckIn(b))

