#
# Arc.tcl
# ------------------------------------------------------------------------
# A CodeObj composed of various ArcElbows to form an arc connecting
# two nodes.
# ------------------------------------------------------------------------
# @(#) $Id: Arc.tcl,v 1.9 1996/03/05 17:07:07 emery Exp $
#
# ------------------------------------------------------------------------
# AUTHOR:
#
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================


class Code::Arc {

    inherit Code::CodeObj

    constructor {w port} { Code::CodeObj::constructor $w } {}
    destructor {
	foreach elbow $elbows {
	    catch { $canvas delete [$elbow outgoing] }
	    catch { $canvas delete [$elbow incoming] }
	    catch { rename $elbow "" }
	}

	set arc ""
	$startport detach-arc $name

	# since we can destroy an arc before it is
	# attached to its endport, we need this check.

	if { $endport != "" } {
	    catch { $endport detach-arc $name }
	}
    }

    public method bitmap {}
    public method add-elbow {x y}
    public method delete-elbow {elbow}
    public method save {out}
    public method select {}
    public method deselect {}
    public method move {dx dy}
    public method arc_end {port}
    public method any_motion {x y}
    public method arc_move_endpoints_by {dx dy}
    public method attach_start {port}
    public method attach_end {port}
    public method translate {ProgramAST GraphAST}
    public method attributes {}
    public method arc_move_startpoints_by {dx dy}
    public method build_arcpoints {arcpoints}
    public method reshape {}
    public method no-reshape {}

    public variable elbows

    public variable Name ""
    public variable Rule ""

    # Rule, above, is simply the concatenation of the
    # below pieces in order (with separators taken from
    # separators array, defined above).

    public variable leftcallindex ""
    public variable leftnodeindex ""
    public variable leftident     ""
    public variable leftportindex ""

    public variable rightcallindex ""
    public variable rightnodeindex ""
    public variable rightident     ""
    public variable rightportindex ""



    common arcclass

    #  This array defines the "features" of the
    #  different arc classes.

    #  The first attribute defines what will be in the
    #  listbox:
    #
    #  0 = <disabled>
    #  1 = input ports
    #  2 = output ports
    #  3 = shared variables from NameSharing node
    #  4 = shared variables from Comp node
    #  5 = InInterface & CreationParameter nodes in subgraph
    #  6 = OutInterface nodes in subgraph

    #  The second attribute defines the kind of indices which
    #  may be entered:
    #
    #  0 = no indices
    #  1 = node index
    #  2 = node & port index
    #  3 = call, node, & port index
    #

    set arcclass(InInterfaceNode,CompNode,left)          {0 0}
    set arcclass(InInterfaceNode,CompNode,right)         {1 0}
    set arcclass(InInterfaceNode,NameSharingNode,left)   {0 0}
    set arcclass(InInterfaceNode,NameSharingNode,right)  {3 0}
    set arcclass(InInterfaceNode,OutInterfaceNode,left)  {0 0}
    set arcclass(InInterfaceNode,OutInterfaceNode,right) {0 0}
    set arcclass(InInterfaceNode,CallNode,left)          {0 0}
    set arcclass(InInterfaceNode,CallNode,right)         {5 0}
    set arcclass(CompNode,OutInterfaceNode,left)         {2 0}
    set arcclass(CompNode,OutInterfaceNode,right)        {0 0}
    set arcclass(CallNode,OutInterfaceNode,left)         {6 0}
    set arcclass(CallNode,OutInterfaceNode,right)        {0 0}
    set arcclass(CompNode,CompNode,left)                 {2 2}
    set arcclass(CompNode,CompNode,right)                {1 2}
    set arcclass(CompNode,NameSharingNode,left)          {4 1}
    set arcclass(CompNode,NameSharingNode,right)         {3 1}
    set arcclass(CompNode,CallNode,left)                 {2 2}
    set arcclass(CompNode,CallNode,right)                {5 3}
    set arcclass(CallNode,CompNode,left)                 {6 3}
    set arcclass(CallNode,CompNode,right)                {1 2}
    set arcclass(CallNode,CallNode,left)                 {6 3}
    set arcclass(CallNode,CallNode,right)                {5 3}
    set arcclass(CallNode,NameSharingNode,left)          {6 2}
    set arcclass(CallNode,NameSharingNode,right)         {3 1}

    common message

    #
    #  This defines the string to be placed above a pane,
    #  depending on the type of node and which pane.
    #

    set message(InInterfaceNode,*,left)        "Connect this input..."
    set message(CallNode,*,left)               "Connect this parameter..."
    set message(CompNode,NameSharingNode,left) "Connect this shared variable..."
    set message(CompNode,*,left)               "Connect this port..."

    set message(*,OutInterfaceNode,right) "...to this output."
    set message(*,NameSharingNode,right)  "...to this shared variable."
    set message(*,CallNode,right)         "...to this parameter."
    set message(*,CompNode,right)         "...to this port."


    common separators

    set separators(InInterfaceNode,CompNode)          {"" "" "" "" ""}
    set separators(InInterfaceNode,NameSharingNode)   {"" "" "" "" ""}
    set separators(InInterfaceNode,OutInterfaceNode)  {"" "" "" "" ""}
    set separators(InInterfaceNode,CallNode)          {"" "" "" "" ""}
    set separators(CompNode,OutInterfaceNode)         {"" "" "" "" ""}
    set separators(CallNode,OutInterfaceNode)         {"" "" "" "" ""}
    set separators(CompNode,CompNode)                 {"" "." "=>" "" "."}
    set separators(CompNode,NameSharingNode)          {"" "." "=>" "" "."}
    set separators(CompNode,CallNode)                 {"" "." "=>" "." "."}
    set separators(CallNode,CompNode)                 {"." "." "=>" "" "."}
    set separators(CallNode,CallNode)                 {"." "." "=>" "." "."}
    set separators(CallNode,NameSharingNode)          {"." "." "=>" "" "."}


    # attribute values

    protected variable attributes {Name Rule}

    protected variable startport ""
    protected variable endport ""

    protected variable current_elbow ""
    protected variable prev_elbow ""

    protected variable selectable 1  ;# the arc is not selectable when we can see the elbows


    private method arc_move_startpoints {x y}
    private method arc_move_endpoints {x y}
    public proc update_annotation {args}


}


body Code::Arc::constructor {w port} {
    attach_start $port
    scan [$port arc_position] "%f %f" x y
    add-elbow $x $y
    eval itk_initialize
}




body Code::Arc::bitmap {} {
    return "@[$canvas info variable ReleaseDirectory -value]/bitmaps/ArcPicture.bit"
}


body Code::Arc::add-elbow {x y} {
    set prev_elbow $current_elbow
    
    if { $prev_elbow != "" } {
	$prev_elbow update_coords "$x $y"
    }
    
    set current_elbow [$canvas create ArcElbow $x $y $this]
    lappend elbows $current_elbow
    
    set newline [$canvas create line $x $y $x $y -arrow last -parent $this]
    
    # add its entire heritage as tags.
    
    $canvas addtag $this withtag $newline

    foreach classname [$this info heritage] {
	$canvas addtag $classname withtag $newline
    }

    $current_elbow set_incoming $newline
    if {$prev_elbow != "" } {
	$prev_elbow set_outgoing $newline
    }
}

body Code::Arc::delete-elbow {elbow} {
    set i 0
    set n [llength $elbows]
    while {$i < $n} {
	if {![string compare [lindex $elbows $i] $elbow]} {

	    # if this is the first elbow
	    if {$i == 0} {
		scan [$canvas coords [$elbow incoming]] "%f %f %f %f" x y d d
		set prevcoords "$x $y"

	    } else {
		set prev [lindex $elbows [expr $i-1]]
		set prevcoords [$prev coords]
	    }

	    # if this isn't the last elbow...

	    if {$i < [expr $n-1]} {
		set next [lindex $elbows [expr $i+1]]
		set line [eval $canvas create line $prevcoords [$next coords] \
			-arrow last -parent $this]

		# add its entire heritage as tags.
		
		foreach classname [$this info heritage] {
		    $canvas addtag $classname withtag $line
		}

		$next set_incoming $line
		$canvas delete [$elbow outgoing]
	    }

	    $canvas delete [$elbow incoming]

	    # if this isn't the first elbow

	    if {$i != 0 } {
		$prev set_outgoing $line
	    }

	    set elbows [lreplace $elbows $i $i]
	    break
	}
	incr i
    }

    #
    # ensure that (in case of deletion) the last elbow has
    # no arrow drawn on the end of the line entering it
    #

    if {$elbows != ""} {
	$canvas itemconfigure [[last $elbows] incoming] -arrow none
    }
}


body Code::Arc::save {out} {

    puts $out "set obj \[\$objindex($startport) create_arc\]"
    puts $out "lappend objlist \$obj"

    set points ""
    foreach elbow $elbows {
	lappend points [$elbow coords]
    }

    if {$points != ""} {
	puts $out "\$obj build_arcpoints \{$points\}"
    }

    # puts $out "set objindex($endport) $endport"
    puts $out "\$objindex($endport) complete_arc"

    #
    # output all attribute values
    #

    set attribs \
	    {Name Rule leftcallindex leftnodeindex leftident leftportindex \
	    rightcallindex rightnodeindex rightident rightportindex}

    foreach i $attribs {
	puts $out "\$obj configure -$i {[$this info variable $i -value]}"
    }

}

body Code::Arc::select {} {
    # return

    if {$selectable} {
	Code::CodeObj::select
	$canvas itemconfigure [[first $elbows] incoming] -fill [$canvas info variable SelectedFGColor -value]
	foreach elbow $elbows {
	    $canvas itemconfigure [$elbow outgoing] -fill [$canvas info variable SelectedFGColor -value]
	}
    }
}

body Code::Arc::reshape {} {
    set selectable 0
    set selected 0
    map "apply reshape" $elbows
}

body Code::Arc::no-reshape {} {
    set selectable 1
    map "apply no-reshape" $elbows
}

body Code::Arc::deselect {} {
    if {$selectable} {
	Code::CodeObj::deselect
	$canvas itemconfigure [[first $elbows] incoming] -fill [$canvas info variable FGColor -value]
	foreach elbow $elbows {
	    $canvas itemconfigure [$elbow outgoing] -fill [$canvas info variable FGColor -value]
	}
    }
}

body Code::Arc::move {dx dy} {

    
    set last [expr [llength $elbows]-1]
    foreach elbow [lrange $elbows 0 [expr $last-1]]  {
	$elbow move $dx $dy
    }

    return ;# disabling everything below

    set items [$canvas selected_items]
    $canvas deselect_all
    [$startport attached-node] select_group
    [$endport attached-node] select_group
    $canvas move_selected_by $dx $dy
    $canvas deselect_all

    $canvas select_items $items

}

body Code::Arc::arc_end {port} {

    # attach the arc end to the port correctly

    set points ""
    foreach elbow $elbows {
	lappend points [$elbow coords]
    }

    attach_end $port
    $port attach-arc $this
    $current_elbow update_coords [$port arc_position]
    $current_elbow configure -editable 0
    $canvas itemconfigure [$current_elbow incoming] -arrow none

    # we are no longer drawing an arc, so...

    $canvas configure -currentDrawingArcItem ""

    # set the default name of the arc

    set Name [concat \
	    [[$startport attached-node] info variable Name -value] \
	    "->" \
	    [[$endport attached-node] info variable Name -value]]

}

body Code::Arc::any_motion {x y} {

    #
    # we need to keep the end of the arc
    # off of the arc itself (so we can
    # click on the canvas and input ports)
    #

    scan [$canvas coords [$current_elbow incoming]] "%f %f %f %f" x1 y1 x2 y2

    if { $x > $x1 } {
	set canvasX [$canvas canvasx [expr $x-1]]
    } else {
	set canvasX [$canvas canvasx [expr $x+1]]
    }

    set canvasY [$canvas canvasy $y]

    $current_elbow update_coords "$canvasX $canvasY"
}

body Code::Arc::arc_move_endpoints {x y} {
    [last $elbows] update_coords "$x $y"
}

body Code::Arc::arc_move_endpoints_by {dx dy} {
    scan [[last $elbows] coords] "%f %f" x y
    arc_move_endpoints [expr $x+$dx] [expr $y+$dy]
}

body Code::Arc::arc_move_startpoints {x y} {
    set line [[first $elbows] incoming]
    scan [$canvas coords $line] "%f %f %f %f" x1 y1 x2 y2
    $canvas coords $line $x $y $x2 $y2
}

body Code::Arc::arc_move_startpoints_by {dx dy} {
    set line [[first $elbows] incoming]
    scan [$canvas coords $line] "%f %f %f %f" x1 y1 x2 y2
    $canvas coords $line [expr $x1+$dx] [expr $y1+$dy] $x2 $y2
}


body Code::Arc::attach_start {port} {
    set startport $port
}

body Code::Arc::attach_end {port} {
    set endport $port
}

body Code::Arc::translate {ProgramAST GraphAST} {
    return [c2_CreateArcAST $ProgramAST $GraphAST $Name $id "$Rule" \
	    [[$startport attached-node] id] \
	    [[$endport attached-node] id]]
}

body Code::Arc::build_arcpoints {arcpoints} {
    set n [llength $arcpoints]
    set coords [lindex $arcpoints 0]
    $current_elbow update_coords $coords
    for {set i 1} {$i < $n} {incr i} {
	eval add-elbow $coords
	set coords [lindex $arcpoints $i]
	$current_elbow update_coords $coords
    }
}

body Code::Arc::attributes {} {


    #
    #  This is a complex procedure,
    #  since in effect there are 12 different
    #  varieties of arcs.


    set leftnode [$startport attached-node]
    set rightnode   [$endport attached-node]

    set leftnodeclass  [info namespace tail [$leftnode info class]]
    set rightnodeclass [info namespace tail [$rightnode info class]]


    #
    #  Copy all relevant values
    #  out to the c2_tmp array for use
    #  by the dialog box.
    #

    global c2_tmp

    # regsub -all "\[\n\t\]" $Rule "" Rule  ;# kill newlines, etc.

    set c2_tmp(annotation) $Rule
    set c2_tmp(separator) $separators($leftnodeclass,$rightnodeclass)

    set c2_tmp(leftcallindex) $leftcallindex
    set c2_tmp(leftnodeindex) $leftnodeindex
    set c2_tmp(leftportindex) $leftportindex
    set c2_tmp(leftident) $leftident

    set c2_tmp(rightcallindex) $rightcallindex
    set c2_tmp(rightnodeindex) $rightnodeindex
    set c2_tmp(rightportindex) $rightportindex
    set c2_tmp(rightident) $rightident
    
    foreach i {left right} {
	trace variable c2_tmp(${i}callindex) w "Code::Arc::update_annotation"
	trace variable c2_tmp(${i}nodeindex) w "Code::Arc::update_annotation"
	trace variable c2_tmp(${i}portindex) w "Code::Arc::update_annotation"
    }

    #
    #  Place the icons and names of the two nodes
    #  up on the dialog box.
    #
    
    
    set attrframe ._Arcattributes

    ##### ******** ######

    foreach i {left right} {

	[$attrframe.topframe.$i.middle.ports subwidget hlist] configure -browsecmd \
		"function {port} \{global c2_tmp ; \
		set c2_tmp(${i}ident) \$port ; \
		Code::Arc::update_annotation \}"
	
	# Display icon and name of node.
	
	$attrframe.topframe.$i.top.node.icon configure \
		-image [[set ${i}node] thisimage]
	$attrframe.topframe.$i.top.node.label configure \
		-text [[set ${i}node] info variable Name -value]

	# Display appropriate pane title.

	if { [catch { set msg $message($leftnodeclass,$rightnodeclass,$i) }] } {
	    switch $i {
		left {
		    set msg $message($leftnodeclass,*,$i)
		}
		right {
		    set msg $message(*,$rightnodeclass,$i)
		}
	    }
	}

	$attrframe.topframe.$i.top.title configure -text $msg


	#
	#  extract the items to appear
	#  in each pane.
	#
	
	set items ""

	set panetype [lindex $arcclass($leftnodeclass,$rightnodeclass,$i) 0]

	switch $panetype {
	    0 {
		# disabled
	    }
	    1 -
	    2 -
	    3 -
	    4 {
		switch $panetype {
		    1 { set ports [[set ${i}node] cget -InputPorts] }
		    2 { set ports [[set ${i}node] cget -OutputPorts] }
		    3 { set ports [[set ${i}node] cget -SharedVars] }
		    4 { set ports [[set ${i}node] cget -SharedVariables] }
		}

		
		#  Remove all tabs & newlines; reduce spaces.
		#  This will put the list into a
		#    type var type var...
		#  format.
		
		regsub -all "\[\n\t\]" $ports " " ports
		regsub -all "\ \ *" $ports " " ports
		regsub -all ";" $ports " " ports

		# NameSharing declarations may have components in braces
		# which need to be removed before processing, so we do that here.

		if { $panetype == 3 } {
		    regsub -all {\{[^\}]*\}} $ports "" ports
		}
		
		#  Add the port names to the list.
		
		switch $panetype {
		    1 -
		    2 -
		    3 {
			set v 0
			foreach port $ports {
			    incr v
			    # Skip the odd entries (they're type names).
			    if { [expr $v/2.0] == [expr round($v/2.0)] } {
				lappend items $port
			    }
			}
		    }
		    4 {
			set v 0
			foreach port $ports {
			    incr v
			    # Include only the 2nd entry on every line.
			    if { [expr int(fmod($v,3))] == 2 } {
				lappend items $port
			    }
			}
		    }
		}
	    }
	    5 {
		# InInterface & CreationParameter nodes from a
		# Call node's subgraph

		set isalias [[set ${i}node] info variable IsAliasToGraph -value]
		if $isalias {
		    set subgraph [[[set ${i}node] info variable AliasGraph -value] info variable subgraph -value]
		} else {
		    set subgraph [[set ${i}node] info variable subgraph -value]
		}
		if { $subgraph != "" } {
		    foreach node [winfo children $subgraph] {
			if { [$node isa ::Code::InInterfaceNode] || \
				[$node isa ::Code::CreationParameterNode] } {
			    lappend items [$node info variable Name -value]
			}
		    }
		}
	    }
	    6 {
		# OutInterface nodes from a Call node's subgraph

		set isalias [[set ${i}node] info variable IsAliasToGraph -value]
		if $isalias {
		    set subgraph [[[set ${i}node] info variable AliasGraph -value] info variable subgraph -value]
		} else {
		    set subgraph [[set ${i}node] info variable subgraph -value]
		}
		if { $subgraph != "" } {
		    foreach node [winfo children $subgraph] {
			if { [$node isa ::Code::OutInterfaceNode] } {
			    lappend items [$node info variable Name -value]
			}
		    }
		}
	    }
	}

	#
	# Enable/disable appropriate index entry widgets.
	#

	set number [lindex $arcclass($leftnodeclass,$rightnodeclass,$i) 1]

	set indices "$attrframe.topframe.$i.bottom.nodeindex $attrframe.topframe.$i.bottom.portindex $attrframe.topframe.$i.bottom.callindex"

	foreach index $indices {
	    $index configure -state disabled
	}

	for { set j 0 } { $j < $number } { incr j } {
	    [lindex $indices $j] configure -state normal
	}

	# add the items to the list
	
	[$attrframe.topframe.$i.middle.ports subwidget hlist] delete all
	
	foreach item $items {
	    [$attrframe.topframe.$i.middle.ports subwidget hlist] add "$item" -text "$item"
	}

	

    }


    #
    #  Set the selections according to the current arc annotation.
    #

    foreach i {left right} {
	if { [set c2_tmp(${i}ident)] != "" } {
	    catch { [$attrframe.topframe.$i.middle.ports subwidget hlist] selection set [set c2_tmp(${i}ident)] }
	}
    }


    #
    #  (finally) we are ready to bring up the dialog box and
    #  handle possible changes.
    #
    
    update_annotation

    if {[._Arcattributes activate]} {
	set Rule $c2_tmp(annotation)
	set leftcallindex $c2_tmp(leftcallindex)
	set leftnodeindex $c2_tmp(leftnodeindex)
	set leftportindex $c2_tmp(leftportindex)
	set leftident $c2_tmp(leftident)
	
	set rightcallindex $c2_tmp(rightcallindex)
	set rightnodeindex $c2_tmp(rightnodeindex)
	set rightportindex $c2_tmp(rightportindex)
	set rightident $c2_tmp(rightident)
    }

    # Remove traces.

    foreach i {left right} {
	trace vdelete c2_tmp(${i}callindex) w "Code::Arc::update_annotation"
	trace vdelete c2_tmp(${i}nodeindex) w "Code::Arc::update_annotation"
	trace vdelete c2_tmp(${i}portindex) w "Code::Arc::update_annotation"
    }

}

body Code::Arc::update_annotation {args} {
    global c2_tmp

    set c2_tmp(annotation) \
	    [format "%s%s%s%s%s%s%s%s%s%s%s%s%s" \
	    $c2_tmp(leftcallindex) \
	    [lindex $c2_tmp(separator) 0] \
	    $c2_tmp(leftnodeindex) \
	    [lindex $c2_tmp(separator) 1] \
	    $c2_tmp(leftident) \
	    $c2_tmp(leftportindex) \
	    [lindex $c2_tmp(separator) 2] \
	    $c2_tmp(rightcallindex) \
	    [lindex $c2_tmp(separator) 3] \
	    $c2_tmp(rightnodeindex) \
	    [lindex $c2_tmp(separator) 4] \
	    $c2_tmp(rightident) \
	    $c2_tmp(rightportindex)]

    # This should NOT be necessary, but we seem to need this in order
    # to force an update.

    ._Arcattributes.bottomframe.annoframe.arc_annotation configure -text "x"

}
