#
# Label.tcl
# ------------------------------------------------------------------------
# An editable text widget. When created, acts like a standard text widget.
# When deselected, acts like a canvas text object.
# ------------------------------------------------------------------------
# @(#) $Id: Label.tcl,v 1.6 1996/03/05 17:07:18 emery Exp $
#
# ------------------------------------------------------------------------
# AUTHOR:
#
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================


class Code::Label {

    inherit Code::CodeObj

    constructor {w x y args} {
	set text [extract_arg "-text" args]
	eval Code::CodeObj::constructor $w $args
    } {}

    destructor {
	# WORKAROUND? Why catch this?
	catch { destroy $entrywidget }

	$canvas delete $dispobject
	$canvas delete $textwin
    }


    public method attributes {}
    public method deselect {}
    public method select {}

    public method editmode {}
    public method dispmode {}
    public method set_text {t}
    public method get_text {}

    public method notify {fn}
    
    public variable text "" {update_text}
    public variable attributes {text}

    protected variable label
    protected variable notify_fn ""

    # If editmode = 1, use the text widget; if it is zero,
    # use a canvas text object.

    protected variable editmode 0
    protected variable dispobject  ;# the canvas text object
    protected variable entrywidget
    protected variable textwin     ;# the canvas window holding the entry widget
    protected variable insertion_point 0  ;# where the cursor last was in the entry widget


    private method update_text {} {}

}


body Code::Label::constructor {w x y args} {

    # Initialize a text widget with minimum allowed ht. & width.

    set entrywidget [uplevel #0 entry [file rootname [info namespace tail $canvas]].text[$canvas new_id] -width 1]

    # Shuffle the widgets a bit so that the text widget
    # will not appear on top of the scrollbars.

    lower [info namespace tail $canvas]
    lower $entrywidget
    raise $entrywidget [info namespace tail $canvas]

    # Get the text from the argument list and insert
    # it into the text widget, adjusting the width.

    $entrywidget insert end $text
    $entrywidget configure -width [string length $text]


    # Create the text widget.

    set textwin \
	    [eval $canvas create window $x $y \
	    -anchor w -parent $this]

    # Create the canvas object which will also contain the text.

    set dispobject [$canvas create text $x $y -anchor w -width 0 -parent $this]

    update_text

    # Add the entire heritage as tags to both objects.

    foreach object "$textwin $dispobject" {
	$canvas addtag $this withtag $object
	foreach classname [$this info heritage] {
	    $canvas addtag $classname withtag $object
	}
    }


    # Have it update the text variable on every key-press.
    # This is stupid but simple.
    # Also update the widget's width based upon the length of the string.

    # We have to change the order of the bindings so that our binding
    # which updates the text variable occurs *after* the insertion of
    # the text into the entry widget.
    #
    # We do this by removing it from the bindtags and then appending
    # it to the end of the bindtags.
    
    set btags [bindtags $entrywidget]
    set btags [lreplace $btags 0 0]     ;# Trash the first item ($entrywidget).
    lappend btags $entrywidget          ;# Put $entrywidget at the end of the list.
    bindtags $entrywidget $btags        ;# Set the bindtags.

    ::bind [info namespace tail $entrywidget] <Any-Key> \
	    "$this set_text \[$entrywidget get\] ; \
	    $entrywidget configure -width \[string length \[$this get_text\]\]"

    eval itk_initialize $args

}

#
# switch to edit mode
#

body Code::Label::editmode {} {
    if {$editmode} return
    set editmode 1

    # this is a kludge so that it will appear on the items to be
    # deselected when we click away from it.
    Code::CodeObj::select
    
    $canvas dchars $dispobject 0 end
    ::focus -force [info namespace tail $entrywidget]
    Code::CodeCanvas::SetCurrentWindow $entrywidget
    update_text
    $entrywidget icursor $insertion_point 	;# restore the old cursor position
    $canvas itemconfigure $textwin -window $entrywidget ;# display the entry widget
}

#
# switch to display mode
# (hides text widget, displays
#  canvas text object)
#

body Code::Label::dispmode {} {
    if {!$editmode} return
    # store the cursor position for next time
    set insertion_point [$entrywidget index insert]
    set editmode 0
    $canvas itemconfigure $textwin -window {} ;# remove the entry widget from the screen
    update_text
    # ::focus -force [info namespace tail $canvas]
    CodeCanvas::SetCurrentWindow $canvas
}

body Code::Label::set_text {t} {
    set text $t
}

body Code::Label::get_text {} {
    return $text
}

body Code::Label::select {} {
    Code::CodeObj::select
    dispmode
    $canvas itemconfigure $dispobject -fill [$canvas info variable SelectedFGColor -value]
}

body Code::Label::deselect {} {
    Code::CodeObj::deselect
    dispmode
    $canvas itemconfigure $dispobject -fill [$canvas info variable FGColor -value]
}

body Code::Label::notify {fn} {
    set notify_fn $fn
}

body Code::Label::attributes {} {
    Code::CodeObj::attributes
    update_text
}

body Code::Label::update_text {} {
    if { [winfo exists [info namespace tail $this]] } {
	if {!$editmode} {
	    $canvas dchars $dispobject 0 end
	    $canvas insert $dispobject insert $text
	}
	$entrywidget delete 0 end
	$entrywidget insert insert $text
	$entrywidget configure -width [string length $text]
	if { [string compare $notify_fn ""] } {
	    catch {eval "$notify_fn \"$text\""}
	}
    }
}
