#
# Node.tcl
# ------------------------------------------------------------------------
# This is the base class for all *Node CODE objects. It provides automatic
# comment creation and management as well as handling ports.
# ------------------------------------------------------------------------
# @(#) $Id: Node.tcl,v 1.6 1996/03/05 17:07:22 emery Exp $
#
# ------------------------------------------------------------------------
# AUTHOR:
#
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================


class Code::Node {

    inherit Code::CodeObj

    constructor {w x y args} { eval Code::CodeObj::constructor $w $args } {}

    destructor {
	$canvas delete $node
	
	# since ports are always attached to nodes,
	# it probably is redundant to try and delete them
	# here.

	foreach port $ports {
	    catch { rename $port "" }
	}
    }


    public method addport {type direction {limit -1}}
    public method attributes {}
    public method bitmap {}
    public method deselect {}
    public method save {out}
    public method scale {}
    public method set_name {n}
    public method select {}
    public method thisimage {}
    public method translate {ProgramAST GraphAST}

    protected method update_node {}
    protected method draw_node {x y}

    protected method update_name {args}



    public variable Name "" {$label configure -text $Name ; update_name}

    private common node_number 0

    protected variable node
    protected variable label
    protected variable attributes {Name}
    protected variable ports ""


}


body Code::Node::constructor {w x y args} {

    # keep a count of the nodes created so far
    incr node_number

    #
    # label creation
    #

    # put up a comment with the text "*Node n"

    set Name "Untitled"
    set label [$w create Label [expr $x+25*[$canvas info variable scale -value]] $y \
	    -text $Name]

    # tell the label to change Node's Name value
    # to whatever the label is changed to.

    $label notify "$this set_name"

    #
    # draw the node itself
    #

    draw_node $x $y

    # add its entire heritage as tags.

    $w addtag $this withtag $node

    foreach classname [$this info heritage] {
	$canvas addtag $classname withtag $node
    }

    $this scale

    #
    # group label together with this
    #

    $canvas group "$label $this"

}

body Code::Node::addport {type direction {limit -1}} {
    # figure out where to put it
    set x [lindex [$canvas coords $this] 0]
    set y [lindex [$canvas coords $this] 1]
    set scale [$canvas info variable scale -value]
    switch $direction {
	bottom {
	    set y [expr round($y+13*$scale)]
	}
	top {
	    set y [expr round($y-12*$scale)]
	}
	left {
	    set x [expr round($x-12*$scale)]
	}
	right {
	    set x [expr round($x+13*$scale)]
	}
    }
    set port [$canvas create $type $direction $this $x $y]
    lappend ports $port
    foreach g [split [$this group_name] .] {
	$canvas add_to_group $g $port
    }

    $port set_group [$this group_name]

    # changed from $port configure -group ...

    $port configure -limit $limit
    $port attach-node $this
    return $port
}


body Code::Node::attributes {} {
    set prevname $Name
    Code::CodeObj::attributes

    # try to modify the label (may not exist)

    if { [catch { $label configure -text $Name }] } {

	# recreate the label if Name has been edited

	if { [string compare $prevname $Name] } {
	    scan [$canvas coords $node] "%d %d %d %d" x y dummy dummy
	    set label [$canvas create Label [expr $x+25] $y -text $Name]

	    # tell the label to change Node's Name value
	    # to whatever the label is changed to.

	    $label notify "$this set_name"

	    # group the label to this

	    $canvas group "$label $this"

	}
    }

}

body Code::Node::deselect {} {
    if { !$selected } return
    Code::CodeObj::deselect
    foreach port $ports {
	$port deselect
    }
    update_node
    $canvas dtag $node selected
}


body Code::Node::draw_node {x y} {}


body Code::Node::save {out} {
    Code::CodeObj::save $out
    puts $out "\[\$obj info variable label -value\] configure -text {$Name}"

    #
    #  output a reference for all the ports
    #

    set index 0
    foreach port $ports {
	puts $out "set objindex($port) \
		\[lindex \[\$obj info variable ports -value\] $index\]"
	incr index
    }
}


body Code::Node::scale {} {
    set s [$canvas info variable scale -value]
    set os [$canvas info variable oldscale -value]
    if { (![string compare $os $s]) && $s } {
	return
    }
    set nodeimage [$canvas itemcget $node -image]
    set baseimage [image create photo -format GIF -file [$nodeimage cget -file]]

    if { $s >= 1 } {
	set ints [expr int($s)]
	$nodeimage copy $baseimage -shrink -zoom $ints $ints
    } else {
	set rs [expr int(1/$s)]
	$nodeimage copy $baseimage -shrink -subsample $rs $rs
    }
}


body Code::Node::select {} {
    if { $selected } return
    Code::CodeObj::select
    foreach port $ports {
	$port select
    }
    update_node
    $canvas addtag selected withtag $node
}

# This is a method used by label
# to alter the name value whenever the label
# is altered. We use this method because
# if label uses config, we end up with too many
# (unnecessary) updates.

body Code::Node::set_name {n} {
    set Name $n
    update_name
}


# we call this method every time
# that Name is updated -- needed by CallNode
# to update its window title.

body Code::Node::update_name {args} {}


body Code::Node::update_node {} {
    $canvas itemconfigure $node -image [thisimage]
    $this scale
}
