/* workers.c */

#include <stdio.h>
#include "c2_misc.h"
#include "c2_workers.h"

#ifdef _C2_DBGR_
#include <setjmp.h>
#include <signal.h>
#include <errno.h>
#include <debugQ.h>
#endif

shared int _c2_Terminated;  /* 1 iff term node has fired */

   /* structure for links in shared ready Q-- actually LIFO */

typedef struct _c2_Link {
   _c2_NodeBase *Node;
   struct _c2_Link *Next;
} _c2_RQLink;

static shared _c2_RQLink *Head;
static shared _c2_RQLink *FreeList;
static shared Lock RQLock;

#define MAXWORKERS 32

static shared Thread *WorkerList[MAXWORKERS];


   /* Signal a halt to the computation */

void _c2_StopAll()
{
   printf("_c2_StopAll called to end computation\n");
   _c2_Terminated = 1;
   fflush(stdout);
   fflush(stderr);
}


void _c2_KillComp(Reason)
   int Reason;
{
   fprintf(stderr, "\n\n*** CODE fatal runtime error: ");
   switch (Reason) {
     case _c2_CREPTWICE:
       fprintf(stderr, "Attempt to bind 2nd value to creation parameter\n");
       break;
     case _c2_RMFROMEMPTY:
       fprintf(stderr, "Attempt to remove from empty arc\n");
       break;
     default:
       fprintf(stderr, "Unknown problem (internal error?)\n");
       break;
   }
   fflush(stdout);
   fflush(stderr);

   _c2_Terminated = 1;
}

   /* Add a node to the Ready Q */

void _c2_EnQueue(NodeToAdd)
   _c2_NodeBase *NodeToAdd;
{
   _c2_RQLink *tmp;

#ifdef DEBUG
/*   printf("Adding node with UID %d to Ready Q\n",
	  NodeToAdd->UID); */
#endif

   LockAcquire(&(NodeToAdd->QueueLock));

   if (!(NodeToAdd->Crepped)) {  /* Node cannot be run until creps are bound */
      LockRelease(&(NodeToAdd->QueueLock));
      return;
   }

   switch (NodeToAdd->QueueStatus) {
    case _c2_IDLE:
      NodeToAdd->QueueStatus = _c2_ONQUEUE;
#     ifdef _C2_DBGR_
         _c2_timeUcPutOnRdyQ(NodeToAdd); /* performance measurement */
#     endif
      LockRelease(&(NodeToAdd->QueueLock));
      break;
    case _c2_RUNNING:
      NodeToAdd->QueueStatus = _c2_NEEDSRUN;
      LockRelease(&(NodeToAdd->QueueLock));
      return;
      break;

#   ifdef _C2_DBGR_
    case _c2_ONDBGQ:		/* Debugger will later put it for running */
#   endif 
    case _c2_ONQUEUE:
    case _c2_NEEDSRUN:
      LockRelease(&(NodeToAdd->QueueLock));
      return;
      break;
    default:
      fprintf(stderr, "Error: Illegal Q status in _c2_EnQueue\n");
      fflush(stdout);
      fflush(stderr);
      LockRelease(&(NodeToAdd->QueueLock));
      _c2_StopAll();
   }

   /* If we get here, we need to Q to node */

   LockAcquire(&RQLock);
   if (FreeList != 0) {
      tmp = FreeList;
      FreeList = FreeList->Next;
   } else
     tmp = (_c2_RQLink *) _c2_shmalloc(sizeof(_c2_RQLink));
   tmp->Node = NodeToAdd;
   tmp->Next = Head;
   Head = tmp;
   LockRelease(&RQLock);
   return;
}


   /* Get a node to run from the ready Q, return 0 if Q empty */

static _c2_NodeBase *_c2_PopNode()
{
   _c2_NodeBase *ReturnTask;
   _c2_RQLink *TmpHead;

   LockAcquire(&RQLock);

   if (Head == 0) {
      LockRelease(&RQLock);
      return 0;
   }

   TmpHead = Head;
   Head = Head->Next;
   ReturnTask = TmpHead->Node;

   TmpHead->Next = FreeList;
   FreeList = TmpHead;

   LockRelease(&RQLock);

   LockAcquire(&(ReturnTask->QueueLock));

   switch (ReturnTask->QueueStatus) {
    case _c2_ONQUEUE:
      ReturnTask->QueueStatus = _c2_RUNNING;
      LockRelease(&(ReturnTask->QueueLock));
      break;
    default:
      fprintf(stderr, "Error: Illegal Q status in _c2_PopNode\n");
      fflush(stderr);
      fflush(stdout);
      LockRelease(&(ReturnTask->QueueLock));
      _c2_StopAll();
   }

   return ReturnTask;
}


   /* Just waste some time-- don't hog lock */

static void _c2_spin(n)
   int n;
{
   int i;
   for (i = 0; i < n; i++);
}

#ifdef _C2_DBGR_


void _c2_emptyRdyQ()
{
   while(_c2_PopNode())
       ;
}

static struct {
   jmp_buf env;			/* map environment with pid of processes */
   int pid;			/* pid of the process running the workertasks */
} WrkrEnv[MAXWORKERS];

/* signal handler */

void _c2_inthandler(sig)
int sig;
{
   int i;
   int pid;

   pid =getpid();
   for (i=0; i < numProcessors; i++){        
      if (WrkrEnv[i].pid == pid) {
	 if (sig != _c2_ERRSIGNAL) { /* not debugger code's error */
	    psignal("caught", sig);
	    if (_c2_debugsignals) {
	       fprintf(stderr, "interrupt caught by pid = %d.\n", pid); 
	    }
	 }
	 _c2_setGlobalState(_c2_STOPPED);
	 _c2_emptyRdyQ();
	 longjmp(WrkrEnv[i].env, 1);
      }
   }
   _c2_reinitWrkrs();
}

/* 
 * To handle the child. Specifically for SIGSEGV that can not be
 * handled because its handler messes up the shbrk() and shmalloc()
 */

#include <sys/wait.h>

void _c2_ChldHndler()
{
   int pid;
   union wait *status;

   pid = wait3(status, WNOHANG);
   if (pid == 0) {
      error("Why a SIGCHLD");
   } else {
      _c2_setGlobalState(_c2_STOPPED);
      if (status->w_termsig == SIGSEGV) {
	 psignal("A worker died", status->w_termsig);
      } else {
	 psignal("detected", status->w_termsig);
      } 
      fflush(stdout);
   }
}


/* in case of abnormal abort, the debugger must exist gracefully */

void _c2_KillWorkers()
{
   int i;
   _c2_StopAll();
   /* 
    * in case some processes are stalled and can not
    * terminate normally, do it the hard way!
    */
   for (i=0; i<numProcessors; i++){
      if (_c2_debugsignals)
	  printf("killing worker %d\n", WrkrEnv[i].pid);
      kill(WrkrEnv[i].pid, SIGKILL);
   }
}   
#endif /* _C2_DBGR_ */



   /* Process for a worker-- a FastThreads Thread that runs nodes on the
      ready Q */


static void _c2_WorkerTask(ID)
   int ID;
{
   _c2_NodeBase *NodeToRun;
   int RunCount = 0;
   int ReRunCount = 0;
   int IdleCount = 0;
   int TotalCount = 0;
   int Skip = 1;



#  ifdef _C2_DBGR_
   WrkrEnv[ID].pid = getpid();	/*  associate pid with env entries */
   if (setjmp(WrkrEnv[ID].env) != 0){
      if (_c2_debugsignals)
	  printf("workerTask %d: Returned from interrupt\n", ID);
   }
   _c2_sig_install(_c2_inthandler);
#  endif/* _C2_DBGR_ */


#  ifdef DEBUG  /* Peter's debugging stuff for worker tasks*/
   printf("Worker %d starts\n", ID);
#  endif


      /*      if (TotalCount % 50000 == 0) {
	      printf("Worker %d: Total %d, Idle %d, Runs %d, Reruns %d\n",
	      ID, TotalCount, IdleCount, RunCount, ReRunCount);
	      fflush(stdout);
	      } */


   while (!_c2_Terminated) {
      TotalCount++;
#     ifdef _C2_DBGR_
      if (_c2_GlobalState == _c2_STOPPED)  continue; 
#     endif

      /* Check ready queue for a node to run */
      if ( Head == 0 || (NodeToRun = _c2_PopNode()) == 0) {
#       ifdef _C2_DBGR_
	 if (_c2_sizeOutDbgQ() == 0 || (NodeToRun = _c2_PopOutDbgQ()) == 0) {
	    _c2_assertWrkrState(ID);
#       endif
	    IdleCount++;
	    Skip <<= 1;
	    if (Skip > 1024) Skip = 32;
	    _c2_spin(Skip); 
	    continue;		/* nothing to do */
#       ifdef _C2_DBGR_
	 }
#       endif
      }
      Skip = 32;
      RunCount++;

#   ifdef DEBUG
      printf("Worker %d Running a %d\n", ID, NodeToRun->UID);
    fflush(stdout);
#     endif

    Again:
      NodeToRun->CompProc(NodeToRun);

      LockAcquire(&(NodeToRun->QueueLock));

      switch (NodeToRun->QueueStatus) {
       case _c2_RUNNING:
	 NodeToRun->QueueStatus = _c2_IDLE;
	 LockRelease(&(NodeToRun->QueueLock));
	 break;

#      ifdef _C2_DBGR_
       case _c2_ONDBGQ:		/* debugger will run it later on */
	 LockRelease(&(NodeToRun->QueueLock));
	 break;
#      endif

       case _c2_NEEDSRUN:
	 NodeToRun->QueueStatus = _c2_RUNNING;
#        ifdef _C2_DBGR_
           _c2_timeUcPutOnRdyQ(NodeToRun); /* performance measurement */
#        endif
	 LockRelease(&(NodeToRun->QueueLock));
#        ifdef DEBUG
	 printf("Worker %d Re-running a %d\n", ID, NodeToRun->UID);
	 fflush(stdout);
#        endif
	 ReRunCount++;
	 TotalCount++;
	 goto Again;
	 break;
       default:
	 fprintf(stderr, "Error: Illegal Q status in _c2_WorkerTask\n");
         fflush(stdout);
         fflush(stderr);
	 LockRelease(&(NodeToRun->QueueLock));
	 _c2_StopAll();
	 break;
      }
   }
   printf("Worker %d: Total %d, Idle %d, Runs %d, Reruns %d\n",
	  ID, TotalCount, IdleCount, RunCount, ReRunCount);
   printf("Worker %d terminates\n", ID);
}

  /* Wait for workers (and the debugger when invoked)*/

void _c2_WaitforThreads()
{
   int i;
   for (i=0; i<numProcessors; i++)
     ThreadJoin(WorkerList[i]);

#ifdef DEBUG
   printf("Workers have all joined.\n");
   printf("_c2_Terminated = %d\n", _c2_Terminated);
#endif

   ProgramDone();

}




void _c2_RunWorkers(InitNode)
  _c2_NodeBase *InitNode;
{
   int i;


   if (numProcessors > MAXWORKERS) {
      fprintf(stderr, "** Only %d workers are allowed\n", MAXWORKERS);
      exit(1);
   }

   LockInit(&RQLock);
   Head = 0;
   FreeList = 0;
   _c2_Terminated = 0;

   /* enqueue the start node */
      _c2_EnQueue(InitNode);


   /* Start the workers; if debugger is installed then  */
   /* reserve one processor for the debugger; indicated by nd = 1*/

   for (i= 0; i< numProcessors; i++)
     WorkerList[i] = ThreadStart(_c2_WorkerTask, JOIN, "worker", 1, i);


#ifdef DEBUG
   printf("Workers have all started\n");
#endif
     _c2_WaitforThreads();
}



#ifdef _C2_DBGR_

void _c2_RunDebugger(InitNode, argc, argv)
   _c2_NodeBase *InitNode;
     int argc;
     char **argv;
{

   int i;
   int debuggerid;

   if (numProcessors < 2) {
      fprintf(stderr, "at least 2 processors needed for c2udbg\n");
      fprintf(stderr, " try running %s with -n# option\n", argv[0]);
      exit(1);

   } else if (numProcessors > MAXWORKERS) {
      fprintf(stderr, "** Only %d workers are allowed\n", MAXWORKERS);
      exit(1);
   }
   _c2_saveInitNode(InitNode);
   _c2_ScopeInit(InitNode->MyGraph);

   debuggerid = numProcessors -1;
   /* workers' ids range from  0..(debuggerid-1) */
   _c2_initWrkrs(debuggerid-1);   
   WorkerList[debuggerid] = ThreadStart(_c2_DebuggerTask, JOIN,
				 "worker",3,  debuggerid, argc, argv);    
   LockInit(&RQLock);
   Head = 0;
   FreeList = 0;
   _c2_Terminated = 0;

   /* Start the workers; The number of workers is one less 
    * than the number of
    * of processors. One processor is reserved for the debugger
    */
   for (i= 0; i< numProcessors - 1; i++) {
       WorkerList[i] = ThreadStart(_c2_WorkerTask, JOIN, "worker", 1, i);
    }

#  ifdef DEBUG
      printf("Workers have all joined.\n");
      printf("_c2_Terminated = %d\n", _c2_Terminated);
#  endif
   _c2_WaitforThreads();
}

#endif /* _C2_DBGR_ */





