/* workers.c */


#include <stdio.h>
#include "c2_misc.h"
#include "c2_workers.h"

#include <thread.h>

#include <limits.h>
#include <sys/times.h>
#include <time.h>


#ifdef _C2_DBGR_
#include <setjmp.h>
#include <signal.h>
#include <errno.h>
#include <debugQ.h>
#endif


int _c2_Terminated;  /* 1 iff term node has fired */

/* structure for links in shared ready Q-- actually LIFO */

typedef struct _c2_Link {
   _c2_NodeBase *Node;
   struct _c2_Link *Next;
} _c2_RQLink;


static _c2_RQLink *Head;
static _c2_RQLink *FreeList;
static Lock RQLock;
static Lock EmptyLock;
static Thread *WorkerList[MAXWORKERS];

static float _c2_SysSec;
static float _c2_UserSec;
static clock_t start_time;
static clock_t end_time;


/* Timing routines */

void _c2_StartClock() {
  struct tms before;

  start_time  = times(&before);
  _c2_SysSec  = (float)before.tms_stime/(float)CLK_TCK;
  _c2_UserSec = (float)before.tms_utime/(float)CLK_TCK;

}

void _c2_StopClock() {
  struct tms after;

  end_time    = times(&after);
  _c2_SysSec  = (float)after.tms_stime/(float)CLK_TCK - _c2_SysSec;
  _c2_UserSec = (float)after.tms_utime/(float)CLK_TCK - _c2_UserSec;
}


/* Signal a halt to the computation */

void _c2_StopAll()
{

  int i;

  printf("Computation complete.\n");
  printf("Time between completion of Start node and firing of Stop node:\n");
  printf("  Wall clock time = %f seconds.\n",(float)(end_time - start_time)/(float)CLK_TCK);
  printf("  System time: %f seconds.\n", _c2_SysSec);
  printf("  User time  : %f seconds.\n", _c2_UserSec);

  _c2_Terminated = 1;

  /* Just make sure that no one is stuck
     waiting for an item to appear
     on the ready queue. */

  for (i = 0; i < numProcessors; i++) {
    LockRelease(&EmptyLock);
  }

  fflush(stdout);
  fflush(stderr);

}


void _c2_KillComp(Reason)
   int Reason;
{
  int i;

   fprintf(stderr, "\n\n*** CODE fatal runtime error: ");
   switch (Reason) {
     case _c2_CREPTWICE:
       fprintf(stderr, "Attempt to bind 2nd value to creation parameter\n");
       break;
     case _c2_RMFROMEMPTY:
       fprintf(stderr, "Attempt to remove from empty arc\n");
       break;
     default:
       fprintf(stderr, "Unknown problem (internal error?)\n");
       break;
   }
   fflush(stdout);
   fflush(stderr);

   _c2_Terminated = 1;

    LockRelease(&EmptyLock);
}

   /* Add a node to the Ready Q */

void _c2_EnQueue(NodeToAdd)
   _c2_NodeBase *NodeToAdd;
{
   _c2_RQLink *tmp;

   LockAcquire(&(NodeToAdd->QueueLock));

   if (!(NodeToAdd->Crepped)) {  /* Node cannot be run until creps are bound */
      LockRelease(&(NodeToAdd->QueueLock));
      return;
   }

   switch (NodeToAdd->QueueStatus) {
    case _c2_IDLE:
      NodeToAdd->QueueStatus = _c2_ONQUEUE;
#     ifdef _C2_DBGR_
         _c2_timeUcPutOnRdyQ(NodeToAdd); /* performance measurement */
#     endif
      LockRelease(&(NodeToAdd->QueueLock));
      break;
    case _c2_RUNNING:
      NodeToAdd->QueueStatus = _c2_NEEDSRUN;
      LockRelease(&(NodeToAdd->QueueLock));
      return;
      break;

#   ifdef _C2_DBGR_
    case _c2_ONDBGQ:		/* Debugger will later put it for running */
#   endif 
    case _c2_ONQUEUE:
    case _c2_NEEDSRUN:
      LockRelease(&(NodeToAdd->QueueLock));
      return;
      break;
    default:
      fprintf(stderr, "Error: Illegal Q status in _c2_EnQueue\n");
      fflush(stdout);
      fflush(stderr);
      LockRelease(&(NodeToAdd->QueueLock));
      _c2_StopAll();
   }

   /* If we get here, we need to Q to node */

   LockAcquire(&RQLock);

   if (FreeList != 0) {
      tmp = FreeList;
      FreeList = FreeList->Next;
   } else
     tmp = (_c2_RQLink *) malloc(sizeof(_c2_RQLink));
   tmp->Node = NodeToAdd;
   tmp->Next = Head;
   Head = tmp;

   LockRelease(&RQLock);
   LockRelease(&EmptyLock);

   return;
}


/* Get a node to run from the ready Q, return 0 if Q empty */

static _c2_NodeBase *_c2_PopNode()
{
   _c2_NodeBase *ReturnTask;
   _c2_RQLink *TmpHead;
   int i;

   /*
   LockAcquire(&RQLock);
   if (Head == 0) {
     LockRelease(&RQLock);
     return 0;
   }
   */

   if (Head == 0) {
     return 0;
   }

   LockAcquire(&RQLock);

   TmpHead = Head;
   Head = Head->Next;
   ReturnTask = TmpHead->Node;

   TmpHead->Next = FreeList;
   FreeList = TmpHead;

   LockRelease(&RQLock);

   LockAcquire(&(ReturnTask->QueueLock));

   switch (ReturnTask->QueueStatus) {
    case _c2_ONQUEUE:
      ReturnTask->QueueStatus = _c2_RUNNING;
      LockRelease(&(ReturnTask->QueueLock));
      break;
    default:
      fprintf(stderr, "Error: Illegal Q status in _c2_PopNode\n");
      fflush(stderr);
      fflush(stdout);
      LockRelease(&(ReturnTask->QueueLock));
      _c2_StopAll();
   }
   return ReturnTask;
}



#ifdef _C2_DBGR_

void _c2_emptyRdyQ()
{
   while(_c2_PopNode())
       ;
}

static struct {
   jmp_buf env;			/* map environment with pid of processes */
   int pid;			/* pid of the process running the workertasks */
} WrkrEnv[MAXWORKERS];

/* signal handler */

void _c2_inthandler(sig)
int sig;
{
   int i;
   int pid;

   pid =getpid();
   for (i=0; i < numProcessors; i++){        
      if (WrkrEnv[i].pid == pid) {
	 if (sig != _c2_ERRSIGNAL) { /* not debugger code's error */
	    psignal("caught", sig);
	    if (_c2_debugsignals) {
	       fprintf(stderr, "interrupt caught by pid = %d.\n", pid); 
	    }
	 }
	 _c2_setGlobalState(_c2_STOPPED);
	 _c2_emptyRdyQ();
	 longjmp(WrkrEnv[i].env, 1);
      }
   }
   _c2_reinitWrkrs();
}

/* 
 * To handle the child. Specifically for SIGSEGV that can not be
 * handled because its handler messes up the shbrk() and shmalloc()
 */

#include <sys/wait.h>

void _c2_ChldHndler()
{
   int pid;
   union wait *status;

   pid = wait3(status, WNOHANG);
   if (pid == 0) {
      error("Why a SIGCHLD");
   } else {
      _c2_setGlobalState(_c2_STOPPED);
      if (status->w_termsig == SIGSEGV) {
	 psignal("A worker died", status->w_termsig);
      } else {
	 psignal("detected", status->w_termsig);
      } 
      fflush(stdout);
   }
}


/* in case of abnormal abort, the debugger must exist gracefully */

void _c2_KillWorkers()
{
   int i;
   _c2_StopAll();
   /* 
    * in case some processes are stalled and can not
    * terminate normally, do it the hard way!
    */
   for (i=0; i<numProcessors; i++){
      if (_c2_debugsignals)
	  printf("killing worker %d\n", WrkrEnv[i].pid);
      kill(WrkrEnv[i].pid, SIGKILL);
   }
}   
#endif /* _C2_DBGR_ */




/* Process for a worker-- a thread that runs nodes on the ready queue. */


static void _c2_WorkerTask(IDptr)
int *IDptr;
{
   _c2_NodeBase *NodeToRun;
   int RunCount = 0;
   int ReRunCount = 0;
   int IdleCount = 0;
   int TotalCount = 0;
   int Skip = 1;
   int ID;

   int pri;

   ID = *IDptr;
   free(IDptr);


#  ifdef _C2_DBGR_
   WrkrEnv[ID].pid = getpid();	/*  associate pid with env entries */
   if (setjmp(WrkrEnv[ID].env) != 0){
      if (_c2_debugsignals)
	  printf("workerTask %d: Returned from interrupt\n", ID);
   }
   _c2_sig_install(_c2_inthandler);
#  endif/* _C2_DBGR_ */


#  ifdef DEBUG
   printf("Worker %d starts\n", ID);
#  endif

   while (!_c2_Terminated) {
     TotalCount++;

#    ifdef _C2_DBGR_
     if (_c2_GlobalState == _c2_STOPPED)  continue; 
#    endif

     /* Check ready queue for a node to run */

     if ( (NodeToRun = _c2_PopNode()) == 0) {
#      ifdef _C2_DBGR_
       if (_c2_sizeOutDbgQ() == 0 || (NodeToRun = _c2_PopOutDbgQ()) == 0) {
	   _c2_assertWrkrState(ID);
#      endif
	   IdleCount++;
	   if (!_c2_Terminated) {
	     LockAcquire(&EmptyLock);
	   }
	   continue;		/* nothing to do */
#      ifdef _C2_DBGR_
       }
#      endif
     }
     RunCount++;

#    ifdef DEBUG
     printf("Worker %d Running a %d\n", ID, NodeToRun->UID);
     fflush(stdout);
#    endif

   Again:

     NodeToRun->CompProc(NodeToRun);
#ifdef DEBUG
     printf("Worker %d Done Running a %d\n", ID, NodeToRun->UID);
     fflush(stdout);
#endif

     LockAcquire(&(NodeToRun->QueueLock));
     
     switch (NodeToRun->QueueStatus) {
     case _c2_RUNNING:
       NodeToRun->QueueStatus = _c2_IDLE;
       LockRelease(&(NodeToRun->QueueLock));
       break;

#    ifdef _C2_DBGR_
     case _c2_ONDBGQ:		/* debugger will run it later on */
       LockRelease(&(NodeToRun->QueueLock));
       break;
#    endif

     case _c2_NEEDSRUN:
       NodeToRun->QueueStatus = _c2_RUNNING;

#      ifdef _C2_DBGR_
       _c2_timeUcPutOnRdyQ(NodeToRun); /* performance measurement */
#      endif

       LockRelease(&(NodeToRun->QueueLock));

#      ifdef DEBUG
       printf("Worker %d Re-running a %d\n", ID, NodeToRun->UID);
       fflush(stdout);
#      endif

       ReRunCount++;
       TotalCount++;
       goto Again;
       break;
     default:
       fprintf(stderr, "Error: Illegal Q status in _c2_WorkerTask\n");
       fflush(stdout);
       fflush(stderr);
       LockRelease(&(NodeToRun->QueueLock));
       _c2_StopAll();
       break;
     }
   }

   /* Execution is complete. */

   printf("Thread %d: Total %d, Idle %d, Runs %d, Reruns %d\n",
	  ID, TotalCount, IdleCount, RunCount, ReRunCount);
   LockRelease(&EmptyLock);
}


/* Wait for all workers to complete. */

void _c2_WaitforThreads()
{
  int i;
  for (i=0; i<numProcessors; i++)
    ThreadJoin(WorkerList[i]);

#ifdef DEBUG
   printf("Workers have all joined.\n");
#endif

}




void _c2_RunWorkers(InitNode)
  _c2_NodeBase *InitNode;
{
   int i;


   if (numProcessors > MAXWORKERS) {
      fprintf(stderr, "** Only %d workers are allowed\n", MAXWORKERS);
      exit(1);
   }

   LockInit(&RQLock);
   LockInit(&EmptyLock);

   Head = 0;
   FreeList = 0;
   _c2_Terminated = 0;

   /* Enqueue the start node. */
   _c2_EnQueue(InitNode);

   /* Start the workers. */

   for (i= 0; i< numProcessors; i++) {
     int *iptr;
     iptr = (int *) malloc(sizeof(int));
     *iptr = i;

     WorkerList[i] = ThreadStart(_c2_WorkerTask, JOIN, "worker", 1, iptr);
#ifdef DEBUG
     printf("Started thread %d.\n", i);
#endif
   }

#ifdef DEBUG
   printf("Workers have all started\n");
#endif

   _c2_WaitforThreads();
}



#ifdef _C2_DBGR_

void _c2_RunDebugger(InitNode, argc, argv)
   _c2_NodeBase *InitNode;
     int argc;
     char **argv;
{

   int i;
   int debuggerid;

   if (numProcessors < 2) {
      fprintf(stderr, "at least 2 processors needed for c2udbg\n");
      fprintf(stderr, " try running %s with -n# option\n", argv[0]);
      exit(1);

   } else if (numProcessors > MAXWORKERS) {
      fprintf(stderr, "** Only %d workers are allowed\n", MAXWORKERS);
      exit(1);
   }
   _c2_saveInitNode(InitNode);
   _c2_ScopeInit(InitNode->MyGraph);

   debuggerid = numProcessors -1;
   /* workers' ids range from  0..(debuggerid-1) */
   _c2_initWrkrs(debuggerid-1);

   printf ("creating workerlist[%d]...\n", debuggerid);

   WorkerList[debuggerid] = ThreadStart(_c2_DebuggerTask, JOIN,
				 "worker",3,  debuggerid, argc, argv);    
   LockInit(&RQLock);
   LockInit(&EmptyLock);
   Head = 0;
   FreeList = 0;
   _c2_Terminated = 0;

   /* Start the workers. The number of workers is one less 
      than the number of processors -- one processor is reserved
      for the debugger.
    */

   for (i= 0; i< numProcessors - 1; i++) {
	int *iptr;
	iptr = (int *) malloc(sizeof(int));
	*iptr = i;
       WorkerList[i] = ThreadStart(_c2_WorkerTask, JOIN, "worker", 1, iptr);
    }

#  ifdef DEBUG
      printf("Workers have all joined.\n");
#  endif
   _c2_WaitforThreads();
}

#endif /* _C2_DBGR_ */





