#
# CallNode.tcl
# ------------------------------------------------------------------------
# This node holds a subgraph, and can be expanded (by a double-click) to
# reveal its sub-canvas.
# ------------------------------------------------------------------------
# @(#) $Id: CallNode.tcl,v 1.8 1996/03/05 17:07:11 emery Exp $
#
# ------------------------------------------------------------------------
# AUTHOR:
#
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================

#  if anything is inside the call node, it
#    will be dark in the middle.


class Code::CallNode {

    inherit Code::Node

    constructor {args} {
	if {!$_initialized} {
	    init
	}
	eval Code::Node::constructor $args
    } {}

    destructor {}

    public method attributes {}
    public method bitmap {}
    public method translate {ProgramAST GraphAST}

    public method save {out}
    public method expand {}
    public method collapse {}
    public method print {}
    public method thisimage {}

    protected method draw_node {x y}

    protected method update_name {args}
    private method occupied {}
    private method init {}
    private method destroy_contents {}

    private common _initialized 0
    common image
    common aliasimage
    
    # attributes

    public variable IsAliasToGraph 0
    public variable AliasGraph     ""

    protected variable attributes {Name TraceCall GraphFunctionSignatures \
	    GraphFunctionDefinitions GraphDocumentation \
	    IsAliasToGraph AliasGraph}
    
    public variable TraceCall 0
    public variable GraphFunctionSignatures ""
    public variable GraphFunctionDefinitions ""
    public variable GraphDocumentation ""


    public variable subgraph ""             ;# the canvas in the expanded window

    common Aliases

    #
    # default dimensions of expanded window
    #

    common width
    common height

    set width 730
    set height 522

    protected variable windowname ""  ;# the name of the expanded window
    protected variable expanded 0     ;# == window is expanded. initially zero
    protected variable graphid        ;# unique id number for subgraph

    common aliasimagefile
    common imagefile        ;# array for the node image files
    common bitmapfile

    #
    # first value == (1) expanded / (0) closed
    # second      == (1) occupied / (0) empty
    # third       == (1) selected / (0) not selected
    #

    set imagefile(0,0,0) "CallNode.gif"
    set imagefile(0,1,0) "CallNodeGraph.gif"
    set imagefile(1,0,0) "CallNodeOpen.gif"
    set imagefile(1,1,0) "CallNodeOpenGraph.gif"

    set imagefile(0,0,1) "CallNode-Selected.gif"
    set imagefile(0,1,1) "CallNodeGraph-Selected.gif"
    set imagefile(1,0,1) "CallNodeOpen-Selected.gif"
    set imagefile(1,1,1) "CallNodeOpenGraph-Selected.gif"

    set aliasimagefile(0) "CallNodeAlias.gif"
    set aliasimagefile(1) "CallNodeAlias-Selected.gif"

    set bitmapfile(0,0) "CallNode.bit"
    set bitmapfile(0,1) "CallNodeGraph.bit"
    set bitmapfile(1,0) "CallNodeOpen.bit"
    set bitmapfile(1,1) "CallNodeOpenGraph.bit"

}



body Code::CallNode::constructor {args} {
    addport InputPort top
    addport OutputPort bottom
    set Aliases($this) ""
    set graphid [$canvas new_id]   ;# maintain an extra identifier for subgraph
    itk_initialize
}


body Code::CallNode::destructor {} {

    if $IsAliasToGraph {

	# Delete the alias from the list of aliases held by
	# the aliased graph.

	set index [lsearch -exact $Aliases($AliasGraph) $this]
	lreplace $Aliases($AliasGraph) $index [expr $index+1]
	return
    }

    destroy_contents
}


body Code::CallNode::destroy_contents {} {
    # if there is a subwindow...
    if { ($subgraph != "") && [winfo exists $windowname] } {
	# delete everything in the subgraph
	$subgraph delete_items [winfo children $subgraph]

	# delete the window
	destroy $windowname

	# If the focus was on this window,
	# move it to the parent window.
	if { ![string compare [$canvas info variable CurrentWindow -value] $subgraph] } {
	    ::focus -force [info namespace tail $canvas]
	    Code::CodeCanvas::SetCurrentWindow $canvas
	}

	set windowname ""
	set subgraph   ""

	foreach attr {GraphFunctionSignatures GraphFunctionDefinitions GraphDocumentation} {
	    set $attr ""
	}

	# Eliminate any aliases to this node.

	foreach alias $Aliases($name) {
	    delete object $alias
	}
	set Aliases($name) ""
    }
}


body Code::CallNode::attributes {} {
    set OldIsAliasToGraph $IsAliasToGraph
    set OldAliasGraph $AliasGraph

    Node::attributes

    update_node

    # We have to handle a few cases:
    #  (1) We were an alias to one graph, but now we're an alias to a different one.
    #  (2) We weren't an alias before, but we are now.
    #  (3) We were an alias before, but we aren't anymore.

    if $IsAliasToGraph {
	if !$OldIsAliasToGraph {
	    # Case (2).

	    # Destroy the contents of this graph (since it's now just an alias).
	    # (We should warn the user with a dialog here, and abort
	    #  the changes if cancelled.)  TO BE CHANGED!!! REMOVE!!!

	    destroy_contents

	} else if [string compare $OldAliasGraph $AliasGraph] {

	    # Case (1).
	    # Remove this object from the alias list of $OldAliasGraph
	    # and add it to $AliasGraph.

	    set index [lsearch -exact $Aliases($OldAliasGraph) $this]
	    set Aliases($OldAliasGraph) [lreplace $Aliases($OldAliasGraph) $index [expr $index+1]]
	    lappend Aliases($AliasGraph) $this

	}

    } else {
	if $OldIsAliasToGraph {

	    # Case (3).
	    # Remove this object from the alias list of $OldAliasGraph.

	    set index [lsearch -exact $Aliases($OldAliasGraph) $this]
	    set Aliases($OldAliasGraph) [lreplace $Aliases($OldAliasGraph) $index $index]

	}
	set AliasGraph ""
    }
}


body Code::CallNode::bitmap {} {

    # change the image to the appropriate call node picture
    # (depending on if it is open or has children)

    return "@[$canvas info variable ReleaseDirectory -value]/bitmaps/$bitmapfile($expanded,[occupied])"

}


body Code::CallNode::collapse {} {
    if $IsAliasToGraph {
	$AliasGraph collapse
	return
    }
    wm withdraw $windowname
    set expanded 0
    update_node
}


body Code::CallNode::draw_node {x y} {
    set node [$canvas create image $x $y -parent $this]
    update_node
}


body Code::CallNode::expand {} {

    global ::c2_MenuFont

    ##
    ## pop up the window with the contents of the
    ## call node subgraph.
    ##

    #
    # change the node to the "open" state
    #

    set expanded 1
    update_node

    if $IsAliasToGraph {
	$AliasGraph expand
	return
    }

    #
    # either reveal the window if it exists,
    # or draw it for the first time.
    #
    if { ($subgraph != "") && ([winfo exists $windowname]) } {
	wm deiconify $windowname ; raise $windowname ; return
    } else {

	## create the window
	
	set windowname "._CallNode$id"
	uplevel #0 toplevel $windowname

	global ::c2_ReleaseDirectory

	wm minsize $windowname $width $height
	wm geometry $windowname ${width}x$height
	wm title $windowname $Name
	wm protocol $windowname WM_DELETE_WINDOW {c2_QuitProgram}
	# wm iconbitmap $windowname "@$c2_ReleaseDirectory/bitmaps/ICON.bit"

	#
	# for brain-damaged window-managers...
	# this forces the minsize constraint
	# to be obeyed immediately.
	#

	wm withdraw $windowname
	wm deiconify $windowname

	uplevel #0 frame $windowname.parent
	uplevel #0 frame $windowname.parent.fone  ;# canvas window must not contain a number
	frame $windowname.parent.menubar -borderwidth 2 -relief raised
	frame $windowname.parent.iconbar -borderwidth 2 -relief raised

	label $windowname.parent.statusbar -text "" -height 1 -relief sunken -borderwidth 2 -font $c2_MenuFont

	build_iconbar $windowname.parent.iconbar $windowname.parent.statusbar

	uplevel #0 "scrollbar $windowname.parent.fone.scrollbar1 \
		-command \"$windowname.parent.fone.canvas xview\" \
		-orient {horizontal} \
		-relief {raised}"
	
	uplevel #0 "scrollbar $windowname.parent.fone.scrollbar2 \
		-command \"$windowname.parent.fone.canvas yview\" \
		-relief {raised}"
	
	set subgraph [\
		uplevel #0 "CodeCanvas $windowname.parent.fone.canvas \
		-closeenough {0.2} \
		-height [expr $height/10] \
		-insertofftime {600} \
		-relief {sunken} \
		-scrollregion {0 0 1000 1000} \
		-width [expr $width/10] \
		-xscrollcommand \"$windowname.parent.fone.scrollbar1 set\" \
		-yscrollcommand \"$windowname.parent.fone.scrollbar2 set\" " \
		]


	build_menubar $windowname.parent.menubar $subgraph


	# Tell the canvas that its parent is this call node.

	$subgraph configure -parentnode $this
	
	pack append $windowname.parent.fone \
		$windowname.parent.fone.scrollbar2 {right frame center filly} \
		$windowname.parent.fone.canvas {top frame center expand fill} \
		$windowname.parent.fone.scrollbar1 {top frame center fillx} \
	        $windowname.parent.statusbar {bottom frame s fillx}
	
	pack append $windowname.parent \
		$windowname.parent.menubar {top frame nw fillx} \
		$windowname.parent.fone {right frame ne expand fill} \
                $windowname.parent.iconbar {left frame nw fill}
	
	pack append $windowname \
		$windowname.parent {top frame center expand fill}

    }   
}


body Code::CallNode::init {} {
    global c2_ReleaseDirectory
    foreach i {0 1} {
	foreach j {0 1} {
	    foreach k {0 1} {
		set image($i,$j,$k) [uplevel #0 image create photo -palette 256x256x256 -format GIF \
	    -file "$c2_ReleaseDirectory/bitmaps/$imagefile($i,$j,$k)"]
	    }
	}
	set aliasimage($i) [uplevel #0 image create photo -palette 256x256x256 -format GIF -file "$c2_ReleaseDirectory/bitmaps/$aliasimagefile($i)"]
    }
    set _initialized 1
}


body Code::CallNode::occupied {} {

    if $IsAliasToGraph {
	return [$AliasGraph occupied]
    }

    if {($subgraph != "") && ([winfo children $subgraph] != "")} {
	return 1
    } else {
	return 0
    }
}


body Code::CallNode::print {} {
    if $IsAliasToGraph {
	$AliasGraph print
	return
    }

    if { $subgraph != "" } {
	$subgraph print
    }
}


body Code::CallNode::save {out} {
    # recursively save subgraphs (which are sub-canvases)
    Node::save $out
    if !$IsAliasToGraph {
	if { ($subgraph != "") && ([winfo children $subgraph] != "") } {
	    # puts $out "\$objindex($this) expand [mangle_group $id]"
	    puts $out "\$objindex($this) expand"
	    puts $out "set oldcanvas \$canvas"
	    puts $out "set canvas \[\$objindex($this) info variable subgraph -value\]"
	    $subgraph save $out
	    puts $out "\$objindex($this) collapse"
	    puts $out "set canvas \$oldcanvas"
	}
    } else {
	puts $out "\$objindex($this) configure -AliasGraph ::\$objindex($AliasGraph)"
    }
}


body Code::CallNode::thisimage {} {
    if $IsAliasToGraph {
	return $aliasimage($selected)
    } else {
	return $image($expanded,[occupied],$selected)
	# change the image to the appropriate call node picture
	# (depending on if it is open or has children or is selected)
    }
}


body Code::CallNode::translate {ProgramAST GraphAST} {

    if $IsAliasToGraph {
	set graphname "graph[$AliasGraph info variable graphid -value]"
    } else {
	set graphname "graph$graphid"
    }

    set result [c2_CreateCallNodeAST $ProgramAST $GraphAST \
	    $Name $id $graphname $TraceCall]
    if { $result != "" } {
	if { ($subgraph != "") && ([winfo children $subgraph] != "") && !$IsAliasToGraph } {
	    set result [$subgraph translate $ProgramAST $graphname $graphid]
	}
    }
    return $result
}


body Code::CallNode::update_name {args} {
    if { [winfo exists $windowname] } {
	wm title $windowname $Name
    }
}

