#
# CodeCanvas.tcl
# ------------------------------------------------------------------------
# An augmented version of the Tk widget "canvas" which provides for the
# creation of CODE objects as well as the standard canvas widgets.
# ------------------------------------------------------------------------
# @(#) $Id: CodeCanvas.tcl,v 1.10 1996/03/15 18:01:20 emery Exp $
#
# ------------------------------------------------------------------------
# AUTHOR:
#
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================

#
#  All CODE objects go onto a CodeCanvas, e.g.:
#
#     CodeCanvas .c
#     .c create CallNode 10 10
#
#  creates a call node on the CodeCanvas at
#  (10,10).
#


#
# Option database default resources:
#

option add *CodeCanvas.background white widgetDefault
option add *CodeCanvas.borderWidth 2 widgetDefault
option add *CodeCanvas.relief sunken widgetDefault
option add *CodeCanvas.scrollMargin 3 widgetDefault
option add *CodeCanvas.vscrollMode static widgetDefault
option add *CodeCanvas.hscrollMode static widgetDefault
option add *CodeCanvas.width 360 widgetDefault
option add *CodeCanvas.height 200 widgetDefault
option add *CodeCanvas.labelPos n widgetDefault
option add *CodeCanvas.autoResize 1 widgetDefault


class Code::CodeCanvas {

    inherit wigwam::itcl_canvas Code::Undo

    constructor {args} { eval wigwam::itcl_canvas::constructor $args } {}

    destructor {
	unset AllCanvases($this)
    }

    itk_option define -current_x current_x Current_x ""
    itk_option define -current_y current_y Current_y ""
    itk_option define -currentDrawingArcItem currentDrawingArcItem CurrentDrawingArcItem ""
    itk_option define -justMoved justMoved JustMoved 0

    public method create {type args}
    public method collapse {}
    public method expand {}

    public method new_file {}
    public method insert_file_dlog {}
    public method open {}
    public method attempt_open {filename}

    public method print {}
    public method save_file {{filename ""}}
    public method save_as {}
    public method translate {programAST graphName graphId}
    public method duplicate {}
    public method cut {}
    public method copy {{file ""}}
    public method can_paste {}
    public method paste {}
    public method save {out}
    public method save_items {out items}

    public proc set_id {val}
    public proc get_id {}

    public method new_id {}

    public method zoom_in {x y}
    public method zoom_out {x y}
    public method select_group {g}
    public method select_all {}
    public method select_items {items}
    public method deselect_all {}
    public method selected_items {}
    public method selected_groups {}
    public method set_selected {item}
    public method unset_selected {item}
    public method delete_selected {}
    public method delete_items {items}
    public method align {direction}
    public method move_selected {x y}
    public method move_selected_by {x y}
    public method attributes {}
    public method print_attributes {}
    public method bitmap {}
    public method all_groups {item}
    public method groupset {items}
    public method items_in_group {g}
    public method items_in_same_group {item}
    public method lock_group {g}
    public method lock {}
    public method unlock_group {g}
    public method unlock {}
    public method ungroup_selected {}
    public method group_selected {}
    public method group {items}
    public method set_group {group items}
    public method add_to_group {group item}
    public method delete_from_group {group item}
    public method select_inside {coords}
    public method lookup_parent {item}
    public method set_cursor {type}
    public method tool {}
    public method cleanup_after {item}
    public method draw_bbox {items tagid}
    public method remove_bbox {tagid}
    public method Canvas_Enter {}
    public method Unshifted_Button_Press_Action {terminalX terminalY}
    public method Button_Press_Action {terminalX terminalY}


    public proc set_tool {tool}
    public proc SetCurrentWindow {win}
    public proc init {undobuf cutbuf maincanv sfg fg rd}



    public variable parentnode ""
    public variable FunctionSignatures ""
    public variable FunctionDefinitions ""
    public variable Documentation ""

    public variable current_x 0
    public variable current_y 0

    public variable scale 1.0                 ;# the magnification of the canvas

    public variable CurrentDrawingArcItem ""  ;# the arc currently being drawn
    public variable RubberBand ""             ;# the rubber-band
    public variable JustMoved 0               ;# have we just moved some objects?
    public variable Name "Untitled"           ;# this will just mirror FileName, below

    common version 2.1b1                      ;# program version number

    common auto    0
    common new_id  0

    private common ToolSelected "Select"    ;# the "mode" we are in
    common UndoBuffer       ;# the filename of the undo buffer
    common CutBuffer        ;# the filename of the cut buffer
    common MainCanvas       ;# the pathname of the main CodeCanvas
    common CurrentWindow    ;# the currently focused canvas
    common SelectedFGColor  ;# the foreground color to use for selected objects
    common FGColor          ;# the normal foreground color
    common ReleaseDirectory ;# the directory for the CODE release

    common cursor_value        ;# the current cursor bitmap

    set cursor_value(Select) ""
    set cursor_value(Arc) ""
    set cursor_value(CallNode) ""
    set cursor_value(CompNode) ""
    set cursor_value(CreationParameterNode) ""
    set cursor_value(NameSharingNode) ""
    set cursor_value(InInterfaceNode) ""
    set cursor_value(OutInterfaceNode) ""
    set cursor_value(Comment) "xterm"
    set cursor_value(Query) ""
    set cursor_value(Waiting) "watch"
    set cursor_value(Drag) ""
    set cursor_value(ZoomIn) ""
    set cursor_value(ZoomOut) ""

    common AllCanvases       ;# list of all canvases = [array names AllCanvases]

    common FileName ""      ;# the name of the file in memory (initially "")

    protected method insert_file {fn}
    protected method insert_callnode {fn}
    protected method load_file {filename}
    protected method save_items_to_file {filename items {header ""} {canvas ""}}
    protected method magnify {}
    protected method shrink {}
    protected method RubberBandDraw {terminalX terminalY}
    protected method RubberBandSelect {}
    public method Button_Motion_Action {terminalX terminalY}
    public method Button_Release_Action {}
    public method Delete_Arc_In_Progress {}
    protected method update_name {name e op}

    protected proc auto_incr {name e op}

    protected variable lookup
    protected variable lock
    protected variable items_in_group
    protected variable selected
    protected variable oldscale 1.0              ;# the previous magnification
    protected variable oldthis
}


#
#
#  OPTIONS
#
#

configbody Code::CodeCanvas::current_x {
    set current_x $itk_option(-current_x)
}

configbody Code::CodeCanvas::current_y {
    set current_y $itk_option(-current_y)
}

configbody Code::CodeCanvas::justMoved {
    set JustMoved $itk_option(-justMoved)
}

configbody Code::CodeCanvas::currentDrawingArcItem {
    set CurrentDrawingArcItem $itk_option(-currentDrawingArcItem)
}


#
#  METHODS
#
#


body Code::CodeCanvas::constructor {args} {
    set AllCanvases($this) ""

    set oldthis [info namespace tail $this]

    $oldthis configure -background white

    # ::focus -force $oldthis
    set CurrentWindow $oldthis

    #  bindings

    ::bind $oldthis <Enter> "if \{\[winfo exists $oldthis\]\} \{$this Canvas_Enter\} ;  break"

    #  bind <Button-1> and <Shift-Button-1> differently to allow
    #  for an unshifted click on the canvas to deselect all
    #  but let a shifted click continue selecting.
    
    ::bind $oldthis <Button-1>             "$this Unshifted_Button_Press_Action %x %y"
    ::bind $oldthis <Shift-Button-1>       "$this Button_Press_Action %x %y"
    
    #  these two sets of bindings (with shift and w/o) are identical
    #  -- button1-motion drags the rubber band around
    
    ::bind $oldthis <Button1-Motion>       "$this Button_Motion_Action %x %y"
    ::bind $oldthis <Shift-Button1-Motion> "$this Button_Motion_Action %x %y"

    #  these handle selecting all the items inside the rubber band
    #  (they are identical)
    
    ::bind $oldthis <ButtonRelease-1>        "$this Button_Release_Action"
    ::bind $oldthis <Shift-ButtonRelease-1>  "$this Button_Release_Action"

    #  make a canvas double-click delete any arc we're in the middle of

    ::bind $oldthis <Double-Button-1> "$this Delete_Arc_In_Progress"
    ::bind $oldthis <Key-Escape>      "$this Delete_Arc_In_Progress"

    # FOR DEBUGGING

    ::bind $oldthis <Double-Button-3> "foreach i \[winfo children $oldthis\] \{ puts \"\$i : \[\$i group_name\] \{ \[$oldthis items_in_group \[\$i group_parent_name\]\] \} \" \}"


    #
    # keyboard shortcuts
    #
    
    ::bind $oldthis <Alt-a> "$this select_all" 
    ::bind $oldthis <Alt-c> "$this copy"
    ::bind $oldthis <Alt-g> "$this group_selected"
    ::bind $oldthis <Alt-n> "c2_NewFile"
    ::bind $oldthis <Alt-o> "$this open"
    ::bind $oldthis <Alt-q> "c2_QuitProgram"
    ::bind $oldthis <Alt-s> "c2_SaveFile"
    ::bind $oldthis <Alt-u> "$this ungroup_selected"
    ::bind $oldthis <Alt-v> "$this paste"
    ::bind $oldthis <Alt-x> "$this cut"
    ::bind $oldthis <Alt-z> "$this undo"
    ::bind $oldthis <Delete>    "$this delete_selected"
    ::bind $oldthis <BackSpace> "$this delete_selected"
    
    ::bind $oldthis <Down>  "$this move_selected_by 0 1"
    ::bind $oldthis <Up>    "$this move_selected_by 0 -1"
    ::bind $oldthis <Right> "$this move_selected_by 1 0"
    ::bind $oldthis <Left>  "$this move_selected_by -1 0"
    
    ::bind $oldthis <Alt-Down>  "$this move_selected_by 0 10"
    ::bind $oldthis <Alt-Up>    "$this move_selected_by 0 -10"
    ::bind $oldthis <Alt-Right> "$this move_selected_by 10 0"
    ::bind $oldthis <Alt-Left>  "$this move_selected_by -10 0"

    
    # CodeObj bindings:

    $this bind ::Code::CodeObj <Shift-Button-1> \
	    "set obj \[$this lookup_parent \[$this find withtag current\]\] ; \
	    if \{ !\[string compare \[$this tool] \"Select\"\] \} \{ \
	    \$obj toggle_group ; \
	    $this configure -justMoved 1 \}"

    $this bind ::Code::CodeObj <Button-1> \
	    "set obj \[$this lookup_parent \[$this find withtag current\]\] ; \
	    if \{ !\[string compare \[$this tool\] \"Select\"\] && !\[\$obj selected\] \} \
	    \{ $this deselect_all ; \$obj select_group \} \
	    elseif \{ !\[string compare \[$this tool\] \"Query\"\] \} \{ \
	    \$obj toggle ; \$obj attributes ; \$obj toggle \}"
    
    $this bind ::Code::CodeObj <ButtonRelease-1> \
	    "set obj \[$this lookup_parent \[$this find withtag current\]\] ; \
	    if \{ !\[string compare \[$this tool\] \"Select\"\] && \
	    !\[$this info variable JustMoved -value\] \} \
	    \{ $this deselect_all ; \
	    \$obj select_group ; \
	    $this configure -justMoved 0 \}"
    
    $this bind ::Code::CodeObj <Button1-Motion> \
	"if \{ !\[string compare \[$this tool\] \"Select\"\] \} \{ \
            $this move_selected \[$this canvasx %x\] \[$this canvasy %y\] ; \
	    $this configure -current_x \[$this canvasx %x\] ; \
	    $this configure -current_y \[$this canvasy %y\] ; \
	    $this configure -justMoved 1 \}"


    # Label bindings:

    # If we are in Comment mode and we click on the object, we enter
    # edit mode and display the text widget.

    $this bind ::Code::Label <Button-1> \
	    "set obj \[$this lookup_parent \[$this find withtag current\]\] ; \
	    if \{ !\[string compare \[$this tool\] \"Comment\"\] \} \{ \
	    $this deselect_all ; \$obj editmode \}"


    # CallNode bindings:

    # A double-click expands the node window.

    $this bind ::Code::CallNode <Double-Button-1> \
	    "set obj \[$this lookup_parent \[$this find withtag current\]\] ; \
	    if \{ !\[string compare \[$this tool\] \"Select\"\] \} \
	    \{ \$obj expand \}"

    eval itk_initialize $args

    trace variable auto r auto_incr
    trace variable new_id r auto_incr
    trace variable Name r update_name
}



body Code::CodeCanvas::create {type args} {

    switch $type {
	arc -
	bitmap -
	image -
	line -
	oval -
	polygon -
	rectangle -
	text -
	window {
	    set parent [extract_arg "-parent" args]
	    set canvobj [eval wigwam::itcl_canvas::create $type $args]
	    
	    #
	    # associate the canvas object with its parent
	    #

	    set lookup($canvobj) $parent
	    return $canvobj
	}
	default {
	    
	    # make the item a child of this canvas.
	    # we need the "_" because window names
	    # can't start with capital letters.
	    
	    set name [extract_arg "-Name" args]
	    if {![string compare $name ""] } {
		set name "$this._$type$auto"
	    }
	    
	    set newobj [eval Code::$type $name $this $args]
	    set lock(.[$newobj id]) 0

	    return $newobj
	}
    }
}


body Code::CodeCanvas::collapse {} {
    if {[string compare $parentnode ""]} {
	$parentnode collapse
    }
}


body Code::CodeCanvas::expand {} {
    if { ![string compare [info namespace tail $this] $MainCanvas] } {
	wm deiconify .
	raise .
    } else {

	#  Find the enclosing window and raise it.
	#  This canvas is a subgraph, so its name is
	#  of the form:
	#
	#     windowname.parent.fone.canvas
	#
	#  (cf. CallNode.tcl)

	set windowname [file rootname [file rootname [file rootname $oldthis]]]
	wm deiconify $windowname
	raise $windowname
    }
}

#
#  destroy the file in memory
#

body Code::CodeCanvas::new_file {} {
    global \
	    ::c2_GlobalTypes \
	    ::c2_GlobalFunctionSignatures \
	    ::c2_GlobalFunctionDefinitions \
	    ::c2_GlobalDocumentation

    set c2_GlobalTypes ""
    set c2_GlobalFunctionSignatures ""
    set c2_GlobalFunctionDefinitions ""
    set c2_GlobalDocumentation ""

    #  ensure that lock is recognized as an array
    set lock(BogusRecord) 0

    #
    #  remove all locks
    #

    foreach name [array names lock] {
	unset lock($name)
    }

    # delete everything on the canvas

    delete_items [winfo children $oldthis]

    set FileName ""
    wm title . "CODE - Untitled"
}

#
#  bring up a dialog to choose a file
#  to insert
#

body Code::CodeCanvas::insert_file_dlog {} {
    set dialog [tix filedialog tixExFileSelectDialog]
    $dialog config -title "Code: Insert file..." -command "$this insert_callnode"
    $dialog popup
}


#
#  insert the file fn into the
#  current workspace
#

body Code::CodeCanvas::insert_file {fn} {

    tixBusy $oldthis on
    ::.framesix.statusbar configure -text "Loading $fn..."
    update

    source $fn
    c2_restore
    tixBusy $oldthis off
    ::.framesix.statusbar configure -text "Loading $fn...done."
    after 5000 {::.framesix.statusbar configure -text ""}
}


#
#  insert the file fn as a Call Node
#  into the current canvas.
#

body Code::CodeCanvas::insert_callnode {fn} {

    tixBusy $oldthis on

    set globals {
	c2_GlobalTypes
	c2_GlobalFunctionSignatures
	c2_GlobalFunctionDefinitions
	c2_GlobalDocumentation
	c2_FilesToLink
    }

    # Save all of the global values
    # to a temporary array for later restoration.

    foreach g $globals {
	global ::$g
	set tmp($g) [set $g]
    }

    # Create the call node and place the program inside of it.

    set newnode [$this create CallNode $current_x $current_y]
    $newnode expand

    source $fn
    c2_restore [$newnode info variable subgraph -value]

    $newnode collapse
    $newnode configure -Name [file tail $fn]

    # Copy all of the program attributes into the call node's attribute slots.

    $newnode configure -GraphFunctionSignatures  $c2_GlobalFunctionSignatures
    $newnode configure -GraphFunctionDefinitions $c2_GlobalFunctionDefinitions
    $newnode configure -GraphDocumentation       $c2_GlobalDocumentation

    set files_to_link $c2_FilesToLink

    # Restore the global values.

    foreach g $globals {
	set $g [set tmp($g)]
    }

    # Add any files which need to be linked to the
    # global files_to_link value.

    if {[string compare $files_to_link ""]} {
	set c2_FilesToLink "$c2_FilesToLink $files_to_link"
    }

    tixBusy $oldthis off
}


#
#  read a file
#

body Code::CodeCanvas::open {} {
    set dialog [tix filedialog tixExFileSelectDialog]
    $dialog config -title "Code: Open file..." -command "$this attempt_open"
    $dialog popup
}

#
# try to read the file before we attempt to insert it
#   (called by open)
#

body Code::CodeCanvas::attempt_open { filename } {
    if { [expr ![file readable $filename]] } {
	::.error configure -message [list "The file $filename is unreadable."]
	::.error activate
	return 0
    }
    new_file
    load_file $filename
}


body Code::CodeCanvas::load_file {filename} {
    set FileName $filename
    tixBusy $oldthis on
    insert_file $FileName
    wm title . "CODE - [file tail $FileName]"
    tixBusy $oldthis off
}

#
#  prints the canvas
#

body Code::CodeCanvas::print {} {
    global \
	    ::c2_PrintToPrinter \
	    ::c2_PrintFilename \
	    ::c2_PrintCommand \
	    ::c2_PrintFirstPageFirst \
	    ::c2_PrintPortraitMode \
	    ::c2_PrintGreyscale \
	    ::c2_PrintPaperSize

    # Determine the name of the canvas
    # ("main" if this is the main canvas or
    #  the title of the enclosing call node).


    set fn [file tail $FileName]
    if { ![string compare $fn ""] } {
	set fn "Untitled"
    }

    if {![string compare $oldthis $MainCanvas]} {
	set canvas_name "$fn:Main"
    } else {
	set canvas_name "$fn:[$parentnode info variable Name -value]"
    }

    if {$c2_PrintGreyscale} {
	set colormode "gray"
    } else {
	set colormode "color"
    }
    
    if {$c2_PrintPortraitMode} {
	set rotate 0
    } else {
	set rotate 1
    }

    if {$c2_PrintToPrinter} {
	set printbuf [::open "|$c2_PrintCommand -J$canvas_name" w ]
    } else {
	set printbuf [::open $c2_PrintFilename w]
    }

    tixBusy $oldthis on

    # Change scale to 1, and replace all images by bitmaps

    set os $scale

    if { $scale < 1 } {
	while { $scale < 1 } {
	    magnify
	}
    }
    
    if { $scale > 1 } {
	while { $scale > 1 } {
	    shrink
	}
    }

    if 1 {
    foreach obj [$this find all] {
	if {![string compare [$this type $obj] "image"]} {

	    # Get all info about image to be able
	    # to bring up its bitmap.

	    set coords [$this coords $obj]
	    set imagename [last [$this itemconfigure $obj -image]]
	    set imagefile [$imagename cget -file]
	    set bitmapfile "@[file root $imagefile].bit"

	    eval $this create bitmap $coords -bitmap $bitmapfile
	}
    }
    }

    # print.

    update
    ::scan [$this bbox all] "%d %d %d %d" leftx topy rightx bottomy
    puts $printbuf [$this postscript -x $leftx -y $topy -width $rightx -height $bottomy -rotate $rotate -colormode $colormode]
    close $printbuf

    # Change scale back to what it was, and trash all bitmaps

    if 1 {
    foreach obj [$this find all] {
	if {![string compare [$this type $obj] "bitmap"]} {
	    $this delete $obj
	}
    }
    }

    if { $scale < $os } {
	while { $scale < $os } {
	    magnify
	}
    }
    
    if { $scale > $os } {
	while { $scale > $os } {
	    shrink
	}
    }
    tixBusy $oldthis off
}


#
#  saves items to a file,
#  with an optional header string
#  to be placed at the beginning of
#  the "c2_restore" procedure
#

body Code::CodeCanvas::save_items_to_file {filename items {header ""} {canvas ""}} {

    set out [::open $filename w]

    #
    # this procedure creates the text for
    # a procedure "c2_restore" which, when sourced
    # and executed, will recreate all the elements
    # in the graph window.
    #

    if {![string compare $canvas ""]} {
	set canvas [Code::CodeObj::mangle_group $this]
    }

    puts $out "#  saved program file ([file tail $filename]) created by CODE $version"
    puts $out ""

    puts $out "proc c2_restore \{\{canvas \"\"\}\} \{"
    puts $out [join $header \n]
    
    #
    # initialize the list of objects to nil
    #
    
    puts $out "set objlist \"\""


    #
    # Set the canvas appropriately if it is not specified.
    #

    puts $out "  if \{ !\[string compare \$canvas \"\"\] \} \{ set canvas $canvas \}" 
    

    #
    # save the current cursor position
    #  (we will use this to paste at the cursor
    #   location -- see "paste")
    #

    puts $out "\$canvas configure -current_x $current_x"
    puts $out "\$canvas configure -current_y $current_y"

    save_items $out $items
    
    puts $out "  return \$objlist"
    puts $out "\}"
    
    close $out
}


#
#  save the file but
#  ask for a file name
#

body Code::CodeCanvas::save_as {} {
    set dialog [tix filedialog tixExFileSelectDialog]
    $dialog config -title "Code: Save as..." -command "$this save_file"
    $dialog popup
}



body Code::CodeCanvas::save_file {{filename ""}} {

    # We don't want to change the FileName until we have
    # saved the file...

    # if we have no file name, we need to come up with one.
    
    if { ![string compare $filename ""] } {
	set filename $FileName
	if { (![string compare $filename ""]) } {
	    save_as
	    return
	}
    }

    if { [file isdir $filename] } {
	return 0
    }

    # Backup old version of file, if any, a la Emacs.

    set error ""
    if { [file exists $filename] } {
	eval "catch \{exec mv $filename ${filename}~\} error"
    }
    if { [string compare $error ""] } {
	::.error configure -message [list $error]
	::.error activate
	return 0
    }

    # Try to write to the file, and return if there is an error
    # (after displaying the error message).

    # **** #

    tixBusy $oldthis on

    cd [file dirname $filename]
    set items [winfo children $oldthis]
    set header ""

    #
    # save global attribute values
    #
    
    global \
	    ::c2_GlobalTypes \
	    ::c2_GlobalFunctionSignatures \
	    ::c2_GlobalFunctionDefinitions \
	    ::c2_GlobalDocumentation \
	    ::c2_FilesToLink
    
    lappend header "global c2_GlobalTypes \
	    c2_GlobalFunctionSignatures \
	    c2_GlobalFunctionDefinitions \
	    c2_GlobalDocumentation \
	    c2_FilesToLink"

    lappend header "set c2_GlobalTypes \"[massage $c2_GlobalTypes]\""
    lappend header "set c2_GlobalFunctionSignatures \"[massage $c2_GlobalFunctionSignatures]\""
    lappend header "set c2_GlobalFunctionDefinitions \"[massage $c2_GlobalFunctionDefinitions]\""
    lappend header "set c2_GlobalDocumentation \"[massage $c2_GlobalDocumentation]\""
    lappend header "set c2_FilesToLink \"[massage $c2_FilesToLink]\""

    set FileName $filename

    save_items_to_file $FileName $items $header [Code::CodeObj::mangle_group $this]

    wm title . "CODE - [file tail $FileName]"
    not_undoable

    tixBusy $oldthis off

    ::.framesix.statusbar configure -text "File \"[file tail $FileName]\" saved."
    after 5000 {::.framesix.statusbar configure -text ""}
    return 1
}


body Code::CodeCanvas::translate {programAST graphName graphId} {

    global ::c2_yylinenum ::c2_yycolumn

    set graphAST [c2_CreateGraphAST $programAST $graphName \
	    $graphId $FunctionSignatures $FunctionDefinitions \
	    $Documentation]

    if { [string compare $graphAST ""] } {

	# go through all the children,
	# and translate the Nodes then the Arcs.
	#   (Nodes before Arcs because the arcs need to
	#    refer to the nodes.)
	
	set children [winfo children $oldthis]

	# Find all of the CallNodes which are aliases
	# and put them at the end of the list. We have to
        # translate them after any nodes they might refer to.

	set aliases ""
	set index -1

	foreach child $children {
	    incr index
	    if { [$child isa ::Code::CallNode] } {
		if { [$child info variable IsAliasToGraph -value] } {
		    lappend aliases $child
		    set children [lreplace $children $index $index]
		}
	    }
	}

	set children "$children $aliases"

	foreach type {::Code::Node ::Code::Arc} {
	    foreach child $children {
		if { [$child isa $type] } {
		    set result [$child translate $programAST $graphAST]
		    if { (![string compare $result ""]) || ($result == "0")} {
		        expand
			$child attributes
			return 0
		    }
		}
	    }
	}
	return 1
    } else {
	return 0
    }
}

#
#  duplicate selected items
#  on the canvas
#

body Code::CodeCanvas::duplicate {} {
    copy
    paste
    not_undoable
}

#
#  cut selected objects from the
#  canvas into the cut buffer file
#

body Code::CodeCanvas::cut {} {
    copy
    delete_selected
    Undo::cut
}


#
#  copy selected objects from the
#  canvas into the cut buffer file
#

body Code::CodeCanvas::copy {{file ""}} {

    if { ![string compare $file ""] } {
	set file $CutBuffer
    }
    set items [selected_items]

    set header ""
    lappend header "global c2_MainCanvas"

    save_items_to_file $file $items $header

    select_items $items
    not_undoable
}

#
#  returns true iff there is something
#  in the cut buffer
#

body Code::CodeCanvas::can_paste {} {
    return [file exists $CutBuffer]
}

#
#  paste the contents of the cut buffer
#  into the canvas
#

body Code::CodeCanvas::paste {} {

    # store current cursor positon

    set oldx $current_x
    set oldy $current_y

    deselect_all

    # read in the items from the cut buffer

    source $CutBuffer
    set items [c2_restore $this]

    #
    # select them and move them to the cursor
    # (save_items_to_file saves the cursor location,
    #  so this will move the items correctly)
    #

    select_items $items
    move_selected $oldx $oldy

    # restore the old cursor position

    set current_x $oldx
    set current_y $oldy

    Undo::paste $items
}


body Code::CodeCanvas::save {out} {
    save_items $out [winfo children $oldthis]
}


body Code::CodeCanvas::save_items {out items} {

    # save those which are Nodes, then Comments,
    # then Arcs.
    # We save Nodes before Arcs because the arcs need to
    # refer to the nodes.

    foreach type {::Code::Node ::Code::Comment ::Code::Arc} {
	foreach item $items {
	    if { [$item isa $type] } {
		$item save $out
	    }
	}
    }
}


####
#
#  returns a new unique id number
#

body Code::CodeCanvas::set_id {val} {
    set new_id $val
}

body Code::CodeCanvas::get_id {} {
    return $new_id
}

body Code::CodeCanvas::new_id {} {
    return $new_id
}

# canvas methods

#
#  zooms in the view of the canvas by 2x
#

body Code::CodeCanvas::magnify {} {
    set oldscale $scale
    set scale [expr $scale*2.0]
    $this scale all 0 0 2.0 2.0
    map "apply scale" [winfo children $oldthis]
}

body Code::CodeCanvas::zoom_in {x y} {
    magnify
    $this scan mark [expr round($x/10.0)] [expr round($y/10.0)]
    $this scan dragto 0 0
    $this configure -scrollregion [map {function {x} {return [expr $x*2]}} [cget -scrollregion]]
    Undo::zoom_in
}

#
#  zooms out the view of the canvas by 2x
#

body Code::CodeCanvas::shrink {} {
    set oldscale $scale
    set scale [expr $scale/2.0]
    $this scale all 0 0 0.5 0.5
    map "apply scale" [winfo children $oldthis]
}

body Code::CodeCanvas::zoom_out {x y} {
    shrink
    $this scan mark 0 0
    $this scan dragto [expr round($x/20.0)] [expr round($y/20.0)]
    $this configure -scrollregion [map {function {x} {return [expr $x/2]}} [cget -scrollregion]]
    Undo::zoom_out
}

#
#  selects everything in group g
#

body Code::CodeCanvas::select_group {g} {
    set groupitems [items_in_group $g]
    map "apply select" $groupitems

    # Because some objects won't let themselves be selected,
    # we check which ones are selected and only draw the box
    # around those objects.

    set selecteditems [filter "apply selected" $groupitems]
    if { [string compare $selecteditems ""] } {
	$this draw_bbox $selecteditems $g
    }
    not_undoable
    return $selecteditems
}

#
#  selects everything on the canvas
#

body Code::CodeCanvas::select_all {} {
    select_items [winfo children $oldthis]
    not_undoable
}

#
#  selects only the given items
#

body Code::CodeCanvas::select_items {items} {
    map "$this select_group" [filter "notempty" [groupset $items]]
}

#
#  deselects everything on the canvas
#

body Code::CodeCanvas::deselect_all {} {
    map "apply deselect" [selected_items]
    $this delete bbox
}

#
#  returns all the currently selected items
#

body Code::CodeCanvas::selected_items {} {
    return [array names selected]
}

body Code::CodeCanvas::selected_groups {} {
    return [groupset [selected_items]]
}

body Code::CodeCanvas::set_selected {item} {
    set selected($item) 1
}

body Code::CodeCanvas::unset_selected {item} {
    catch { unset selected($item) }
}


#
#  deletes all selected items
#

body Code::CodeCanvas::delete_selected {} {
    copy $UndoBuffer
    delete_items [selected_items]
    Undo::delete_selected
}

#
#  delete "items"
#

body Code::CodeCanvas::delete_items {items} {
    foreach item $items {
	catch { cleanup_after $item }
	catch { rename $item "" }
    }
    $this delete bbox
}

body Code::CodeCanvas::cleanup_after {item} {

    # remove this item from the selected and items_in_group arrays

    unset_selected $item
    set groups [$item group_name]
    foreach g [lrange [split $groups .] 1 end] {

	# delete this from all groups

	set index [lsearch -exact items_in_group(.$g) $this]
	lreplace items_in_group(.$g) $index $index

    }
    
    catch { unset items_in_group(.[$item info variable id -value]) }
    
}


#
#  moves the selected items to (x,y)
#  on the canvas
#

body Code::CodeCanvas::move_selected {x y} {

    set dx [expr $x - $current_x]
    set dy [expr $y - $current_y]

    move_selected_by $dx $dy
}

#
#  moves the selected items by
#  (dx,dy) canvas units
#

body Code::CodeCanvas::move_selected_by {dx dy} {
    $this move bbox $dx $dy
    map "apply \"move $dx $dy\"" [selected_items]
    Undo::move_selected_by $dx $dy
}

#
#  aligns the selected objects to each other
#  in the specified direction
#

body Code::CodeCanvas::align {direction} {

    set allitemgroups [selected_groups]

    #
    #  how we align:
    #
    #     top = set all to minimum y-axis
    #     bottom = set all to maximum y-axis
    #     left = set all to minimum x-axis
    #     right = set all to maximum x-axis
    #     vcenter = set all to avg of bbox y-axis
    #     hcenter = set all to avg of bbox x-axis
    #

    ::scan [$this bbox selected] "%d %d %d %d" x1 y1 x2 y2

    #
    #  create an appropriate "coordinate transform" string,
    #  to be evaluated for each object,
    #  which adjusts dx/dy appropriately.
    #

    switch $direction {
	top {
	    set coordxform "set dy \[expr $y1-\$topy\]"
	}
	bottom {
	    set coordxform "set dy \[expr $y2-\$boty\]"
	}
	left {
	    set coordxform "set dx \[expr $x1-\$leftx\]"
	}
	right {
	    set coordxform "set dx \[expr $x2-\$rightx\]"
	}
	vcenter {
	    set vcenter [expr ($y1+$y2)/2]
	    set coordxform \
		    "set thisvcenter \
		    \[expr (\$topy+\$boty)/2\] ; \
		    set dy \[expr $vcenter-\$thisvcenter\]"
	}
	hcenter {
	    set hcenter [expr ($x1+$x2)/2]
	    set coordxform \
		    "set thishcenter \
		    \[expr (\$leftx+\$rightx)/2\] ; \
		    set dx \[expr $hcenter-\$thishcenter\]"
	}
	default {
	    set coordxform ""
	}
    }
    
    ## process by groups
    
    deselect_all
    
    foreach g $allitemgroups {
	
	set selecteditems [select_group $g]

	::scan [eval $this bbox $selecteditems] "%d %d %d %d" leftx topy rightx boty
	set dx 0
	set dy 0
	eval $coordxform
	move_selected_by $dx $dy

	map "apply deselect" $selecteditems
	remove_bbox $g

    }

    ## reselect these items

    map "$this select_group" $allitemgroups

    not_undoable
}

# Note: unintuitive stuff here.
# these are the GLOBAL attributes for the program.
# (this is so the top-level canvas acts
#  like a call-node without a collapse button.)

body Code::CodeCanvas::attributes {} {
    #
    # brings up the global attribute window
    #

    global \
	    ::c2_tmp

    set attribs \
	    {c2_GlobalTypes \
	    c2_GlobalFunctionSignatures \
	    c2_GlobalFunctionDefinitions \
	    c2_GlobalDocumentation \
	    c2_FilesToLink}

    foreach i $attribs {
	global ::$i
	eval "global c2_tmp ; \
		set c2_tmp($i) \$$i"
    }
    if {[._Globalattributes activate]} {
	global c2_tmp

	# Trash any trailing newlines after the last file in "FilesToLink"
	# because otherwise we'll screw up the Makefile.

	regsub -all "\[\n\t\]*\$" $c2_tmp(c2_FilesToLink) "" c2_tmp(c2_FilesToLink)

	foreach i $attribs {
	    eval "global c2_tmp ; set $i \"\$c2_tmp($i)\""
	}
    }
}

#  This print_attributes, like the attributes body Code::CodeCanvas::above,
#  is a specialized version of the one from CodeObj. I
#  still am uncomfortable about making Canvas a CodeObj, so...

body Code::CodeCanvas::print_attributes {} {
    global \
	    ::c2_PrintToPrinter \
	    ::c2_PrintFilename \
	    ::c2_PrintCommand \
	    ::c2_PrintFirstPageFirst \
	    ::c2_PrintPortraitMode \
	    ::c2_PrintGreyscale \
	    ::c2_PrintPaperSize
    
    if {$c2_PrintToPrinter} {
	set printbuf [::open "|$c2_PrintCommand -T[file tail $FileName]" w]
    } else {
	set printbuf [::open $c2_PrintFilename w]
    }
    
    set attribs \
	    {c2_GlobalTypes \
	    c2_GlobalFunctionSignatures \
	    c2_GlobalFunctionDefinitions \
	    c2_GlobalDocumentation \
	    c2_FilesToLink}
    
    puts $printbuf "-------------------------"
    puts $printbuf "Code Main Canvas"
    puts $printbuf "-------------------------"

    foreach attr $attribs {
	global ::$attr
	puts $printbuf [concat "---" $attr "---"]
	puts $printbuf [set $attr]
	puts $printbuf ""
    }

    close $printbuf
}


body Code::CodeCanvas::bitmap {} {
    return "@$ReleaseDirectory/bitmaps/Canvas.bit"
}


#
#  put little boxes at the corners of the bounding rectangle
#  around these items (and give them the tag "tagid")
#

body Code::CodeCanvas::draw_bbox {items tagid} {

    ## we will put a temporary tag on each item and then
    ## take advantage of the builtin bbox canvas function
    ## to determine the bounding box
    
    # don't draw boxes around arcs (delete them from the items list)
    set items [filter {function {x} {return [expr ![$x isa ::Code::Arc]]}} $items]
    if {![string compare $items ""]} return

    set corners [eval $this bbox $items]

    ::scan $corners "%f %f %f %f" x1 y1 x2 y2

    if { ![string compare $corners ""] } {
	return
    }

    ## draw the little rectangles on the corners of the box,
    ## with the tags bbox and bbox+tag identifier given as a parameter.
    
    set x1_minus_5 [expr $x1-5]
    set y1_minus_5 [expr $y1-5]
    set x2_plus_5  [expr $x2+5]
    set y2_plus_5  [expr $y2+5]

    set b(0) [$this create rectangle $x1_minus_5 $y1_minus_5 $x1 $y1 \
	    -fill Black -tags bbox]
    set b(1) [$this create rectangle $x1_minus_5 $y2_plus_5 $x1 $y2 \
	    -fill Black -tags bbox]
    set b(2) [$this create rectangle $x2 $y1 $x2_plus_5 $y1_minus_5 \
	    -fill Black -tags bbox]
    set b(3) [$this create rectangle $x2 $y2 $x2_plus_5 $y2_plus_5 \
	    -fill Black -tags bbox]
    
    foreach i {0 1 2 3} {
	$this addtag bbox$tagid withtag $b($i)
    }
    
}


body Code::CodeCanvas::remove_bbox {tagid} {
    $this delete bbox$tagid
}

##

#
#  returns all groups in group list
#

body Code::CodeCanvas::all_groups {item} {
    set g [$item group_name]
    regsub -all {(\.)} $g { .} newg
    return $newg
}

#
#  returns the set of all group-parents
#

body Code::CodeCanvas::groupset {items} {
    return [unique [map "apply group_parent_name" $items]]
}


#
#  returns every item in group g
#

body Code::CodeCanvas::items_in_group {g} {
    if {[info exists items_in_group($g)]} {
	return $items_in_group($g)
    } else {
	return ""
    }
}

#
#  returns every item in the same group as "item"
#

body Code::CodeCanvas::items_in_same_group {item} {
    return [items_in_group [$item group_parent_name]]
}


#
#  locks the groups for the selected items
#  so that they cannot be ungrouped without
#  first being unlocked
#

body Code::CodeCanvas::lock_group {g} {
    set lock($g) 1
}

body Code::CodeCanvas::lock {} {
    map "$this lock_group" [selected_groups]
    Undo::lock
}

#
#  unlocks all selected groups
#

body Code::CodeCanvas::unlock_group {g} {
    set lock($g) 0
}

body Code::CodeCanvas::unlock {} {
    map "$this unlock_group" [selected_groups]
    Undo::unlock
}

#
#  breaks the selected items into their
#  constituent groups
#

body Code::CodeCanvas::ungroup_selected {} {
    set groups [selected_groups]
    deselect_all

    foreach group $groups {

	# If the group is not locked, break
	# it up.

	if {!$lock($group)} {
	    set items [items_in_group $group]
	    map "apply ungroup" $items
	    map "apply select_group" $items
	} else {
	    # Otherwise, we just re-select it.
	    select_group $group
	}
    }
    Undo::ungroup
}


#
#  puts the selected items into one group
#

body Code::CodeCanvas::group_selected {} {

    set items [selected_items]
    set newgroup [group $items]

    ## remove all bboxes and redraw the new one

    $this delete bbox
    draw_bbox $items $newgroup
}


body Code::CodeCanvas::group {items} {

    #  take all the items and put them into the same group
    #  by appending a new group to their group tags

    set newgroup ""

    #  don't try to group if no items are selected

    if { [string compare $items ""] } {

	#  don't make a new group if all items
	#  are in the same group already
	#  (this has the desirable side effect of never
	#   grouping a singleton)
	
	set allinsamegroup 1
	set currentgroup [[lindex $items 0] group_parent_name]
	foreach i $items {
	    if { [string compare $currentgroup [$i group_parent_name]] } {
		set allinsamegroup 0
		break
	    }
	}

	if { !$allinsamegroup } {

	    set newgroup [$this new_id]
	    foreach i $items {
		
		# replace this item's group name
		#  with the old name + .$newgroup
		
		set gname [$i group_name]
		set gname "$gname.$newgroup"
		$i set_group $gname
	    }
	    set_group $newgroup $items
	}
    }
    return $newgroup
}

body Code::CodeCanvas::set_group {group items} {
    set items_in_group(.$group) $items
    # make group initially unlocked
    set lock(.$group) 0
}

body Code::CodeCanvas::add_to_group {group item} {
    lappend items_in_group(.$group) $item
}

body Code::CodeCanvas::delete_from_group {group item} {
    set index [lsearch -exact items_in_group($group) $this]
    lreplace items_in_group(.$group) $index $index
}

#
# draws/updates the rubber band rectangle
# used for selecting
#

body Code::CodeCanvas::RubberBandDraw { terminalX terminalY } {

    set canvasX [$this canvasx $terminalX]
    set canvasY [$this canvasy $terminalY]

    if { ![string compare $RubberBand ""] } {
	set RubberBand [$this create rectangle \
		$current_x $current_y \
		$canvasX $canvasY]
    } else {
	
	#
	#  we use the current position as the anchor
	#  for the rubber band rectangle.
	#
	
	set x1 [min $canvasX $current_x]
	set y1 [min $canvasY $current_y]
	set x2 [max $canvasX $current_x]
	set y2 [max $canvasY $current_y]
	
	eval "$this coords $RubberBand \
		$x1 $y1 $x2 $y2"
	
    }
}

#
# selects every group with a child inside these coords
#

body Code::CodeCanvas::select_inside {coords} {

    set enclosed_items "[eval $this find enclosed $coords]"
    set enclosed_objects [filter "notempty" \
	    [map "$this lookup_parent" $enclosed_items]]
    set groups [groupset $enclosed_objects]
    map "$this select_group" [filter "notempty" $groups]
}

body Code::CodeCanvas::lookup_parent {item} {
    return $lookup($item)
}

#
#  find every object within the
#  rubber band and select it
#

body Code::CodeCanvas::RubberBandSelect {} {

    set coords [$this coords $RubberBand]
    
    ## get rid of the rubber band
    
    $this delete $RubberBand
    set RubberBand ""

    select_inside $coords
    
    not_undoable
}

# binding actions

body Code::CodeCanvas::Canvas_Enter {} {
    set_cursor $ToolSelected
    if {![string compare $ToolSelected "Query"]} {
	[winfo command $oldthis] configure -closeenough 5.0
    } else {
	[winfo command $oldthis] configure -closeenough 0.2
    }

    # ::focus -force $CurrentWindow
}

#
#  button-1 pushed, no shift
#

body Code::CodeCanvas::Unshifted_Button_Press_Action { terminalX terminalY } {

    #
    #  A click on an empty part of the
    #  canvas while in select mode or comment mode
    #  deselects everything.
    #

    if { ((![string compare $ToolSelected "Select"]) || (![string compare $ToolSelected "Comment"])) && \
	     (![string compare [$this find withtag current] ""])} {
	deselect_all
    }
    Button_Press_Action $terminalX $terminalY
}

#
#  button-1 pushed (whether shifted or not)
#

body Code::CodeCanvas::Button_Press_Action { terminalX terminalY} {

    #
    #  Make sure we are now the "CurrentWindow",
    #  since we've just received a button click inside.
    #

    set current [$this find withtag current]

    set current_x [canvasx $terminalX]
    set current_y [canvasy $terminalY]

    if 0 {
    if { [string compare $CurrentWindow [info namespace tail $this]]} {
	set switchfocus 1
	if { [string compare $current ""] } {
	    set item [lookup_parent $current]
	    if { [$item isa ::Code::Comment] || [$item isa ::Code::Label] } {
		set switchfocus 0
	    }
	}
    }
    }
    
    set switchfocus 1
    if { [string compare $current ""] } {
	set item [lookup_parent $current]
	if { [$item isa ::Code::Comment] || [$item isa ::Code::Label] } {
	    set switchfocus 0
	}
    }

    if { $switchfocus } {
	Code::CodeCanvas::SetCurrentWindow $this
	::focus -force $oldthis
	set_cursor $ToolSelected
    }

    #
    # zoom in or out
    #

    switch $ToolSelected {
	ZoomIn { zoom_in $current_x $current_y ; return }
	ZoomOut { zoom_out $current_x $current_y ; return }
	default {}
    }


    #
    # if we are not on an object...
    #   create an item or an arc elbow
    #   or do nothing at all.
    #

    if { ![string compare $current ""] } {

	switch $ToolSelected {
	    Comment {
		$this create Comment $current_x $current_y \
			-text "// Comment"
		not_undoable
	    }
	    CompNode -
	    CallNode -
	    CreationParameterNode -
	    NameSharingNode -
	    InInterfaceNode -
	    OutInterfaceNode {
		$this create $ToolSelected $current_x $current_y
		not_undoable
	    }
	    Arc {
		$CurrentDrawingArcItem add-elbow $current_x $current_y
	    }
	    default {}
	}
    }
}


#
#  draw rubber-band or
#  drag around the canvas within
#  the scroll window
#

body Code::CodeCanvas::Button_Motion_Action { terminalX terminalY } {

    if { (![string compare $ToolSelected "Select"]) && \
	     (![string compare [$this find withtag current] ""])} {
	RubberBandDraw $terminalX $terminalY
    } elseif { (![string compare $ToolSelected "Drag"] ) } {
	set canvasX [$this canvasx $terminalX]
	set canvasY [$this canvasy $terminalY]
	scan mark 0 0
	scan dragto \
		[expr round(($canvasX-$current_x)/10.0)] \
		[expr round(($canvasY-$current_y)/10.0)]
    }
}

#
#  we have released the button --
#   if we were making a rubber-band,
#   select everything inside it
#

body Code::CodeCanvas::Button_Release_Action {} {
    if { (![string compare $ToolSelected "Select"]) && ([string compare $RubberBand ""]) } {
	RubberBandSelect
    }
}

#
#  double-click aborts arc
#

body Code::CodeCanvas::Delete_Arc_In_Progress {} {
    if { (![string compare $CurrentDrawingArcItem {}]) && (![string compare $ToolSelected "Arc"]) } {
	set arc $CurrentDrawingArcItem
	set CurrentDrawingArcItem ""

	catch { rename $arc "" }

	::bind $oldthis <Any-Motion> ""
	deselect_all
	Code::CodeCanvas::set_tool Select
    }
}


body Code::CodeCanvas::set_id {val} {
    set new_id $val
}

body Code::CodeCanvas::get_id {} {
    return $new_id
}

body Code::CodeCanvas::set_cursor {type} {
    [winfo command $oldthis] configure -cursor "$cursor_value($type)"
}

body Code::CodeCanvas::auto_incr { name e op } {
	upvar $name x
	incr x
}

body Code::CodeCanvas::update_name { name e op} {
    set Name [file tail [$this info variable FileName -value]]
    if { ![string compare $Name ""] } {
	set Name "Untitled"
    }
}


body Code::CodeCanvas::set_tool {tool} {
    set Code::CodeCanvas::ToolSelected $tool
    foreach canv [array names AllCanvases] {
	catch { $canv set_cursor $tool }
    }
}

body Code::CodeCanvas::tool {} {
    return [set Code::CodeCanvas::ToolSelected]
}

body Code::CodeCanvas::SetCurrentWindow {win} {
    set CurrentWindow [info namespace tail $win]
}

body Code::CodeCanvas::init {undobuf cutbuf maincanv sfg fg rd} {
    set UndoBuffer $undobuf
    set CutBuffer  $cutbuf
    set MainCanvas $maincanv
    set SelectedFGColor $sfg
    set FGColor    $fg
    set ReleaseDirectory $rd

    # makes these arrays to begin with

    set lookup(BogusObject) ""
    set lock(BogusObject) 0
    
    # initialize the cursors for all the possible modes
    # (note that these are held in a common array)
    
    set cursor_value(Select) ""
    set cursor_value(Arc)        "@$ReleaseDirectory/bitmaps/Arc.bit Black"
    set cursor_value(CallNode)  \
	    "@$ReleaseDirectory/bitmaps/CallNode.bit Black"
    set cursor_value(CompNode)  \
	    "@$ReleaseDirectory/bitmaps/CompNode.bit Black"
    set cursor_value(CreationParameterNode)  \
	    "@$ReleaseDirectory/bitmaps/CreationParameterNode.bit Black"
    set cursor_value(NameSharingNode)  \
	    "@$ReleaseDirectory/bitmaps/NameSharingNode.bit Black"
    set cursor_value(InInterfaceNode)  \
	    "@$ReleaseDirectory/bitmaps/InInterfaceNode.bit Black"
    set cursor_value(OutInterfaceNode)  \
	    "@$ReleaseDirectory/bitmaps/OutInterfaceNode.bit Black"
    set cursor_value(Comment) "xterm"
    set cursor_value(Query) "@$ReleaseDirectory/bitmaps/info Black"
    set cursor_value(Waiting) "watch"
    set cursor_value(Drag) "@$ReleaseDirectory/bitmaps/HandDrag.bit Black"
    set cursor_value(ZoomIn) "@$ReleaseDirectory/bitmaps/ZoomIn.bit Black"
    set cursor_value(ZoomOut) "@$ReleaseDirectory/bitmaps/ZoomOut.bit Black"
    set CurrentWindow [info namespace tail $maincanv]
}
