#
# DebuggerCodeCanvas.tcl
# ------------------------------------------------------------------------
# A canvas derived from CodeCanvas which implements additional
# functionality for debugger animations.
# ------------------------------------------------------------------------
# @(#) $Id: DebuggerCodeCanvas.tcl,v 1.3 1996/01/09 18:32:37 emery Exp $
#
# ------------------------------------------------------------------------
# AUTHOR:
#
# Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
# Systems Analyst                 @           <mailto:emery@cs.utexas.edu>
# Parallel Programming Group      |  <http://www.cs.utexas.edu/users/code>
# Department of Computer Sciences |             <http://www.cs.utexas.edu>
# University of Texas at Austin   |                <http://www.utexas.edu>
# ========================================================================


class Code::DebuggerCodeCanvas {

    inherit Code::CodeCanvas

    constructor {args} { eval Code::CodeCanvas::constructor $args } {}

    public method attributes {}
    public method create {type args}

    public method relayout {}      ;# redraw and redisplay the canvas
    public method graph_update {path indices instance paths}
    public method select_others {obj}

    # The following methods are bound to <Enter> and <Leave>,
    # respectively.

    public method block_tool {}    ;# prevents certain tools from use
    public method restore_tool {}  ;# restores access to all tools

    public variable canvas_type "" ;# "events" or "instances" -- eventually a new class?
    public variable dot_canvas     ;# the scratch canvas for Dot operations

    protected method group_bbox {object}
    protected method group_move {object dx dy}

    protected variable SavedTool "Select"
    protected variable dot_graph ;# the Dot graph
    protected common   node      ;# array of objects : node(full name,dot_graph)
    protected common   edge      ;# array of arcs    : edge(to,from,dot_graph)
    protected common   path_to_object ;# array of objects : path_to_object(path name)
    protected common   instances ;# array of int     : instances(full name)

    protected common   pixels_per_inch

}


#             #
# CONSTRUCTOR #
#             #

body Code::DebuggerCodeCanvas::constructor {args} {

    set pixels_per_inch  [winfo pixels . 1i]
    set dot_graph [dotnew digraph]


    #
    # modify object bindings
    #

    # prevent object motion
    # $this bind ::Code::CodeObj <Button1-Motion> {}

    # prevent editing of labels
    $this bind ::Code::Label <Button-1> {}

    # Have clicking on an object select related debugger objects.

    $this bind ::Code::CodeObj <Button-1> \
	    "set obj \[$this lookup_parent \[$this find withtag current\]\] ; \
	    if \{ \[$this tool\] == \"Select\" && !\[\$obj selected\] \} \
	    \{ $this deselect_all ; \
	    \$obj select_group ; \
	    $this select_others \$obj; \
	    \} elseif \{ \[$this tool\] == \"Query\" \} \{ \
	    \$obj attributes \
	    \} ; break"

    #
    # modify canvas bindings
    #

    # prevent object deletion, etc. via keyboard

    ::bind $oldthis <Control-g> {}
    ::bind $oldthis <Control-u> {}
    ::bind $oldthis <Control-v> {}

    ::bind $oldthis <Delete> {}
    ::bind $oldthis <BackSpace> {}

    ::bind $oldthis <Down>  {}
    ::bind $oldthis <Up>    {}
    ::bind $oldthis <Right> {}
    ::bind $oldthis <Left>  {}

    ::bind $oldthis <Control-Down>  {}
    ::bind $oldthis <Control-Up>    {}
    ::bind $oldthis <Control-Right> {}
    ::bind $oldthis <Control-Left>  {}

    # prevent user creation of new Code objects

    ::bind $oldthis <Enter> "$oldthis block_tool"
    ::bind $oldthis <Leave> "$oldthis restore_tool"

}


#         #
# METHODS #
#         #


body Code::DebuggerCodeCanvas::attributes {} {
}


if 0 {
    body Code::DebuggerCodeCanvas::add_instance {codeobj} {
	
	# Find the type information for the (normal) code object.
	set objtype [info namespace tail [$codeobj info class]]

	# Create the appropriate debugger object (prepend "Debugger").
	set debuggerobj [create Debugger$objtype 0 0 -id [$codeobj id] -object $codeobj]
    }
}


#
# Disallow user creation of new Code objects on
# the debugger canvas.
#

body Code::DebuggerCodeCanvas::block_tool {} {

    set SavedTool [tool]

    switch $SavedTool {
	Arc -
	Comment -
	CompNode -
	CallNode -
	CreationParameterNode -
	NameSharingNode -
	InInterfaceNode -
	OutInterfaceNode {
	    Code::CodeCanvas::set_tool "Select"
	}
	default {}
    }
}


#
# Create the object on the canvas,
# doing some internal bookkeeping.
#

body Code::DebuggerCodeCanvas::create {type args} {

    set id     [extract_arg -id args]
    set object [extract_arg -object args]

    set debuggerobj [eval Code::CodeCanvas::create $type $args]

    switch $type {
	arc -
	bitmap -
	image -
	line -
	oval -
	polygon -
	rectangle -
	text -
	window {
	    return $debuggerobj
	}
	default {

	    # Put a reference to this id number so we can later reference it by id.
	    set path_to_object($id) $object
	    
	    # Update the total number of instances of this id.
	    if {![info exists instances]} {
		set instances($id) 1
	    } else {
		incr instances($id)
	    }
	    
	    # Update the object's information.

	    # $debuggerobj configure -instance $instances($id)
	    # $debuggerobj configure -code_id  $id
	    
	    return $debuggerobj
	}
    }
}


#
# Add new nodes and arcs to the Dot graph.
#

body Code::DebuggerCodeCanvas::graph_update {path indices instance paths} {
    # Deselect all; we will highlight only new nodes and edges.

    deselect_all

    set path "$path$indices"

    switch $canvas_type {
	instances {
	    set myname [file tail $path]
	}
	events {
	    set myname [file tail $path]:$instance
	}
	default {
	    error "what da heck?"
	}
    }

    #
    # Add the new node to the graphs.
    # and add the new node to both canvases (out of sight for now).
    #
    # Set the node's width to the width of the bbox
    # around the Debugger object.
    #

    set graphnode [$dot_graph addnode $myname shape box]
    if {![info exists node($myname,$dot_graph)]} {
	set newnode [create DebuggerCompNode -100 -100]
	$newnode configure -name $myname
	set node($myname,$dot_graph) $newnode
	set newnodebbox [group_bbox $newnode]
	$graphnode setattributes width \
		[expr 2*([lindex $newnodebbox 2]-[lindex $newnodebbox 0]) \
		/ $pixels_per_inch]

    }

    #
    # Add the predecessor edges and draw the Arcs.
    #

    foreach n $paths {
	regexp {(/[^\:]*):([0-9]*)} $n dummy npath ninstance

	switch $canvas_type {
	    instances {
		set toname [file tail $npath]
	    }
	    events {
		set toname [file tail $npath]:$ninstance
	    }
	    default {
		error "what da heck?"
	    }
	}

	$dot_graph addedge $toname $myname

	# Add the arcs.

	if {![info exists edge($myname,$toname,$dot_graph)]} {
	    
	    # Find an output port on the "from" node and start an arc,
	    # storing it in the edge array.
	    
	    set outputport [first [filter "apply \"isa ::Code::OutputPort\"" [$node($toname,$dot_graph) info variable ports -value]]]
	    set arc [$outputport create_arc]
	    set edge($myname,$toname,$dot_graph) $arc
	    
	    # add an elbow
	    # eval $arc add-elbow [$canvas coords [$arc info variable current_elbow -value]]
	    
	    # Find (any) input port on the "to" node and complete the arc.
	    
	    set inputport [first [filter "apply \"isa ::Code::InputPort\"" [$node($myname,$dot_graph) info variable ports -value]]]
	    $inputport complete_arc
	    
	}

	# Highlight the current arc.
	$edge($myname,$toname,$dot_graph) select
    }

    # Highlight the current node.
    $node($myname,$dot_graph) select_group
}


#
# Return the bounding box around an object and all objects
# grouped together with it on the canvas.
#

body Code::DebuggerCodeCanvas::group_bbox {object} {
    set items [items_in_group [$object group_parent_name]]
    foreach i $items {
	addtag groupbox withtag $i
    }
    set bbox [$this bbox groupbox]
    foreach i $items {
	dtag $i groupbox
    }
    return $bbox
}


#
# Move an object and all objects grouped together with it.
#

body Code::DebuggerCodeCanvas::group_move {object dx dy} {
    foreach item [items_in_group [$object group_parent_name]] {
	$item move $dx $dy
    }
}


#
# Fix the graph layout, using Dot extensions.
#

body Code::DebuggerCodeCanvas::relayout {} {

    # Clean off the dot canvas and
    # draw the graph onto it.

    $dot_canvas delete all
    set c $dot_canvas
    $dot_graph layout
    eval [$dot_graph render]

    # Trash any errant box corners, created by select_group
    # in graph_update, above.

    delete withtag bbox

    #
    # Using the dot canvas as a reference,
    # move all of the nodes to their correct locations.
    #

    foreach nodename [$dot_graph listnodes] {

	set node_fullname [$nodename showname]
	
	#
	# Move the existing node to its new place.
	#

	# Find the bounding box around the node on the dot canvas.
	
	set bbox [$dot_canvas bbox 1$nodename]
	set xcor [expr ([lindex $bbox 0] + [lindex $bbox 2]) / 2]
	set ycor [expr ([lindex $bbox 1] + [lindex $bbox 3]) / 2]

	# Move the node to where it should be.
	
	set oldcoords [$this bbox $node($node_fullname,$dot_graph)]
	set oldxcor [lindex $oldcoords 0]
	set oldycor [lindex $oldcoords 1]

	if {($oldxcor != $xcor) || ($oldycor != $ycor)} {
	    group_move $node($node_fullname,$dot_graph) [expr $xcor-$oldxcor] [expr $ycor-$oldycor]
	}
    }
}


#
# When we leave the debugger canvas, restore
# the old tool.
#

body Code::DebuggerCodeCanvas::restore_tool {} {
    Code::CodeCanvas::set_tool $SavedTool
}


body Code::DebuggerCodeCanvas::select_others {obj} {

    if {[$obj isa ::Code::Label]} {
    } else {
	puts "$obj : select others!"

    }
}
