/*
   2-D Barnes-Hut particle simulator-- aux files for CODE program

   The main data structure is a quad tree of treenodes.  Leaf nodes represent
   lone particles and have no children.  The 4 pointers to children are
   arranged as follows.

          ---------
          | 0 | 1 |
          ---------
          | 3 | 2 |
          ---------

   There is also an array of particles.

*/

#include <stdio.h>
#include <math.h>
/* #include <usclkc.h> */
#include "c2_globtype.h"

#define NULLPTR -1

#define pi 3.1415926536
#define nearness 1.0
#define G 0.0000000667
#define time_int 20.0

#define MAXPART 1000

#define INFINITY 10000000000.0
#define MAXMASS 500.0
#define MAXINITPOS 100.0
#define MAXINITVEL 2.0


void compute_acc(Part, Node, Acc_Mag, Acc_Angle)
     PartNode Part;
     TreeNode Node;
     double *Acc_Mag;
     double *Acc_Angle;
{
   double dx, dy;

   dx = Node->Pos->x - Part->Pos->x; 
   dy = Node->Pos->y - Part->Pos->y;
   *Acc_Angle = atan2(dy, dx);
   dy = dy*dy;
   dx = dx*dx;
   dx = dx+dy;
   *Acc_Mag = G * Node->Mass / dx;
}



int far(p1, p2, ll, ur)
     Point p1;
     Point p2;
     Point ll;
     Point ur;
{
   double cell_len, dist, dx, dy;

   dx = ur->x - ll->x; dx = dx*dx;
   dy = ur->y - ll->y; dy = dy*dy;

   cell_len = dx + dy;
   dx = p1->x - p2->x; dx = dx*dx;
   dy = p1->y - p2->y; dy = dy*dy;
   dist = dx + dy;

   return(cell_len/dist < nearness);
}


   /* Vector add:  a1 = a1 + a2 */

void vector_sum(a1_Mag, a1_Angle, a2_Mag, a2_Angle)
     double *a1_Mag;
     double *a1_Angle;
     double a2_Mag;
     double a2_Angle;
{
   double x_comp, y_comp;

   x_comp = *(a1_Mag) * cos(*(a1_Angle))  +  a2_Mag * cos(a2_Angle);
   y_comp = *(a1_Mag) * sin(*(a1_Angle))  +  a2_Mag * sin(a2_Angle);
   *(a1_Mag) = sqrt(x_comp*x_comp + y_comp*y_comp);
   if (y_comp == 0.0 && x_comp == 0.0)
     *(a1_Angle) = 0.0;
   else
     *(a1_Angle) = atan2(y_comp, x_comp);
}


int leaf(Node)
     TreeNode Node;
{
   int i;

   for (i = 0; i < 4; i++)
     if (Node->Child[i] != NULLPTR) return 0;
   return 1;
}


void acc(Part, Tree, CurIndex, a_Mag, a_Angle)
     PartNode Part;
     TreeType Tree;
     int CurIndex;
     double *a_Mag;
     double *a_Angle;
{
   int i;
   double tmp_Mag;
   double tmp_Angle;
   TreeNode Node;

   Node = Tree->Nodes[CurIndex];

   if (Part->Pos->x != Node->Pos->x || Part->Pos->y != Node->Pos->y) {
      if (leaf(Node))
	compute_acc(Part, Node, a_Mag, a_Angle);
      else if (far(Part->Pos, Node->Pos, Node->BoundBox->LL,
		   Node->BoundBox->UR))
	compute_acc(Part, Node, a_Mag, a_Angle);
      else {
	 tmp_Mag = 0.0; tmp_Angle = 0.0;
	 for (i = 0; i < 4; i++) {
	    if (Node->Child[i] != NULLPTR) {
	       acc(Part, Tree, Node->Child[i], &tmp_Mag, &tmp_Angle);
	       vector_sum(a_Mag, a_Angle, tmp_Mag, tmp_Angle);
	    }
	 }
      }
   }
}



void compute_pos(Part)
     PartNode Part;
{
   double delta_mag, delta_angle;

   delta_mag = time_int * Part->Vel->Mag;
   delta_angle = Part->Vel->Angle;

   Part->Pos->x += delta_mag*cos(delta_angle);
   Part->Pos->y += delta_mag*sin(delta_angle);
}


void Move(Tree, SubPart, NumPart, SpaceBox)
     TreeType Tree;
     PartArray SubPart;
     int NumPart;
     Box SpaceBox;
{
   double a_Mag;
   double a_Angle;
   int i;

   SpaceBox->LL->x = INFINITY;
   SpaceBox->LL->y = INFINITY;
   SpaceBox->UR->x = -INFINITY;
   SpaceBox->UR->y = -INFINITY;;

   for (i = 0; i < NumPart; i++)
        if (SubPart[i]->Mass >= 0.0) {
	   a_Mag = 0.0; a_Angle = 0.0;
	   acc(SubPart[i], Tree, 0, &a_Mag, &a_Angle);
	   compute_pos(SubPart[i]);

	   if (SubPart[i]->Pos->x > SpaceBox->UR->x)
	     SpaceBox->UR->x = SubPart[i]->Pos->x;
	   if (SubPart[i]->Pos->x < SpaceBox->LL->x)
	     SpaceBox->LL->x = SubPart[i]->Pos->x;

	   if (SubPart[i]->Pos->y > SpaceBox->UR->y) 
	     SpaceBox->UR->y = SubPart[i]->Pos->y;
	   if (SubPart[i]->Pos->y < SpaceBox->LL->y)
	     SpaceBox->LL->y = SubPart[i]->Pos->y;

	   a_Mag = time_int * a_Mag;
	   vector_sum(&(SubPart[i]->Vel->Mag), &(SubPart[i]->Vel->Angle),
		      a_Mag, a_Angle);
	}
}

   /* Determine index of quad in Node in which Pos lies.  Also
      return bounding box of this quad */

void get_quad(Node, Pos, quad_index, bb)
     TreeNode Node;
     Point Pos;
     int *quad_index;
     Box bb;
{
   double center_x, center_y;

   center_x = (Node->BoundBox->LL->x + Node->BoundBox->UR->x)/2.0;
   center_y = (Node->BoundBox->LL->y + Node->BoundBox->UR->y)/2.0;

   if (Pos->x < center_x) {
      if (Pos->y < center_y) {
         *quad_index = 3;
         bb->LL->x = Node->BoundBox->LL->x;
         bb->LL->y = Node->BoundBox->LL->y;
         bb->UR->x = center_x;
         bb->UR->y = center_y;
      } else {
	*quad_index =  0;
	bb->LL->x = Node->BoundBox->LL->x;
	bb->LL->y = center_y;
	bb->UR->x = center_x;
	bb->UR->y = Node->BoundBox->UR->y;
     }
   } else if (Pos->y < center_y) {
      *quad_index = 2;
	bb->LL->x = center_x;
	bb->LL->y = Node->BoundBox->LL->y;
	bb->UR->x = Node->BoundBox->UR->x;
	bb->UR->y = center_y;
   } else {
      *quad_index = 1;
      bb->LL->x = center_x;
      bb->LL->y = center_y;
      bb->UR->x = Node->BoundBox->UR->x;
      bb->UR->y = Node->BoundBox->UR->y;
   }
   
}



void update(Node, Part)
     TreeNode Node;
     PartNode Part;
{

   Node->Pos->x = Node->Pos->x*Node->Mass + Part->Pos->x*Part->Mass;
   Node->Pos->y = Node->Pos->y*Node->Mass + Part->Pos->y*Part->Mass;
   Node->Mass = Node->Mass + Part->Mass;

   Node->Pos->x = Node->Pos->x/Node->Mass;
   Node->Pos->y = Node->Pos->y/Node->Mass;
}


void InitNode(NewNode, Part, new_bb)
     TreeNode NewNode;
     PartNode Part;
     Box new_bb;
{
   int i;

   NewNode->Mass = Part->Mass;
   *(NewNode->Pos) = *(Part->Pos);
   *(NewNode->BoundBox->LL) = *(new_bb->LL);
   *(NewNode->BoundBox->UR) = *(new_bb->UR);
   for (i = 0; i < 4; i++)
     NewNode->Child[i] = NULLPTR;
}


void enter_particle(Tree, NewPart, SpaceBox, bb1, bb2)
     TreeType Tree;
     PartNode NewPart;
     Box SpaceBox;
     Box bb1;        /* These are params because it is a pain to alloc */
     Box bb2;        /* storage for them outside of CODE */
{

   int quad_index1, quad_index2, i;
   TreeNode Node, TmpNode;

   if (NewPart->Mass == 0.0)  /* Do not enter "null" particles */
     return;

   if (Tree->NextFree == 0) {  /* Tree is empty, add particle */
      Node = Tree->Nodes[0];
      InitNode(Node, NewPart, SpaceBox);
      Tree->NextFree = 1;
      return;
   }

   /* Got to search down the tree */

   Node = Tree->Nodes[0];

   for (;;) {

      if (leaf(Node)) {

	 if (NewPart->Pos->x == Node->Pos->x && 
	     NewPart->Pos->y == Node->Pos->y) {
	    /* Destroyed in (poorly modeled!) collision */
	    NewPart->Mass = 0.0;
	    return;
	 } else {
	    /* leaf node becomes non-leaf with two particles below */
	    get_quad(Node, Node->Pos, &quad_index2, bb2);
	    get_quad(Node, NewPart->Pos, &quad_index1, bb1);

	    TmpNode = Tree->Nodes[Tree->NextFree];
	    Node->Child[quad_index2] = Tree->NextFree;
	    Tree->NextFree += 1;

	    /* Add new tree node for old leaf node */
	    TmpNode->Mass = Node->Mass;
	    *(TmpNode->Pos) = *(Node->Pos);
	    *(TmpNode->BoundBox->LL) = *(bb2->LL);
	    *(TmpNode->BoundBox->UR) = *(bb2->UR);
	    for (i = 0; i < 4; i++)
	      TmpNode->Child[i] = NULLPTR;
	    
	    update(Node, NewPart);
	    
	    if (Node->Child[quad_index1] == NULLPTR) {
	       TmpNode = Tree->Nodes[Tree->NextFree];
	       Node->Child[quad_index1] = Tree->NextFree;
	       Tree->NextFree += 1;
	       InitNode(TmpNode, NewPart, bb1);
	       return;
	    } else {
	       Node = Tree->Nodes[Node->Child[quad_index1]];
	    }
	 }

      } else /* not a leaf node */ {

	 get_quad(Node, NewPart->Pos, &quad_index1, bb1);
	 update(Node, NewPart);
	 if (Node->Child[quad_index1] == NULLPTR) {
	    TmpNode = Tree->Nodes[Tree->NextFree];
	    Node->Child[quad_index1] = Tree->NextFree;
	    Tree->NextFree += 1;
	    InitNode(TmpNode, NewPart, bb1);
	    return;
	 } else {
	    Node = Tree->Nodes[Node->Child[quad_index1]];
	}

      }

   } /* for */

}



void  print_tree(Tree, CurIndex, Level, Count)
     TreeType Tree;
     int CurIndex;
     int Level;
     int *Count;
{
   TreeNode Node;

   if (CurIndex != NULLPTR) {
      *Count += 1;
      Node = Tree->Nodes[CurIndex];
      printf("quad_tree <%d> has node [%f, (%f, %f)]\n", Level, Node->Mass, 
	     Node->Pos->x, Node->Pos->y);
      printf("bounding box is (%f, %f, %f, %f)\n\n", Node->BoundBox->LL->x,
	     Node->BoundBox->LL->y, Node->BoundBox->UR->x, 
	     Node->BoundBox->UR->y);

      print_tree(Tree, Node->Child[0], Level+1, Count);
      print_tree(Tree, Node->Child[1], Level+1, Count);
      print_tree(Tree, Node->Child[2], Level+1, Count);
      print_tree(Tree, Node->Child[3], Level+1, Count);
   }

}



  /* Generate a particle's initial state */

void InitializeParticle(Part, SpaceBox)
   PartNode Part;
   Box SpaceBox;
{
   double base = 0x7fffffff;

   Part->Mass = random() / base * MAXMASS;

   Part->Pos->x = random() / base * MAXINITPOS;
   if (Part->Pos->x > SpaceBox->UR->x) SpaceBox->UR->x = Part->Pos->x;
   if (Part->Pos->x < SpaceBox->LL->x) SpaceBox->LL->x = Part->Pos->x;

   Part->Pos->y = random() / base * MAXINITPOS;
   if (Part->Pos->y > SpaceBox->UR->y) SpaceBox->UR->y = Part->Pos->y;
   if (Part->Pos->y < SpaceBox->LL->y) SpaceBox->LL->y = Part->Pos->y;

/*   Part->Vel->Mag = random() / base * MAXINITVEL; */
   Part->Vel->Mag = 0.0;
   Part->Vel->Angle = random() / base * 2.0 * pi;
}


void ReadInputs(NumPart, NumIt, NumProc)
   int *NumPart;
   int *NumIt;
   int *NumProc;
{
   printf("Enter number of particles: ");
   scanf("%d", NumPart);
   printf("Enter number of iterations: ");
   scanf("%d", NumIt);
   printf("Enter number of processes: ");
   scanf("%d", NumProc);
}



void PrintParticle(Part)
   PartNode Part;
{
      printf("\n");
      printf("  Mass = %f\n", Part->Mass);
      printf("  vel = (%f, %f)\n", Part->Vel->Mag,
	                           Part->Vel->Angle);
      printf("  pos = (%f, %f)\n", Part->Pos->x,
	                           Part->Pos->y);
}


void PrintInt(n)
     int n;
{
   printf("Int = %d\n", n);
}


void get_new_space_box(Part, SpaceBox, NumProc)
     SplitPartArray Part;
     Box SpaceBox;
     int NumProc;
{
   int i;

   SpaceBox->LL->x = Part[0]->SpaceBox->LL->x;
   SpaceBox->LL->y = Part[0]->SpaceBox->LL->y;
   SpaceBox->UR->x = Part[0]->SpaceBox->UR->x;
   SpaceBox->UR->y = Part[0]->SpaceBox->UR->y;

   for (i = 1; i < NumProc; i++) {
      if (Part[i]->SpaceBox->LL->x < SpaceBox->LL->x)
	SpaceBox->LL->x = Part[i]->SpaceBox->LL->x;
      if (Part[i]->SpaceBox->LL->y < SpaceBox->LL->y)
	SpaceBox->LL->y = Part[i]->SpaceBox->LL->y;
      if (Part[i]->SpaceBox->UR->x > SpaceBox->UR->x)
	SpaceBox->UR->x = Part[i]->SpaceBox->UR->x;
      if (Part[i]->SpaceBox->UR->y > SpaceBox->UR->y)
	SpaceBox->UR->y = Part[i]->SpaceBox->UR->y;
   }
}


void PrintPos(x, y)
   double x;
   double y;
{
   printf("%f %f\n", x, y);
}
