
(* ================== TempEst v1.2 ====================== *)

(* (C) 1995, 1996, Bell Laboratories and University of Texas *)

(* Carlos Puchol, cpg@cs.utexas.edu *)

structure OcCheck =
    struct

    val makestring = Int.toString
    val output = TextIO.output

	exception OcCheck of string;

	datatype reaction =
		(* outputs, next state, locals *)	
	    TERM of int list * int * int list
		(* outputs, input check, present branch, false branch *)
	  | CHECK of int list * int * reaction * reaction

	datatype state = state of int * reaction

	datatype input = PRESENT of int | ABSENT of int

	(* ================= *)

	fun occheck _ = ()

	(* return a pair of poterntial outputs and potential local
	 signals emitted in this state *)

	fun potential_outputs (TERM (ol, _, loc)) = (ol, loc)
	  | potential_outputs (CHECK (ol, _, t1, t2)) =
	    let
		val po1 = potential_outputs t1
		val po2 = potential_outputs t2
	    in
		((ol @ (#1 po1) @ (#1 po2)), ((#2 po1) @ (#2 po2)))
	    end

	fun potential_destinations (TERM (_, n, _)) = [n]
	  | potential_destinations (CHECK (_, _, t1, t2)) =
	    (potential_destinations t1) @ (potential_destinations t2)

	(* checks whether signal can be emitted from the given state *)
 	fun emits signal (state (_, tree)) =
	    case List.find (fn x => x = signal) (#1 (potential_outputs tree)) of
		SOME _		=> true
	      | _		=> false

	(* checks whether state 'to' can be reached directly from state *)
	fun goes_to to (state (i, tree)) =
	    case  List.find (fn x => x = to) (potential_destinations tree) of
		SOME _		=> true
	      | _		=> false

	(*
		 transition_outputs to from:

		 return the inputs required, outputs and local signal emitted
		 while taking a transition from "from" to state 'to'. return
		 NONE if there can be no transition out to 'to'.

		 int -> state -> (input list * int list * int list) option

	*)
	fun transition_outputs to (from as state (i, t)) =
	    let
		fun compute_outputs (TERM (ol, dest, loc)) =
		    if to = dest then SOME ([], ol, loc)
		    else NONE
		  | compute_outputs (CHECK (output, check, t1, t2)) =
			case compute_outputs t1 of
			    SOME (il, ol, ll)	=> SOME ((PRESENT check)::il, output @ ol, ll)
			  | NONE		=>
				case compute_outputs t2 of (* not-present path *)
				   SOME (il, ol, ll)	=> SOME ((ABSENT check)::il, output @ ol, ll)
				 | NONE			=> NONE
	    in
		compute_outputs t
	    end

	(* return a path from the start state to the emission of
	'signal' in the machine 'fsm'. possibly empty list if there is
	no path *)

	fun compute_path (signal, fsm) =
	    let
		fun find_first_emit [] = NONE
		  | find_first_emit (f::r) =
		    if emits signal f then SOME f
		    else find_first_emit r
	    in
		case find_first_emit fsm of
		    NONE			=> []
		  | SOME (s as state (i, _))	=>
			let
			    exception NoPath;
			    val start as state (si, _) = hd fsm
			    (* get a path from the start of the fsm to 'state' *)
			    fun get_path [] _ = raise NoPath
			      | get_path ((f as (state (i, _)))::rest) st =
				if i = st then  (* gone too far! we must decrease! *)
				    raise NoPath
				else
				    if goes_to st f then
					if i = si then [f]
					else f :: (get_path fsm i)
				    else get_path rest st
			in
			    if s = start then [start]
			    else (List.rev (s :: get_path fsm i)) handle _ => []
			end
	    end

	(*
		 emission_outputs signal from:

		 return the inputs required, outputs and local signal emitted
		 while taking a transition from "from" for emiting 'signal'.
		 NONE if there can be no transition out emitting it.

		 int -> state -> (input list * int list * int list) option

	*)
	fun emission_outputs signal (from as state (i, t)) =
	    let
		fun compute_outputs (TERM (ol, _, loc)) =
		    (case (List.find (fn x => x = signal) ol) of
			SOME _		=> SOME ([], ol, loc)
		      | _		=> NONE)
		  | compute_outputs (CHECK (output, check, t1, t2)) =
			 case (List.find (fn x => x = signal) output) of
			     SOME _		=> SOME ([], output, [])
			   | _		=> 
				 case compute_outputs t1 of
				     SOME (il, ol, ll) =>
					 SOME ((PRESENT check)::il, output @ ol, ll)
				   | NONE		=>
					 case compute_outputs t2 of (* not-present path *)
					     SOME (il, ol, ll)	=>
						 SOME ((ABSENT check)::il, output @ ol, ll)
					   | NONE		 => NONE
	    in
		compute_outputs t
	    end

	(* given a path, trace it back to all the signals involved
	 int * state list -> (int * int * (input list * int list * int list) option) list
	 *)
	fun trace_path (signal, path) =
	    let
		fun do_path [] = []
		  | do_path [x as state (i, _)] =[(i, ~1, emission_outputs signal x)]
		  | do_path ((x as state(i, _)):: (y as state (j, _)) :: rest) =
		    (i, j, (transition_outputs j x)) :: (do_path (y :: rest))
	    in
		do_path path
	    end

 	fun print_trace (file, trace) =
	    let
		(* int * (input list * int list * int list) option *)
		fun do_input [] = " "
		  | do_input ((PRESENT i)::r) = (" " ^ makestring i) ^ do_input r
		  | do_input ((ABSENT i)::r) = (" # " ^ makestring i) ^ do_input r
		fun do_output [] = " "
		  | do_output ((i:int) ::r) = (" " ^ makestring i) ^ do_output r
		fun do_one_step (_, _, NONE) = ()
		  | do_one_step (from:int, to:int, SOME (il, ol, ll)) =
		    output(file, (makestring from) ^ " : " ^
			   (if to < 0 then "" else makestring to) ^ " : " ^
				(do_input il) ^ " : " ^
				(do_output ol) ^ " : " ^
				(do_output (rev ll)) ^ ";\n")
		val _ = output(file, "\n\ntrace:\n");
		val _ = map do_one_step trace;
		val _ = output(file, "done.\n");
	    in
		()
	    end
   end
