
(* ================== TempEst v1.2 ====================== *)

(* (C) 1995, 1996, Bell Laboratories and University of Texas *)

(* Carlos Puchol, cpg@cs.utexas.edu *)

(* This module is the entry point for tl2strl *)

(* Carlos Puchol, cpg@research.att.com, June 1994 *)

structure TlToStrl =
struct

(********************************************

    Variables

 ********************************************)

    val version = "@(#)tl2strl v1.2"

    val makestring = Int.toString
    val output = TextIO.output
    val input = TextIO.inputN
    val open_in = TextIO.openIn
    val open_out = TextIO.openOut
    val close_in = TextIO.closeIn
    val close_out = TextIO.closeOut
    val std_err = TextIO.stdErr
    val exportFn = SMLofNJ.exportFn

(********************************************

    Functions

 ********************************************)

    fun rmfile s =			(* Remove a file silently *)
	Posix.FileSys.unlink s
	handle SystemCall => ()

    (* revindex - return the index of the position (starting from l and
     going backwards) in s where ch is found. if it is not found the exception
     Subscript is raised (by sub()) *)

    fun revindex ch (s, l) =
	let
	    fun check pos = 
		if (String.sub(s, pos)) = ch then
		    pos
		else check (pos - 1)
	in
	    check l
	end

(* Returns the last portion of an absolute file name.
	All from right to left of the name, until, *but not including*,
	the last '/', if any.

	basename "/usr/cddcd/cdcdc/cpg.cpg"	= "cpg.cpg"
	basename "/usr/cddcd/cdcdc/"		= ""
	basename "/usr/cddcd/cdcdc"		= "cdcdc"
	basename "/"				= ""
	basename ""				= ""
	basename "jhj"				= "jhj"	
*)

    fun basename s =
	let
	    val l = size s
	    val i = revindex #"/" (s, l-1) handle _ => ~1
	in
	    substring(s, i+1, l-i-1) handle _ => s
	end

(* dirname:
	dirname "/usr/cddcd/cdcdc/cpg.cpg"	= "/usr/cddcd/cdcdc/"
	dirname "/usr/cddcd/cdcdc/"		= "/usr/cddcd/cdcdc/"
	dirname "/usr/cddcd/cdcdc"		= "/usr/cddcd/"
	dirname "/"				= "/"
	dirname ""				= ""
	dirname "jhj"				= ""	
*)

    fun dirname s =
	let
	    val l = size s
	    val i = revindex #"/" (s, l-1) handle _ => ~1
	in
	    substring(s, 0, i+1) handle _ => s
	end

(* remove_suffix removes a suffix, if it exists, from the end of a string *)

    fun remove_suffix (s, suffix) =
	let
	    val l1 = size s
	    val l2 = size suffix
	in
	    if l2 >= l1 then s
	    else
		(if (substring (s, l1 - l2, l2) = suffix) then
		     substring (s, 0, l1 - l2)
		 else s) handle _ => s
	end


    and	main (arg0, argv) =	(* main: string * string list -> unit *)
	let
	    val success = OS.Process.success
	    val failure = OS.Process.failure

	    exception Tl2Strl of string;
	    (* Output code is optimized by default
	     for response and ensures properties *)
	    val	optimized_response = ref true
	    val	optimized_holding = ref true
	    val	want_version = ref false
	    val debug_mode = ref false
	    val	help_mode = ref false

	    fun process_options [] = []
	      | process_options (h::rest) =
		let
		    val first_arg = ref false
		    val _ =
			case h of
			    (* canonical response *)
			    "-cr"	=> optimized_response := false
			  (* canonical holding *)
			  | "-ch"	=> optimized_holding := false
			  | "-v"	=> want_version := true
			  | "-d"	=> debug_mode := true
			  | "-h"	=> help_mode := true
			  | _		=> first_arg := true
		in
		    if !first_arg then
			h :: rest
		    else
			process_options rest
		end
	    val filelist = process_options argv
	    fun do_file fullname =
		let
		    val _ =
			if !debug_mode then
			    output(std_err, "file: " ^ fullname ^"\n")
			else ()
		    val sname = remove_suffix (fullname, ".tl")
		    val iname = sname ^ ".tl"
		    val oname = (basename sname) ^ ".strl";
		    val infile = open_in iname
			handle _ => raise Tl2Strl ("cannot open input file " ^ iname ^ ".")
		    val outputfile = open_out oname
			handle _ => raise Tl2Strl ("cannot open output file " ^ oname ^ ".")
		    val _ = (((TlStruct.tl2strl (outputfile,
						 !optimized_response,
						 !optimized_holding,
						 Tlogic.parse (iname, infile)))
			      before ((close_out outputfile ;
				       close_in infile)))
			     handle TlStruct.Tl2Strl s =>
				 (output(std_err, "tl2strl: " ^ iname ^ ": " ^ s ^ ".\n");
				  output(std_err, "tl2strl: removing " ^ oname ^ "...\n");
				  rmfile oname))
			handle _ => rmfile oname
		in
		    success
		end
	in
	    if (!want_version) then (
		 output(std_err, (substring (version, 4, (size version)-4)) ^ "\n");
		 success)
	    else
		if ((!help_mode) orelse (filelist = [])) then
		    (output(std_err, "Usage: tl2strl [-v] [-cr] [-ch] [-h] <file>\n");
		     success)
		else
		    (((map do_file filelist); success) handle Tl2Strl s =>
			(output(std_err, "tl2strl: " ^ s ^ "\n"); failure))
	end


    fun dump_heap fname =
            (exportFn (fname, main); ())

end
