
(* ================== TempEst v1.2 ====================== *)

(* (C) 1995, 1996, Bell Laboratories and University of Texas *)

(* Carlos Puchol, cpg@cs.utexas.edu *)

(* Main module for the tl2strl translation *)

structure TlStruct =
struct

    val makestring = Int.toString
    val output = TextIO.output

    exception Tl2Strl of string		 	(* Error message *)

    datatype tl_assertion =
	Assertions of (	string list		* (* Type list *)
		       string list		* (* Input list *)
		       Relation list		* (* Relations list *)
		       PastForm list		* (* Past formulas *)
		       ResponseForm list	* (* Response formulas *)
		       PastForm list)		  (* Always formulae *)

    and PastForm = PastF of (string option * Expr)

    and ResponseType = BResponse | BHolds

    and ResponseForm =  Response of (ResponseType		* (* Type of formula *)
				     string option		* (* ID for the formula *)
				     Expr			* (* brexpr 1 *)
				     Expr			* (* brexpr 2 *)
				     int			* (* Deadline *)
				     string			* (* Signal to wait for *)
				     Expr option)		  (* SubjectTo opt brexpr *)

    and Expr =	Boolean of (Boolean * bool)		|
		Past of Past				|
		Id of string

    and Boolean = And of ((Expr * Expr) * bool)		|
		  Or of ((Expr * Expr) * bool)		|
		  Implies of ((Expr * Expr) * bool)	|
		  Not of (Expr * bool)

    and Past =	Once of Expr			|
		AllPast of Expr			|
		Previous of Expr		|
		BackTo of (Expr * Expr)		|
		Since of (Expr * Expr)

    and Relation = Pound of string list		|
		   Then of string list

    val	SPLIT_EVERY = 15;

    val	VIOLATED_SIGNALS: string list ref = ref [];

    val	optimized_r = ref true

    val	optimized_h = ref true

    val	outfile = ref TextIO.stdOut

    val	dummy_prefix = "T2S_"

    fun	oprint s = output (!outfile, s)

    fun	plus (x : int, y : int) = x + y

    fun PFName (PastF (SOME id, _)) = id
      | PFName _ = "Safety"

    fun VName (PastF (SOME id, _)) = id ^ "_VIOLATED"
      | VName _ = "Safety_VIOLATED"

    fun	dummy_list n =
	let
	    fun rest i = if (i = 0)
			     then  ", TRUE"
			 else if ((i mod SPLIT_EVERY) = 0) then
			     (",\n\t\t" ^ dummy_prefix  ^ (makestring i)
			      ^ (rest (i - 1)))
			      else 	(", " ^ dummy_prefix  ^ (makestring i)
					 ^ (rest (i - 1)))
	in
	    if (n < 1 ) then "TRUE"
	    else (dummy_prefix ^ (makestring n) ^ rest (n - 1))
	end

    fun	do_interface (modulename, typel, inputl, rellist, outputl, n_dummies) =
	let val _ = oprint ("module " ^ modulename ^ ":\n\n")
	    fun   str_list (i, []) = ""
	      | str_list (i, (""::y)) = str_list (i, y) (*Eliminate blanks *)
	      | str_list (i, (x::[])) = x
	      | str_list (i, (x::y)) =
		let val breakline = if ((i mod SPLIT_EVERY) = 0)
					then "\n\t\t"
				    else ""
		in
		    ((x ^ ", ") ^ breakline ^ (str_list (1 + i, y)))
		end
	    val _ = let
			fun do_one_type t =
			    oprint ("\ttype\t" ^ t ^ ";\n")
			val _ = map do_one_type typel
		    in
			oprint "\n"
		    end
	    val _ =
		if inputl <> nil then
		    oprint ("\tinput " ^ (str_list (1, inputl)) ^ ";\n\n")
		else ();
	    val _ = oprint "\n";
	    fun GetOutput [] = "NO_OUTPUT"
	      | GetOutput ([f]) = f
	      | GetOutput (f::r) = (f ^ ", " ^ (GetOutput r))
	    val _ = oprint ("\toutput " ^ (GetOutput (!VIOLATED_SIGNALS)) ^ ";\n\n");
	    val _ = let
			fun RelList c [] = ""
			  | RelList c (f::[]) = f
			  | RelList c (f::l) = f ^ c ^ (RelList c l)
			fun DoOneRel (Pound []) = ()
			  | DoOneRel (Then []) = ()
			  | DoOneRel (Pound f) =
			    oprint ("\trelation " ^ (RelList " # " f) ^ ";\n")
			  | DoOneRel (Then f) =
			    oprint ("\trelation " ^ (RelList " => " f) ^ ";\n")
		    in
			map DoOneRel rellist
		    end
	    val _ = oprint ("\n\tsignal " ^ (str_list (1, outputl )) ^ ", " ^
			    (dummy_list n_dummies) ^ " in")
	    val _ = oprint "\n\n"
	in
	    oprint ("\tsustain TRUE\n")
	end

    val count = ref 0

    fun	next_dummy_id () =
	let
	    val _ = (count := (!count) + 1)
	in
	    (dummy_prefix ^ (makestring (!count)))
	end

    and	in_parallel ()	= oprint "||\n"

    and	  all_bool _ (id :: []) = id
      | all_bool s (id :: rest) = id ^ s ^ (all_bool s rest)
      | all_bool _ [] = ""
	
    and	all_and idl = all_bool " and " idl

    and	all_or idl = all_bool " or " idl

    and	every_tick (condition, then_else, signal) =
	let
	    val _ = in_parallel ();
	    val _ = oprint ("\t[\n\t every immediate tick do\n" ^
			    "\t\tpresent [ " ^ condition ^ " ] " ^
			    then_else ^ "\n" ^
			    "\t\t\temit " ^ signal ^ "\n" ^
			    "\t\tend present\n" ^ "\t end\n\t]\n"
			    )
	in
	    signal
	end

    and	every_tick_then(cond, signal) =
	every_tick (cond, "then", signal)

    and	every_tick_else(cond, signal) =
	every_tick (cond, "else", signal)

    and	do_past (Once e) = 
	let
	    val previd = do_expr e
	    val newid = next_dummy_id ()
	    val _ = in_parallel ();
	    val _ = oprint ("\t[\n\t await immediate " ^ previd ^ ";\n" ^
			    "\t sustain " ^ newid ^ "\n\t]\n")
	in
	    newid
	end
      | do_past (AllPast e) =
	let val previd = do_expr e
	    val newid = next_dummy_id ()
	    val _ = in_parallel ();
	    val _ = oprint ("\t[\n\t do\n\t\tsustain " ^ newid ^";\n" ^
			    "\t watching immediate [ not ( " ^
			    previd ^ " )]\n\t]\n")
	in
	    newid
	end
      | do_past (Previous e) =
	let val previd = do_expr e
	    val newid = next_dummy_id ()
	    val _ = in_parallel ();
	    val _ = oprint ("\t[\n\t loop\n" ^
			    "\t\tpresent " ^ previd ^ " then\n" ^
			    "\t\t\tawait tick;\n" ^
			    "\t\t\temit " ^ newid ^ "\n" ^
			    "\t\telse await tick\n" ^
			    "\t\tend present\n" ^
			    "\t end\n\t]\n")
	in
	    newid
	end
      | do_past (Since (e1, e2)) =
	let val (id1, id2) = (do_expr e1, do_expr e2)
	    val newid = next_dummy_id ()
	    val _ = in_parallel ();
	    val _ = oprint ("\t[\n\t every immediate " ^ id2 ^ " do\n" ^
			    "\t\tdo\n\t\t\tsustain " ^ newid ^ "\n" ^
			    "\t\twatching [ not ( " ^ id1 ^ " )]\n" ^
			    "\t end\n\t]\n")
	in
	    newid
	end
      | do_past (BackTo (e1, e2)) =
	do_expr (Boolean ((Or
			   ((Past (Since (e1, e2)),
			     (Past (AllPast e1))), false)), false))

    and	process_ground ( x, operator,  y) =
	let
	    fun bottom_ground (Id x) = x
	      | bottom_ground (Boolean (x, true)) = do_ground x
	      | bottom_ground _ =
		raise Tl2Strl ("internal error 02 while processing" ^
			       " ground boolean formulas.\n" ^
			       "Please send the file that caused this error to\n"^
			       "cpg@research.att.com\n")
	    val sx = bottom_ground x
	    val sy = bottom_ground y
	in
	    if ( operator =	"not") then
		("(not " ^ sx ^ " )")
	    else if (operator = "implies") then
		("((not " ^ sx ^ ") or " ^ sy ^ ")")
		 else	("(" ^ sx ^ " " ^ operator ^ " " ^ sy ^ ")")
	end

    and	do_ground formula =	
	case formula of
	    (And ((x,y), true))	=> process_ground (x, "and", y)
	  | (Or  ((x,y), true))	=> process_ground (x, "or", y)
	  | (Implies ((x,y), true))	=> process_ground (x, "implies", y)
	  | (Not (x,true))		=> process_ground (x, "not", x) 
	  | _			=> 
		raise Tl2Strl ("internal error 01 while processing ground" ^
			       " boolean formulas.\nPlease send the " ^
			       "file that caused this error to\n"^
			       "cpg@research.att.com\n")


    and do_bool (And ((e1, e2),_)) =
	let
	    val (id1, id2) = (do_expr e1, do_expr e2)
	in
	    every_tick_then ( "(" ^ id1 ^ " and " ^ id2 ^ ")",
			     next_dummy_id ())
	end
      | do_bool (Or ((e1, e2),_)) =
	let
	    val (id1, id2) = (do_expr e1, do_expr e2)
	in
	    every_tick_then ( "(" ^ id1 ^ " or " ^ id2 ^ ")",
			     next_dummy_id () )
	end
      | do_bool (Not (e,_)) =
	let
	    val newid = do_expr e
	in
	    every_tick_then ( "(not " ^ newid ^ ")",
			     next_dummy_id ())
	end
      | do_bool (Implies ((e1, e2),_)) =
	let
	    val (id1, id2) = (do_expr e1, do_expr e2)
	in
	    every_tick_then ( "((not " ^ id1 ^ " ) or " ^ id2 ^ ")",
			     next_dummy_id())
	end

    (* Do one expression *)

    and	do_expr (Boolean (b, is_ground)) =
	if is_ground then	(* Ground boolean formula *)
	    every_tick_then (do_ground b, next_dummy_id())
	else	do_bool b
      | do_expr (Past p) = do_past p
      | do_expr (Id id) = id

    (* every_tick (id, next_dummy_id())*)
			

    and	do_past_formula (id, exprl) = 
	let
	    val id2 = do_expr exprl
	in
	    every_tick_then(id2, id)
	end

    and	past_formula (PastF (SOME id, expr)) =
	do_past_formula (id, expr)
      | past_formula (PastF (NONE, expr)) =
	do_past_formula (next_dummy_id (), expr)

    and do_previous marker (i, pid) =
	let
	    fun internal_previous ident =
		let
		    val newid = next_dummy_id ()
		    val _ = in_parallel ();
		    val _ = oprint ("\t[\n\t loop\n" ^
				    "\t\tpresent " ^ ident ^ " then\n" ^
				    "\t\t\tawait " ^ marker ^ ";\n" ^
				    "\t\t\temit " ^ newid ^ "\n" ^
				    "\t\telse await " ^ marker ^ "\n" ^
				    "\t\tend present\n" ^
				    "\t end\n\t]\n")
		in
		    newid
		end
	    fun f (x, p) =
		if x <= 0 then []
		else
		    let
			val newid = internal_previous p
		    in
			newid :: f (x-1, newid)
		    end
	in
	    f (i, pid)
	end

    fun awaits signal x =
	let val s = ("\t\t\t\tawait "^ signal ^";\n")
	    fun f x =
		if (x<=0) then ""
		else if (x = 1) then s
		     else (s ^ (f (x - 1)))
	in
	    f x
	end

    and	do_response_formula x =
	if (!optimized_r) then
	    do_response_formula_optimized x
	else
	    do_response_formula_canonical x

    and	do_holds_formula x =
	if (!optimized_h) then
	    do_holds_formula_optimized x
	else
	    do_holds_formula_canonical x

    and	do_response_formula_optimized_OLD (id, e1, e2, n, signal) =
	let
	    val _ =
		if (n < 0) then
		    raise Tl2Strl ("the number of counts for signal " ^ signal ^
				   " has to be positive in leads response formula " ^
				   id ^ ".\n")
		else ()
	    val Q = do_expr e1
	    val P = do_expr e2

	    val _ = oprint(
			   "\t||\n\t[\n" ^
			   "\t loop\n" ^
			   "\t\ttrap TERMINATE in\n" ^
			   "\t\t\tsustain " ^ id ^ " ||\n" ^
			   "\t\t\t[ loop\n" ^
			   "\t\t\t\ttrap RESTART in\n" ^
			   "\t\t\t\t\tawait immediate " ^ P ^ ";\n" ^
			   "\t\t\t\t\tpresent " ^ Q ^ " then\n" ^
			   "\t\t\t\t\t\tawait tick;\n" ^
			   "\t\t\t\t\t\texit RESTART\n" ^
			   "\t\t\t\t\tend;\n" ^
			   "\t\t\t\t\t[\n" ^
			   "\t\t\t\t\t\t[await " ^ Q ^ "; exit RESTART]\n" ^
			   "\t\t\t\t\t\t||\n" ^
			   "\t\t\t\t\t\t[\n" ^ (awaits signal (n + 1)) ^
			   "\t\t\t\t\t\texit TERMINATE]]\n" ^
			   "\t\t\t\tend trap\n" ^
			   "\t\t\t end loop]\n" ^
			   "\t\tend trap;\n" ^
			   "\t\tawait tick;\n" ^
			   "\t\tawait tick\n" ^
			   "\t end loop\n" ^
			   "\t]\n")
	in
	    id
	end

    and	do_response_formula_optimized (id, e1, e2, n, signal, subject) =
	let
	    val _ =
		if (n < 0) then
		    raise Tl2Strl ("the number of counts for signal " ^ signal ^
				   " has to be positive in response formula " ^
				   id ^ ".\n")
		else ()
	    val Q = do_expr e1
	    val P = do_expr e2
	    val fairness_expr =
		case subject of
		    SOME e	=> "\t\tawait immediate " ^ (do_expr e) ^ ";\n"
		  | NONE	=> ""
	    val _ = oprint(
			   "||\nsignal TERM in\n" ^
			   "\tloop\n" ^
			   "\t\tdo\n" ^
			   "\t\t\tsustain " ^ id ^ ";\n" ^
			   "\t\twatching immediate " ^ P ^ ";\n" ^
			   fairness_expr ^
			   "\t\ttrap QUIT in\n" ^
			   "\t\t\t[\n" ^
			   "\t\t\t\tdo\n" ^
			   "\t\t\t\t\tdo\n" ^
			   "\t\t\t\t\t\tsustain " ^ id ^ ";\n" ^
			   "\t\t\t\t\twatching immediate " ^ Q ^ "\n" ^
			   "\t\t\t\t\ttimeout\n" ^
			   "\t\t\t\t\t\temit " ^ id ^ ";\n" ^
			   "\t\t\t\t\tend\n" ^
			   "\t\t\t\twatching immediate TERM\n" ^
			   "\t\t\t\ttimeout\n" ^
			   "\t\t\t\t\tpresent " ^ Q ^ " then\n" ^
			   "\t\t\t\t\t\temit " ^ id ^ ";\n" ^
			   "\t\t\t\t\tend\n" ^
			   "\t\t\t\tend;\n" ^
			   "\t\t\t\texit QUIT;\n" ^
			   "\t\t\t]\n" ^
			   "\t\t||\n" ^
			   "\t\t\t[\n" ^
			   (awaits signal n) ^
			   "\t\t\t\temit TERM;\n" ^
			   "\t\t\t]\n" ^
			   "\t\tend trap;\n" ^
			   "\t\tawait tick\n" ^
			   "\tend loop\n" ^
			   "end signal\n")
	in
	    id
	end

    and	do_holds_formula_optimized (id, e1, e2, n, signal) =
    	let
	    val _ =
		if (n < 1) then
		    raise Tl2Strl ("the number of counts for signal " ^ signal ^
				   " has to be greater than zero in holding response formula " ^
				   id ^ ".\n")
		else ()
	    val P = do_expr e1
	    val Q = do_expr e2
	    val _ = oprint(
			   "||\n" ^
			   "signal TERM in\n" ^
			   "\tloop\n" ^
			   "\t\tdo\n" ^
			   "\t\t\tsustain " ^ id ^ "\n" ^
			   "\t\twatching immediate " ^ P ^ ";\n" ^
			   "\t\ttrap QUIT in\n" ^
			   "\t\t[\n" ^
			   "\t\t\tdo\n" ^
			   "\t\t\t\tdo\n" ^
			   "\t\t\t\t\tsustain " ^ id ^ "\n" ^
			   "\t\t\t\twatching immediate [not " ^ P ^ " ]\n" ^
			   "\t\t\t\ttimeout\n" ^
			   "\t\t\t\t\temit " ^ id ^ "\n" ^
			   "\t\t\t\tend\n" ^
			   "\t\t\twatching immediate TERM\n" ^
			   "\t\t\ttimeout\n" ^
			   "\t\t\t\tdo\n" ^
			   "\t\t\t\t\tevery immediate " ^ signal ^ " do\n" ^
			   "\t\t\t\t\t\tpresent " ^ Q ^ " then emit " ^ id ^ " end\n" ^
			   "\t\t\t\t\tend\n" ^
			   "\t\t\t\twatching immediate [not " ^ P ^ "];\n" ^
			   "\t\t\t\temit " ^ id ^ ";\n" ^
			   "\t\t\tend;\n" ^
			   "\t\t\texit QUIT\n" ^
			   "\t\t]\n" ^
			   "\t||\n" ^
			   "\t\t[\n" ^
			   (awaits signal (n - 1)) ^
			   "\t\t\t\temit TERM;\n" ^
			   "\t\t]\n" ^
			   "\t\tend trap;\n" ^
			   "\t\tawait tick;\n" ^
			   "\tend loop\n" ^
			   "end signal\n"
			   )
	in
	    id
	end

    and	do_response_formula_canonical (id, e1, e2, n, signal, _) =
    	let
	    val _ =
		if (n < 0) then
		    raise Tl2Strl ("the number of counts for signal " ^ signal ^
				   " has to be positive in leads response formula " ^
				   id ^ ".\n")
		else ()
	    val Q = do_expr e1
	    val P = do_expr e2
	    val Pid = hd (do_previous signal (n, P)) handle _ => P
	    val OrQ = if n = 0 then Q else all_or (do_previous signal (n, Q))
	    val _ = oprint(
			   "||\n" ^
			   "\tevery immediate " ^ signal ^ " do\n" ^
			   "\t\tpresent [(not (" ^ Pid ^")) or (" ^ OrQ ^ ")] then\n" ^
			   "\t\t\temit " ^ id ^ "\n" ^
			   "\t\tend present\n" ^
			   "\tend every\n"
			   )
	in
	    id (* for now *)
	end


    and do_holds_formula_canonical (id, e1, e2, n, signal) =
    	let
	    val _ =
		if (n < 1) then
		    raise Tl2Strl ("the number of counts for signal " ^ signal ^
				   " has to be greater than zero in holding response formula " ^
				   id ^ ".\n")
		else ()
	    val P = do_expr e1
	    val Q = do_expr e2
			
	    val AndP = all_and ( P :: (do_previous signal (n - 1, P)))
		
	    val _ = oprint(
			   "||\n" ^
			   "\tevery immediate " ^ signal ^ " do\n" ^
			   "\t\tpresent [(not (" ^ AndP ^")) or " ^ Q ^ "] then\n" ^
			   "\t\t\temit " ^ id ^ "\n" ^
			   "\t\tend present\n" ^
			   "\tend every\n"
			   )
	in
	    id (* for now *)
	end

    and	response_formula (Response (BResponse, SOME id, e1, e2, n, signal, subject)) =
	do_response_formula (id, e1, e2, n, signal, subject)
      | response_formula (Response (BResponse, NONE, e1, e2, n, signal, subject)) =
	do_response_formula (next_dummy_id (), e1, e2, n, signal, subject)
      | response_formula (Response (BHolds, SOME id, e1, e2, n, signal, _)) =
	do_holds_formula (id, e1, e2, n, signal)
      | response_formula (Response (BHolds, NONE, e1, e2, n, signal, _)) =
	do_holds_formula (next_dummy_id (), e1, e2, n, signal)

    and do_endl (f::[]) = do_end f
      | do_endl (f::r) = (do_end f ; do_endl r)
      | do_endl _ = ""

    and do_end (f as PastF (_, (Boolean (x, true)))) =
	every_tick_else(do_ground x, VName f)
      | do_end (f as PastF (_, (Id id))) =
	every_tick_else( id, VName f)
      | do_end (_) =
	raise Tl2Strl ("the `Always' clauses can only enclose a boolean formula.\n")

    and bool_ids (And ((e1, e2),_)) = (expr_ids e1) @ (expr_ids e2)
      | bool_ids (Or ((e1, e2),_)) = (expr_ids e1) @ (expr_ids e2)
      | bool_ids (Implies ((e1, e2),_)) = (expr_ids e1) @ (expr_ids e2)
      | bool_ids (Not (e,_)) = (expr_ids e)

    and past_ids (Since (e1, e2)) = (expr_ids e1) @ (expr_ids e2)
      | past_ids (BackTo (e1, e2)) = (expr_ids e1) @ (expr_ids e2)
      | past_ids (Previous e) = expr_ids e
      | past_ids (Once e) = expr_ids e
      | past_ids (AllPast e) = expr_ids e

    and expr_ids (Id x) = [x]
      | expr_ids (Boolean (b, _)) = bool_ids b
      | expr_ids (Past p) = past_ids p

    and past_clause_ids (PastF (id, _)) = 
	(case id of NONE => "??" | SOME s => s)

    and	response_ids (Response (_, id, _, _, _, _, _)) =
	(case id of NONE => "??" | SOME s => s)

    and nexprl_dummies (Boolean (b, true)) = 1
      | nexprl_dummies (Boolean (b, false)) =
	( 1 + (case  b of
		   Not (x,_) =>		(nexprl_dummies x)
		 | And ((x, y),_) =>	((nexprl_dummies x) +
					 (nexprl_dummies y))
		       
		 | Or ((x, y),_) =>	((nexprl_dummies x) +
					 (nexprl_dummies y))
		 | Implies ((x, y),_) =>	((nexprl_dummies x) +
						 (nexprl_dummies y))
		       ))
      | nexprl_dummies (Past p) =
	( 1 + (case p of
		   Since (x, y) =>	((nexprl_dummies x) +
					 (nexprl_dummies y))
		 | BackTo (x, y) =>	((2 * (nexprl_dummies x)) +
					 (nexprl_dummies y) + 2)
		 | Previous x =>	(nexprl_dummies x)
		 | Once x =>	(nexprl_dummies x)
		 | AllPast x =>	(nexprl_dummies x))
	 )
      | nexprl_dummies (Id id) = 0

    and npast_dummies (PastF (id, expr)) =
	((nexprl_dummies expr) + (case id of NONE => 1 | SOME _ => 0))

    and	nresponse_dummies (Response (BHolds, id, e1, e2, n, _, _)) =
	let
	    val res = (nexprl_dummies e1) + (nexprl_dummies e2)
	in
	    (n + res + (case id of NONE => 1 | SOME _ => 0))
	end
      | nresponse_dummies (Response (BResponse, id, e1, e2, n, _, fair)) =
	let
	    val fair_dummies =
		case fair of
		    SOME e	=> (nexprl_dummies e)
		  | _		=> 0
	    val res = (nexprl_dummies e1) + (nexprl_dummies e2) + fair_dummies
	in
	    (n + n + res + (case id of NONE => 1 | SOME _ => 0))
	end

(* name of the module, past formula list, response formula list, always formula *)

    and tl2strl (ofile, o_r, o_h, Assertions (tl, il, rll, pfl, rfl, afl)) =
	let
	    val _ = optimized_r := o_r
	    val _ = optimized_h := o_h
	    val _ = outfile := ofile
	    val _ = count := 0
		
	    val modulename = PFName (hd afl)
		
	    val _ = VIOLATED_SIGNALS := map VName afl
		
	    (* Find the number of dummies needed in past formulas *)
	    fun npast_dummies_l pfl =  foldr plus 0 (map npast_dummies pfl)
		
	    (* Find the number of dummies needed in response formulas *)
	    fun nresponse_dummies_l pfl = foldr plus 0 (map nresponse_dummies pfl)
		
	    (* Find the number of auxiliary signals *)
	    val nsignals =	(npast_dummies_l pfl) +
		(nresponse_dummies_l rfl)
		
	    (* Create the interface for the program *)
	    val _ = do_interface (modulename,
				  tl,
				  il,
				  rll,
				  (map past_clause_ids pfl) @
				  (map response_ids rfl),
				  nsignals)
		
	    (* Write the program itself *)
		
	    (* First the past formulas *)
	    val past_idlist = map past_formula pfl

	    (* Then the response formulas *)
	    val response_idlist = map response_formula rfl

	    (* And terminate the program *)
	    val _ = do_endl (afl)

	    val _ = oprint "end signal\nend module\n"
	in
	    () (* System.Print.flush () *)
	end

end
