/* File: defs.h
 *
 * Defines size parameters and global data structures for the SPEC system.
 *
 * Copyright (C) 1994 Risto Miikkulainen
 *
 *  This software can be copied, modified and distributed freely for
 *  educational and research purposes, provided that this notice is included
 *  in the code, and the author is acknowledged in any materials and reports
 *  that result from its use. It may not be used for commercial purposes
 *  without expressed permission from the author.
 *
 * $Id: defs.h,v 1.41 1994/09/20 10:48:27 risto Exp $
 */

/*********** size constants *************/

/* These constants define the maximum sizes of tables that hold data
   and the network weights and representations */
#define MAXREP 25		/* max size of word representations */
/* max(nhidrep[PARSMOD] + max(nwordrep, ncontrol, nhidrep[STACKMOD]),
       noutrep[PARSMOD]) */
#define MAXLAYER 250		/* max size of any layer */
#define MAXSEQ 20		/* max number of words in fragment */
#define MAXCASE 10		/* max number of assemblies anywhere */
#define MAXFRAGS 10		/* max number of clause fragments in sentence*/
#define MAXSENTS 3000		/* max number of sentences in epoch */
#define MAXWORDS 100		/* max number of words */
#define MAXNOUNS MAXWORDS	/* max number of nouns */
#define MAXPHASE 10		/* max number of phases in simulation */
#define MAXSNAPS 50		/* max number of snapshots */
#define MAXWORDL 50		/* max length of input words (chars) */
#define MAXFILENAMEL 100	/* max length of file names (chars) */
#define MAXSTRL 1000		/* max length of input lines (chars) */

/* system constants (not maxsize but actual size) */
#define NMODULES 3		/* parser, segmenter, stack */
#define NCONTROL 3		/* number of control outputs */


/*********** module, parameter, return, and range constants *************/

/* module numbers are used to index data structures where module-specific
   graphics values are kept */
#define PARSMOD 0		/* module # for the parser */
#define SEGMOD 1		/* module # for the segmenter */
#define STACKMOD 2		/* module # for the stack */

/* initialization, index, return code constants */
#define LARGEFLOAT 999999999.9	/* used to initialize bestvalues */
#define BLANKINDEX (-1)		/* index for the blank word */
#define TRUE 1
#define FALSE 0
#define NONE (-2)		/* indicates no value; note -1 = BLANKINDEX */
#define NOT_USED 0		/* this parameter is not in use */
#define REQUIRED 1		/* whether to require a keyword */
#define NOT_REQUIRED 0		/* or not */

/* ranges for initial representation and weight values */
#define REPLOW 0.0
#define REPHIGH 1.0
#define REPSPAN (REPHIGH - REPLOW)
#define WEIGHTLOW (-1.0)
#define WEIGHTHIGH 1.0
#define WEIGHTSPAN (WEIGHTHIGH - WEIGHTLOW)

/* exit codes; 1 is used by the system */
#define EXIT_NORMAL 0
#define EXIT_SIZE_ERROR 2
#define EXIT_DATA_ERROR 3
#define EXIT_COMMAND_ERROR 4
#define EXIT_FILE_ERROR 5
#define EXIT_X_ERROR 6


/************ lexicon structure *************/

/* the lexicon data structure: an array of word strings and reps */
typedef struct WORDSTRUCT
  {
    char chars[MAXWORDL + 1];
    double rep[MAXREP];
    int got_rep;		/* records that a rep was obtained */
  }
WORDSTRUCT;


/************ input data structures *************/

/* definition of one sentence fragment */
typedef struct FRAGSTRUCT
  {
    int inps[MAXSEQ],		/* input sequence indices */
      nseq,			/* number of words in this fragment */
      tgts[MAXCASE],		/* target indices */
      pushc,			/* push after the fragment? */
      popc,			/* pop after the fragment? */
      outc;			/* output after the fragment? */
  }
FRAGSTRUCT;

/* one sentence */
typedef struct SENTSTRUCT
  {
    FRAGSTRUCT frag[MAXFRAGS];	/* fragments that make up the sentence */
    int nfrag;			/* number of fragments in this sentence */
  }
SENTSTRUCT;


/************ graphics definitions *************/

/* geometry of a network display */
typedef struct NETSTRUCT
  {
    int
      width, height, 		/* window width and height */
      uwidth, uhght,		/* unit width and height */
      wwidth, whght,		/* weight width and height */
      marg,			/* left margin */
      inpx,			/* left side of input layer */
      inpwx,			/* left side of input weight layer */
      prevx,			/* left side of prev hid layer */
      hidx,			/* left side of hidden layer */
      outx,			/* left size of output layer */
      outwx,			/* left side of output weight layer */
      tgtx,			/* left side of target layer */
      inpy,			/* top of input layer */
      inpwy,			/* top of input weight layer */
      prevy,			/* top of previous hidden layer */
      hidy,			/* top of hidden layer */
      outy,			/* top of output layer */
      outwy,			/* top of output weight layer */
      tgty,			/* top of target layer */
      hsp,			/* separation of assemblies */
      columns;			/* max number of assemblies */
    char sequence[MAXSEQ * (2 * MAXWORDL + 5) + 1], /* I/O word sequence */
      newitem[2 * MAXWORDL + 5 + 1]; /* new item in that sequence */
    char log[MAXSTRL + 1];	/* current I/O item and error */
  }
NETSTRUCT;

/* the resource data for the X display */
typedef struct RESOURCE_DATA
  {
    Boolean bringupdisplay,	/* whether to start display */
      weightdisplay;		/* display weights as well */
    int delay;			/* seconds to sleep in major simulation steps*/
    Dimension netwidth,		/* width of network widgets */
      netheight,		/* height if weights are not displayed */
      weightnetheight;		/* ..with weights displayed */
    int weightsPerUnit;		/* how many weightboxes per each unitbox */
    Boolean owncmap;		/* use private colormap */
    Pixel textColor,		/* color of the text on display */
      netColor;			/* color of network outlines */
    float weightcolorrange;	/* green-red within this range */
    String commandfont,		/* command line */
      titlefont,		/* network title */
      logfont,			/* I/O item and error */
      asmfont,			/* words in assemblies */
      asmerrorfont;		/* erroneous words in assemblies */
    Boolean help,		/* user help command line option */
      testing;			/* testing/training mode command line option */
    String chaining,		/* whether to chain the nets option */
      stacknouns,		/* whether to train stack with nouns option */
      includeall;		/* whether to run all modules option */
  }
RESOURCE_DATA, *RESOURCE_DATA_PTR;


/************ function prototype macro *************/

#if __STDC__
#define __P(a) a
#else
#ifndef __P			/* Sun ANSI compiler defines this */
#define __P(a) ()
#endif
#endif
