/* File: graph.c
 *
 * X interface for SPEC
 * 
 * Copyright (C) 1994 Risto Miikkulainen
 *
 *  This software can be copied, modified and distributed freely for
 *  educational and research purposes, provided that this notice is included
 *  in the code, and the author is acknowledged in any materials and reports
 *  that result from its use. It may not be used for commercial purposes
 *  without expressed permission from the author.
 *
 * $Id: graph.c,v 1.37 1994/09/20 10:48:27 risto Exp risto $
 */

#include <stdio.h>
#include <math.h>
#include <setjmp.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/AsciiText.h>

#include "defs.h"
#include "Gwin.h"
#include "globals.c"


/************ graphics constants *************/

/* color parameters */
#define UNITCOLORS 0		/* chooses bryw colorscale */
#define WEIGHTCOLORS 1		/* chooses gbryw colorscale */
#define MINCOLORS 10		/* if fewer free colors, use existing */
#define MAXCOLORS 256		/* max # of colors */
#define MAXCOLGRAN 65535	/* granularity of color values */

/* colormap entry */
typedef struct RGB
  {
    short red, green, blue;	/* hold possible rgb values in XColor struct */
  }
RGB;

/* graphics separator constants */
#define HIDSP 1			/* separation of hidden layers,pixels */
#define VERSP 3			/* generic vertical separation */
#define HORSP 3			/* generic horizontal separation */
#define BOXSP 1			/* vertical space around boxed text */
#define HFMSP 3			/* separation of hfm boxes */
#define PREVSP (3*HORSP)	/* indentation of prev hidden layers */
#define ABOVE  (-asmboxhght)	/* distance of labels from units */
#define ABOVE2 (-net[modi].uhght - asmboxhght)	/* same when beyond a layer */
#define BELOW  net[modi].uhght	/* distance of labels from units */
#define BELOW2 (2 * net[modi].uhght)	/* same when beyond a layer */

#define NWINS NMODULES		/* number of windows */


/******************** Function prototypes ************************/

/* global functions */
#include "prototypes.h"
extern unsigned sleep __P ((unsigned seconds));

/* functions local to this file */
static void handle_events __P ((void));
static void runstop_callback __P ((Widget w, XtPointer client_data,
				   XtPointer call_data));
static void start_running __P ((void));
static void toggle_callback __P ((Widget w, XtPointer client_data,
				  XtPointer call_data));
static void clear_callback __P ((Widget w, XtPointer client_data,
				 XtPointer call_data));
static void clear_networks_display __P ((void));
static void quit_callback __P ((Widget w, XtPointer client_data,
				XtPointer call_data));
static void close_display __P ((void));
static void init_rgbtab_weights __P ((void));
static void init_rgbtab_noweights __P ((void));
static void init_rgbtab_bw __P((void));
static void alloc_col __P ((void));
static void clean_color_map __P((int k));
static void create_colormap __P ((void));
static int createGC __P ((Window New_win, GC * New_GC, Font fid, Pixel FGpix,
			  Pixel theBGpix));
static XFontStruct *loadFont __P ((char fontName[]));
static void common_resize __P ((int modi, Widget w));
static void init_parser_display_params __P ((int modi, Widget w));
static void expose_parser __P ((Widget w, XtPointer client_data,
				XtPointer call_data));
static void resize_parser __P ((Widget w, XtPointer client_data,
				XtPointer call_data));
static void init_segmenter_display_params __P ((int modi, Widget w));
static void expose_segmenter __P ((Widget w, XtPointer client_data,
				   XtPointer call_data));
static void resize_segmenter __P ((Widget w, XtPointer client_data,
				   XtPointer call_data));
static void init_stack_display_params __P ((int modi, Widget w));
static void expose_stack __P ((Widget w, XtPointer client_data,
			       XtPointer call_data));
static void resize_stack __P ((Widget w, XtPointer client_data,
			       XtPointer call_data));
static void display_sequence __P ((int modi, int x, int y));
static void display_labels __P ((int modi, int x, int y,
				 char *labels[], int nitem));
static void display_labeled_layer __P ((int modi, int nas, double rep[],
					int nums[], int x, int y,
					int labeloffset));
static void display_layer __P ((int modi, int nas, double layer[], int x,
				int y, int nrep));
static void display_assembly __P ((int modi, int x, int y, double assembly[],
				   int nrep));
static void display_boxword __P((int modi, int x, int y, int width, int height,
				 char wordchars[], int dodisplay,
				 XFontStruct *fontStruct, GC currGC));
static void display_normalweights __P ((int modi, int x, int y,
					double weights[MAXLAYER][MAXLAYER],
					int nsource, int ndest));
static void display_reversedweights __P ((int modi, int x, int y,
					  double weights[MAXLAYER][MAXLAYER],
					  int nsource, int ndest));
static void display_title __P ((int modi, char name[]));
static void display_log __P ((int modi));
static int trans_to_color __P ((double value, int map));
static void clearRectangle __P ((int modi, int x, int y, int width,
				 int height));
static void fillRectangle __P ((int modi, int x, int y, int width, int height,
				int colorindex));
static void drawRectangle __P ((int modi, int x, int y,
				int width, int height));
static void drawText __P ((int modi, int x, int y, char text[], GC currGC));
static void drawoverText __P ((int modi, int x, int y,
			       char text[], GC currGC));


/******************** static variables ******************** */

/* current simulator state */
static int simulator_running, stepping;

static XColor colors[MAXCOLORS];
static int actual_color_range;	/* number of colors in use */

/* button, command, and graphics widgets */
static Widget
  runstop, step, clear, quit,	/* buttons */
  command,			/* command input widget */
  parser, segmenter, stack;	/* network graphics widgets */

/* useful pointers to windows */
static Window
  theMain, runstopWin, commandWin, Win[NWINS];	/* network graphics windows */

/* text constants to be displayed */
static char
/* labels of the case role assemblies */
 *caselabels[] =
{"AGENT", "ACT", "PATIENT"},
/* names of the network modules */
 *titles[] =
{"PARSER", "SEGMENTER", "STACK"};

/* graphics contexts */
static GC
  titleGC,			/* window name */
  logGC,			/* log line: item and error */
  asmGC,			/* text in assemblies */
  asmerrorGC,			/* error text in assemblies (smaller font) */
  clearGC,			/* for clearing areas */
  boxGC,			/* for drawing network boxes */
  activityGC;			/* for displaying activations */

/* corresponding fonts */
static XFontStruct
  *titlefontStruct, *logfontStruct, *asmfontStruct, *asmerrorfontStruct;

static int asmboxhght, titleboxhght;	/* size of text boxes */

/* Array of rgb values that represents a linear color spectrum */
static RGB rgbtab[MAXCOLORS];
/* the weight/unit value colormap */
static short cmap[MAXCOLORS];


/********************* general initialization ***********************/

void
display_init ()
/* initialize the X display */
{
  Arg args[3];
  char s[MAXSTRL + 1];
  Pixel theBGpix;
  Dimension borderwidth,
    height, width,		/* various widget heights and widths given */
    tot_width;			/* computed total width of display */

  printf ("Initializing graphics...\n");
  fflush (stdout);

  XtSetArg (args[0], XtNborderWidth, &borderwidth);
  XtGetValues (main_widget, args, 1);	/* get the current border width  */
  /* create a form with no space between widgets */
  XtSetArg (args[0], XtNdefaultDistance, 0);
  form = XtCreateManagedWidget ("form", formWidgetClass, main_widget, args, 1);

  /* the command button for running and stopping the simulation */
  runstop = XtCreateManagedWidget ("runstop", commandWidgetClass,
				   form, args, 1);
  /* toggle switch: if on, the simulation will stop after each propagation */
  step = XtCreateManagedWidget ("step", toggleWidgetClass, form, args, 1);
  /* the command button for clearing the networks */
  clear = XtCreateManagedWidget ("clear", commandWidgetClass, form, args, 1);
  /* the command button for exiting the program */
  quit = XtCreateManagedWidget ("quit", commandWidgetClass, form, args, 1);

  /* create command line widget: */
  /* first get the height from runstop */
  XtSetArg (args[0], XtNheight, &height);
  /* then figure out the total width of the display by */
  /* getting the widths of each widget, adding them up, and subtracting
     from the total width of the display */
  XtSetArg (args[1], XtNwidth, &width);
  XtGetValues (runstop, args, 2);
  tot_width = width + 2 * borderwidth;
  XtSetArg (args[0], XtNwidth, &width);
  XtGetValues (step, args, 1);
  tot_width += width + 2 * borderwidth;
  XtSetArg (args[0], XtNwidth, &width);
  XtGetValues (clear, args, 1);
  tot_width += width + 2 * borderwidth;
  XtSetArg (args[0], XtNwidth, &width);
  XtGetValues (quit, args, 1);
  tot_width += width + 2 * borderwidth;
  XtSetArg (args[0], XtNheight, height);
  XtSetArg (args[1], XtNwidth, data.netwidth - tot_width);
  /* display the name of the simufile in the command line widget */
  sprintf (s, "simulation file: %s", simufile);
  XtSetArg (args[2], XtNstring, s);
  command = XtCreateManagedWidget ("command", asciiTextWidgetClass,
				   form, args, 3);

  /* create the network displays */
  /* get the size from resourses */
  XtSetArg (args[0], XtNwidth, data.netwidth);
  /* use different height depending on whether weights are displayed */
  if (!data.weightdisplay)
    XtSetArg (args[1], XtNheight, data.netheight);
  else
    XtSetArg (args[1], XtNheight, data.weightnetheight);
  parser = XtCreateManagedWidget ("parser", gwinWidgetClass,
				  form, args, 2);
  if (!parseronly)
    {
      segmenter = XtCreateManagedWidget ("segmenter", gwinWidgetClass,
					 form, args, 2);
      stack = XtCreateManagedWidget ("stack", gwinWidgetClass,
				     form, args, 2);
    }

  /* callbacks: what to do when a button is pressed */
  XtAddCallback (runstop, XtNcallback, runstop_callback, NULL);
  XtAddCallback (clear, XtNcallback, clear_callback, NULL);
  XtAddCallback (quit, XtNcallback, quit_callback, NULL);
  XtAddCallback (step, XtNcallback, toggle_callback, NULL);

  /* network callbacks: redrawing the state */
  XtAddCallback (parser, XtNexposeCallback, expose_parser, NULL);
  if (!parseronly)
    {
      XtAddCallback (segmenter, XtNexposeCallback, expose_segmenter, NULL);
      XtAddCallback (stack, XtNexposeCallback, expose_stack, NULL);
    }

  /* network callbacks for resizing */
  XtAddCallback (parser, XtNresizeCallback, resize_parser, NULL);
  if (!parseronly)
    {
      XtAddCallback (segmenter, XtNresizeCallback, resize_segmenter, NULL);
      XtAddCallback (stack, XtNresizeCallback, resize_stack, NULL);
    }

  /* figure out the display type and allocate colors */
  create_colormap ();
  /* put the display on screen */
  XtRealizeWidget (main_widget);
  theMain = XtWindow (main_widget);
  runstopWin = XtWindow (runstop);
  commandWin = XtWindow (command);

  /* calculate some network geometry variables */
  init_parser_display_params (PARSMOD, parser);
  if (!parseronly)
    {
      init_segmenter_display_params (SEGMOD, segmenter);
      init_stack_display_params (STACKMOD, stack);
    }

  /* set a common font for all buttons and command line */
  XtSetArg (args[0], XtNfont, loadFont (data.commandfont));
  XtSetValues (runstop, args, 1);
  XtSetValues (clear, args, 1);
  XtSetValues (quit, args, 1);
  XtSetValues (step, args, 1);
  XtSetValues (command, args, 1);

  /* load the other fonts */
  titlefontStruct = loadFont (data.titlefont);
  logfontStruct = loadFont (data.logfont);
  asmfontStruct = loadFont (data.asmfont);
  asmerrorfontStruct = loadFont (data.asmerrorfont);

  /* figure out space needed for the title and labels */
  titleboxhght = titlefontStruct->ascent + titlefontStruct->descent
    + 2 * BOXSP;
  asmboxhght = asmfontStruct->ascent + asmfontStruct->descent + 2 * BOXSP;

  /* get the background color */
  XtSetArg (args[0], XtNbackground, &theBGpix);
  XtGetValues (main_widget, args, 1);

  /* create graphics context for all fonts */
  createGC (theMain, &titleGC, titlefontStruct->fid, data.textColor, theBGpix);
  createGC (theMain, &logGC, logfontStruct->fid, data.textColor, theBGpix);
  createGC (theMain, &asmGC, asmfontStruct->fid, data.textColor, theBGpix);
  createGC (theMain, &asmerrorGC, asmerrorfontStruct->fid, data.textColor,
	    theBGpix);

  /* clearing areas */
  createGC (theMain, &clearGC, logfontStruct->fid, theBGpix, theBGpix);
  /* network boxes */
  createGC (theMain, &boxGC, logfontStruct->fid, data.netColor, theBGpix);
  /* generic context for displaying unit activity */
  createGC (theMain, &activityGC, logfontStruct->fid, theBGpix, theBGpix);

  /* calculate all network geometries and put them on screen */
  resize_parser (parser, NULL, NULL);
  if (!parseronly)
    {
      resize_segmenter (segmenter, NULL, NULL);
      resize_stack (stack, NULL, NULL);
    }

  printf ("Graphics initialization complete.\n");
}


/*********************  event handler  ***********************/

static void
handle_events ()
  /* event handling loop */
  /* we need this instead of XtMainLoop because the main task is to run
     the simulation, and only occasionally check for events */
{
  XEvent theEvent;

  while (XtAppPending (app_con))
    /* as long as there are unprocessed events */
    {
      XtAppNextEvent (app_con, &theEvent);
      if (!(theEvent.type == Expose && theEvent.xexpose.count > 0))
	/* only process the last expose event */
	{
	  XtDispatchEvent (&theEvent);
	  XFlush (theDisplay);
	}
    }
}


void
wait_and_handle_events ()
/* this is called after each main step of the simulation
   to stop and wait for "run" if stepping, and to deal with
   any other possible events before displaying simulation results */
{
  if (stepping)
    wait_for_run ();		/* wait until user is ready */
  else
    /* slow down the display this many seconds */
    sleep ((unsigned) abs (data.delay));
  handle_events ();		/* process all pending events before proceeding
				   to display the simulation results */
}


/*********************  button callback routines  ***********************/

static void
runstop_callback (w, client_data, call_data)
  /* "Run/Stop" button press handler:
     if simulation is running, stop it; otherwise, continue simulation.
     from wherever handle_events was called */
  /* standard callback parameters, not used here */
     Widget w;
     XtPointer client_data, call_data;
{
  if (simulator_running)
    wait_for_run ();
  else
    start_running ();
}


void
wait_for_run ()
  /* stop the simulation and start processing events */
{
  Arg args[1];

  simulator_running = FALSE;		/* process events */
  XtSetArg (args[0], XtNlabel, "Run");	/* change label on "Run/Stop" button */
  XtSetValues (runstop, args, 1);
  XFlush (theDisplay);
  while (!simulator_running)		/* process events until Run pressed */
    {
      /* Process one event if any or block */
      XtAppProcessEvent (app_con, XtIMAll);
      /* Process more events, but don't block */
      handle_events ();
    }
}


static void
start_running ()
  /* set up to continue or start simulation */
{
  Arg args[1];

  simulator_running = TRUE;		/* allow exit from wait_for_run */
  XtSetArg (args[0], XtNlabel, "Stop");	/* change label on "Run/Stop" button */
  XtSetValues (runstop, args, 1);
}


static void
toggle_callback (w, client_data, call_data)
  /* "Step" button press handler: turn stepping on or off */
  /* standard callback parameters, not used here */
     Widget w;
     XtPointer client_data, call_data;
{
  stepping = !stepping;
}


static void
clear_callback (w, client_data, call_data)
  /* "Clear" button press handler: clear the display and network activations,
     start a fresh command processing loop */
  /* standard callback parameters, not used here */
     Widget w;
     XtPointer client_data, call_data;
{
  extern jmp_buf loop_sents_env;

  clear_networks_display ();
  longjmp (loop_sents_env, 0);
}


static void
clear_networks_display ()
  /* clear the network activations, logs, stack, and the display */
{
  int modi;					/* module number */

  /* clear each module and its display */
  for (modi = 0; modi < NMODULES; modi++)
    if (!parseronly || modi == PARSMOD)
      {      
	clear_network (modi);
	XClearArea (theDisplay, Win[modi], 0, 0, 0, 0, True);
      }

  /* clear the stack pointer */
  top = 0;

  /* update the output immediately before data changes */
  XFlush (theDisplay);
  handle_events ();
}


static void
quit_callback (w, client_data, call_data)
  /* "Quit" button press handler: exit the program */
  /* standard callback parameters, not used here */
     Widget w;
     XtPointer client_data, call_data;
{
  close_display ();
  exit (EXIT_NORMAL);
}


static void
close_display ()
  /* free fonts and close the display */
{
  XFreeFont (theDisplay, titlefontStruct);
  XFreeFont (theDisplay, logfontStruct);
  XFreeFont (theDisplay, asmfontStruct);
  XFreeFont (theDisplay, asmerrorfontStruct);
  XCloseDisplay (theDisplay);
}


/*********************  colormap allocation  ***********************/

static void
create_colormap ()
  /* allocate colors for activation values: depending on the display type
     and available colors, allocate a continuous spectrum the best you can */
{
  if (!(visual->class == GrayScale || visual->class == StaticGray))
    /* we have color display */
    if (data.weightdisplay)
      init_rgbtab_weights ();
    else
      init_rgbtab_noweights ();
  else
    /* black and white screen; same scale with or without weights */
    init_rgbtab_bw ();
  alloc_col ();
}


static void
init_rgbtab_noweights ()
/* calculates values for the linear color spectrum black-red-yellow-white
   and stores them in rbgtab */
{
  double multiplier;
  int rangeby3, i;

  /* divide the range into three sections:
     black-red, red-yellow, yellow-white */
  rangeby3 = MAXCOLORS / 3;
  multiplier = ((double) MAXCOLGRAN) / rangeby3;
  for (i = 0; i < MAXCOLORS; i++)
    {
      rgbtab[i].green = 0;
      rgbtab[i].blue = 0;
      if (i < rangeby3)
	rgbtab[i].red = (int) (i * multiplier);
      else
	/* second section: red to yellow */
	{
	  rgbtab[i].red = MAXCOLGRAN;
	  if (i < 2 * rangeby3)
	    rgbtab[i].green = (int) ((i - rangeby3) * multiplier);
	  else
	    /* third section: yellow to white */
	    {
	      rgbtab[i].green = MAXCOLGRAN;
	      rgbtab[i].blue = (int) ((i - 2 * rangeby3) * multiplier);
	    }
	}
    }
}


static void
init_rgbtab_weights ()
/* calculates values for the linear color spectrum green-black-red-yellow-white
   and stores them in rbgtab */
{
  double multiplier;
  int rangeby4, i;

  /* divide the range into four sections:
     green-black, black-red, red-yellow, yellow-white */
  rangeby4 = MAXCOLORS / 4;
  multiplier = ((double) MAXCOLGRAN) / rangeby4;
  for (i = 0; i < MAXCOLORS; i++)
    {
      rgbtab[i].red = 0;
      rgbtab[i].blue = 0;
      if (i < rangeby4)
	rgbtab[i].green = MAXCOLGRAN - (int) (i * multiplier);
      else
	/* second section: black to red */
	{
	  if (i < 2 * rangeby4)
	    {
	      rgbtab[i].red = (int) ((i - rangeby4) * multiplier);
	      rgbtab[i].green = 0;
	    }
	  else
	    /* third section: red to yellow */
	    {
	      rgbtab[i].red = MAXCOLGRAN;
	      if (i < 3 * rangeby4)
		rgbtab[i].green = (int) ((i - 2 * rangeby4) * multiplier);
	      else
		{
		  rgbtab[i].green = MAXCOLGRAN;
		  rgbtab[i].blue = (int) ((i - 3 * rangeby4) * multiplier);
		}
	    }
	}
    }
}


static void
init_rgbtab_bw ()
/* calculates values for the linear gray scale and stores them in rbgtab */
{
  double multiplier;
  int i;

  /* straight scale from black to white */
  multiplier = ((double) MAXCOLGRAN) / MAXCOLORS;
  for (i = 0; i < MAXCOLORS; i++)
    rgbtab[i].green =  rgbtab[i].blue = rgbtab[i].red
      = (int) (i * multiplier);
}


static void
alloc_col ()
/* allocate colors from rgbtab until no more free cells,
   using existing colors if they match what we want */
{
  int start = MAXCOLORS / 2;		/* starting offset in an alloc sweep */
  int d = MAXCOLORS;			/* increment in an alloc sweep */
  int j,				/* index to the linear spectrum */
    k;					/* number of colors allocated */

  for (j = 0; j < MAXCOLORS; j++)
    cmap[j] = NONE;
  k = 0;

  /* add colors to cmap, keep them uniformly distributed in the range,
     and gradually make the spectrum more refined */
  while (d > 1)
    {
      /* add colors to cmap with d as the current distance
	 between new colors in the spectrum and start as the first location */
      j = start;
      while (j < MAXCOLORS)		/* completed a sweep of new colors */
	{
	  colors[k].flags = DoRed | DoGreen | DoBlue;	/* use all planes */
	  colors[k].red = rgbtab[j].red;
	  colors[k].green = rgbtab[j].green;
	  colors[k].blue = rgbtab[j].blue;

	  cmap[j] = k;
	  if (XAllocColor (theDisplay, colormap, &(colors[k])) == 0)
	    {
	      cmap[j] = NONE;
	      clean_color_map(k);
	      return;
	    }
	  k++;				/* allocated one new color */
	  j += d;			/* next location in the spectrum */
	}
      d /= 2;				/* set up a tighter distance */
      start /= 2;			/* start lower in the spectrum */
    }
  clean_color_map(k);	                /* set # of colors, clean up */
}

static void
clean_color_map(k)
/* set the number of colors, print message, and clean up the map */
     int k;				/* number of colors allocated */
{
  int i, m;			/* counters for cleaning up cmap */

  /* colors[k-1] is the last valid colorcell */
  actual_color_range = k;

  if (actual_color_range < MINCOLORS)
    {
      fprintf (stderr, "Warning: obtained only %d colors\n", k);
      fprintf (stderr, "(consider using a private colormap)\n");
    }
  else
    printf ("Obtained %d colors.\n", k);

  /* clean up cmap; move all entries to the beginning */
  m = 0;
  while (m < MAXCOLORS && cmap[m] != NONE)
    ++m;
  for (i = m + 1; i < MAXCOLORS; i++)
    if (cmap[i] == NONE)	/* no colorcell */
      continue;
    else
      cmap[m] = cmap[i], ++m;
}


/*********************  GCs, fonts, resizing  ***********************/

static int
createGC (New_win, New_GC, fid, theFGpix, theBGpix)
  /* create a new graphics context for the given window with given
     font and foreground and background colors */
     Window New_win;		/* window for the GC */
     GC *New_GC;		/* return pointer to the created GC here */
     Font fid;			/* font id  */
     Pixel theFGpix, theBGpix;	/* foreground and background colors */
{
  XGCValues GCValues;		/* graphics context parameters; not used */

  *New_GC = XCreateGC (theDisplay, New_win, (unsigned long) 0, &GCValues);

  if (*New_GC == 0)		/* could not create */
    return (FALSE);
  else
    {
      /* set font, foreground and background */
      XSetFont (theDisplay, *New_GC, fid);
      XSetForeground (theDisplay, *New_GC, theFGpix);
      XSetBackground (theDisplay, *New_GC, theBGpix);
      return (TRUE);
    }
}


static XFontStruct *
loadFont (fontName)
  /* load a given font */
     char fontName[];		/* name of the font */
{
  XFontStruct *fontStruct;	/* return font here */

  if ((fontStruct = XLoadQueryFont (theDisplay, fontName)) == NULL)
    {
      fprintf (stderr, "Cannot load font: %s, using fixed\n", fontName);
      if ((fontStruct = XLoadQueryFont (theDisplay, "fixed")) == NULL)
	{
	  fprintf (stderr, "Cannot load fixed font\n");
	  exit (EXIT_X_ERROR);
	}
    }
  return (fontStruct);
}


static void
common_resize (modi, w)
  /* when resizing any net, first get the new window width and height */
     int modi;			/* module number */
     Widget w;			/* and its widget */
{
  Arg args[2];
  Dimension width, height;

  /* get the current width and height from the server */
  XtSetArg (args[0], XtNwidth, &width);
  XtSetArg (args[1], XtNheight, &height);
  XtGetValues (w, args, 2);
  /* and store them for further calculations */
  net[modi].width = width;
  net[modi].height = height;
}


/********************* parser module operations ***********************/

static void
init_parser_display_params (modi, w)
  /* initialize the parameters for the parser display */
     int modi;			/* module number */
     Widget w;			/* and its widget */
{
  int i;

  Win[modi] = XtWindow (w);	/* get a pointer to the window */
  /* calculate the number of slots at the input and output */
  for (i = 0; i < ninputs[modi]; i++)
    inputs[modi][i] = BLANKINDEX;	/* initially all blank words */
  for (i = 0; i < noutputs[modi]; i++)
    targets[modi][i] = BLANKINDEX;/* initially all blank words */

  /* number of assembly columns on the display */
  net[modi].columns = imax (ninputs[modi], noutputs[modi]);
}


static void
expose_parser (w, client_data, call_data)
  /* expose event handler for the parser: redraw the window */
  /* standard callback parameters, not used here */
     Widget w;
     XtPointer client_data, call_data;
{
  int modi = PARSMOD;

  XClearWindow (theDisplay, Win[modi]);
  display_title (modi, titles[modi]);
  display_labels (modi, net[modi].tgtx,
		  net[modi].tgty + net[modi].uhght + asmboxhght,
		  caselabels, ncase);
  display_current_parser (modi, TRUE);
}


void
display_current_parser (modi, weights_included)
  /* display all activations and the current log */
     int modi;			/* module number */
     int weights_included;	/* whether to update weight display */
{
  display_labeled_layer (modi, ninputs[modi], inprep[modi], inputs[modi],
			 net[modi].inpx, net[modi].inpy, ABOVE);
  display_assembly (modi, net[modi].prevx, net[modi].prevy,
		    &inprep[modi][nwordrep], nhidrep[modi]);
  if (data.weightdisplay && weights_included)
    display_normalweights (modi, net[modi].inpwx, net[modi].inpwy,
			   wih[modi], ninprep[modi], nhidrep[modi]);
  display_assembly (modi, net[modi].hidx, net[modi].hidy,
		    hidrep[modi], nhidrep[modi]);
  if (data.weightdisplay && weights_included)
    display_reversedweights (modi, net[modi].outwx, net[modi].outwy,
			     who[modi], nhidrep[modi], noutrep[modi]);
  display_labeled_layer (modi, noutputs[modi], outrep[modi], targets[modi],
			 net[modi].outx, net[modi].outy, BELOW2);
  display_layer (modi, noutputs[modi], tgtrep[modi],
		 net[modi].tgtx, net[modi].tgty, nwordrep);
  display_sequence (modi, net[modi].inpx, net[modi].inpy);
  display_log (modi);
}


static void
resize_parser (w, client_data, call_data)
  /* resize event handler for sentence parser: recalculate geometry */
  /* standard callback parameters, not used here */
     Widget w;
     XtPointer client_data, call_data;
{
  int modi = PARSMOD;

  common_resize (modi, parser);	/* get new window size */

  /* width of the boxes representing unit activities and weights */
  if (!data.weightdisplay)
    net[modi].uwidth =
    /* either the I/O columns fill the window or the previous hidden layer */
      imin ((net[modi].width - 2 * HORSP) / (net[modi].columns * nwordrep),
	    (net[modi].width - 2 * HORSP - PREVSP) / nhidrep[modi]);
  else
    {
      net[modi].uwidth =
      /* either the I/O columns or the previous hidden layer + input word */
	imin ((net[modi].width - 2 * HORSP) / (net[modi].columns * nwordrep),
	 (net[modi].width - 2 * HORSP - PREVSP) / (nhidrep[modi] + nwordrep));
      net[modi].wwidth = net[modi].uwidth;
    }

  net[modi].hsp = nwordrep * net[modi].uwidth;	/* space between assemblies */
  net[modi].marg = HORSP;	/* horizontal margin */

  /* x-coordinates of the left side of each layer */
  net[modi].hidx =		/* hidden layer */
    (net[modi].width - nhidrep[modi] * net[modi].uwidth) / 2;
  net[modi].outx =		/* output layer */
    (net[modi].width - net[modi].hsp * noutputs[modi]) / 2;
  net[modi].tgtx =		/* target pattern, same x as output */
    net[modi].outx;
  if (!data.weightdisplay)
    {
      net[modi].inpx =		/* input layer */
	(net[modi].width - net[modi].hsp * ninputs[modi]) / 2;
      net[modi].prevx =		/* previous hidden layer is displaced right */
	net[modi].hidx + PREVSP;
    }
  else
    {
      net[modi].inpx =		/* input layer */
	(net[modi].width - net[modi].hsp * ninputs[modi] -
	 nhidrep[modi] * net[modi].uwidth) / 2;
      net[modi].inpwx = net[modi].inpx;
      net[modi].prevx = net[modi].inpx + ninputs[modi] * net[modi].hsp;
      net[modi].outwx = net[modi].outx;
    }

  /* heights of the boxes representing unit activities and weights */
  if (!data.weightdisplay)
    net[modi].uhght = imax (0, (net[modi].height - titleboxhght -
				3 * asmboxhght - HIDSP - 3 * VERSP) / 5);
  else
    {
      net[modi].whght =
	imax (0,
	      (net[modi].height - titleboxhght - 3 * asmboxhght - 3 * VERSP) /
	      (2 * nhidrep[modi] + 4 * data.weightsPerUnit));
      net[modi].uhght = net[modi].whght * data.weightsPerUnit;
    }

  /* y-coordinates of the tops of each layer */
  net[modi].inpy = titleboxhght + asmboxhght;
  if (!data.weightdisplay)
    {
      net[modi].prevy = net[modi].inpy + net[modi].uhght + VERSP;
      net[modi].hidy = net[modi].prevy + net[modi].uhght + HIDSP;
      net[modi].outy = net[modi].hidy + net[modi].uhght + VERSP;
    }
  else
    {
      net[modi].prevy = net[modi].inpy;
      net[modi].inpwy = net[modi].inpy + net[modi].uhght;
      net[modi].hidy =
	net[modi].inpwy + nhidrep[modi] * net[modi].whght + VERSP;
      net[modi].outwy = net[modi].hidy + net[modi].uhght + VERSP;
      net[modi].outy = net[modi].outwy + nhidrep[modi] * net[modi].whght;
    }
  net[modi].tgty = net[modi].outy + net[modi].uhght;

}


/********************* segmenter module operations ***********************/

static void
init_segmenter_display_params (modi, w)
  /* initialize the parameters for the segmenter display */
     int modi;			/* module number */
     Widget w;			/* and its widget */
{
  int i;

  Win[modi] = XtWindow (w);	/* get a pointer to the window */
  /* calculate the number of slots at the input and output */
  for (i = 0; i < ninputs[modi]; i++)
    inputs[modi][i] = BLANKINDEX;	/* initially blank */
}


static void
expose_segmenter (w, client_data, call_data)
  /* expose event handler for the segmenter: redraw the window */
  /* standard callback parameters, not used here */
     Widget w;
     XtPointer client_data, call_data;
{
  int modi = SEGMOD;

  XClearWindow (theDisplay, Win[modi]);
  display_title (modi, titles[modi]);
  display_current_segmenter (modi, TRUE);
}


void
display_current_segmenter (modi, weights_included)
  /* display all activations and the current log */
     int modi;			/* module number */
     int weights_included;	/* whether to update weight display */
{
  display_labeled_layer (modi, ninputs[modi], inprep[modi], inputs[modi],
			 net[modi].inpx, net[modi].inpy, ABOVE);
  display_assembly (modi, net[modi].outx, net[modi].inpy,
		    &inprep[modi][nwordrep], nhidrep[modi]);
  if (data.weightdisplay && weights_included)
    display_normalweights (modi, net[modi].inpx, net[modi].inpwy,
			   wih[modi], ninprep[modi], nhidrep[modi]);
  display_assembly (modi, net[modi].hidx, net[modi].hidy,
		    hidrep[modi], nhidrep[modi]);
  if (data.weightdisplay && weights_included)
    display_reversedweights (modi, net[modi].outwx, net[modi].outwy,
			     who[modi], nhidrep[modi], noutrep[modi]);
  display_assembly (modi, net[modi].outx, net[modi].outy,
		    outrep[modi], noutrep[modi]);
  display_assembly (modi, net[modi].tgtx, net[modi].tgty,
		    tgtrep[modi], noutrep[modi]);
  display_log (modi);
}


static void
resize_segmenter (w, client_data, call_data)
  /* resize event handler for the segmenter: recalculate geometry */
  /* standard callback parameters, not used here */
     Widget w;
     XtPointer client_data, call_data;
{
  int modi = SEGMOD;

  common_resize (modi, segmenter);	/* get new window size */

  /* width of the boxes representing unit activities and weights */
  net[modi].uwidth = (net[modi].width - 2 * HORSP) /
    (nhidrep[PARSMOD] + nwordrep + NCONTROL);
  if (data.weightdisplay)
    net[modi].wwidth = net[modi].uwidth;

  net[modi].hsp = nwordrep * net[modi].uwidth;	/* space between assemblies */
  net[modi].marg = HORSP;	/* horizontal margin */

  /* x-coordinates of the left sides of each layer, centered in the window */
  net[modi].inpx =		/* input layer */
    (net[modi].width - (nhidrep[PARSMOD] + nwordrep + NCONTROL) *
     net[modi].uwidth) / 2;
  net[modi].hidx =		/* hidden layer */
    (net[modi].width - nhidrep[modi] * net[modi].uwidth) / 2;
  net[modi].outx =		/* output layer */
    (net[modi].inpx + nwordrep * net[modi].uwidth);
  net[modi].tgtx =		/* target pattern, same x as output */
    net[modi].outx;
  if (data.weightdisplay)
    {
      net[modi].inpwx = net[modi].inpx;
      net[modi].outwx = net[modi].outx;
    }

  /* heights of the boxes representing unit activities and weights */
  if (!data.weightdisplay)
    net[modi].uhght = net[PARSMOD].uhght;
  else
    {
      net[modi].whght =
	imax (0,
	      (net[modi].height - titleboxhght - asmboxhght - 3 * VERSP) /
	      (2 * nhidrep[modi] + 4 * data.weightsPerUnit));
      net[modi].uhght = net[modi].whght * data.weightsPerUnit;
    }

  /* y-coordinates of the tops of each layer */
  net[modi].inpy = titleboxhght + asmboxhght;
  if (!data.weightdisplay)
    {
      net[modi].hidy = net[modi].inpy + net[modi].uhght + VERSP;
      net[modi].outy = net[modi].hidy + net[modi].uhght + VERSP;
    }
  else
    {
      net[modi].inpwy = net[modi].inpy + net[modi].uhght;
      net[modi].hidy =
	net[modi].inpwy + nhidrep[modi] * net[modi].whght + VERSP;
      net[modi].outwy = net[modi].hidy + net[modi].uhght + VERSP;
      net[modi].outy = net[modi].outwy + nhidrep[modi] * net[modi].whght;
    }
  net[modi].tgty = net[modi].outy + net[modi].uhght;
}


/********************* stack module operations ***********************/

static void
init_stack_display_params (modi, w)
  /* initialize the parameters for the stack display */
     int modi;			/* module number */
     Widget w;			/* and its widget */
{
  Win[modi] = XtWindow (w);	/* get a pointer to the window */
}


static void
expose_stack (w, client_data, call_data)
  /* expose event handler for the stack: redraw the window */
  /* standard callback parameters, not used here */
     Widget w;
     XtPointer client_data, call_data;
{
  int modi = STACKMOD;

  XClearWindow (theDisplay, Win[modi]);
  display_title (modi, titles[modi]);
  display_current_stack (modi, TRUE);
}


void
display_current_stack (modi, weights_included)
  /* display all activations and the current log */
     int modi;			/* moduel number */
     int weights_included;	/* whether to update weight display */
{
  display_assembly (modi, net[modi].inpx, net[modi].inpy,
		    inprep[modi], ninprep[modi]);
  if (data.weightdisplay && weights_included)
    display_normalweights (modi, net[modi].inpx, net[modi].inpwy,
			   wih[modi], ninprep[modi], nhidrep[modi]);
  display_assembly (modi, net[modi].hidx, net[modi].hidy,
		    hidrep[modi], nhidrep[modi]);
  if (data.weightdisplay && weights_included)
    display_reversedweights (modi, net[modi].outwx, net[modi].outwy,
			     who[modi], nhidrep[modi], noutrep[modi]);
  display_assembly (modi, net[modi].outx, net[modi].outy,
		    outrep[modi], noutrep[modi]);
  if (!testing)
    display_assembly (modi, net[modi].tgtx, net[modi].tgty,
		      tgtrep[modi], noutrep[modi]);
  else
    display_assembly (modi, net[modi].tgtx, net[modi].tgty,
		      tgtrep[modi], nhidrep[PARSMOD]);
  display_log (modi);
}


static void
resize_stack (w, client_data, call_data)
  /* resize event handler for the stack: recalculate geometry */
  /* standard callback parameters, not used here */
     Widget w;
     XtPointer client_data, call_data;
{
  int modi = STACKMOD;

  common_resize (modi, stack);	/* get new window size */

  /* width of the boxes representing unit activities and weights */
  net[modi].uwidth =
    (net[modi].width - 2 * HORSP) / ninprep[modi];
  if (data.weightdisplay)
    net[modi].wwidth = net[modi].uwidth;

  net[modi].marg = HORSP;	/* horizontal margin */

  /* x-coordinates of the left sides of each layer, centered in the window */
  net[modi].inpx =		/* input layer */
    (net[modi].width - ninprep[modi] * net[modi].uwidth) / 2;
  net[modi].hidx =		/* hidden layer */
    (net[modi].width - (nhidrep[modi]) * net[modi].uwidth) / 2;
  net[modi].outx = net[modi].tgtx = net[modi].inpx;
  if (data.weightdisplay)
    {
      net[modi].inpwx = net[modi].inpx;
      net[modi].outwx = net[modi].outx;
    }

  /* heights of the boxes representing unit activities and weights */
  if (!data.weightdisplay)
    net[modi].uhght = net[PARSMOD].uhght;
  else
    {
      net[modi].whght =
	imax (0,
	      (net[modi].height - titleboxhght - 3 * VERSP) /
	      (2 * nhidrep[modi] + 4 * data.weightsPerUnit));
      net[modi].uhght = net[modi].whght * data.weightsPerUnit;
    }

  /* y-coordinates of the tops of each layer */
  net[modi].inpy = titleboxhght;
  if (!data.weightdisplay)
    {
      net[modi].hidy = net[modi].inpy + net[modi].uhght + VERSP;
      net[modi].outy = net[modi].hidy + net[modi].uhght + VERSP;
    }
  else
    {
      net[modi].inpwy = net[modi].inpy + net[modi].uhght;
      net[modi].hidy =
	net[modi].inpwy + nhidrep[modi] * net[modi].whght + VERSP;
      net[modi].outwy = net[modi].hidy + net[modi].uhght + VERSP;
      net[modi].outy = net[modi].outwy + nhidrep[modi] * net[modi].whght;
    }
  net[modi].tgty = net[modi].outy + net[modi].uhght;
}


/********************* general display subroutines */

static void
display_sequence (modi, x, y)
  /* write out the word sequence so far */
     int modi,			/* module number */
       x, y;			/* top left corner of input/output assembly */
{
  /* first clear out the old sequence from the display */
  clearRectangle (modi, 0, titleboxhght, x, asmboxhght);
  drawText (modi,
  /* calculate bottom left corner of the text */
	 x - net[modi].marg - XTextWidth (asmfontStruct, net[modi].sequence,
					  strlen (net[modi].sequence)),
	    y - asmboxhght + asmfontStruct->ascent + BOXSP,
	    net[modi].sequence, asmGC);
  XFlush (theDisplay);
}


static void
display_labels (modi, x, y, labels, nitem)
  /* write out the names of the assemblies in a layer */
     int modi, x, y;		/* module number, top left of the labels */
     char *labels[];		/* labels */
     int nitem;			/* number of assemblies */
{
  int i;

  for (i = 0; i < nitem; i++)
    /* enclose the name inside a box */
    display_boxword (modi, x + i * net[modi].hsp, y,
		     net[modi].hsp, asmboxhght,
		     labels[i], TRUE, asmfontStruct, asmGC);
}


static void
display_labeled_layer (modi, nas, rep, nums, x, y, labeloffset)
  /* display a layer of activations and the labels of the closest words */
     int modi, nas;		/* module number, number of assemblies */
     double rep[];		/* activations in the layer */
     int nums[],		/* indices of the correct words */
       x, y,			/* top left of the layer */
       labeloffset;		/* vertical displacement of the labels */
{
  int i;
  char s[2 * MAXWORDL + 4 + 1];	/* label: may contain correct label also */
  int nearest;			/* index of the nearest rep in the lexicon */

  /* display the unit activations */
  display_layer (modi, nas, rep, x, y, nwordrep);
  /* display the labels of the closest words and correct words */
  for (i = 0; i < nas; i++)
    {
      /* get the index of the closest word representation in the lexicon */
      nearest = find_nearest (&rep[i * nwordrep], words, nwordrep, nwords);
      if (nearest == nums[i])
	/* if it is correct only output the label of the closest word */
	/* do not write out the word if it is blank */
	{
	  sprintf (s, "%s", words[nearest].chars);
	  display_boxword (modi, x + i * net[modi].hsp, y + labeloffset,
			   net[modi].hsp, asmboxhght,
			   words[nearest].chars, nearest != BLANKINDEX,
			   asmfontStruct, asmGC);
	}
      else
	/* closest word is not the correct one; output the closest label
	   followed by the correct label in parenthesis, enclosed in ** */
	{
	  sprintf (s, "*%s(%s)*", words[nearest].chars,
		   words[nums[i]].chars);
	  display_boxword (modi, x + i * net[modi].hsp, y + labeloffset,
			   net[modi].hsp, asmboxhght,
			   s, TRUE, asmerrorfontStruct, asmerrorGC);
	}
    }
  /* update the input/output word sequence */
  if (modi == PARSMOD && nas == 1)
    sprintf (net[modi].newitem, "%s", s);
  XFlush (theDisplay);
}


static void
display_layer (modi, nas, layer, x, y, nrep)
  /* display a layer of unit activities */
     int modi, nas;		/* module number, number of assemblies */
     double layer[];		/* activities */
     int x, y,			/* top left corner of the layer */
       nrep;			/* number of units in word representation */
{
  int i;

  for (i = 0; i < nas; i++)
    /* display one assembly at a time */
    display_assembly (modi, x + i * net[modi].hsp, y, &layer[i * nrep], nrep);
}


static void
display_assembly (modi, x, y, assembly, nrep)
  /* display the activities of the units in one assembly */
     int modi,			/* module number */
       x, y;			/* top left corner of the assembly */
     double assembly[];		/* activities */
     int nrep;			/* number of units in assembly */
{
  int i;

  for (i = 0; i < nrep; i++)
    /* display one unit at a time */
    /* translate activation to color using the black-red-yellow-white scale */
    fillRectangle (modi, x + i * net[modi].uwidth, y,
		   net[modi].uwidth, net[modi].uhght,
		   trans_to_color (assembly[i], UNITCOLORS));
  /* draw a boundary around the assembly */
  drawRectangle (modi, x, y, nrep * net[modi].uwidth, net[modi].uhght);
  XFlush (theDisplay);
}


static void
display_boxword (modi, x, y, width, height, wordchars, dodisplay, fontStruct, 
		 currGC)
  /* write out a string inside a box */
     int modi,			/* module number */
       x, y,			/* top left corner of the box */
       width, height;		/* of the box */
     char wordchars[];		/* the string to be written */
     int dodisplay;		/* whether to display the word */
     XFontStruct *fontStruct;	/* font to be used */
     GC currGC;			/* write it in this context */
{
  int i;			/* last char in the string */
  char s[2 * MAXWORDL + 4 + 1];	/* copy the string here */

  /* first clear the area of the box */
  clearRectangle (modi, x, y, width, height);

  if (dodisplay)		/* if really writing the word */
    {
      /* if necessary, cut the text from the right to make it fit */
      /* first copy it to something we can always modify */
      sprintf (s, "%s", wordchars);
      for (i = strlen (s);
	   XTextWidth (fontStruct, s, strlen (s)) > width;
	   s[--i] = '\0')
	;
      /* write the word neatly centered inside the box area */
      drawoverText (modi,
		    x + (width - XTextWidth (fontStruct, s, strlen (s))) / 2,
		    y + height - BOXSP - asmfontStruct->descent, s, currGC);
    }
  
  /* draw the box around the word */
  drawRectangle (modi, x, y, width, height);
  XFlush (theDisplay);
}


void
display_all_weights ()
  /* display weights of all modules */
{
  int modi;

  for (modi = 0; modi < NMODULES; modi++)
    if (!parseronly || modi == PARSMOD)
      {
	display_normalweights (modi, net[modi].inpwx, net[modi].inpwy,
			       wih[modi], ninprep[modi], nhidrep[modi]);
	display_reversedweights (modi, net[modi].outwx, net[modi].outwy,
				 who[modi], nhidrep[modi], noutrep[modi]);
      }
}


static void
display_normalweights (modi, x, y, weights, nsource, ndest)
  /* display the weight values between two layers with x=source, y=dest */
     int modi,			/* module number */
       x, y;			/* top left corner of the weight block */
     double weights[MAXLAYER][MAXLAYER];	/* weight values */
     int nsource,		/* number of units in the source assembly */
       ndest;			/* number of units in the destin. assembly */
{
  int i, j;

  /* weight display may take a lot of time; make sure events are uptodate */
  handle_events ();

  for (i = 0; i < nsource; i++)
    for (j = 0; j < ndest; j++)
      /* display one weight at a time */
      /* translate activation to color using the green-black-red scale */
      fillRectangle (modi, x + i * net[modi].wwidth, y + j * net[modi].whght,
		     net[modi].wwidth, net[modi].whght,
		     trans_to_color (weights[i][j], WEIGHTCOLORS));
  /* draw a boundary around the weight display */
  drawRectangle (modi, x, y,
		 nsource * net[modi].wwidth, ndest * net[modi].whght);
  XFlush (theDisplay);
}


static void
display_reversedweights (modi, x, y, weights, nsource, ndest)
  /* display the weight values between two layers with x=dest, y=source  */
     int modi,			/* module number */
       x, y;			/* top left corner of the weight block */
     double weights[MAXLAYER][MAXLAYER];	/* weight values */
     int nsource,		/* number of units in the source assembly */
       ndest;			/* number of units in the destin. assembly */
{
  int i, j;

  /* weight display may take a lot of time; make sure events are uptodate */
  handle_events ();

  for (i = 0; i < ndest; i++)
    for (j = 0; j < nsource; j++)
      /* display one weight at a time */
      /* translate activation to color using the green-black-red scale */
      fillRectangle (modi, x + i * net[modi].wwidth, y + j * net[modi].whght,
		     net[modi].wwidth, net[modi].whght,
		     trans_to_color (weights[j][i], WEIGHTCOLORS));
  /* draw a boundary around the weight display */
  drawRectangle (modi, x, y,
		 ndest * net[modi].wwidth, nsource * net[modi].whght);
  XFlush (theDisplay);
}


static void
display_title (modi, name)
/* write the name of the module on top center of the graphics window */
     int modi;			/* module number */
     char name[];		/* module name string */
{
  drawText (modi,
  (net[modi].width - XTextWidth (titlefontStruct, name, strlen (name))) / 2,
	    BOXSP + titlefontStruct->ascent,
	    name, titleGC);
}


static void
display_log (modi)
/* write the log line on top left of the graphics window */
     int modi;				/* module number */
{ 
  clearRectangle (modi, 0, 0, net[modi].width, titleboxhght);
  drawText (modi, net[modi].marg, BOXSP + titlefontStruct->ascent,
	    net[modi].log, logGC);
  /* if we did not overwrite the title, rewrite it */
  if (net[modi].marg +
      XTextWidth (logfontStruct, net[modi].log, strlen (net[modi].log)) <
      (net[modi].width -
       XTextWidth (titlefontStruct, titles[modi], strlen (titles[modi]))) / 2)
    display_title (modi, titles[modi]);
}


static int
trans_to_color (value, map)
/* translate an activation or weight value into color index */
     double value;		/* activation/weight */
     int map;			/* selects a colormap */
{
  double range;			/* how much of the scale is used for weights */
  
  if (!data.weightdisplay)
    /* map the number [0,1] to corresponding color */
    return ((int) (((actual_color_range - 1) * value) + 0.499999));
  else
    {
      if (map == UNITCOLORS)
	/* map the number [0,1] to corresponding color */
	return ((int) (((actual_color_range - 1) / 4.0 +
		(actual_color_range - 1) * 3.0 / 4.0 * value) + 0.499999));
      else
	{
	  if ((visual->class == GrayScale || visual->class == StaticGray))
	    range = 1.0;
	  else
	    range = 0.5;
	  /* map [-lim,lim] to corresponding color, clip */
	  if (value < -data.weightcolorrange)
	    return (0);
	  else if (value > data.weightcolorrange)
	    return ((int) ((actual_color_range - 1) * range));
	  else
	    return ((int) ((actual_color_range - 1) * range *
			   (value + data.weightcolorrange) /
			   (2.0 * data.weightcolorrange) + 0.499999));
	}
    }
}


/********************* low-level operations ***********************/

static void
clearRectangle (modi, x, y, width, height)
/* draw a rectangle in the background color */
     int modi, x, y, width, height;
{
  XFillRectangle (theDisplay, Win[modi], clearGC, x, y, width, height);
}


static void
fillRectangle (modi, x, y, width, height, colorindex)
/* draw a filled rectangle in given color */
     int modi, x, y, width, height, colorindex;
{
  XSetForeground (theDisplay, activityGC, colors[cmap[colorindex]].pixel);
  XFillRectangle (theDisplay, Win[modi], activityGC, x, y, width, height);
}


static void
drawRectangle (modi, x, y, width, height)
/* draw a rectangle in the box color */
     int modi, x, y, width, height;
{
  XDrawRectangle (theDisplay, Win[modi], boxGC, x, y, width, height);
}


static void
drawText (modi, x, y, text, currGC)
/* draw text image according to given graphics context */
     int modi, x, y;
     char text[];
     GC currGC;
{
  XDrawImageString (theDisplay, Win[modi], currGC, x, y, text, strlen (text));
}


static void
drawoverText (modi, x, y, text, currGC)
/* overwrite text on screen according to give graphics context */
     int modi, x, y;
     char text[];
     GC currGC;
{
  XDrawString (theDisplay, Win[modi], currGC, x, y, text, strlen (text));
}
