/* File: main.c
 *
 * Main sentence processing loop, simulation management, and I/O for SPEC.
 *
 * Copyright (C) 1994 Risto Miikkulainen
 *
 *  This software can be copied, modified and distributed freely for
 *  educational and research purposes, provided that this notice is included
 *  in the code, and the author is acknowledged in any materials and reports
 *  that result from its use. It may not be used for commercial purposes
 *  without expressed permission from the author.
 *
 * $Id: main.c,v 1.79 1994/09/20 10:48:27 risto Exp $
 */

#include <stdio.h>
#include <math.h>
#include <setjmp.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Cardinals.h>

#include "defs.h"
#define DEFINE_GLOBALS		/* so global variables get defined here only */
#include "globals.c"


/************ simu, input, option, and parameter keyword strings *************/

/* name defaults */
#define APP_CLASS "Spec"		/* class of this application */
#define DEFAULT_SIMUFILENAME "simu"	/* simulation file */

/* keywords in the simulation specification file */
#define SIMU_INPUTFILE "inputfile"
#define SIMU_SEED "seed"
#define SIMU_PARSERONLY "parseronly"
#define SIMU_STACKNOUNS "stacknouns"
#define SIMU_CHAIN "chain"
#define SIMU_SHUFFLING "shuffling"
#define SIMU_NWORDREP "nwordrep"
#define SIMU_NHIDREP "network-nhidrep"
#define SIMU_FGREPPING "network-fgrepping"
#define SIMU_SIMULATIONENDEPOCH "simulationendepoch"
#define SIMU_SNAPSHOTEPOCHS "snapshotepochs"
#define SIMU_PHASE_LASTEPOCHS "phase-lastepochs"
#define SIMU_ETAS "etas"
#define SIMU_WORDETARATIOS "wordetaratios"
#define SIMU_PARSER_RUNNING "parser-running"
#define SIMU_SEGMENTER_RUNNING "segmenter-running"
#define SIMU_STACK_RUNNING "stack-running"
#define SIMU_EPOCH "epoch"
#define SIMU_NETWORK_ERRORS "network-errors"
#define SIMU_WORD_REPRESENTATIONS "word-representations"
#define SIMU_NETWORK_WEIGHTS "network-weights"

/* keywords and symbols in the input file */
#define INP_NOUNS "nouns"
#define INP_SENTENCES "sentences"
#define SENTDELIM "."		/* sentence delimiter in input files */
#define FRAGDELIM ","		/* fragment delimiter in input files */
#define BLANKLABEL "_"		/* blank rep symbol in input files */
#define WHOLABEL "who"		/* "who" symbol in input files */
#define THELABEL "the"		/* "the" symbol in input files */

/* option keywords */
#define OPT_HELP "-help"
#define OPT_TEST "-test"
#define OPT_TRAIN "-train"
#define OPT_CHAIN "-chain"
#define OPT_NOCHAIN "-nochain"
#define OPT_STACKNOUNS "-stacknouns"
#define OPT_NOSTACKNOUNS "-nostacknouns"
#define OPT_INCLUDEALL "-includeall"
#define OPT_NOINCLUDEALL "-noincludeall"
#define OPT_GRAPHICS "-graphics"
#define OPT_NOGRAPHICS "-nographics"
#define OPT_WEIGHTS "-weights"
#define OPT_NOWEIGHTS "-noweights"
#define OPT_OWNCMAP "-owncmap"
#define OPT_NOOWNCMAP "-noowncmap"
#define OPT_DELAY "-delay"


/******************** Function prototypes ************************/

/* global functions */
#include "prototypes.h"
extern double drand48 __P ((void));
extern void srand48 __P ((long seed));
extern long lrand48 __P ((void));
extern int strcasecmp __P ((const char *s1, const char *s2));

/* functions local to this file */
static void run_simulation __P ((void));
static void run_simulation_once __P ((void));
static void read_and_process_snapshots __P ((int *epoch));
static void test_snapshot __P ((int epoch));
static void training __P ((int epoch));
static void create_toplevel_widget __P ((int argc, char **argv));
static void process_remaining_arguments __P ((int argc, char **argv));
static void process_display_options __P ((int argc, char **argv));
static void process_nodisplay_options __P ((int *argc, char **argv));
static char *get_option __P ((XrmDatabase db,
			      char *app_name, char *app_class,
			      char *res_name, char *res_class));
static void set_option __P ((int *var, char *res_str, int testd));
static void usage __P ((char *app_name));
static void init_system __P ((void));
static void read_params __P ((FILE * fp));
static void read_inputs __P ((FILE * fp));
static int list2indices __P((int itemarray[], char rest[],
			     int maxitems, char listname[]));
static int frag2indices __P ((FRAGSTRUCT * frag, char rest[]));
static int text2floats __P((double itemarray[], int nitems, char nums[]));
static int text2ints __P((int itemarray[], int nitems, char nums[]));
static int determine_index __P ((char wordstring[]));
static int wordindex __P ((char wordstring[]));
static int read_till_keyword __P ((FILE * fp, char keyword[], int required));
static char *rid_sspace __P ((char rest[]));
static void fgetline __P ((FILE * fp, char *s, int lim));
static void fgl __P ((FILE * fp));
static int open_file __P((char *filename, char *mode, FILE **fp,int required));
static void read_reps __P ((FILE * fp, int epoch));
static void check_reps_valid __P ((int epoch));
static void write_reps __P ((FILE * fp));
static void randomize_reps __P ((double par1, double par2));
static void readfun __P ((FILE * fp, double *place, double par1, double par2));
static void writefun __P((FILE * fp, double *place, double par1, double par2));
static void randfun __P ((FILE * fp, double *place, double low, double span));
static double distance __P ((int *foo, double v1[], double v2[], int nrep));
static void get_current_params __P ((int epoch));
static void save_current __P ((int epoch));
static void update_nextsnapshot __P ((int epoch));
static void shuffle __P ((void));
static double f01rnd __P ((void));


/******************** static variables ******************** */

/* space for the lexicon (including blank word -1) */
static WORDSTRUCT wordarray[MAXWORDS + 1];

static int phaseends[MAXPHASE],	/* phase end epochs */
  snapshots[MAXSNAPS],		/* snapshot epochs */
  runnings[NMODULES][MAXPHASE];	/* what modules are run and when */

static int nphase,		/* number of phases */
  nsnaps,			/* number of snapshots */
  nextsnapshot;			/* index of the next snapshot */

static int includeall;		/* whether to always run all modules */

static double etas[MAXPHASE],	/* etas for each phase */
  wordetaratios[MAXPHASE];	/* ratio of weight eta and word eta */

/* define the geometry of the display */
static String fallback_resources[] =
{
  "*runstop.left: 	ChainLeft",
  "*runstop.right: 	ChainLeft",
  "*runstop.top:	ChainTop",
  "*runstop.bottom:	ChainTop",
  "*step.fromHoriz: 	runstop",
  "*step.left:	 	ChainLeft",
  "*step.right: 	ChainLeft",
  "*step.top:		ChainTop",
  "*step.bottom:	ChainTop",
  "*clear.fromHoriz: 	step",
  "*clear.left: 	ChainLeft",
  "*clear.right: 	ChainLeft",
  "*clear.top:		ChainTop",
  "*clear.bottom:	ChainTop",
  "*quit.fromHoriz: 	clear",
  "*quit.left:	 	ChainLeft",
  "*quit.right: 	ChainLeft",
  "*quit.top:		ChainTop",
  "*quit.bottom:	ChainTop",
  "*command.fromHoriz: 	quit",
  "*command.left: 	ChainLeft",
  "*command.right: 	ChainRight",
  "*command.top:	ChainTop",
  "*command.bottom:	ChainTop",
  "*parser.fromVert: 	runstop",
  "*parser.top:		ChainTop",
  "*segmenter.fromVert:	parser",
  "*stack.fromVert: 	segmenter",

  /* define the color defaults */
  "*foreground:	        white",
  "*background:		black",
  "*borderColor:	white",

  NULL
};

/* these are the possible command line options */
static XrmOptionDescRec options[] = 
{
  {OPT_HELP, ".help", XrmoptionNoArg, "true"},
  {OPT_TEST, ".testing", XrmoptionNoArg, "true"},
  {OPT_TRAIN, ".testing", XrmoptionNoArg, "false"},
  {OPT_CHAIN, ".chaining", XrmoptionNoArg, "true"},
  {OPT_NOCHAIN, ".chaining", XrmoptionNoArg, "false"},
  {OPT_STACKNOUNS, ".stacknouns", XrmoptionNoArg, "true"},
  {OPT_NOSTACKNOUNS, ".stacknouns", XrmoptionNoArg, "false"},
  {OPT_INCLUDEALL, ".includeall", XrmoptionNoArg, "true"},
  {OPT_NOINCLUDEALL, ".includeall", XrmoptionNoArg, "false"},
  {OPT_GRAPHICS, ".bringupDisplay", XrmoptionNoArg, "true"},
  {OPT_NOGRAPHICS, ".bringupDisplay", XrmoptionNoArg, "false"},
  {OPT_WEIGHTS, ".weightDisplay", XrmoptionNoArg, "true"},
  {OPT_NOWEIGHTS, ".weightDisplay", XrmoptionNoArg, "false"},
  {OPT_OWNCMAP, ".owncmap", XrmoptionNoArg, "true"},
  {OPT_NOOWNCMAP, ".owncmap", XrmoptionNoArg, "false"},
  {OPT_DELAY, ".delay", XrmoptionSepArg, NULL},
};

/* the default values for the application-specific resources;
   see defs.h for component descriptions */
static XtResource resources[] =
{
  {"bringupDisplay", "BringupDisplay", XtRBoolean, sizeof (Boolean),
   XtOffset (RESOURCE_DATA_PTR, bringupdisplay), XtRImmediate,
   (XtPointer) True},
  {"weightDisplay", "WeightDisplay", XtRBoolean, sizeof (Boolean),
   XtOffset (RESOURCE_DATA_PTR, weightdisplay), XtRImmediate,
   (XtPointer) False},
  {"owncmap", "Owncmap", XtRBoolean, sizeof (Boolean),
   XtOffset (RESOURCE_DATA_PTR, owncmap), XtRImmediate, (XtPointer) False},
  {"delay", "Delay", XtRInt, sizeof (int),
   XtOffset (RESOURCE_DATA_PTR, delay), XtRString, "0"},

  {"netwidth", "Netwidth", XtRDimension, sizeof (Dimension),
   XtOffset (RESOURCE_DATA_PTR, netwidth), XtRString, "512"},
  {"netheight", "Netheight", XtRDimension, sizeof (Dimension),
   XtOffset (RESOURCE_DATA_PTR, netheight), XtRString, "128"},
  {"weightnetheight", "Weightnetheight", XtRDimension, sizeof (Dimension),
   XtOffset (RESOURCE_DATA_PTR, weightnetheight), XtRString, "248"},
  {"weightsPerUnit", "WeightsPerUnit", XtRInt, sizeof (int),
   XtOffset (RESOURCE_DATA_PTR, weightsPerUnit), XtRString, "3"},

  {"textColor", "TextColor", XtRPixel, sizeof (Pixel),
   XtOffset (RESOURCE_DATA_PTR, textColor), XtRString, "green"},
  {"netColor", "NetColor", XtRPixel, sizeof (Pixel),
   XtOffset (RESOURCE_DATA_PTR, netColor), XtRString, "red"},
  {"weightColorRange", "WeightColorRange", XtRFloat, sizeof (float),
   XtOffset (RESOURCE_DATA_PTR, weightcolorrange), XtRString, "1.0"},

  {"commandfont", "Commandfont", XtRString, sizeof (String),
   XtOffset (RESOURCE_DATA_PTR, commandfont), XtRString, "7x13bold"},
  {"titlefont", "Titlefont", XtRString, sizeof (String),
   XtOffset (RESOURCE_DATA_PTR, titlefont), XtRString, "8x13bold"},
  {"logfont", "Logfont", XtRString, sizeof (String),
   XtOffset (RESOURCE_DATA_PTR, logfont), XtRString, "6x10"},
  {"asmfont", "Asmfont", XtRString, sizeof (String),
   XtOffset (RESOURCE_DATA_PTR, asmfont), XtRString, "6x10"},
  {"asmerrorfont", "Asmerrorfont", XtRString, sizeof (String),
   XtOffset (RESOURCE_DATA_PTR, asmerrorfont), XtRString, "5x8"},

  /* command line options */
  {"help", "Help", XtRBoolean, sizeof (Boolean),
   XtOffset (RESOURCE_DATA_PTR, help), XtRImmediate, (XtPointer) False},
  {"testing", "Testing", XtRBoolean, sizeof (Boolean),
   XtOffset (RESOURCE_DATA_PTR, testing), XtRImmediate, (XtPointer) False},
  {"chaining", "Chaining", XtRString, sizeof (String),
   XtOffset (RESOURCE_DATA_PTR, chaining), XtRString, NULL},
  {"stacknouns", "Stacknouns", XtRString, sizeof (String),
   XtOffset (RESOURCE_DATA_PTR, stacknouns), XtRString, NULL},
  {"includeall", "Includeall", XtRString, sizeof (String),
   XtOffset (RESOURCE_DATA_PTR, includeall), XtRString, NULL},
};

/* interrupt handling */
jmp_buf loop_sents_env;		/* jump here from after interrupt */


/*********************  main processing loops ******************************/

void
main (argc, argv)
/* initialize X display, system parameters, read inputs, process them */
     int argc;
     char **argv;
{
  int xargc;			/* saved argc & argv */
  char **xargv;
  int i;

  /* save command line args so we can parse them later */
  xargc = argc;
  xargv = (char **) XtMalloc (argc * sizeof (char *));
  for (i = 0; i < argc; i++)
    xargv[i] = argv[i];

  /* try to open the display */
  XtToolkitInitialize ();
  app_con = XtCreateApplicationContext ();
  XtAppSetFallbackResources (app_con, fallback_resources);
  theDisplay = XtOpenDisplay (app_con, NULL, NULL, APP_CLASS,
			      options, XtNumber (options), &argc, argv);
  if (theDisplay != NULL)
    /* create the top-level widget and get application resources */
    create_toplevel_widget(xargc, xargv);
  else
    fprintf (stderr, "No display: running in text mode.\n");
  
  process_remaining_arguments (argc, argv);
  init_system ();
  run_simulation ();
  exit (EXIT_NORMAL);
}


static void
run_simulation ()
/* run once through the simulation if no graphics, otherwise
   multiple times until user hits "quit" */
{
  if (setjmp (loop_sents_env))
    {
      /* longjmp gets here */
    }
  else
    {
      /* return from setjmp */
    }
  if (displaying)
    while (TRUE)
      {
	/* if the Xdisplay is up, process events until the user hits "Run" */
	/* user hitting "Quit" will terminate the program */
	wait_for_run ();
	/* user hit "Run"; start processing snapshots */
	run_simulation_once ();
      }
  else
    /* no Xdisplay; iterate snapshots right away */
    run_simulation_once ();
}


static void
run_simulation_once ()
/* iterate once through the snapshots and training if required */
{
  int i,
  epoch = NONE;		/* indicates no snapshots were saved */

  if (!testing)
    {
      /* start a random number sequence */
      srand48 (seed);
      /* always randomize weights again
         so that the random number sequence stays the same */
      randomize_reps (REPLOW, REPSPAN);
      iterate_weights (randfun, NULL, WEIGHTLOW, WEIGHTSPAN);

      /* read all snapshots, getting current weights and epoch */
      read_and_process_snapshots (&epoch);

      /* update shuffling (so that we can continue where we left off) */
      if (shuffling)
	{
	  /* reinitialize shuffletable */
	  for (i = 0; i < nsents; i++)
	    shuffletable[i] = i;
	  if (epoch != NONE)
	    for (i = 0; i < epoch; i++)
	      shuffle ();
	}

      /* save the initial weights if required */
      nextsnapshot = 0;
      if (epoch == NONE)
	{
	  /* no snapshots read; check if initial one needs to be saved */
	  epoch = 0;
	  if (nsnaps > 0)
	    if (snapshots[nextsnapshot] == 0)
	      save_current (epoch);
	}
      else
	update_nextsnapshot (epoch);

      /* continue training from the next epoch on */
      training (++epoch);
    }
  else
    /* process each snapshot */
    read_and_process_snapshots (&epoch);
}


static void
read_and_process_snapshots (epoch)
/* iterate through all snapshots; test each one if in testing mode */
     int *epoch;		/* return the epoch of the last snapshot */
{
  FILE *fp;			/* pointer to the simufile */
  
  /* reopen the simufile (necessary because the user may want to */
  /* do the same simulation multiple times in one sitting */
  fp = fopen (simufile, "r");
  read_params (fp);

  while (read_till_keyword (fp, SIMU_EPOCH, NOT_REQUIRED))
    {
      /* read the epoch number of snapshot */
      fscanf (fp, "%d", epoch);
      if (displaying)
	{
	  printf ("Reading snapshot %d, hold on...", *epoch);
	  fflush (stdout);
	}
      read_till_keyword (fp, SIMU_WORD_REPRESENTATIONS, REQUIRED);
      /* read the current word representations and weights */
      read_reps (fp, *epoch);
      check_reps_valid (*epoch);/* make sure got all reps */
      iterate_weights (readfun, fp, NOT_USED, NOT_USED);
      if (displaying)
	printf ("Done.\n");
      if (testing)
	/* run through the test set: display and collect stats */
	test_snapshot (*epoch);
    }
  fclose (fp);
}


static void
test_snapshot (epoch)
/* run through all sentences and collect statistics about performance */
     int epoch;			/* epoch of the snapshot */
{
  if (displaying && data.weightdisplay)
    display_all_weights ();	/* first display the current weights */
  get_current_params (epoch);	/* current learning rates and modules run */
  init_stats ();
  iterate_sentences ();		/* run through the test set */
  print_summary (epoch);	/* average performance */
  print_through_sentence ();	/* accuracy through a sentence */
}


static void
training (epoch)
/* train the modules until simulationendepoch */
     int epoch;			/* starting from this epoch */
{
  if (displaying && data.weightdisplay)
    display_all_weights ();	/* display initial weights */
  while (epoch <= simulationendepoch)
    {
      get_current_params (epoch);/* current learning rates and modules run */
      init_stats ();
      if (shuffling)		/* change the order of sentences */
	shuffle ();
      iterate_sentences ();	/* run through the training set */

      /* log the progress of training error */
      printf ("Epoch %d errors: ", epoch);
      write_error (stdout);
      if (nextsnapshot < nsnaps)
	if (epoch >= snapshots[nextsnapshot])
	  save_current (epoch);
      epoch++;
    }
}


/*********************  initializations ******************************/

/**************** X interface, command line */

static void
create_toplevel_widget (argc, argv)
  /* retrieve resources, create a colormap, and start the top-level widget */
  int argc;
  char **argv;
{
  Widget dummy;			/* dummy top-level widget */
  Arg args[10];
  int scr;			/* temporary screen (for obtaining colormap) */
  int n = 0;			/* argument counter */

  /* Create a dummy top-level widget to retrieve resources */
  /* (necessary to get the right netcolor etc with owncmap) */
  dummy = XtAppCreateShell (NULL, APP_CLASS, applicationShellWidgetClass,
			    theDisplay, NULL, ZERO);
  XtGetApplicationResources (dummy, &data, resources,
			     XtNumber (resources), NULL, ZERO);
  scr = DefaultScreen (theDisplay);
  visual = DefaultVisual (theDisplay, scr);
  
  /* Select colormap; data.owncmap was specified in resources
     or as an option */
  if (data.owncmap)
    {
      colormap = XCreateColormap (theDisplay, DefaultRootWindow (theDisplay),
				  visual, AllocNone);
      XtSetArg (args[n], XtNcolormap, colormap);
      n++;
    }
  else
    colormap = DefaultColormap (theDisplay, scr);
  XtDestroyWidget (dummy);
  
  /* Create the real top-level widget */
  XtSetArg (args[n], XtNargv, argv);
  n++;
  XtSetArg (args[n], XtNargc, argc);
  n++;
  main_widget = XtAppCreateShell (NULL, APP_CLASS, applicationShellWidgetClass,
				  theDisplay, args, n);
  XtGetApplicationResources (main_widget, &data, resources,
			     XtNumber (resources), NULL, ZERO);
}


static void
process_remaining_arguments (argc, argv)
  /* parse nongraphics options, simufile and inputfile, setup displaying */
  int argc;
  char **argv;
{
  int i;

  /* if opendisplay was successful, all options were parsed into "data" */
  /* otherwise, we have to get the options from command-line argument list */
  if (theDisplay != NULL)
    process_display_options (argc, argv);
  else
    process_nodisplay_options (&argc, argv);

  /* figure out the simufile and input file names */
  sprintf (simufile, "%s", "");
  sprintf (current_inpfile, "%s", "");
  for (i = 1; i < argc; i++)
    if (argv[i][0] == '-')
      {
	fprintf (stderr, "Unknown option %s\n", argv[i]);
	usage (argv[0]);
	exit (EXIT_COMMAND_ERROR);
      }
    else if (!strlen (simufile))	/* first argument is simufile */
      sprintf (simufile, "%s", argv[i]);
    else if (!strlen (current_inpfile))	/* second is inputfile */
      sprintf (current_inpfile, "%s", argv[i]);
    else
      {
	fprintf (stderr, "Too many arguments\n");
	usage (argv[0]);
	exit (EXIT_COMMAND_ERROR);
      }
  if (!strlen (simufile))		/* if no simufile given */
    sprintf (simufile, "%s", DEFAULT_SIMUFILENAME);
  /* if no inputfilename is specified, the one in the simufile is used */

  /* decide whether to bring up display */
  if (theDisplay && data.bringupdisplay)
    displaying = TRUE;
  else
    displaying = FALSE;
}


static void
process_display_options (argc, argv)
/* get the non-graphics-related options from the "data" structure */
  int argc;
  char **argv;
{
  /* quick user help */
  if (data.help)
    {
      usage (argv[0]);
      exit (EXIT_NORMAL);
    }
  /* training or testing mode */
  testing = data.testing;
  /* chaining the modules or processing each in isolation */
  set_option (&chain, data.chaining, TRUE);
  /* whether the stack is trained with nouns or sentences */
  set_option (&stacknouns, data.stacknouns, FALSE);
  /* whether to run all modules or according to the defs in the simufile */
  set_option (&includeall, data.includeall, TRUE);
}


static void
process_nodisplay_options (argc, argv)
/* get the non-graphics-related options from the command string */
  int *argc;
  char **argv;
{
  char *res_str;		/* string value of an option */
  XrmDatabase db = NULL;	/* resource database for options */

  XrmParseCommand (&db, options, XtNumber (options), APP_CLASS, argc, argv);

  /* quick user help */
  res_str = get_option (db, argv[0], APP_CLASS, "help", "Help");
  if (res_str != NULL)
    if (!strcasecmp ("true", res_str))
      {
	usage (argv[0]);
	exit (EXIT_NORMAL);
      }
  
  /* training or testing mode */
  res_str = get_option (db, argv[0], APP_CLASS, "testing", "Testing");
  if (res_str != NULL)
    testing = !strcasecmp ("true", res_str);
  else
    testing = FALSE;
  
  /* chaining the modules or processing each in isolation */
  res_str = get_option (db, argv[0], APP_CLASS, "chaining", "Chaining");
  set_option (&chain, res_str, TRUE);

  /* whether the stack is trained with nouns or sentences */
  res_str = get_option (db, argv[0], APP_CLASS, "stacknouns", "Stacknouns");
  set_option (&stacknouns, res_str, FALSE);
  
  /* whether to run all modules or according to the defs in the simufile */
  res_str = get_option (db, argv[0], APP_CLASS, "includeall", "Includeall");
  set_option (&includeall, res_str, TRUE);
}


static char *
get_option (db, app_name, app_class, res_name, res_class)
  /* return the pointer to the string value of the resource */
  XrmDatabase db;			/* resource database */
  char *res_name, *res_class,		/* resource name and class */
    *app_name, *app_class;		/* application name and class */
{
  XrmValue value;			/* value of the resource */
  char *type,				/* resource type */
    name[MAXSTRL + 1], class[MAXSTRL + 1];/* full name and class of resource */

  sprintf (name, "%s.%s", app_name, res_name);
  sprintf (class, "%s.%s", app_class, res_class);
  XrmGetResource(db, name, class, &type, &value);
  return (value.addr);
}


static void
set_option (var, res_str, testd)
/* decide whether to use the simufile value, testing default, or option */
  int *var,				/* simulation variable */
    testd;				/* default value in test mode */
  char *res_str;			/* truth value for the variable */
{
  if (res_str != NULL)			/* use the option if given */
    *var = !strcasecmp ("true", res_str);
  else if (testing)			/* in testing mode use the default */
    *var = testd;
  else
    *var = NONE;			/* in training take it from simufile */
}


static void
usage (app_name)
  /* print out the list of options and arguments */
  char *app_name;			/* name of the program */
{
  char s[MAXSTRL + 1];

  sprintf(s, "%s %s", OPT_DELAY, "<sec>");
  fprintf (stderr, "Usage: %s [options] [simulation file] [input file]\n\
where the options are\n\
  %-20s  Prints this message\n\
  %-20s  Testing mode\n\
  %-20s  Training mode (default)\n\
  %-20s  Networks are connected in a chain (default in testing)\n\
  %-20s  Networks are run in isolation\n\
  %-20s  Stack data specified by the nounlist\n\
  %-20s  Stack data specified by sentences (default in testing)\n\
  %-20s  Include all modules (default in testing)\n\
  %-20s  Include only those specified in the simufile\n\
  %-20s  Bring up graphics display\n\
  %-20s  Text output only\n\
  %-20s  Display weights\n\
  %-20s  Display only activations\n\
  %-20s  Use a private colormap\n\
  %-20s  Use the existing colormap\n\
  %-20s  Delay in updating the screen (in seconds)\n",
	   app_name, OPT_HELP, OPT_TEST, OPT_TRAIN, OPT_CHAIN, OPT_NOCHAIN,
	   OPT_STACKNOUNS, OPT_NOSTACKNOUNS, OPT_INCLUDEALL, OPT_NOINCLUDEALL,
	   OPT_GRAPHICS, OPT_NOGRAPHICS, OPT_WEIGHTS, OPT_NOWEIGHTS,
	   OPT_OWNCMAP, OPT_NOOWNCMAP, s);
}


/********************* system setup */

static void
init_system ()
/* set up blank word, read simulation parameters and input data */
{
  FILE *fp;			/* pointer to the simufile and inputfile */
  int i;

  words = wordarray + 1;	/* reserve index -1 for the blank */
  sprintf (words[BLANKINDEX].chars, "%s", BLANKLABEL);
  for(i = 0; i < MAXREP; i++)	/* make sure its rep is all-0 */
    words[BLANKINDEX].rep[i] = 0.0;

  /* read simulation parameters from simufile */
  printf ("Initializing SPEC simulation from %s...\n", simufile);
  open_file (simufile, "r", &fp, REQUIRED);
  read_params (fp);
  fclose (fp);

  /* read the input data from the input file*/
  printf ("Reading input from %s...", current_inpfile);
  fflush (stdout);
  open_file (current_inpfile, "r", &fp, REQUIRED);
  read_inputs (fp);		/* read the input data */
  fclose (fp);
  printf ("%d sentences, %d words.\n", nsents, nwords);

  /* set up numbers of input/output words and reps for convenience */
  ninputs[PARSMOD] = ninputs[SEGMOD] = 1;
  noutputs[PARSMOD] = ncase;
  noutputs[SEGMOD] = ninputs[STACKMOD] = noutputs[STACKMOD] = 0;
  ninprep[PARSMOD] = ninprep[SEGMOD] = nwordrep + nhidrep[PARSMOD];
  ninprep[STACKMOD] = nhidrep[PARSMOD] + nhidrep[STACKMOD];
  noutrep[PARSMOD] = ncase * nwordrep;
  noutrep[SEGMOD] = nhidrep[PARSMOD] + NCONTROL;
  noutrep[STACKMOD] = nhidrep[PARSMOD] + nhidrep[STACKMOD];
  
  /* check that no layer exceeds max size */
  if (noutrep[PARSMOD] > MAXLAYER)
    {
      fprintf (stderr, "Parser output layer exceeds array size\n");
      exit (EXIT_SIZE_ERROR);
    }
  if (ninprep[SEGMOD] > MAXLAYER && !parseronly)
    {
      fprintf (stderr, "Segmenter input layer exceeds array size\n");
      exit (EXIT_SIZE_ERROR);
    }
  if (noutrep[SEGMOD] > MAXLAYER && !parseronly)
    {
      fprintf (stderr, "Segmenter output layer exceeds array size\n");
      exit (EXIT_SIZE_ERROR);
    }
  if (ninprep[STACKMOD] > MAXLAYER && !parseronly)
    {
      fprintf (stderr, "Stack I/O layers exceed array size\n");
      exit (EXIT_SIZE_ERROR);
    }

  /* initialize the shuffle table */
  for (i = 0; i < nsents; i++)
    shuffletable[i] = i;

  /* initialize graphics */
  if (displaying)
    display_init ();

  /* print out the switch settings so we can recreate the simulation later */
  printf ("Simulation switches: %s %s %s %s\n",
	  testing ? OPT_TEST : OPT_TRAIN,
	  chain ? OPT_CHAIN : OPT_NOCHAIN,
	  stacknouns ? OPT_STACKNOUNS : OPT_NOSTACKNOUNS,
	  (includeall == TRUE) ? OPT_INCLUDEALL : OPT_NOINCLUDEALL);
  printf ("System initialization complete.\n");
}


static void
read_params (fp)
/* read the simulation parameters from the simufile:
   read (ignore everything) until the next keyword, read the value
   of the parameter, and check that it makes sense. Make sure
   all keywords are found; abort if not. */
     FILE *fp;			/* pointer to the simufile */
{
  char rest[MAXSTRL + 1];	/* parameter string */
  int modi,			/* module number */
    i, j;

  read_till_keyword (fp, SIMU_INPUTFILE, REQUIRED);
  /* read the input file name if it was not given as a command line param */
  if (!strlen(current_inpfile))
    fscanf (fp, "%s", current_inpfile);

  read_till_keyword (fp, SIMU_PARSERONLY, REQUIRED);
  fscanf (fp, "%d", &parseronly);

  read_till_keyword (fp, SIMU_STACKNOUNS, REQUIRED);
  /* if stack training was not specified as an option, read it now */
  if (stacknouns == NONE)
    fscanf (fp, "%d", &stacknouns);

  read_till_keyword (fp, SIMU_CHAIN, REQUIRED);
  /* if chaining was not specified as an option, read it now */
  if (chain == NONE)
    fscanf (fp, "%d", &chain);

  read_till_keyword (fp, SIMU_SEED, REQUIRED);
  fscanf (fp, "%d", &seed);

  read_till_keyword (fp, SIMU_SHUFFLING, REQUIRED);
  fscanf (fp, "%d", &shuffling);

  read_till_keyword (fp, SIMU_NWORDREP, REQUIRED);
  fscanf (fp, "%d", &nwordrep);
  if (nwordrep > MAXREP || nwordrep <= 0)
    {
      fprintf (stderr, "%s exceeds array size\n", SIMU_NWORDREP);
      exit (EXIT_SIZE_ERROR);
    }

  read_till_keyword (fp, SIMU_NHIDREP, REQUIRED);
  fgetline (fp, rest, MAXSTRL);
  /* convert the string to numbers and load */
  /* check that the number of nets and hidreps match */
  if (text2ints (nhidrep, NMODULES, rest) != NMODULES)
    {
      fprintf (stderr, "Wrong number of %s specifications\n", SIMU_NHIDREP);
      exit (EXIT_DATA_ERROR);
    }
  for (modi = 0; modi < NMODULES; modi++)
    if (nhidrep[modi] > MAXLAYER || nhidrep[modi] <= 0)
      {
	fprintf (stderr, "%s[%d] exceeds array size\n",
		 SIMU_NHIDREP, modi);
	exit (EXIT_SIZE_ERROR);
      }

  read_till_keyword (fp, SIMU_FGREPPING, REQUIRED);
  fgetline (fp, rest, MAXSTRL);
  if (text2ints (fgrepping, NMODULES, rest) != NMODULES)
    {
      fprintf (stderr, "Wrong number of %s specifications\n", SIMU_FGREPPING);
      exit (EXIT_DATA_ERROR);
    }

  read_till_keyword (fp, SIMU_SIMULATIONENDEPOCH, REQUIRED);
  fscanf (fp, "%d", &simulationendepoch);

  read_till_keyword (fp, SIMU_SNAPSHOTEPOCHS, REQUIRED);
  fgetline (fp, rest, MAXSTRL);
  /* as a side effect, establish the number of snapshots */
  nsnaps = text2ints (snapshots, MAXSNAPS, rest);
  if (nsnaps > MAXSNAPS)
    {
      fprintf (stderr, "Number of %s exceeds array size\n",
	       SIMU_SNAPSHOTEPOCHS);
      exit (EXIT_SIZE_ERROR);
    }

  read_till_keyword (fp, SIMU_PHASE_LASTEPOCHS, REQUIRED);
  fgetline (fp, rest, MAXSTRL);
  /* as a side effect, establish the number of phases */
  nphase = text2ints (phaseends, MAXPHASE, rest);
  if (nphase > MAXPHASE)
    {
      fprintf (stderr, "Number of %s exceeds array size\n",
	       SIMU_PHASE_LASTEPOCHS);
      exit (EXIT_SIZE_ERROR);
    }
  if (phaseends[nphase - 1] < simulationendepoch)
    {
      fprintf (stderr, "%s exceeds last %s\n",
	       SIMU_SIMULATIONENDEPOCH,
	       SIMU_PHASE_LASTEPOCHS);
      exit (EXIT_DATA_ERROR);
    }

  read_till_keyword (fp, SIMU_ETAS, REQUIRED);
  fgetline (fp, rest, MAXSTRL);
  if (text2floats (etas, MAXPHASE, rest) != nphase)
    {
      fprintf (stderr, "Number of %s does not match the number of phases\n",
	       SIMU_ETAS);
      exit (EXIT_DATA_ERROR);
    }

  read_till_keyword (fp, SIMU_WORDETARATIOS, REQUIRED);
  fgetline (fp, rest, MAXSTRL);
  if (text2floats (wordetaratios, MAXPHASE, rest) != nphase)
    {
      fprintf (stderr, "Number of %s does not match the number of phases\n",
	       SIMU_WORDETARATIOS);
      exit (EXIT_DATA_ERROR);
    }

  if (includeall != TRUE)
    /* if includeall was not requested, read the running data */
    {
      read_till_keyword (fp, SIMU_PARSER_RUNNING, REQUIRED);
      fgetline (fp, rest, MAXSTRL);
      if (text2ints (runnings[PARSMOD], MAXPHASE, rest) != nphase)
	{
	  fprintf (stderr,"Number of %s does not match the number of phases\n",
		   SIMU_PARSER_RUNNING);
	  exit (EXIT_DATA_ERROR);
	}
      read_till_keyword (fp, SIMU_SEGMENTER_RUNNING, REQUIRED);
      fgetline (fp, rest, MAXSTRL);
      if (text2ints (runnings[SEGMOD], MAXPHASE, rest) != nphase)
	{
	  fprintf (stderr,"Number of %s does not match the number of phases\n",
		   SIMU_SEGMENTER_RUNNING);
	  exit (EXIT_DATA_ERROR);
	}
      read_till_keyword (fp, SIMU_STACK_RUNNING, REQUIRED);
      fgetline (fp, rest, MAXSTRL);
      if (text2ints (runnings[STACKMOD], MAXPHASE, rest) != nphase)
	{
	  fprintf (stderr,"Number of %s does not match the number of phases\n",
		   SIMU_STACK_RUNNING);
	  exit (EXIT_DATA_ERROR);
	}
    }
  else
    /* otherwise set it all up to true */
    for (i = 0; i < NMODULES; i++)
      for (j = 0; j < nphase; j++)
	runnings[i][j] = TRUE;
}


/********************* inputfile */

static void
read_inputs (fp)
/* read the sentence data, make sure the array sizes are not exceeded.
   As a side effect, set up ncase */
     FILE *fp;			/* pointer to the inputfile */
{
  int i, k;
  char rest[MAXSTRL + 1];	/* holds one line of input data */
  char pushstr[MAXSTRL + 1],	/* temporarily hold push, pop and output*/
       popstr[MAXSTRL + 1],
       outstr[MAXSTRL + 1];
  int lastfrag;			/* whether this was last fragment of sent */

  /* first find the list of nouns (for training the stack separately) */
  read_till_keyword (fp, INP_NOUNS, REQUIRED);
  fgetline (fp, rest, MAXSTRL);
  nnouns = list2indices (nouns, rest, MAXNOUNS, INP_NOUNS);

  /* read the input sentences */
  read_till_keyword (fp, INP_SENTENCES, REQUIRED);
  /* first the push command of the first fragment */
  for (i = 0; fscanf (fp, "%s", pushstr) != EOF; i++)
    {
      if (i >= MAXSENTS)
	{
	  fprintf (stderr, "Number of input sentences exceeds array size\n");
	  exit (EXIT_SIZE_ERROR);
	}
      lastfrag = FALSE;
      /* read the fragments of the sentence */
      for (k = 0; !lastfrag; k++)
	{
	  if (k >= MAXFRAGS)
	    {
	      fprintf (stderr,
		       "Number of sentence fragments exceeds array size\n");
	      exit (EXIT_SIZE_ERROR);
	    }
	  /* read the push, pop, and output commands */
	  /* the push command of the first frag was read above */
	  if (k != 0)
	    fscanf (fp, "%s", pushstr);
	  fscanf (fp, "%s %s", popstr, outstr);
	  sprintf (rest, "%s %s %s", pushstr, popstr, outstr);
	  if (sscanf (rest, "%d %d %d", &sents[i].frag[k].pushc,
		      &sents[i].frag[k].popc, &sents[i].frag[k].outc) != 3)
	    {
	      fprintf (stderr, "Could not get push, pop, or output command\n");
	      exit (EXIT_DATA_ERROR);
	    }
	  /* read the fragment data */
	  fgetline (fp, rest, MAXSTRL);
	  /* convert the words into indices and store */
	  lastfrag = frag2indices (&sents[i].frag[k], rest);
	}
      sents[i].nfrag = k;	/* number of fragments in this sent */
    }
  nsents = i;			/* number of sentences */
  if (ncase <= 0)
    {
      fprintf (stderr, "No case roles specified\n");
      exit (EXIT_DATA_ERROR);
    }
}


static int
list2indices (itemarray, rest, maxitems, listname)
/* convert at most maxitems words in the rest list to indices,
   store, and add to lexicon if necessary. Return the number of items */
     int itemarray[];		/* array where the indices will be stored */
     int maxitems;		/* size of the array */
     char rest[];		/* the input string */
     char listname[];		/* name of the variable list */
{
  int j = 0, index;		/* index of the word in the lexicon */
  char wordstring[MAXSTRL + 1];	/* string for one word */

  while (TRUE)
    {
      rest = rid_sspace (rest);	/* first remove blanks */
      if (rest != NULL)		/* if there is anything left */
	{
	  if (j >= maxitems)
	    {
	      fprintf (stderr, "Number of %s exceeds array size\n",
		       listname);
	      exit (EXIT_SIZE_ERROR);
	    }
	  sscanf (rest, "%s", wordstring);	/* get the next word */
	  rest += strlen (wordstring);		/* remove from the string */
	  index = determine_index (wordstring);	/* get or set its index */
	  itemarray[j++] = index;/* put it in the list */
	}
      else
	break;			/* if the string has been exhausted */
    }
  return (j);			/* number of words read */
}


static int
frag2indices (frag, rest)
/* convert the words in the fragment definition into indices,
   store, and add to lexicon if necessary; returns whether
   the last delimiter was a sentence delim (i.e., this is the last fragment),
   and as a side effect, computes the number of cases.
   Returns whether this was the last fragment in the sentence */
     FRAGSTRUCT *frag;		/* structure for the fragment data */
     char rest[];		/* the input string */
{
  int j,			/* word counter */
    lastfrag = FALSE;		/* whether this was the last fragment */
  char wordstring[MAXSTRL + 1];	/* string for one word */

  sprintf (wordstring, "%s", "");
  /* read input word sequence until delimiter and check that it fits array */
  for (j = 0; j < MAXSEQ + 1; j++)
    {
      rest = rid_sspace (rest);		/* first get rid of blanks */
      if (rest != NULL)		/* if there is anything left */
	{
	  sscanf (rest, "%s", wordstring);	/* get the next word */
	  rest += strlen (wordstring);	/* remove from the string */

	  /* if the word was a delimiter stop */
	  /* without taking it as part of input */
	  if (!strcasecmp (wordstring, FRAGDELIM) ||
	      !strcasecmp (wordstring, SENTDELIM))
	    break;
	  else
	    {
	      /* too many non-delimiter words in sequence */
	      if (j >= MAXSEQ)
		{
		  fprintf (stderr,
			   "Number of words in sequence exceeds array size\n");
		  exit (EXIT_SIZE_ERROR);
		}
	      /* otherwise establish its index (add in the lexicon if
		 necessary) and make it part of the input data */
	      frag->inps[j] = determine_index (wordstring);
	    }
	}
      else
	break;
    }
  /* set up number of words in this fragment */
  frag->nseq = j;
  /* was this the last fragment? */
  lastfrag = !strcasecmp (wordstring, SENTDELIM);

  /* read the case-role assignment and check that it does not exceed array */
  for (j = 0; j < MAXCASE + 1; j++)
    {
      rest = rid_sspace (rest);	/* first get rid of blanks */
      if (rest != NULL)		/* if there is anything left */
	{
	  if (j >= MAXCASE)
	    {
	      fprintf (stderr, "Number of cases exceeds array size\n");
	      exit (EXIT_SIZE_ERROR);
	    }
	  sscanf (rest, "%s", wordstring);/* get the next word */
	  rest += strlen (wordstring);	/* remove from the string */

	  /* establish index of the word (and add in lexicon if necessary)
	     make part of input data */
	  frag->tgts[j] = determine_index (wordstring);
	}
      else
	break;			/* if the string has been exhausted */
    }
  if (ncase == NONE)
    ncase = j;			/* establish the number of cases */
  else if (j != ncase)
    {
      fprintf (stderr, "Wrong number of case roles\n");
      exit (EXIT_DATA_ERROR);
    }
  return (lastfrag);		/* whether this was the last fragment */
}


static int
text2floats (itemarray, nitems, nums)
/* convert the string of numbers to floats and stores in an array, returning
   the number of floats read, or nitems + 1 if there where too many */
     double itemarray[];	/* array where the numbers will be stored */
     int nitems;		/* max number of items */
     char nums[];		/* the input string */
{
  int j = 0;
  char onenum[MAXSTRL + 1];	/* string for one float */

  while (j < nitems)
    {
      nums = rid_sspace (nums);	/* first get rid of blanks */
      if (nums != NULL)		/* if there is anything left */
	{
	  sscanf (nums, "%s", onenum);  /* get the next number */
	  nums += strlen (onenum);	/* remove from the string */
	  if (sscanf (onenum, "%lf", &itemarray[j]) != 1)
	    break;			/* if could not get a number */
	  else
	    j++;
	}
      else
	break;			/* if the string has been exhausted */
    }
  /* check if there was more stuff in the input string that was not read */
  if (j == nitems && strlen (nums))
    {
      nums = rid_sspace (nums);	/* first get rid of blanks */
      if (nums != NULL)		/* if there is anything left */
	j++;
    }
  return (j);
}


static int
text2ints (itemarray, nitems, nums)
/* convert the string of numbers to ints and stores in an array, returning
   the number of ints read, or nitems + 1 if there where too many */
     int itemarray[];		/* array where the numbers will be stored */
     int nitems;		/* max number of items */
     char nums[];		/* the input string */
{
  int j = 0;
  char onenum[MAXSTRL + 1];	/* string for one float */

  while (j < nitems)
    {
      nums = rid_sspace (nums);	/* first get rid of blanks */
      if (nums != NULL)		/* if there is anything left */
	{
	  sscanf (nums, "%s", onenum);  /* get the next number */
	  nums += strlen (onenum);	/* remove from the string */
	  if (sscanf (onenum, "%d", &itemarray[j]) != 1)
	    break;			/* if could not get a number */
	  else
	    j++;
	}
      else
	break;			/* if the string has been exhausted */
    }
  /* check if there was more stuff in the input string that was not read */
  if (j == nitems && strlen (nums))
    {
      nums = rid_sspace (nums);	/* first get rid of blanks */
      if (nums != NULL)		/* if there is anything left */
	j++;
    }
  return (j);
}


/********************* lexicon entries */

static void
read_reps (fp, epoch)
/* read the values for the word representations */
     FILE *fp;			/* pointer to the simulation file */
     int epoch;			/* epoch of the snapshot */
{
  int index,			/* word index */
     i;
  char wordstring[MAXSTRL + 1],	/* string for one word */
     rest[MAXSTRL + 1];		/* string for the representation components */

  /* initialize the completeness check */
  for (i = 0; i < nwords; i++)
    words[i].got_rep = FALSE;

  sprintf (wordstring, "%s", "");
  /* read representations until the weight definitions begin */
  while (strcasecmp (wordstring, SIMU_NETWORK_WEIGHTS))
    {
      if (fscanf (fp, "%s", wordstring) == EOF) /* read the word string */
	/* tell that cannot find weights and exit */
	read_till_keyword (fp, SIMU_NETWORK_WEIGHTS, REQUIRED);
      if (strcasecmp (wordstring, SIMU_NETWORK_WEIGHTS))
	/* if it wasn't the keyword indicating weights, it is a new word */
	{
	  /* find its index (or add to lexicon if it is not already there) */
	  index = determine_index (wordstring);
	  /* read its representation */
	  fgetline (fp, rest, MAXSTRL);
	  /* convert the string to numbers and load */
	  /* as a side effect, check the number of components */
	  if (text2floats (words[index].rep, nwordrep, rest) != nwordrep)
	    {
	      fprintf (stderr, "Wrong number of representation components\n");
	      exit (EXIT_DATA_ERROR);
	    }
	  /* make a note that a rep was read for this word */
	  words[index].got_rep = TRUE;
	}
    }
}


static int
determine_index (wordstring)
/* get the index of the word if it is in the lexicon;
   otherwise add it in the lexicon, update nwordrep, return the new index */
     char *wordstring;		/* the word to be checked */
{
  int index;

  if (strlen (wordstring) > MAXWORDL)
    {
      fprintf (stderr, "Word length exceeds array size\n");
      exit (EXIT_SIZE_ERROR);
    }
  /* see if already in lexicon */
  index = wordindex (wordstring);
  /* add the word to the lexicon if it is not already there */
  if (index == NONE)
    {
      if (nwords >= MAXWORDS)
	{
	  fprintf (stderr, "Number of words exceeds array size\n");
	  exit (EXIT_SIZE_ERROR);
	}
      sprintf (words[nwords++].chars, "%s", wordstring);
      index = nwords - 1;
      words[index].got_rep = FALSE;	/* no rep yet */
      /* keep track of where "who" and "the" end up;
         they are used in setting up targets for the stack */
      if (!strcasecmp (wordstring, WHOLABEL))
	whoindex = index;
      if (!strcasecmp (wordstring, THELABEL))
	theindex = index;
    }
  return (index);		/* the old or new index for the word */
}


static int
wordindex (wordstring)
/* find the lexicon index of a word given as a string */
     char wordstring[];		/* text word */
{
  int i;

  for (i = BLANKINDEX; i < nwords; i++) /* scan through the lexicon */
    if (!strcasecmp (wordstring, words[i].chars))
      return (i);		/* if the word found, return its index */
  return (NONE);		/* otherwise return failure signal */
}


/*********************  I/O and low-level routines  *********************/

/***************** inputfile */

static int
read_till_keyword (fp, keyword, required)
/* read text in the file until you find the given keyword
   in the beginning of the line; if the keyword is required but
   not found, terminate program; otherwise return if it was found or not */
     FILE *fp;			/* pointer to the file */
     char keyword[];		/* the keyword string */
     int required;		/* whether to abort if keyword not found */
{
  char s[MAXSTRL + 1];

  while (TRUE)
    if (fscanf (fp, "%s", s) == EOF)
      /* file ran out, keyword was not found */
      if (required)
	{
	  fprintf (stderr, "Could not find keyword: %s\n", keyword);
	  exit (EXIT_DATA_ERROR);
	}
      else
	/* that's ok; return the signal indicating that it was not found */
	return (FALSE);
    else
      /* something was read from the file */
      {
	/* if it was the keyword, return success signal */
	if (!strcasecmp (s, keyword))
	  return (TRUE);
	/* otherwise get rid of the rest of the line */
	fgl (fp);
      }
}


static char *
rid_sspace (rest)
/* read the blanks off the string */
     char rest[];		/* string to be cleaned up */
{
  int i = 0;
  while (i < strlen (rest) && (rest[i] == ' ' || rest[i] == '\t'))
    i++;
  if (i < strlen (rest))	/* if there is anything left */
    return (rest + i);
  else
    return (NULL);
}


static void
fgetline (fp, s, lim)
/* read a line of at most lim characters from the file fp into string s */
     FILE *fp;
     char *s;
     int lim;
{
  int c = 0;
  int i = 0;

  while (--lim > 0 && (c = getc (fp)) != EOF && c != '\n')
    s[i++] = (char) c;
  s[i] = '\0';
  if (lim == 0 && c != EOF && c != '\n')
    {
      fprintf (stderr, "Line character limit exceeded\n");
      exit (EXIT_SIZE_ERROR);
    }
}


static void
fgl (fp)
/* get rid of the rest of the line and the newline in the end */
     FILE *fp;
{
  int c;

  while ((c = getc (fp)) != EOF && c != '\n')
    ;
}

static int
open_file (filename, mode, fpp, required)
/* open a file, exit or return FALSE if failure */
     char *filename;		/* file name */
     FILE **fpp;		/* return the file pointer */
     char *mode;		/* open mode: "r", "a", "w" */
     int required;		/* whether to exit if cannot open */
{
  if ((*fpp = fopen (filename, mode)) == NULL)
    {
      fprintf (stderr, "Cannot open %s\n", filename);
      if (required)
	exit (EXIT_FILE_ERROR);
      else
	return (FALSE);
    }
  return (TRUE);
}


/***************** representations */

static void
check_reps_valid (epoch)
     int epoch;			/* epoch of the snapshot */
{
  int i;

  for (i = 0; i < nwords; i++)
    if (!words[i].got_rep)
      {
	fprintf (stderr, "No representation for %s in epoch %d\n",
		 words[i].chars, epoch);
	exit (EXIT_DATA_ERROR);
      }
}


static void
write_reps (fp)
/* write the current word representations into the simulation file */
     FILE *fp;			/* pointer to the simulation file */
{
  int i, j;
  for (i = 0; i < nwords; i++)
    {
      fprintf (fp, "%s", words[i].chars);
      for (j = 0; j < nwordrep; j++)
	fprintf (fp, " %f", words[i].rep[j]);
      fprintf (fp, "\n");
    }
}


static void
randomize_reps (par1, par2)
/* set up initial random word representations */
     double par1, par2;		/* distribution parameters */
{
  int i, j;
  for (i = 0; i < nwords; i++)
    for (j = 0; j < nwordrep; j++)
      randfun (NULL, &words[i].rep[j], par1, par2);
}


/***************** weights */

static void
readfun (fp, place, par1, par2)
/* function for reading a float from a file */
/* given as a parameter for reading in weights */
     FILE *fp;			/* weight file */
     double *place,		/* return the float read */
       par1, par2;		/* unused */
{
  if (fscanf (fp, "%lf", place) != 1)
    {
      fprintf (stderr, "Error reading weights\n");
      exit (EXIT_DATA_ERROR);
    }
}


static void
writefun (fp, place, par1, par2)
/* function for writing a float to a file */
/* given as a parameter for writing weights */
     FILE *fp;			/* weight file */
     double *place,		/* the float to be written */
       par1, par2;		/* unused */
{
  fprintf (fp, "%f\n", *place);
}


static void
randfun (fp, place, low, span)
/* function for assigning a random float value */
/* given as a parameter for initializing weights and word reps */
     FILE *fp;			/* not used */
     double *place,		/* the float to be written */
       low, span;		/* bias and scale for the range */
{
  *place = low + span * f01rnd ();
}


/***************** lexicon comparison */

int
find_nearest (rep, words, nrep, nwords)
/* find the index of the nearest representation in the lexicon */
     double rep[];		/* word representation */
     WORDSTRUCT words[];	/* lexicon */
     int nrep,			/* rep dimension */
       nwords;			/* number of words in lexicon */
{
  int i,
    bestindex = BLANKINDEX;	/* index of the closest rep so far */
  double lbest,			/* closest distance so far */
    dist;			/* last distance */

  lbest = LARGEFLOAT;
  /* linearly search through the lexicon */
  /* starting from the blank */
  for (i = BLANKINDEX; i < nwords; i++)
    {
      dist = distance (NULL, rep, words[i].rep, nrep);
      if (dist < lbest)		/* keep track of the best one so far */
	{
	  bestindex = i;
	  lbest = dist;
	}
    }
  return (bestindex);
}


static double
distance (foo, v1, v2, nrep)
/* compute the Euclidean distance of two nrep dimensional vectors */
     double v1[], v2[];		/* the vectors */
     int nrep;			/* their dimensionality */
     int *foo;			/* not used */
{
  double sum = 0.0;
  int i;

  for (i = 0; i < nrep; i++)
    sum += (v1[i] - v2[i]) * (v1[i] - v2[i]);
  return (sqrt (sum));
}


/**************** general */

int
imin (a, b)
  /* return the smaller of two ints */
     int a, b;
{
  return ((a > b) ? b : a);
}


int
imax (a, b)
  /* return the larger of two ints */
     int a, b;
{
  return ((a > b) ? a : b);
}


/*********************  simulation management  *************************/

static void
get_current_params (epoch)
/* determine the current learning rates and which modules to run
   depending on epoch and phase definitions */
     int epoch;			/* current epoch */
{
  int phase,
    modi;			/* module number */

  /* first figure out what phase we are in */
  for (phase = nphase - 1; phase > 0 && phaseends[phase - 1] >= epoch; phase--)
    ;
  eta = etas[phase];			   /* current weight learning rate */
  wordeta = wordetaratios[phase] * eta;	   /* current word rep learning rate */
  for (modi = 0; modi < NMODULES; modi++)
    running[modi] = runnings[modi][phase]; /* current modules to be run */
}


static void
save_current (epoch)
/* save the current word representations and weights */
     int epoch;			/* current epoch */
{
  FILE *fp;			/* simulation file pointer */

  /* open the file for appending */
  open_file (simufile, "a", &fp, REQUIRED);

  /* first write the current epoch number and network errors */
  fprintf (fp, "%s %d\n", SIMU_EPOCH, epoch);
  fprintf (fp, "%s", SIMU_NETWORK_ERRORS);
  write_error (fp);
  fprintf (fp, "%s\n", SIMU_WORD_REPRESENTATIONS);
  write_reps (fp);
  fprintf (fp, "%s\n", SIMU_NETWORK_WEIGHTS);
  iterate_weights (writefun, fp, NOT_USED, NOT_USED);
  fclose (fp);
  /* update the next snapshot epoch */
  update_nextsnapshot (epoch);
}


static void
update_nextsnapshot (epoch)
/* find out when the next snapshot should be saved */
     int epoch;			/* current epoch */
{
  while (nextsnapshot < nsnaps)
    {
      if (snapshots[nextsnapshot] <= epoch)
	nextsnapshot++;
      else
	break;
    }
}


static void
shuffle ()
/* change the order of training sentences */
{
  int i;
  int dest, destvalue;
  /* sequentially find a new random place for each sentence */
  for (i = 0; i < nsents; i++)
    {
      dest = lrand48 () % nsents;
      destvalue = shuffletable[dest];
      shuffletable[dest] = shuffletable[i];
      shuffletable[i] = destvalue;
    }
}


static double
f01rnd ()
{
  /* random double between 0.0 and 1.0 */
  return (drand48 ());
}
