/* File: nets.c
 *
 * Network propagation
 *
 * Copyright (C) 1994 Risto Miikkulainen
 *
 *  This software can be copied, modified and distributed freely for
 *  educational and research purposes, provided that this notice is included
 *  in the code, and the author is acknowledged in any materials and reports
 *  that result from its use. It may not be used for commercial purposes
 *  without expressed permission from the author.
 *
 * $Id: nets.c,v 1.64 1994/09/21 11:10:16 risto Exp $
 */

#include <stdio.h>
#include <math.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>

#include "defs.h"
#include "globals.c"


/******************** Function prototypes ************************/

/* global functions */
#include "prototypes.h"

/* functions local to this file */
static void process_sentence __P ((int senti));
static void pars_prop __P ((int seqi));
static void setup_pars_target __P ((int targetnums[]));
static void setup_pars_input __P ((int wordindex));
static void setup_pars_prevhid __P ((void));
static void init_parser __P ((void));
static void segmenter_prop __P ((int seqi));
static void seg_cleanup_target __P((int nounindex, double tgtrep[]));
static int get_nextword __P ((int senti, int fragi, int seqi));
static void setup_seg_input __P ((int nextwordindex));
static void setup_seg_target __P ((int senti, int fragi, int seqi,
				   int wordindex, int nextwordindex));
static void stack_push_prop __P ((void));
static void stack_pop_prop __P ((void));
static void set_stack_inprep __P ((void));
static void set_stack_tgtrep __P ((void));
static void run_stack_one __P ((double therep[]));
static void stack_prop_through __P ((void));
static void init_stack_targets __P ((void));
static void forward_propagate_layer __P ((double inprep[], int ninprep,
					  double outrep[], int noutrep,
					  double weights[MAXLAYER][MAXLAYER]));
static void backward_propagate __P ((int modi));
static double clip __P ((double activity));


/******************** static variables ******************** */

/* symbolic stack (for keeping track of stack targets) */
static double symstack[MAXFRAGS][MAXLAYER];

/* targets for the stack */
static double
  stacktgts1[MAXWORDS][MAXLAYER],		/* first-level targets */
  stacktgts2[MAXWORDS][MAXWORDS][MAXLAYER];	/* second-level targets */


/******************** main sentence processing loop  ************************/

void
iterate_sentences ()
{
  int senti;			/* sentence number */
  static int
    stackdata_update = TRUE;	/* stack I/O data needs update */

  /* to train/test the parser or the segmenter, or the stack concurrently
     with the parser, we need to run through the sentence set */
  if (running[PARSMOD] || running[SEGMOD] ||
      (running[STACKMOD] && !stacknouns))
    for (senti = 0; senti < nsents; senti++)
      process_sentence (shuffletable[senti]);
  /* stack I/O data needs updating at first or if the parser has changed */
  if (!parseronly && stacknouns && !testing && !stackdata_update)
    stackdata_update = running[PARSMOD];

  if (!parseronly && running[STACKMOD] && stacknouns)
    /* stack is trained/tested in isolation through the nounlist */
    {
      /* check if we need to (re)compute the stack training data */
      if (stackdata_update)
	{
	  init_stack_targets ();
	  stackdata_update = FALSE;
	}
      run_stack_all ();
    }
}


static void
process_sentence (senti)
/* propagate fragments one at a time, train parser, seg and stack if
   necessary, collect statistics */
     int senti;			/* sentence number */
{
  int fragi,			/* fragment number */
    seqi,			/* word number (in fragment) */
    k;
  int wordindex,		/* index of the current word */
    nextwordindex;		/* index of the next word */

  init_parser ();		/* clean up parser input */
  top = 0;			/* start with an empty stack */
  for (fragi = 0; fragi < sents[senti].nfrag; fragi++)
    /* go through all fragments of the sentence */
    {
      if (running[PARSMOD])
	setup_pars_target (sents[senti].frag[fragi].tgts);
      for (seqi = 0; seqi < sents[senti].frag[fragi].nseq; seqi++)
	/* go through all words in the fragment sequence */
	{
	  /* propagate through parser */
	  wordindex = sents[senti].frag[fragi].inps[seqi];
	  setup_pars_input (wordindex);
	  if (running[PARSMOD])
	    pars_prop (seqi);
	  else
	    /* if parser is not running, we only need the hidden layer rep */
	    forward_propagate_layer (inprep[PARSMOD], ninprep[PARSMOD],
				     hidrep[PARSMOD], nhidrep[PARSMOD],
				     wih[PARSMOD]);
	  if (!parseronly)
	    /* propagate through segmenter */
	    {
	      nextwordindex = get_nextword (senti, fragi, seqi);
	      setup_seg_target (senti, fragi, seqi, wordindex, nextwordindex);
	      if (running[SEGMOD])
		{
		  setup_seg_input (nextwordindex);
		  segmenter_prop (seqi);
		  seg_cumulate_stats (fragi, seqi);
		  cont_cumulate_stats (fragi, seqi);
		}
	    }
	  setup_pars_prevhid ();
	}

      if (!parseronly)
	if (sents[senti].frag[fragi].pushc)
	  /* push required after reading the fragment */
	  {
	    /* push into the symbolic stack */
	    for (k = 0; k < nhidrep[PARSMOD]; k++)
	      symstack[top][k] = hidrep[PARSMOD][k];
	    top++;
	    if (running[STACKMOD] && !stacknouns)
	      /* propagate into stack */
	      {
		set_stack_inprep ();
		if (testing)
		  stack_push_prop ();
		else
		  /* train the stack */
		  {
		    for (k = 0; k < noutrep[STACKMOD]; k++)
		      tgtrep[STACKMOD][k] = inprep[STACKMOD][k];
		    stack_prop_through ();
		    stack_cumulate_stats (NOT_USED);
		  }
	      }
	  }
	else if (sents[senti].frag[fragi].popc)
	  /* pop required after reading the fragment */
	  {
	    /* pop the symbolic stack */
	    top--;
	    /* propagate and set up parser prevhidrep */
	    if (!chain || !(running[STACKMOD] && !stacknouns))
	      {
		if (testing && running[STACKMOD] && !stacknouns)
		  {
		    /* propagate out of the stack */
		    set_stack_tgtrep ();
		    stack_pop_prop ();
		    stack_cumulate_stats (fragi);
		  }
		/* use the correct representations */
		for (k = 0; k < nhidrep[PARSMOD]; k++)
		  inprep[PARSMOD][nwordrep + k] = symstack[top][k];
	      }
	    else
	      {
		/* propagate out of the stack */
		set_stack_tgtrep ();
		stack_pop_prop ();
		if (testing)
		  stack_cumulate_stats (fragi);
		for (k = 0; k < nhidrep[PARSMOD]; k++)
		  inprep[PARSMOD][nwordrep + k] = outrep[STACKMOD][k];
	      }
	  }
      if (sents[senti].frag[fragi].outc && running[PARSMOD])
	/* output required; cumulate error statistics */
	pars_cumulate_stats (senti, fragi);
    }
}


/*************************   parser   ****************************/

static void
pars_prop (seqi)
/* propagate the current input through the parser */
     int seqi;			/* current step in the sequence */
{
  int i, j, modi = PARSMOD;
  double pars_error = 0.0;	/* error in current propagation */

  /* forward propagate from inprep */
  forward_propagate_layer (inprep[modi], ninprep[modi],
			   hidrep[modi], nhidrep[modi], wih[modi]);
  forward_propagate_layer (hidrep[modi], nhidrep[modi],
			   outrep[modi], noutrep[modi], who[modi]);
  if (displaying)
    {
      /* calculate the error sum and average it per unit */
      for (i = 0; i < noutputs[modi]; i++)
	for (j = 0; j < nwordrep; j++)
	  pars_error += fabs (words[targets[modi][i]].rep[j] -
			      outrep[modi][i * nwordrep + j]);
      /* display error on the log line */
      sprintf (net[modi].log, "Step %d: Eavg %.3f", seqi + 1,
	       pars_error / noutrep[modi]);
      /* update the input sequence display */
      sprintf (net[modi].sequence, "%s %s",
	       net[modi].sequence, net[modi].newitem);
      /* print the current activations */
      display_current_parser (modi, !testing);
      wait_and_handle_events ();
    }

  if (!testing)
    backward_propagate (modi);
}


static void
setup_pars_target (targetnums)
/* get the current word representations from the lexicon
   and load it in the parser's target layer */
     int targetnums[];		/* target data indices */
{
  int i, j, modi = PARSMOD;

  for (i = 0; i < noutputs[modi]; i++)
    {
      for (j = 0; j < nwordrep; j++)
	tgtrep[modi][i * nwordrep + j] = words[targetnums[i]].rep[j];
      /* remember the current target indices (for display) */
      targets[modi][i] = targetnums[i];
    }
}


static void
setup_pars_input (wordindex)
/* get the current word rep and load it in the parser's input layer */
     int wordindex;		/* index of the current input word */
{
  int j, modi = PARSMOD;

  inputs[modi][0] = wordindex;	/* remember the index (for display) */
  for (j = 0; j < nwordrep; j++)
    inprep[modi][j] = words[wordindex].rep[j];
}


static void
setup_pars_prevhid ()
/* decide where to take the previous hidden layer activation */
{
  int i, k, modi = PARSMOD;

  if (parseronly)
    /* just use the previous hidden layer activation */
    for (k = 0; k < nhidrep[modi]; k++)
      inprep[modi][nwordrep + k] = hidrep[modi][k];
  else if (!chain || !running[SEGMOD])
    /* use the correct segmenter output */
    for (k = 0; k < nhidrep[modi]; k++)
      inprep[modi][nwordrep + k] = tgtrep[SEGMOD][k];
  else
    /* use the actual segmenter output */
    for (i = 0; i < nhidrep[modi]; i++)
      inprep[modi][nwordrep + i] = outrep[SEGMOD][i];
}


static void
init_parser ()
/* clean up parser activation and display vars */
{
  int i, modi = PARSMOD;
  sprintf (net[modi].sequence, "%s", "");
  sprintf (net[modi].newitem, "%s", "");
  /* start with a blank previous hidden layer */
  for (i = 0; i < nhidrep[modi]; i++)
    inprep[modi][nwordrep + i] = 0.0;
}


/******************** segmenter ************************/

static void
segmenter_prop (seqi)
/* propagate activation through the segmenter */
     int seqi;			/* current step in the sequence */
{
  int i, modi = SEGMOD;
  double seg_error = 0.0,	/* error in the segmenter output */
    cont_error = 0.0;		/* error in the control output */

  /* propagate activation forward through the segmenter */
  forward_propagate_layer (inprep[modi], ninprep[modi],
			   hidrep[modi], nhidrep[modi], wih[modi]);
  forward_propagate_layer (hidrep[modi], nhidrep[modi],
			   outrep[modi], noutrep[modi], who[modi]);
  if (displaying)
    {
      /* calculate the average error per output unit */
      for (i = 0; i < nhidrep[PARSMOD]; i++)
	seg_error += fabs (tgtrep[modi][i] - outrep[modi][i]);
      for (i = nhidrep[PARSMOD]; i < nhidrep[PARSMOD] + NCONTROL; i++)
	cont_error += fabs (tgtrep[modi][i] - outrep[modi][i]);
      /* print the errors in the log line */
      sprintf (net[modi].log, "Step %d: Eavg %.3f",
	       seqi + 1, seg_error / nhidrep[PARSMOD]);
      sprintf (net[modi].log, "%s %.3f",
	       net[modi].log, cont_error / NCONTROL);
      /* print the current activations */
      display_current_segmenter (modi, !testing);
      wait_and_handle_events ();
    }
  if (!testing)
    backward_propagate (modi);
}


static void
seg_cleanup_target (nounindex, tgtrep)
/* figure out the segmenter clean-up target for this noun,
   that is, the hidden layer pattern after "THE <noun>", by propagating
   through parser's weights (don't change parser activities though) */
       int nounindex;		/* index of the head noun */
       double tgtrep[];		/* result returned here */
{
  int i, j;
  double inprep[MAXLAYER];	/* stands for the parser input rep */

  /* propagate "the" to the target rep through the input weights */
  for (j = 0; j < nwordrep; j++)
    inprep[j] = words[theindex].rep[j];
  forward_propagate_layer (inprep, nwordrep,
			   tgtrep, nhidrep[PARSMOD], wih[PARSMOD]);

  /* update previous hidden layer */
  for (i = 0; i < nhidrep[PARSMOD]; i++)
    inprep[nwordrep + i] = tgtrep[i];

  /* propagate the <noun> to the target rep through the input weights */
  for (j = 0; j < nwordrep; j++)
    inprep[j] = words[nounindex].rep[j];
  forward_propagate_layer (inprep, ninprep[PARSMOD],
			   tgtrep, nhidrep[PARSMOD], wih[PARSMOD]);
}


static int
get_nextword (senti, fragi, seqi)
/* figure out the index of the next word in the sentence */
     int senti,			/* sentence number */
       fragi,			/* fragment number */
       seqi;			/* word number in the fragment */
{
  if (seqi + 1 < sents[senti].frag[fragi].nseq)
    /* next word in the same fragment */
    return (sents[senti].frag[fragi].inps[seqi + 1]);
  else if (fragi + 1 < sents[senti].nfrag)
    /* first word in the next fragment */
    return (sents[senti].frag[fragi + 1].inps[0]);
  else
    /* no more words in this sentence */
    return (BLANKINDEX);
}


static void
setup_seg_input (nextwordindex)
/* get the current rep for next word, parser hidrep and load them up
   in the segmenter's input layer */
     int nextwordindex;
{
  int j, modi = SEGMOD;

  for (j = 0; j < nwordrep; j++)
    inprep[modi][j] = words[nextwordindex].rep[j];
  for (j = 0; j < nhidrep[PARSMOD]; j++)
    inprep[modi][nwordrep + j] = hidrep[PARSMOD][j];
  /* remember the word index (for display) */
  inputs[modi][0] = nextwordindex;
}


static void
setup_seg_target (senti, fragi, seqi, wordindex, nextwordindex)
/* determine what the segmenter target should be */
     int senti,			/* sentence number */
       fragi,			/* fragment number */
       seqi,			/* word number in the fragment */
       wordindex,		/* index of the current word */
       nextwordindex;		/* index of the next word */
{
  int k, modi = SEGMOD;

  /* make all control targets 0 to begin with */
  for (k = nhidrep[PARSMOD]; k < nhidrep[PARSMOD] + NCONTROL; k++)
    tgtrep[modi][k] = 0.0;

  if (nextwordindex == whoindex)
    /* relative clause transition; clean up context */
    seg_cleanup_target (wordindex, tgtrep[modi]);
  else if (nextwordindex == BLANKINDEX)
    /* end of sentence; don't care */
    for (k = 0; k < nhidrep[PARSMOD]; k++)
      tgtrep[modi][k] = 0.5;
  else if (seqi == sents[senti].frag[fragi].nseq - 1
	   && sents[senti].frag[fragi].popc)
    /* popping; don't care */
    for (k = 0; k < nhidrep[PARSMOD]; k++)
      tgtrep[modi][k] = 0.5;
  else
    /* propagate parser's hidden rep without modification */
    for (k = 0; k < nhidrep[PARSMOD]; k++)
      tgtrep[modi][k] = hidrep[PARSMOD][k];

  if (seqi == sents[senti].frag[fragi].nseq - 1)
    /* if this was the last word in the fragment, set up control outputs */
    {
      tgtrep[modi][nhidrep[PARSMOD]] = sents[senti].frag[fragi].pushc;
      tgtrep[modi][nhidrep[PARSMOD] + 1] = sents[senti].frag[fragi].popc;
      tgtrep[modi][nhidrep[PARSMOD] + 2] = sents[senti].frag[fragi].outc;
    }
}


/*********************  stack  ******************************/

/************* propagation when chained */

static void
stack_push_prop ()
/* propagate into the stack hidden layer and display */
{
  int k, modi = STACKMOD;

  forward_propagate_layer (inprep[modi], ninprep[modi],
			   hidrep[modi], nhidrep[modi], wih[modi]);
  if (displaying)
    {
      /* display the depth of stack */
      sprintf (net[modi].log, "Push level %d", top);
      /* clean up the output */
      for (k = 0; k < noutrep[modi]; k++)
	outrep[modi][k] = 0.0;
      for (k = 0; k < nhidrep[PARSMOD]; k++)
	tgtrep[modi][k] = 0.0;
      /* display the current activations */
      display_current_stack (modi, !testing);
      wait_and_handle_events ();
    }
}


static void
stack_pop_prop ()
/* propagate from stack hidden to output and display; update hidden */
{
  int i, k, modi = STACKMOD;
  double stack_error = 0.0;	/* current output error */

  forward_propagate_layer (hidrep[modi], nhidrep[modi],
			   outrep[modi], noutrep[modi], who[modi]);
  if (displaying)
    {
      for (i = 0; i < nhidrep[PARSMOD]; i++)
	stack_error += fabs (tgtrep[modi][i] - outrep[modi][i]);
      /* display the depth of stack in the log line with error */
      sprintf (net[modi].log, "Pop level %d: Eavg %.3f",
	       top + 1, stack_error / nhidrep[PARSMOD]);
      /* clean up the input layer activation */
      for (k = 0; k < ninprep[modi]; k++)
	inprep[modi][k] = 0.0;
      /* clean up the stackhidrep target */
      for (k = nhidrep[PARSMOD]; k < noutrep[modi]; k++)
	tgtrep[modi][k] = 0.0;
      /* display the current activations */
      display_current_stack (modi, !testing);
      wait_and_handle_events ();
    }
  /* update the RAAM hidden layer */
  for (k = 0; k < nhidrep[modi]; k++)
    hidrep[modi][k] = outrep[modi][k + nhidrep[PARSMOD]];
}


static void
set_stack_inprep ()
/* decide what the input representation should be and set it up */
{
  int k, modi = STACKMOD;

  /* the left side is always parser's hidden layer rep */
  for (k = 0; k < nhidrep[PARSMOD]; k++)
    inprep[modi][k] = hidrep[PARSMOD][k];
  if (top == 1)
    /* if stack is empty, make right side "don't care" */
    for (k = nhidrep[PARSMOD]; k < ninprep[modi]; k++)
      inprep[modi][k] = 0.5;
  else
    /* otherwise use current stack rep */
    for (k = 0; k < nhidrep[modi]; k++)
      inprep[modi][nhidrep[PARSMOD] + k] = hidrep[modi][k];
}


static void
set_stack_tgtrep ()
/* set up stack target according to the correct rep in the symbolic stack */
{
  int k, modi = STACKMOD;

  for (k = 0; k < nhidrep[PARSMOD]; k++)
    tgtrep[modi][k] = symstack[top][k];
}


/************* stack in isolation */

void
run_stack_all ()
/* run the stack with all possible center embeddings
   that the nounlist specify up to four clauses */
{
  int j, k, p, jj, pp;		/* noun counters */

  for (k = 0; k < nnouns; k++)
    for (j = 0; j < nnouns; j++)
      for (p = 0; p < nnouns; p++)
	for (jj = 0; jj < nnouns; jj++)
	  for (pp = 0; pp < nnouns; pp++)
	    {
	      /* straight center embeddings */
	      top = 1;
	      run_stack_one (stacktgts1[nouns[k]]);
	      top++;
	      run_stack_one (stacktgts2[nouns[j]][nouns[p]]);
	      top++;
	      run_stack_one (stacktgts2[nouns[jj]][nouns[pp]]);

	      /* center embeddings after a tail embedding */
	      top = 1;
	      run_stack_one (stacktgts2[nouns[j]][nouns[p]]);
	      top++;
	      run_stack_one (stacktgts2[nouns[jj]][nouns[pp]]);
	    }
}


static void
run_stack_one (therep)
/* run the stack in one case of embedding */
     double therep[];		/* representation to be pushed/popped */
{
  int i, modi = STACKMOD;

  for (i = 0; i < nhidrep[PARSMOD]; i++)
    hidrep[PARSMOD][i] = therep[i];
  set_stack_inprep ();

  /* set up the target */
  for (i = 0; i < noutrep[modi]; i++)
    tgtrep[modi][i] = inprep[modi][i];

  stack_prop_through ();
  stack_cumulate_stats (NOT_USED);
}


static void
stack_prop_through ()
/* propagate forward through the stack and maybe backpropagate, display */
{
  int i, modi = STACKMOD,
    nstackstatsrep;		/* part where stats are collected */
  double stack_error = 0.0;	/* current output error */

  /* forward propagate through the stack */
  forward_propagate_layer (inprep[modi], ninprep[modi],
			   hidrep[modi], nhidrep[modi], wih[modi]);
  forward_propagate_layer (hidrep[modi], nhidrep[modi],
			   outrep[modi], noutrep[modi], who[modi]);

  if (displaying)
    {
      if (testing)
	nstackstatsrep = nhidrep[PARSMOD];
      else
	nstackstatsrep = noutrep[modi];
	  
      /* calculate average error per output unit */
      for (i = 0; i < nstackstatsrep; i++)
	stack_error += fabs (tgtrep[modi][i] - outrep[modi][i]);
      /* display depth of stack and current error */
      sprintf (net[modi].log, "Level %d: Eavg %.3f",
	       top, stack_error / nstackstatsrep);
      /* display current activations */
      display_current_stack (modi, !testing);
      wait_and_handle_events ();
    }
  if (!testing)
    backward_propagate (modi);
}


static void
init_stack_targets ()
/* given the current parser, figure out all possible representations
   that could be pushed to the stack */
{
  int i, j, k, modi = PARSMOD;
  double savehidrep[MAXLAYER];	/* first-level push */

  for (k = 0; k < nnouns; k++)
    {
      /* first-level embeddings for all nouns */
      /* start with a blank parser previous hidden layer */
      for (i = 0; i < nhidrep[modi]; i++)
	inprep[modi][nwordrep + i] = 0.0;
      /* propagate "the" */
      setup_pars_input (theindex);
      forward_propagate_layer (inprep[modi], ninprep[modi],
			       hidrep[modi], nhidrep[modi], wih[modi]);
      for (i = 0; i < nhidrep[modi]; i++)
	inprep[modi][nwordrep + i] = hidrep[modi][i];
      /* propagate the first noun */
      setup_pars_input (nouns[k]);
      forward_propagate_layer (inprep[modi], ninprep[modi],
			       hidrep[modi], nhidrep[modi], wih[modi]);
      /* setup first level target "the <noun1>" */
      for (i = 0; i < nhidrep[modi]; i++)
	stacktgts1[nouns[k]][i] = hidrep[modi][i];

      /* all second-level embeddings for each <noun1> */
      /* setup previous hidden layer */
      for (i = 0; i < nhidrep[modi]; i++)
	inprep[modi][nwordrep + i] = hidrep[modi][i];
      /* propagate "who" */
      setup_pars_input (whoindex);
      forward_propagate_layer (inprep[modi], ninprep[modi],
			       hidrep[modi], nhidrep[modi], wih[modi]);
      for (i = 0; i < nhidrep[modi]; i++)
	inprep[modi][nwordrep + i] = hidrep[modi][i];
      /* propagate "the" */
      setup_pars_input (theindex);
      forward_propagate_layer (inprep[modi], ninprep[modi],
			       hidrep[modi], nhidrep[modi], wih[modi]);
      /* save the current rep so that all second level embeddings
	 can be calculated */
      for (i = 0; i < nhidrep[modi]; i++)
	savehidrep[i] = hidrep[modi][i];
      for (j = 0; j < nnouns; j++)
	/* all second-level embeddings */
	{
	  for (i = 0; i < nhidrep[modi]; i++)
	    inprep[modi][nwordrep + i] = savehidrep[i];
	  /* propagate the second-level noun */
	  setup_pars_input (nouns[j]);
	  forward_propagate_layer (inprep[modi], ninprep[modi],
				   hidrep[modi], nhidrep[modi], wih[modi]);
	  /* setup second level target "the <noun1> who the <noun2>" */
	  for (i = 0; i < nhidrep[modi]; i++)
	    stacktgts2[nouns[k]][nouns[j]][i] = hidrep[modi][i];
	}
    }
}


/*********************  propagation  ******************************/

static void
forward_propagate_layer (inprep, ninprep, outrep, noutrep, weights)
/* propagate from a layer of units through weights to another layer
   and take the sigmoid of the weighted sum */
     int ninprep, noutrep;	/* number of units in the input and output */
     double inprep[], outrep[],	/* input and output layer activations */
       weights[MAXLAYER][MAXLAYER];	/* weights from input to output */
{
  int i, j;

  for (j = 0; j < noutrep; j++)
    outrep[j] = 0.0;
  for (i = 0; i < ninprep; i++)
    for (j = 0; j < noutrep; j++)
      outrep[j] += inprep[i] * weights[i][j];
  for (j = 0; j < noutrep; j++)
    outrep[j] = 1.0 / (1.0 + exp (-outrep[j]));
}


static void
backward_propagate (modi)
/* determine output error, backpropagate error signal, change weights,
   and change input word representations if required */
     int modi;			/* module number */
{
  int i, j, k;
  double
    outsig[MAXLAYER],		/* error signal at output layer */
    outfact[MAXLAYER],		/* output weight change factor */
    hidsumsig[MAXLAYER],	/* weighted sum of error signals in hidden l */
    hidsig[MAXLAYER],		/* error signal at hidden layer */
    inpfact[MAXLAYER],		/* input weight change factor */
    inpsumsig[MAXLAYER];	/* weighted sum of error signals in input l */

  /* clean up the error signal sums */
  for (k = 0; k < nhidrep[modi]; k++)
    hidsumsig[k] = 0.0;
  if (fgrepping[modi])
    for (i = 0; i < ninputs[modi]; i++)
      for (j = 0; j < nwordrep; j++)
	inpsumsig[i * nwordrep + j] = 0.0;

  /* change output weights */
  for (i = 0; i < noutrep[modi]; i++)
    {
      /* figure out output error signals */
      outsig[i] = (tgtrep[modi][i] - outrep[modi][i]) *
	outrep[modi][i] * (1.0 - outrep[modi][i]);
      outfact[i] = eta * outsig[i];
    }
  for (k = 0; k < nhidrep[modi]; k++)
    for (i = 0; i < noutrep[modi]; i++)
      {
	/* propagate error signal to hidden layer */
	hidsumsig[k] += outsig[i] * who[modi][k][i];
	/* then change weights */
	who[modi][k][i] += outfact[i] * hidrep[modi][k];
      }

  /* change input weights */
  for (k = 0; k < nhidrep[modi]; k++)
    {
      /* calculate hidden error signal (multiply by act derivative) */
      hidsig[k] = hidsumsig[k] * hidrep[modi][k] * (1.0 - hidrep[modi][k]);
      inpfact[k] = eta * hidsig[k];
    }
  /* word input part */
  for (i = 0; i < ninputs[modi] * nwordrep; i++)
    for (k = 0; k < nhidrep[modi]; k++)
      {
	if (fgrepping[modi])
	  /* propagate error signal to input layer */
	  inpsumsig[i] += hidsig[k] * wih[modi][i][k];
	/* then change input weights */
	wih[modi][i][k] += inpfact[k] * inprep[modi][i];
      }
  /* other than word input part */
  for (i = ninputs[modi] * nwordrep; i < ninprep[modi]; i++)
    for (k = 0; k < nhidrep[modi]; k++)
      wih[modi][i][k] += inpfact[k] * inprep[modi][i];

  /* update the lexicon representations */
  if (fgrepping[modi])
    for (i = 0; i < ninputs[modi]; i++)
      if (inputs[modi][i] != BLANKINDEX)
	{
	  k = i * nwordrep;
	  for (j = 0; j < nwordrep; j++)
	    words[inputs[modi][i]].rep[j] =
	      clip (inprep[modi][k + j] + wordeta * inpsumsig[k + j]);
	}
}


static double
clip (activity)
/* limit word representation component within 0 and 1 */
     double activity;		/* word representation value */
{
  if (activity < 0.0)
    return (0.0);
  else if (activity > 1.0)
    return (1.0);
  else
    return (activity);
}


/*********************  initializations ******************************/

void
iterate_weights (dofun, fp, par1, par2)
/* read, write, or randomize all weights */
     void (*dofun) ();		/* read, write, or rand function */
     FILE *fp;			/* simulation file pointer */
     double par1, par2;		/* distribution parameters */
{
  int i, k, modi;

  for (modi = 0; modi < NMODULES; modi++)
    if (!parseronly || modi == PARSMOD)
      {
	/* input-to-hidden weights */
	for (i = 0; i < ninprep[modi]; i++)
	  for (k = 0; k < nhidrep[modi]; k++)
	    (*dofun) (fp, &wih[modi][i][k], par1, par2);

	/* hidden-to-output weights */
	for (i = 0; i < nhidrep[modi]; i++)
	  for (k = 0; k < noutrep[modi]; k++)
	    (*dofun) (fp, &who[modi][i][k], par1, par2);
      }
}


void
clear_network (modi)
/* clear the activations and input/target indices and display strings */
     int modi;				/* module number */
{
  int i;

  for (i = 0; i < ninputs[modi]; i++)
    inputs[modi][i] = BLANKINDEX;		/* blank word */
  for (i = 0; i < noutputs[modi]; i++)
    targets[modi][i] = BLANKINDEX;
  for (i = 0; i < ninprep[modi]; i++)
    inprep[modi][i] = 0.0;
  for (i = 0; i < noutrep[modi]; i++)
    outrep[modi][i] = 0.0;
  for (i = 0; i < noutrep[modi]; i++)
    tgtrep[modi][i] = 0.0;
  for (i = 0; i < nhidrep[modi]; i++)
    hidrep[modi][i] = 0.0;
  sprintf (net[modi].sequence, "%s", "");	/* clear word sequence */
  sprintf (net[modi].newitem, "%s", "");	/* clear new item in seq */
  sprintf (net[modi].log, "%s", "");		/* clear log line */
}
