#include "mpi.h"

#define TRUE 1
#define MAX_MESSAGE_LENGTH 1000000

void time_allgather
               ( MPI_Datatype datatype,    /* datatype to time              */
                 int ntrials,              /* number of trials              */
                 int *lengths,             /* vector lengths to try         */
		 double *timings,          /* timings                       */
                 MPI_Comm comm,            /* communicator                  */
		 int version )
{
  int  
    me, nprocs, trial, i, j, k, n, typesize;
  int 
    *data_in, *data_out;
  double 
    time;
  MPI_Status
    status;

  /* me = this node's index in the communicator */
  MPI_Comm_rank( comm, &me );          
  MPI_Comm_size( comm, &nprocs );          

  MPI_Type_size( datatype, &typesize );

  data_in = (int *) malloc( lengths[ntrials-1] * typesize );
  data_out = (int *) malloc( lengths[ntrials-1] * typesize );

  for ( trial=0; trial<ntrials; trial++ ){
    n = lengths[ trial ];
    /* Execute each trial twice, timing the second one */
    for ( j=0; j<2; j++ ){
      /* Initialize data */
      for ( i=0; i<n/nprocs; i++ ) data_in[i] = me*(n/nprocs) + i;

      for ( i=0; i<n; i++ ) data_out[i] = -1;

      /* start timing */
      MPI_Barrier( comm );
      time = MPI_Wtime();

      switch( version ){
      case 0:
	MPI_Allgather( data_in, n/nprocs, datatype, 
		      data_out, n/nprocs, datatype, comm );
	break;
      case 1:
	my_allgather( data_in, n/nprocs, datatype, 
		      data_out, n/nprocs, datatype, comm );
	break;
      case 2:
	my_allgather2( data_in, n/nprocs, datatype, 
		      data_out, n/nprocs, datatype, comm );
	break;
      case 3:
	my_allgather3( data_in, n/nprocs, datatype, 
		      data_out, n/nprocs, datatype, comm );
	break;
      }
      
      /* stop timing */
      time = MPI_Wtime() - time;

      /* check if result is correct */
#ifndef NOCHECK      
      for ( i=0; i<(n/nprocs)*nprocs; i++ )
	if ( data_out[ i ] != i ){
	  printf( "allgather %d data_out[ %d ] = %d \n", me, i, data_out[ i ] );
	}
#endif
    }

    /* determine max time spent in bcast over all nodes */
    MPI_Allreduce( &time, &timings[ trial ], 1, MPI_DOUBLE, MPI_MAX, 
		   comm );
  }

  free( data_in );
  free( data_out );

  return;
}
