#include "mpi.h"

#define TRUE 1
#define MAX_MESSAGE_LENGTH 1000000

void time_bounce( MPI_Datatype datatype,    /* datatype to time              */
                  int ntrials,              /* number of trials              */
                  int *lengths,             /* vector lengths to try         */
                  int nrepeats,             /* number of bounces to time     */
		  double *timings,          /* timings                       */
                  MPI_Comm comm,            /* communicator                  */
		  int version )
{
  int  
    me, nprocs, trial, i, j, k, n, typesize, partner;
  int 
    *data;
  double 
    time;
  MPI_Status
    status;
  MPI_Request
    request;

  /* me = this node's index in the communicator */
  MPI_Comm_rank( comm, &me );          

  /* np = number of nodes in the communicator */
  MPI_Comm_size( comm, &nprocs );          
  
  if ( nprocs < 2 ) {
    printf( "time_bounce must be executed with more than 2 nodes\n" );
    exit( 0 );
  }
   
  partner = nprocs/2+1;

  MPI_Type_size( datatype, &typesize );

  data = (void *) malloc( lengths[ntrials-1] * typesize );

  for ( trial=0; trial<ntrials; trial++ ){
    n = lengths[ trial ];
    /* Execute each trial twice, timing the second one */
    for ( j=0; j<2; j++ ){
      /* Initialize data */
      if ( 0 == me ) 
	for ( i=0; i<n; i++ ) data[i] = i;
      else 
	for ( i=0; i<n; i++ ) data[i] = -1;

      /* start timing */
      if ( me == 0 ) time = MPI_Wtime();

      if ( version == 0 ) {
	for ( k=0; k<nrepeats; k++ ){
	  if ( me == 0 ){
	    MPI_Send( data, n, datatype, partner, 999, comm );
	    MPI_Recv( data, n, datatype, partner, 999, comm, &status );
	  }

	  if ( me == partner ) {
	    MPI_Recv( data, n, datatype, 0, 999, comm, &status );
	    MPI_Send( data, n, datatype, 0, 999, comm );
	  }
	}
      } else {
	for ( k=0; k<nrepeats; k++ ){
	  if ( me == 0 ){
	    MPI_Ssend( data, n, datatype, partner, 999, comm );
	    MPI_Recv( data, n, datatype, partner, 999, comm, &status );
	  }

	  if ( me == partner ) {
	    MPI_Recv( data, n, datatype, 0, 999, comm, &status );
	    MPI_Ssend( data, n, datatype, 0, 999, comm );
	  }
	}
      }

      /* stop timing */
      if ( me == 0 ) time = MPI_Wtime() - time;
    }

    if ( me == 0 ) time = time / nrepeats / 2;
    /* Broadcast time to all nodes */
    MPI_Bcast( &time, 1, MPI_DOUBLE, 0, comm );

    /* record result */
    timings[ trial ] = time;
  }

  free( data );

  return;
}
