#include <assert.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <rpc/rpc.h>
#include <rpc/auth.h>

//Kotla : pthread includes
#include <pthread.h>
#include <sched.h>

#include "libbyz.h"
#include "nfs.h"
#include "nfsd.h"
#include "fs_interf.h"
#include "inode.h"
#include "svc.h"

#include "stats.h"

//Kotla : New header file to support scheduler
#include "arg.h"
#include "Parallelizer.h"
#include "Req.h"

//Kotla : *****WARNING******** 
// This has to be the same value as defined in Replica.h
#define ALIGNMENT_BYTES 2
#define THREAD_PRIO 8
// Kotla : Setting this caches the results for each op type
//#define CACHE_RESULT 1

extern pthread_mutex_t thread_mutex;
extern pthread_cond_t req_ready_cond;
extern pthread_mutex_t req_queue_mutex; 
extern pthread_mutex_t fs_mutex; 
extern pthread_mutex_t chkpt_mutex; 
extern pthread_mutex_t client_mutex[];
extern int released;
extern Parallelizer *parallelizer;

// Kotla : Thread specific file attribute variables
extern Thread_fattr *tattr;


// Kotla : Diagnostic purposes
//extern FILE *dptr; 
extern int sleep_time;
extern int num_threads;
int global_tid =-1;

#ifdef STATS
int numcalls[NUM_CALLS];
#endif

#define RQCRED_SIZE     400             /* this size is excessive */

#define MAX_ENTRIES_READDIR 600

nfscookie cookies[MAX_ENTRIES_READDIR];
int prev_fid = 0;

/*
 *  argument conversion routines.  
 */
int nfsargconv_noconv(nfsproc_argument *, int);
int nfsargconv_fhandle(nfsproc_argument *, int);
int nfsargconv_readirargs(nfsproc_argument *, int);
int nfsargconv_diropargs(nfsproc_argument *, int);
int nfsargconv_createargs(nfsproc_argument *, int);
int nfsargconv_writeargs(nfsproc_argument *, int);
int nfsargconv_readargs(nfsproc_argument *, int);
int nfsargconv_sattrargs(nfsproc_argument *, int);
int nfsargconv_renameargs(nfsproc_argument *, int);
int nfsargconv_symlinkargs(nfsproc_argument *, int);
int nfsargconv_linkargs(nfsproc_argument *, int);

/*
 * Read fhandle stuff
 *
 */
int read_fhandle_null(nfsproc_argument *);
int read_fhandle(nfsproc_argument *);
int read_fhandle_readdirargs(nfsproc_argument *);
int read_fhandle_diropargs(nfsproc_argument *);
int read_fhandle_createargs(nfsproc_argument *);
int read_fhandle_writeargs(nfsproc_argument *);
int read_fhandle_readargs(nfsproc_argument *);
int read_fhandle_sattrargs(nfsproc_argument *);
int read_fhandle_renameargs(nfsproc_argument *);
int read_fhandle_symlinkargs(nfsproc_argument *);
int read_fhandle_linkargs(nfsproc_argument *);

/*
 * state update (before NFS RPC) routines
 */
void nfs_pre_update_void(nfsproc_argument *, int);
void nfs_pre_update_rm(nfsproc_argument *, int);
void nfs_pre_update_last(nfsproc_argument *, int);
void nfs_pre_update_rename(nfsproc_argument *, int);

/*
 * state update (after NFS RPC) routines
 */
void nfs_pos_update_sattr(nfsproc_argument *, nfsproc_result *, int);
void nfs_pos_update_create(nfsproc_argument *, nfsproc_result *, int);
void nfs_pos_update_write(nfsproc_argument *, nfsproc_result *, int);
void nfs_pos_update_read(nfsproc_argument *, nfsproc_result *, int);
void nfs_pos_update_void(nfsproc_argument *, nfsproc_result *, int);
void nfs_pos_update_lkup(nfsproc_argument *, nfsproc_result *, int);
void nfs_pos_update_rm(nfsproc_argument *, nfsproc_result *, int);
void nfs_pos_update_mkdir(nfsproc_argument *, nfsproc_result *, int);
void nfs_pos_update_rename(nfsproc_argument *, nfsproc_result *, int);
void nfs_pos_update_symlink(nfsproc_argument *, nfsproc_result *, int);

/*
 *  result conversion routines.  
 */
void nfsres_void(nfsproc_result *, int);
void nfsresconv_attrstat(nfsproc_result *, int);
void nfsresconv_readdirres(nfsproc_result *, int);
void nfsresconv_diropres(nfsproc_result *, int);
void nfsresconv_readres(nfsproc_result *, int);


/*
 *  All the information necessary to handle any NFS request.
 */
struct nfsproc {
  int  (*read_handle)(nfsproc_argument *);
  int  (*argument_conv)(nfsproc_argument *, int);
  void (*pre_update_replica_state)(nfsproc_argument *, int);
  void (*result_conv)(nfsproc_result *, int);
  void (*pos_update_replica_result)(nfsproc_argument *, nfsproc_result *, int);
  xdrproc_t xdr_arg_type;
  xdrproc_t xdr_res_type;
};

#define X xdrproc_t

static struct nfsproc const nfsproc_table[] = {
    // 0. NULL : No changes for this function is required for cbase
    { read_fhandle_null,       nfsargconv_noconv,       nfs_pre_update_void,       nfsres_void,
      nfs_pos_update_void,     (X) xdr_void,       (X) xdr_void },
    // 1. Get Attr : last_inum is made per thread variable, nfs_pre_update_void 
    // pre_update/attrstat should pass thread_id as the argument
    {  read_fhandle,           nfsargconv_fhandle,      nfs_pre_update_void,    nfsresconv_attrstat,
      nfs_pos_update_void,     (X) xdr_fhandle,    (X) xdr_attrstat },
    // 2. Set Attr: pre_update_last/attrstat/pos_update_sattr needs thread_id as the argument
    { read_fhandle_sattrargs,  nfsargconv_sattrargs,    nfs_pre_update_last,     nfsresconv_attrstat,
      nfs_pos_update_sattr,    (X) xdr_sattrargs,  (X) xdr_attrstat },
    // 3. Root : obsolete. Not implemented : Nothing to change for Request ID = 3
    { read_fhandle_null,       nfsargconv_noconv,       nfs_pre_update_void,       nfsres_void,
      nfs_pos_update_void,     (X) xdr_void,       (X) xdr_void },
    // 4. Lookup : Modify argconv to pass thread_id as arg
    { read_fhandle_diropargs,  nfsargconv_diropargs,    nfs_pre_update_void,     nfsresconv_diropres,
      nfs_pos_update_lkup,     (X) xdr_diropargs,  (X) xdr_diropres },
    // 5. Read Link : No new changes
    { read_fhandle,            nfsargconv_fhandle,      nfs_pre_update_void,   nfsres_void,
      nfs_pos_update_void,     (X) xdr_fhandle,    (X) xdr_readlinkres },
    // 6. Read : No new changes 
    { read_fhandle_readargs,   nfsargconv_readargs,     nfs_pre_update_void,       nfsresconv_readres,
      nfs_pos_update_read,     (X) xdr_readargs,   (X) xdr_readres },
    // 7. Write Cache: Not implemented in NFS 2.0 
    { read_fhandle_null,     nfsargconv_noconv,       nfs_pre_update_void, nfsres_void,
      nfs_pos_update_void,     (X) xdr_void,       (X) xdr_void },
    // 8. Write : pos_update_write : Pass thread_id as arg
    { read_fhandle_writeargs,  nfsargconv_writeargs,    nfs_pre_update_last,       nfsresconv_attrstat,
      nfs_pos_update_write,    (X) xdr_writeargs,  (X) xdr_attrstat },
    // 9. Create : Creates new entry in fs
    { read_fhandle_createargs, nfsargconv_createargs,   nfs_pre_update_last,      nfsresconv_diropres,
      nfs_pos_update_create,   (X) xdr_createargs, (X) xdr_diropres },
    // 10. Remove : Removes an entry in fs
    { read_fhandle_diropargs,  nfsargconv_diropargs,    nfs_pre_update_rm,       nfsres_void,
      nfs_pos_update_rm,       (X) xdr_diropargs,  (X) xdr_nfsstat },
    // 11. Rename :
    { read_fhandle_renameargs, nfsargconv_renameargs,   nfs_pre_update_rename,     nfsres_void,
      nfs_pos_update_rename,   (X) xdr_renameargs, (X) xdr_nfsstat },
    // 12. Link : Creates hard link 
    { read_fhandle_null,       nfsargconv_linkargs,     nfs_pre_update_void,       nfsres_void,
      nfs_pos_update_void,     (X) xdr_linkargs,   (X) xdr_nfsstat },
    // 13. Symbolic link :
    { read_fhandle_symlinkargs,nfsargconv_symlinkargs,  nfs_pre_update_last,     nfsres_void,
      nfs_pos_update_symlink,  (X) xdr_symlinkargs,(X) xdr_nfsstat },
    // 14. Make dir:
    { read_fhandle_createargs, nfsargconv_createargs ,  nfs_pre_update_last,       nfsresconv_diropres, 
      nfs_pos_update_mkdir,    (X) xdr_createargs, (X) xdr_diropres },
    // 15. Remove dir:
    { read_fhandle_diropargs,  nfsargconv_diropargs,    nfs_pre_update_rm,      nfsres_void,
      nfs_pos_update_rm,       (X) xdr_diropargs,  (X) xdr_nfsstat },
    // 16. Read dir : 
    { read_fhandle_readdirargs,nfsargconv_readirargs,   nfs_pre_update_void, nfsresconv_readdirres,
      nfs_pos_update_void,     (X) xdr_readdirargs,(X) xdr_readdirres },
/* RAF * changed nfsargconv_noconv to _fhandle */
    // 17. Stat fs : File system attr
    { read_fhandle_null,       nfsargconv_fhandle,       nfs_pre_update_void,     nfsres_void, /* RAF */
      nfs_pos_update_void,     (X) xdr_fhandle,    (X) xdr_statfsres },
};

#undef X

const int num_nfsprocs = sizeof (nfsproc_table) / sizeof (struct nfsproc);

static void set_access(struct svc_req *r);

/*
 *   nfsd_dispatch -- This function is called for each NFS request.
 *		    It dispatches the NFS request to a handler function.
 */

/* TODO: Cannot be globals if we allow concurrency */
extern char hostname[];

//Kotla : made byz_svc Thread specific
// static SVCXPRT *byz_svc = 0;

// Kotla: ??? Make it array of number of server worker threads
extern CLIENT *nfs_clnt[];

// Kotla : All these variables are made thread specific
//int last_inum, last_inum_from, last_inum_to, inum_clobbered;
// extern struct timeval cur_time;
#ifndef MAX_MACHINE_NAME
#define MAX_MACHINE_NAME 256
#endif


void perform_RPC_call(int function, char *arg, char *res,int tid)
{
  enum clnt_stat repval;
  struct nfsproc const *np = nfsproc_table + function;
  struct timeval t = {100, 0};
  char *err;

  if (!nfs_clnt[tid]) {
     assert(0);  
  }

  repval = clnt_call(nfs_clnt[tid], function,
		     np->xdr_arg_type, arg,
		     np->xdr_res_type, res,
		     t);
  if (repval != RPC_SUCCESS) {
    clnt_perror(nfs_clnt[tid], err);
    fprintf(stderr, "Error %d in RPC call. %s\n", repval, err);
  }
}

void free_RPC_res(int function, char *res, int tid)
{
  if (!clnt_freeres(nfs_clnt[tid], nfsproc_table[function].xdr_res_type,
		    res))
    fprintf(stderr, "clnt_freeres failed\n");
}


void perform_RPC_call(int function, char *arg, char *res) {
  // Kotla : Agreement stage calls this function when it modifies state.
  // get_obj() calls this functions, which gets called from 
  // state.cow(). Give  it a different tid than the worker threads
  // (0,1.,,,(num_threads-1)) so that RPC calls do not get mangled.
  //pthread_mutex_lock(&chkpt_mutex);
  perform_RPC_call(function, arg, res,num_threads);
  //pthread_mutex_unlock(&chkpt_mutex);
}

void free_RPC_res(int function, char *res) {
     free_RPC_res(function,res,0);
}

// debug - RR-TODO XXX erase
int lookup_cached = 0, read_cached = 0;
typedef struct pres_struct
{
  int  isSet;
  nfsproc_result result;
} pres;

// Two dimensional array 
// old_results[max_op_types][max_num_clients]
// Kotla : change the second dimension to max_num_clients
#define MAX_NUM_CLIENTS 256
pres old_results[20][MAX_NUM_CLIENTS];

int first_time_isSet = 0;
// Kotla : Used by agreement stage (just one thread). 
SVCXPRT *ins_svc; 

int read_state(Byz_req inb,int *op, int *handle) {
  struct rpc_msg m;
  char cred_area[2*MAX_AUTH_BYTES + RQCRED_SIZE]; 
  struct nfsproc const *np;
  nfsproc_argument argument;
  XDR in;

  // Initialize the state 

  // 1. Extract the operation type 
  xdrmem_create(&in, inb.contents, inb.size, XDR_DECODE);

  /* Decode rpc_msg after initializing credential pointers into cred_area */
  m.rm_call.cb_cred.oa_base = cred_area;
  m.rm_call.cb_verf.oa_base = &(cred_area[MAX_AUTH_BYTES]);
  xdr_callmsg(&in, &m);

  // Op type
  *op = m.rm_call.cb_proc;
  //fprintf(stderr,"OP type : %d \n", *op);
 
  // 2. Get the NFS Handle 
    
  /* Compute dispatch table pointer, and decode arguments */
  np = nfsproc_table + m.rm_call.cb_proc;
    
  // Extract the argument 
  memset(&argument, 0, sizeof (argument));
  (np->xdr_arg_type)(&in,(char *) &argument);
  
  /* Convert arguments */
  *handle = (*np->read_handle)(&argument);
  //fprintf(stderr,"Handle : %d \n", *handle);
  return 1;
}


int nfsd_dispatch(Byz_req *inb, Byz_rep *outb, Byz_buffer *non_det, SVCXPRT *byz_svc, int client, int ro, int tid) {
  struct svc_req r;
  struct rpc_msg m;
  char cred_area[2*MAX_AUTH_BYTES + RQCRED_SIZE]; 
  nfsproc_argument argument;
  nfsproc_result *result ;
  struct nfsproc const *np;
  enum auth_stat astat;
  //char *dir;   //debug - XXX erase
  //struct direntry *de;

#ifdef STATS
  start_counter(INIT_REQ);
#endif

  // Kotla : Cur time has to be made thread specific data
  if (!ro) {
    /* Read in current time as chosen by primary into a global */
    bcopy(non_det->contents, (char*)&(tattr[tid].cur_time), sizeof(struct timeval));
  }

  if (byz_svc == 0) {
    // Kotla : byz_svc is made thread specific
    // byz_svc = svcbyz_create(inb->contents, inb->size, outb->contents, outb->size);
    th_assert(0,"byz_svc is null.");
  } else {
    svcbyz_recycle(byz_svc, inb->contents, inb->size, outb->contents, outb->size);
  }

 /* Decode rpc_msg after initializing credential pointers into cred_area */
  m.rm_call.cb_cred.oa_base = cred_area;
  m.rm_call.cb_verf.oa_base = &(cred_area[MAX_AUTH_BYTES]);
  r.rq_clntcred = &(cred_area[2*MAX_AUTH_BYTES]);
  if (!SVC_RECV(byz_svc, &m)) {
    //fprintf(dptr," *******FAILED IN SVC_RECV *************\n");
    svcerr_noprog(byz_svc);
    outb->size = svcbyz_reply_bytes(byz_svc);
    return 0;
  }

  /* Fill in request */
  r.rq_xprt = byz_svc;
  r.rq_prog = m.rm_call.cb_prog;
  r.rq_vers = m.rm_call.cb_vers;
  r.rq_proc = m.rm_call.cb_proc;

#ifdef STATS
  stop_counter(INIT_REQ);
  numcalls[r.rq_proc]++;
  start_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 0);
#endif

  /* Initialize request credentials to cooked form */
  if ((astat = svcbyz_authenticate(&r, &m)) != AUTH_OK) {
    svcerr_auth(byz_svc, astat);
    outb->size = svcbyz_reply_bytes(byz_svc);
    //fprintf(dptr," *******FAILED IN SVC_AUTH *************\n");
    return 0;
  }
 
  // Note that we check if the request is truly read only here and
  // disallow execution if it is not with a noproc error.

#ifndef NO_READ_ONLY_OPT
  if (r.rq_proc < 0 || r.rq_proc >= (unsigned int) num_nfsprocs 
      || (r.rq_proc != 1 && r.rq_proc != 4 && r.rq_proc != 6 && ro)) {
#else
  if (r.rq_proc < 0 || r.rq_proc >= num_nfsprocs 
      || (r.rq_proc != 1 && ro)) {
#endif
    svcerr_noproc(byz_svc);
    outb->size = svcbyz_reply_bytes(byz_svc);
    //fprintf(dptr," *******FAILED IN READ ONLY CHECK OP : %d *************\n",r.rq_proc);
    return 0;
  }
 
  /* Get credential out of RPC message into a global */
  set_access(&r);

 /* Compute dispatch table pointer, and decode arguments */
  np = nfsproc_table + r.rq_proc;
  //fprintf(stderr," NFS REQUEST : %ld", r.rq_proc);
  memset(&argument, 0, sizeof (argument));
  if (!svc_getargs(byz_svc, np->xdr_arg_type, (char *) &argument)) {
    svcerr_decode(byz_svc);
    outb->size = svcbyz_reply_bytes(byz_svc);
    //fprintf(dptr," *******FAILED IN SVC_GET ARGS OP : %d *************\n",r.rq_proc);
    return 0;
  }

  //memset (&result, 0, sizeof (result));

#ifdef STATS
  stop_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 0);
  start_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 1);
#endif

  /* Perform RPC call */
  // For experiments Sid/Chaitanya 12/05
  // perform_RPC_call(r.rq_proc, (char *)&argument, (char *)&result);
#ifdef CACHE_RESULT
  if ( old_results[ r.rq_proc ][client].isSet ==1)
  {
    //printf("Request type is %d, sending previously computed values. \n", r.rq_proc  );
      result = &old_results[r.rq_proc][client].result;
  }
  else
#endif

  {
    result = &old_results[r.rq_proc][client].result;
    old_results[ r.rq_proc ][client].isSet=1;

  /* Convert arguments */
  if (((*np->argument_conv)(&argument,tid)) < 0) {
    /* could not convert arguments */
    fprintf(stderr, "oi ");
    *((nfsstat *)(result)) = NFSERR_STALE;
    if (!svc_sendreply(byz_svc, np->xdr_res_type, (char *) result))
      svcerr_systemerr(byz_svc);
    outb->size = svcbyz_reply_bytes(byz_svc);
    svc_freeargs(byz_svc, np->xdr_arg_type, (char *) &argument);
#ifdef STATS
    stop_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 1);
#endif
    //fprintf(dptr," *******FAILED IN SVC_ARG CONV OP : %d *************\n",r.rq_proc);
    return 0;
  }

#ifdef STATS
  stop_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 1);
  start_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 2);
#endif

  /* Update local state */
  (*np->pre_update_replica_state)(&argument,tid);

#ifdef STATS
  stop_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 2);
  start_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 3);
#endif

  /* Perform RPC call */
  perform_RPC_call(r.rq_proc, (char *)&argument, (char *)result, tid);

#ifdef STATS
  stop_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 3);
  start_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 4);
#endif

  /* Update local state with the result of the NFS call */
  (*np->pos_update_replica_result)(&argument, result,tid);

#ifdef STATS
  stop_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 4);
  start_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 5);
#endif

  /* Convert result */
  (*np->result_conv)(result,tid);
 }

#ifdef STATS
  stop_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 5);
  start_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 6);
#endif

  if (!svc_sendreply(byz_svc, np->xdr_res_type, (char *) result))
    svcerr_systemerr(byz_svc);

  outb->size = svcbyz_reply_bytes(byz_svc);

#if 0
  fprintf(stderr, "Size %d\n", outb->size);
  for (i=0; i<outb->size; i+=4)
    fprintf(stderr, "%10x", *((int*)&outb->contents[i]));
  fprintf(stderr, "\n");
#endif

#ifndef CACHE_RESULT
  /* Free any data allocated by XDR library */
  svc_freeargs(byz_svc, np->xdr_arg_type, (char *) &argument);

  free_RPC_res(r.rq_proc, (char *)result,tid);
#endif

#ifdef STATS
  stop_counter(MIN_CALL_STATS + r.rq_proc * NUM_STATS_PER_CALL + 6);
#endif

  // Simulate a disk model
  if (sleep_time) {
      usleep(sleep_time);
  }
  /* return number of bytes in out_stream */ 
  return 0;

}

// Kotla : Main routine for worker threads, which service ready requests in
// a loop 
void * start_request (void *t) {

   Thread_arg *targ;
   SVCXPRT * t_svc = NULL;
   int tid = -1;
   
   // Set the priority of this thread to be high so that 
   // if there is work to do this gets priority over 
   // BASE thread
   //struct sched_param param;
   //param.sched_priority = THREAD_PRIO;
   //if (sched_setscheduler(getpid(),SCHED_FIFO,&param)) {
   //exit(-1);
   //}
   // Raise the piority of worker threads;
   nice(-10);
   

   pthread_mutex_lock(&thread_mutex);
   tid = ++global_tid;

   if (  first_time_isSet == 0 )
    {
      first_time_isSet=1;
      for (int i=0;i<20;i++)
        {
	  // Kotla : Replica 100 with max_num_clients
	  for (int j=0; j<MAX_NUM_CLIENTS; j++) {
	    old_results[i][j].isSet=0;
	  }
        }
    }
   pthread_mutex_unlock(&thread_mutex);  
   printf(" Started thread %d . \n",tid);
   while (1) {
     Req* req = NULL;
     //printf(" Worker : Started working %d \n",tid);
     pthread_mutex_lock(&req_queue_mutex);
     while ((req =  parallelizer->next_request()) == NULL) { 
       // Kotla *******Warning ********
       // : Could become bottleneck if kernel scheduler
       // does linear search. May be true in 7.2 kernel
       //printf(" Worker : Blocked as there is no ready request.\n");
       pthread_cond_wait(&req_ready_cond,&req_queue_mutex);
     }
     pthread_mutex_unlock(&req_queue_mutex);

     // Get the request state 
     targ = req->get_state();

     int cid = targ->client;
     Request_id rid = targ->req_id;
     Rep_info *replies = targ->rep;
     Reply *rep = NULL;

     //////////////////////////////////////////////////////////////
     // 1.  Execute the request first before updating 
     // replies->last_cont_reply as checkpoint is taken as soon 
     // as  replies->last_cont_reply % checkpoint_interval == 0
     // in the agreement stage thread.
     /////////////////////////////////////////////////////////////

     // Kotla : ----------Enhancement -------------
     // Can't we make this more fine grained where requests from
     // same client can excute in parallel as slow replica can
     // have multiple outstanding requests from the same client

     pthread_mutex_lock(&client_mutex[cid]);

     // Kotla : Make a new reply here as this thread could be 
     // executing immediately after another worker thread in which 
     // case, replies.reply_size is polluted for this pid
     if (targ->ro) {
       // Kotla : Read only request
       rep = new Reply(targ->view, rid, targ->my_id);
       targ->outb.contents = rep->store_reply(targ->outb.size);
     }
     else {
       targ->outb.contents = replies->new_reply(cid, targ->outb.size);
     }

     if (t_svc == NULL) {
       t_svc = svcbyz_create(targ->inb.contents, targ->inb.size, targ->outb.contents, targ->outb.size);
     }
       
     targ->res = nfsd_dispatch(&targ->inb,&targ->outb,&targ->non_det,t_svc,targ->client,targ->ro,tid);
     //fprintf (stderr,"After nfsd dispatch outb size : %d \n", targ->outb.size);

     // Kotla : If it is error then do not send reply
     if (targ->res) th_assert(0,"NFSD dispatch should not return non zero value");
     else {
//        fprintf (dptr," Request executed client %d rid %qu seq no : %qu size : %d Read only : %d\n",
// 		cid,rid,targ->seqno,targ->outb.size,targ->ro);
//        fflush(dptr);
     }
     
     if (targ->outb.size % ALIGNMENT_BYTES) {
       for (int i=0; i < ALIGNMENT_BYTES - (targ->outb.size % ALIGNMENT_BYTES); i++) {
	 targ->outb.contents[targ->outb.size+i] = 0;
       }
     }

     if (targ->ro) {
       rep->authenticate(replies->i_to_p(cid), targ->outb.size, true);
       if (targ->outb.size < 50 || targ->rep_id == targ->my_id || targ->rep_id < 0) {
	 replies->send(rep, cid);
       }
       else {
	 Reply empty(targ->view, rid, targ->my_id,
		     rep->digest(), replies->i_to_p(cid), false);
	 replies->send(&empty, cid);
       }
       delete rep;

       // Kotla : For read only requests we do not need to update replies->window
       // After sending the reply just continue to service the next request
       //fprintf(stderr, " ######Just Executed Read only request %qu. \n", rid);
       // Remove the request from the request queue
       pthread_mutex_unlock(&client_mutex[cid]);
       parallelizer->remove(req);
       continue; 
     }
     else {
       //fprintf (stderr," Before end reply client %d rid %qu seq no : %qu size : %d \n",
       // cid,rid,targ->seqno,targ->outb.size);

       // ----------- --FIX THIS -----------------------
       // Kotla : If reply_size is not -1 then do not send the reply.
       // I do not the "exact" reason why it can have other value which
       // should not be the case  for normal execution as it is set to -1 in 
       // new_reply. But this seems to happen for a slow replica which 
       // fetches state from other replicas
       // when it falls behind other replicas. In that case just drop this
       // request.
       if (!replies->is_reply_stable(cid)) {
	 	 
	 replies->end_reply(cid,rid, targ->outb.size);
	 if ((targ->outb.size != 0) && (cid != targ->my_id)) {
	   // Kotla : ************WARNING****************
	   // All the replies are sent as committed. Change this if we enqueue
	   // requests tentatively.
       
	   // Kotla : Optimization : Only the assigned replier sends the reply. 
	   if (targ->outb.size < 50 || targ->rep_id == targ->my_id || targ->rep_id < 0) {
	     // Send full reply.
	     // replies.send_reply(cid, view(), id(), false);
	     //printf (" ----Sending Full reply for client %d rid %qu seq no : %qu size : %d \n",
	     //cid,rid,targ->seqno,targ->outb.size);
	     replies->send_reply(cid, targ->view, targ->my_id, false);
	   } 
	   else {
	     // Send empty reply.
	     //printf (" *****Sending Empty reply for client %d rid %qu seq no : %qu size : %d \n",
	     // cid,rid,targ->seqno,targ->outb.size);
	     Reply empty(targ->view, rid, targ->my_id,
			 replies->digest(cid), replies->i_to_p(cid), false);
	     replies->send(&empty, cid);
	   }
	 }
       }
       else {
	 th_assert(0,"Reply size is corrupted \n");
       }
       pthread_mutex_unlock(&client_mutex[cid]);
     }
     

     /////////////////////////////////////////////////////////////////////
     // 2. Update the replies structure of the client after executing 
     //    the request  
     ////////////////////////////////////////////////////////////////////

     pthread_mutex_lock(&thread_mutex);
     // printf("Start request :  Seq no : %qu last_cont_reply : %qu last_chkpt : %qu interval : %qu Windows : %d\n", 
     //targ->seqno, replies->last_cont_reply, replies->last_chkpt,replies->chkpt_interval, replies->window[0]);

     if (targ->seqno >= (replies->last_cont_reply+1)) {
       // To execute the request out of order we need to set the bit
       // before that
       // Find the word and then the bit in the window bit vector.
       int num_entry = (targ->seqno - replies->last_chkpt-1);
       int num_word = num_entry/32; // For word = 32 bit
       int num_bit = num_entry%32;  
     
       // Kotla : Multiple requests can belong to the same sequence no. as primary batches
       // the requests to reduce protocol/computation overhead. We need to extend the replies->window
       // structure to keep track of the batched requests for a given sequence no. Update the 
       // window bit for each sequence number after all the requests batched for this request is 
       // executed. We can keep track of how many requests are batched in executed_prepared() for
       // each sequence no. When we call execute command we can update the thread argument and that 
       // can be used here to keep track of the batched requests.
       
       
       //printf("Before assertion Seq : %qu num_word : %d num_bit : %d window : %d last cont : %qu\n",
       // targ->seqno, num_word, num_bit,replies->window[num_word],replies->last_cont_reply);

       //fprintf(stderr," Request seq : %qu batch_count %d Request id : %qu last_cont : %qu \n", 
       //targ->seqno,replies->batch[num_entry],rid,replies->last_cont_reply);

       if (replies->batch[num_entry]) {
	 replies->batch[num_entry]--;
       }
       else {
	 th_assert(0,"Received more requests than they are batched");
       }
       
       if (!replies->batch[num_entry]) {
	 // If batch count is one, there can be no more than one request
	 // Cannot have this bit set already
	 th_assert(!(replies->window[num_word] & (1 << num_bit)),
		     "Request is already processed");
	 replies->window[num_word] |= (1<<num_bit);
       
	 // First check to see if the replies->last_cont_reply+1 bit is 
	 // already set if so update the last_cont until the targ->seqno
	 // Find the word and then the bit
     
	 num_entry = ((replies->last_cont_reply+1) - replies->last_chkpt-1);
	 num_word = num_entry/32; // For word = 32 bits
	 num_bit = num_entry%32;
     
	 // Update the last_cont_reply      
	 while ((replies->window[num_word] & (1 << num_bit))) {
	   if (num_bit == 31) {
	     num_bit = 0;
	     num_word++;
	   }
	   else {
	     num_bit++;
	   }
	   replies->last_cont_reply++; 
	   //fprintf(stderr,"=== LATEST Seq no : %qu cont rep : %qu num_word %d num_bit %d window %d ====== \n",  
	   //  targ->seqno, replies->last_cont_reply, num_word, (num_bit-1), replies->window[num_word]);
	   // printf(" ITER : cont rep : %qu \n", replies->last_cont_reply); 
	   if (((num_word)*32 + num_bit) == replies->chkpt_interval) {
	     // Break if we reached the fence
	     //fprintf(dptr,"*********REACHED CHECKPOINT INTERVAL *****************\n");
	     break;
	   }

	 }
       }
     }
     else {
       th_assert(0," Executing an old request again \n");
     }
     
     pthread_mutex_unlock(&thread_mutex);
     
     // Call the update dependency on req
     // printf("Removing from start request : Req : %x Seq no : %qu Rid : %qu thread id %d\n",req,req->req_num,req->rid,targ->tid);
     
     parallelizer->remove(req);
     //usleep(10);
   }
}


/* Cannot be globals if we allow concurrency */

// Kotla : Currently these are not made thread specific as they are 
// just written into and nver read. So it does not matter.

extern uid_t cred_uid;
extern gid_t cred_gid;

#define NOBODY_UID 65534
#define NOBODY_GID 65534

static void set_access(struct svc_req *r) {
  if (r->rq_cred.oa_flavor == AUTH_UNIX) {
    struct authunix_parms *unix_cred;
    
    unix_cred = (struct authunix_parms *) r->rq_clntcred;
    cred_uid = unix_cred->aup_uid;
    cred_gid = unix_cred->aup_gid;
    /* TODO: check other groups */
  } else {
    cred_uid = NOBODY_UID;
    cred_gid = NOBODY_GID;
  }
}


/*
 * state update (before NFS RPC) routines
 */
void nfs_pre_update_void(nfsproc_argument *ap, int tid) {}

// Kotla : Pass thread_id as the argument
void nfs_pre_update_rm(nfsproc_argument *ap, int tid) 
{
  diropargs *dargs = (diropargs *) ap;
  diropres dres;
  int last_inum = tattr[tid].last_inum;

  // Kotla : Do we need to lock this  ??
  modified_inode(last_inum); // going to change dir

  perform_RPC_call(NFSPROC_LOOKUP, (char *)dargs, (char *)&dres, tid);

  if (dres.status != NFS_OK) {
    fprintf(stderr, "Error in update NFS RPC call\n");
    last_inum = -1;
  }
  else
    if ((dres.diropok.attributes.type != NFDIR && dres.diropok.attributes.nlink == 1) || (dres.diropok.attributes.type == NFDIR && dres.diropok.attributes.nlink == 2)) {
      last_inum = FS_NFS_handle_to_client_handle(&dres.diropok.file);
      modified_inode(last_inum);  // going to change entry
    }
    else
      last_inum = 0; // Entry still exists: don't delete!
  free_RPC_res(NFSPROC_LOOKUP, (char *)&dres,tid);

  tattr[tid].last_inum = last_inum;
  
}

//  Kotla : ********WARNING ******
// Not sure if we need to lock this as it modifies the state of 
// the replica??
//  Also we need to pass thread id as the argument.

void nfs_pre_update_last(nfsproc_argument *ap, int tid) 
{
  int last_inum = tattr[tid].last_inum;
  modified_inode(last_inum);
}

// Kotla : Pass thread id 
void nfs_pre_update_rename(nfsproc_argument *ap, int tid)
{
  int last_inum = tattr[tid].last_inum;
  int last_inum_to = tattr[tid].last_inum_to;
  int last_inum_from = tattr[tid].last_inum_from;
  int inum_clobbered = tattr[tid].inum_clobbered;

  // Kotla : Do we need to lock here ??
  modified_inode(last_inum); // going to change file
  modified_inode(last_inum_from); // going to change from dir
  modified_inode(last_inum_to); // going to change to dir
  if (inum_clobbered)
    modified_inode(inum_clobbered);
}


/*
 *  argument conversion routines.  
 */
int nfsargconv_noconv(nfsproc_argument *ap, int tid) { return 0; }

// Kotla : We need to make last_inum as per thread variable 
// This function gets thread_id as the argument
int nfsargconv_fhandle(nfsproc_argument *ap, int tid) {
  fhandle *argp = (fhandle *) ap;

  if ((tattr[tid].last_inum = FS_fhandle_to_NFS_handle(argp)) < 0) {
    fprintf(stderr,"Failed in conversion of FS handle to local NFS handle.\n");
    return -1;
  }
  
  return 0;
}

int read_fhandle(nfsproc_argument *ap) {
  fhandle *argp = (fhandle *) ap;
  int handle;
  if ((handle = FS_fhandle_to_NFS_handle(argp)) < 0) {
    return -1;
  }

  return handle; 
} 

int read_fhandle_null(nfsproc_argument *ap) {
  return -1;
}

// Kotla : Pass on thread_id as the argument for all of the following
// functions which convert the arguments  

// Kotla : Read entries of a directory starting from the one stored
// in cookie.

int nfsargconv_readirargs(nfsproc_argument *ap, int tid) {
  readdirargs *argp = (readdirargs *) ap;
  if (nfsargconv_fhandle((nfsproc_argument *)&(argp->dir),tid) < 0)
    return -1;

  prev_fid = *((int *)&argp->cookie);
  if (prev_fid) {
    memcpy(&argp->cookie, &cookies[prev_fid], sizeof(nfscookie));
  }
  return 0;
}

int read_fhandle_readdirargs(nfsproc_argument *ap) {
  readdirargs *argp = (readdirargs *) ap;
  return read_fhandle((nfsproc_argument *)&(argp->dir));
} 
 
int nfsargconv_diropargs(nfsproc_argument *ap, int tid) {
  diropargs *argp = (diropargs *) ap;
  //fprintf(stderr,"Dir args : file name : %s dir handle : %d \n", argp->name,argp->dir);	
  return nfsargconv_fhandle((nfsproc_argument *)&(argp->dir),tid);
}

int read_fhandle_diropargs(nfsproc_argument *ap) {
  diropargs *argp = (diropargs *) ap;
  return read_fhandle((nfsproc_argument *)&(argp->dir));
} 

int nfsargconv_createargs(nfsproc_argument *ap, int tid) {
  createargs *argp = (createargs *) ap;
  return nfsargconv_fhandle((nfsproc_argument *)&(argp->where.dir),tid);
}

int read_fhandle_createargs(nfsproc_argument *ap) {
  createargs *argp = (createargs *) ap;
  return read_fhandle((nfsproc_argument *)&(argp->where.dir));
}

int nfsargconv_writeargs(nfsproc_argument *ap, int tid)
{
  writeargs *argp = (writeargs *) ap;
  return nfsargconv_fhandle((nfsproc_argument *)&(argp->file),tid);
}

int read_fhandle_writeargs(nfsproc_argument *ap) {
  writeargs *argp = (writeargs *) ap;
  return read_fhandle((nfsproc_argument *)&(argp->file));
} 

int nfsargconv_readargs(nfsproc_argument *ap, int tid)
{
  readargs *argp = (readargs *) ap;
  return nfsargconv_fhandle((nfsproc_argument *)&(argp->file),tid);
}

int read_fhandle_readargs(nfsproc_argument *ap) {
  readargs *argp = (readargs *) ap;
  return read_fhandle((nfsproc_argument *)&(argp->file));
} 

int nfsargconv_sattrargs(nfsproc_argument *ap, int tid)
{
  sattrargs *argp = (sattrargs *) ap;
#ifdef OMMIT_UID_GID
  argp->attributes.uid = (unsigned int) -1;
  argp->attributes.gid = (unsigned int) -1;
#endif
  return nfsargconv_fhandle((nfsproc_argument *)&(argp->file), tid);
}

int read_fhandle_sattrargs(nfsproc_argument *ap) {
  sattrargs *argp = (sattrargs *) ap;
  return read_fhandle((nfsproc_argument *)&(argp->file));
} 

//  Kotla : Pass thread_id 
int nfsargconv_renameargs(nfsproc_argument *ap, int tid)
{
  renameargs *argp = (renameargs *) ap;
  diropargs *dargs = &argp->from;
  diropres dres;

  if ((tattr[tid].last_inum_from = FS_fhandle_to_NFS_handle(&(argp->from.dir))) < 0)
    return -1;
  if ((tattr[tid].last_inum_to = FS_fhandle_to_NFS_handle(&(argp->to.dir))) < 0)
    return -1;

  perform_RPC_call(NFSPROC_LOOKUP,(char *)dargs, (char *)&dres, tid);

  if (dres.status != NFS_OK) {
    fprintf(stderr, "Error in update NFS RPC call\n");
    tattr[tid].last_inum = -1;
  }
  else 
    tattr[tid].last_inum = FS_NFS_handle_to_client_handle(&dres.diropok.file);

  free_RPC_res(NFSPROC_LOOKUP, (char *)&dres,tid);

  // Are we going to overwrite a file? Let's check:
  dargs = &argp->to;

  perform_RPC_call(NFSPROC_LOOKUP,(char *)dargs, (char *)&dres,tid);

  if (dres.status != NFS_OK) {
    tattr[tid].inum_clobbered = 0;
  }
  else
    if ((dres.diropok.attributes.type != NFDIR && dres.diropok.attributes.nlink == 1) || (dres.diropok.attributes.type == NFDIR && dres.diropok.attributes.nlink == 2))
    tattr[tid].inum_clobbered = FS_NFS_handle_to_client_handle(&dres.diropok.file);
  free_RPC_res(NFSPROC_LOOKUP, (char *)&dres,tid);

    return 0;
}

// Kotla : ----------WARNING-------
// We just pass the destination directory handle
// Pass both the handles to fix this ..

int read_fhandle_renameargs(nfsproc_argument *ap) {
  renameargs *argp = (renameargs *) ap;
  return read_fhandle((nfsproc_argument *)&(argp->to.dir));
} 
  
int nfsargconv_symlinkargs(nfsproc_argument *ap, int tid)
{
  symlinkargs *argp = (symlinkargs *) ap;
  return nfsargconv_fhandle((nfsproc_argument *)&(argp->from.dir), tid);
}

int read_fhandle_symlinkargs(nfsproc_argument *ap) {
  symlinkargs *argp = (symlinkargs *) ap;
  return read_fhandle((nfsproc_argument *)&(argp->from.dir));
}

int nfsargconv_linkargs(nfsproc_argument *ap, int tid)
{
  linkargs *argp = (linkargs *) ap;
  if ((tattr[tid].last_inum_from = FS_fhandle_to_NFS_handle(&argp->from)) < 0)
    return -1;
  if ((tattr[tid].last_inum_to = FS_fhandle_to_NFS_handle(&argp->to.dir)) < 0)
    return -1;
  return 0;
}

// Kotla : ----------WARNING-------
// We just pass the destination directory handle
// Pass both the handles to fix this ..

int read_fhandle_linkargs(nfsproc_argument *ap) {
  linkargs *argp = (linkargs *) ap;
  return read_fhandle((nfsproc_argument *)&(argp->to.dir));
}


/*
 *  result conversion routines.  
 */
void nfsres_void(nfsproc_result *rp, int tid) {}

// Kotla : This function also gets thread_id as the argument
void nfsresconv_attrstat(nfsproc_result *rp, int tid) {
  attrstat *resp = (attrstat *) rp;

  if (resp->status != NFS_OK) {
    fprintf(stderr,"ERR %d ", resp->status);
    return;
  }

  if (FS_attr_NFS_to_client(tattr[tid].last_inum, &(resp->attributes)) < 0) 
    fprintf(stderr, "Conversion was not possible. Screwed up!\n");
}

// Kotla : How about a better sorting algo.??
void bubbleSort(entry *list)
{
  entry *pFirst, *pSecond;
  char *tmp;

  for (pFirst = list; pFirst; pFirst = pFirst->nextentry) {
    for (pSecond = pFirst; pSecond; pSecond = pSecond->nextentry) {

      // if we find a node out of place with the following node

      if (strcmp(pFirst->name, pSecond->name) > 0) {
        // swap em... this will sort increasing values
        tmp = pFirst->name;
        pFirst->name = pSecond->name;
        pSecond->name = tmp;
      }
    }
  }
}

// Kotla : ****** WARNING ********
// Cookies are global variables. Checkout if we need 
// to make them thread specific ?? Read README.CBASE
// for detailed explanation of the problem

void nfsresconv_readdirres(nfsproc_result *rp, int tid )
{
  //int i;

  entry *ptr;
  readdirres *res = (readdirres *) rp;
  if (res->status != NFS_OK) {
    fprintf(stderr,"ERR %d ", res->status);
    return;
  }
  ptr = res->readdirok.entries;
  while (ptr) {
    ptr->fileid = ++prev_fid; /* XXX - This is not according to the NFS spec */
    memcpy(&cookies[prev_fid], &ptr->cookie, sizeof(nfscookie));
    memset(&ptr->cookie, 0, sizeof(nfscookie));
    *((int *)&ptr->cookie) = prev_fid;
    ptr = ptr->nextentry;
  }

#ifdef ORDER_DIR_ENTRIES
  /* Bubble sort this list XXX Inefficient - change? */
  bubbleSort(res->readdirok.entries);
#endif
}

// Kotla : Not to change this function
void nfsresconv_diropres(nfsproc_result *rp, int tid)
{
  diropres *resp = (diropres *) rp;
  int last_inum; // Kotla : We don't ned to update tattr??

  if (resp->status != NFS_OK) {
    if (resp->status != 2)
      fprintf(stderr,"ERR %d ", resp->status);
    return;
  }

  /* convert file handle */
  if ((last_inum = FS_NFS_handle_to_client_handle(&(resp->diropok.file))) < 0)
    fprintf(stderr, "fh Conversion was not possible. Screwed up!\n");

  /* convert attributes */
  if (FS_attr_NFS_to_client(last_inum, &(resp->diropok.attributes)) < 0) 
    fprintf(stderr, "attr Conversion was not possible. Screwed up!\n");

  tattr[tid].last_inum = last_inum;
}

// Kotla : Not to change this function
void nfsresconv_readres(nfsproc_result *rp, int tid)
{
  attrstat *resp = (attrstat *) rp;

  if (resp->status != NFS_OK) {
    fprintf(stderr,"ERR %d ", resp->status);
    return;
  }

  if (FS_attr_NFS_to_client(tattr[tid].last_inum, &(resp->attributes)) < 0) 
    fprintf(stderr, "Conversion was not possible. Screwed up!\n");
}


/*
 * state update (after NFS RPC) routines
 */
void nfs_pos_update_void(nfsproc_argument *ap, nfsproc_result *rp, int tid) { }

// Kotla : Important function it is here inum is assigned.
// Pass on thread_id as arg

void nfs_pos_update_create(nfsproc_argument *ap, nfsproc_result *rp, int tid)
{
  //  createargs *argp = (createargs *) ap;
  diropres *resp = (diropres *) rp;
  if (resp->status != NFS_OK) {
    fprintf(stderr,"ERR %d ", resp->status);
    return;
  }
  
  // Kotla : Lock before creating an entry as it accesses the global fs->last_free_inum var
  pthread_mutex_lock(&fs_mutex);
  if ((tattr[tid].last_inum = FS_create_entry(&(resp->diropok.file), &(resp->diropok.attributes), &tattr[tid].cur_time, FILE_INODE_TYPE, tattr[tid].last_inum)) < 0)
    fprintf(stderr, "Create file update (after RPC) returned error!\n");
  pthread_mutex_unlock(&fs_mutex);
}

// Kotla : Pass on thread id here. Cur time is to be made thread specific
void nfs_pos_update_write(nfsproc_argument *ap, nfsproc_result *rp, int tid)
{
  /*  writeargs *argp = (writeargs *) ap; */
  attrstat *resp = (attrstat *) rp;
  if (resp->status != NFS_OK) {
    fprintf(stderr,"ERR %d ", resp->status);
    return;
  }
  if (FS_update_time_modified(tattr[tid].last_inum, &(tattr[tid].cur_time)) < 0)
    fprintf(stderr, "Problem updating local state after write");
}

// Kotla : Pass thread id as the argument
void nfs_pos_update_sattr(nfsproc_argument *ap, nfsproc_result *rp, int tid)
{
  sattrargs *argp = (sattrargs *) ap;
  attrstat *resp = (attrstat *) rp;

  if (resp->status != NFS_OK) {
    fprintf(stderr,"SATTR ERR %d ", resp->status);
    return;
  }
  if (FS_set_attr(tattr[tid].last_inum, &(argp->attributes)) < 0)
    fprintf(stderr, "Problem updating local attr after sattr");
}

void nfs_pos_update_read(nfsproc_argument *ap, nfsproc_result *rp, int tid)
{
#ifdef NO_READ_ONLY_OPT
  /* RR-TODO: XXX update atime here */
#endif
}

void nfs_pos_update_lkup(nfsproc_argument *ap, nfsproc_result *rp, int tid)
{
#ifdef NO_READ_ONLY_OPT
  /* RR-TODO: XXX update atime here */
#endif
}

// Kotla : Important here inum is freed
// Pass on the thread_id as the argument 
void nfs_pos_update_rm(nfsproc_argument *ap, nfsproc_result *rp, int tid)
{
  //  diropargs *argp = (diropargs *) ap;
  nfsstat *resp = (nfsstat *) rp;

  if (*resp != NFS_OK) {
    fprintf(stderr, "NFS remove returned error %d\n", *resp);
    return;
  }

  if (tattr[tid].last_inum > 0) {
    // Kotla : Lock before accessing fs->last_free_inode
    pthread_mutex_lock(&fs_mutex);
    if (FS_remove_entry(tattr[tid].last_inum) < 0)
      fprintf(stderr, "Error updating local state after remove\n");
    pthread_mutex_unlock(&fs_mutex);
  }
}

// Kotla : Pass thread_id as argument
void nfs_pos_update_rename(nfsproc_argument *ap, nfsproc_result *rp, int tid)
{
  renameargs *argp = (renameargs *) ap;
  nfsstat *resp = (nfsstat *) rp;

  diropargs *dargs;
  diropres dres;

  if (*resp != NFS_OK) {
    fprintf(stderr, "NFS rename returned error %d\n", *resp);
    return;
  }

  dargs = &(argp->to);
  perform_RPC_call(NFSPROC_LOOKUP, (char *)dargs, (char *)&dres,tid);

  if (dres.status != NFS_OK)
    fprintf(stderr, "Error in update NFS RPC call\n");
  else {
    // Kotla : Lock before accessing fs->last_free_inode
    pthread_mutex_lock(&fs_mutex);

    if (tattr[tid].inum_clobbered > 0) {      
      if (FS_remove_entry(tattr[tid].inum_clobbered) < 0)
	fprintf(stderr, "Error updating local state after overwrite rename\n");
      
    }

    if (FS_update_file_info(tattr[tid].last_inum, &dres.diropok.file, 
			    tattr[tid].last_inum_to, dres.diropok.attributes.fileid, 
			    dres.diropok.attributes.fsid) < 0)
      fprintf(stderr, "Could not update file info after rename\n");

    pthread_mutex_unlock(&fs_mutex);
  }
  
  free_RPC_res(NFSPROC_LOOKUP, (char *)&dres,tid);
}

// Kotla : Pass on thread id. Creates entry in fs.
void nfs_pos_update_mkdir(nfsproc_argument *ap, nfsproc_result *rp, int tid)
{
  diropres *resp = (diropres *) rp;
  if (resp->status != NFS_OK) {
    fprintf(stderr,"ERR %d ", resp->status);
    return;
  }
  
  // Kotla : Lock before accessing fs->last_free_inode
  pthread_mutex_lock(&fs_mutex);

  if ((tattr[tid].last_inum = FS_create_entry(&(resp->diropok.file), &(resp->diropok.attributes), 
				   &tattr[tid].cur_time, DIR_INODE_TYPE, tattr[tid].last_inum)) < 0)
    fprintf(stderr, "Create file update (after RPC) returned error!\n");  
  
  pthread_mutex_unlock(&fs_mutex);
}

// Kotla : Pass on thread id. Creates entry in fs. 
void nfs_pos_update_symlink(nfsproc_argument *ap, nfsproc_result *rp, int tid)
{
  symlinkargs *argp = (symlinkargs *) ap;
  nfsstat *resp = (nfsstat *) rp;

  diropargs *dargs;
  diropres dres;

  if (*resp != NFS_OK) {
    fprintf(stderr, "NFS create symlink returned error %d\n", *resp);
    return;
  }

  dargs = &(argp->from);
  perform_RPC_call(NFSPROC_LOOKUP, (char *)dargs, (char *)&dres,tid);

  if (dres.status != NFS_OK)
    fprintf(stderr, "Error in update NFS RPC call\n");
  else {
    // Kotla : Lock before accessing fs->last_free_inode
    pthread_mutex_lock(&fs_mutex);
    if ((tattr[tid].last_inum = FS_create_entry(&dres.diropok.file, &dres.diropok.attributes, &tattr[tid].cur_time, SYMLINK_INODE_TYPE, tattr[tid].last_inum)) < 0)
      fprintf(stderr, "could not update state after creating symlink\n");
    pthread_mutex_unlock(&fs_mutex);
  }
  free_RPC_res(NFSPROC_LOOKUP, (char *)&dres,tid);
}
