#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <rpc/rpc.h>
#include <signal.h>
#include <unistd.h>
// Kotla : pthread header files here
#include <pthread.h>
#include "Replica.h"
#include "th_assert.h"
#include "libbyz.h"
#include "Statistics.h"

// Kotla : scheduler header file
#include "arg.h"
#include "Parallelizer.h"
#include "parameters.h"
#include "nfs.h"

// thresholds to accept timestamps from primary (in seconds)
#define LOWER_THRESHOLD 10
#define UPPER_THRESHOLD 10


// Service specific functions.

extern int nfsd_dispatch(Byz_req *inb, Byz_rep *outb, Byz_buffer *non_det, 
			     int client, int ro);
extern int read_state(Byz_req inb, int *op, int *handle);

extern "C" int get_obj(int n, char **obj);
extern "C" void put_objs(int num_objs, int *sizes, int *indices, char **objs);

extern "C" void shutdown_state(FILE *o);
extern "C" void restart_state(FILE *i);

// Kotla : Globals for threads
// **WARNING** : Change this to larger value if maximum 
// outstanding requests are more than 64
#define MAX_WORKER_THREADS 256
#define MAX_NUM_CLIENTS 256

int thread_id = 0;
Thread_arg *thread_arg;
pthread_t req_thread[MAX_WORKER_THREADS];
pthread_mutex_t thread_mutex = PTHREAD_MUTEX_INITIALIZER ;
pthread_mutex_t req_queue_mutex = PTHREAD_MUTEX_INITIALIZER ;
pthread_mutex_t fs_mutex = PTHREAD_MUTEX_INITIALIZER ;
pthread_mutex_t chkpt_mutex = PTHREAD_MUTEX_INITIALIZER ;
pthread_cond_t req_ready_cond =  PTHREAD_COND_INITIALIZER ;
pthread_mutex_t client_mutex[MAX_NUM_CLIENTS];
int released = 0;

// Kotla : Thread specific file attribute variables
Thread_fattr *tattr;
CLIENT *nfs_clnt[MAX_WORKER_THREADS+1]; 

// Global Parallelizer shared by AGREEMENT, 
// PARALLELIZER and EXEC stages
Parallelizer *parallelizer = NULL;
int num_threads = 16;
int sleep_time=0;

// Kotla : Diagnostic purposes
//FILE *dptr;

extern void * (start_request)(void *);


extern "C" void create_thread_arg(Thread_arg *t, int tid, Byz_req inb, Byz_rep outb, 
		       Byz_buffer non_det, int client, bool ro, Request_id req_id,
		       Rep_info *rep, int my_id,int view,Seqno seqno,int rep_id) {
  t->tid = tid;
  t->inb = inb;
  t->outb = outb;
  t->non_det = non_det;
  t->client = client;
  t->ro = (ro) ? 1 : 0;
  t->req_id = req_id;
  t->rep = rep;
  t->rep_id = rep_id;
  t->my_id = my_id;
  t->view = view;
  t->seqno = seqno;
  t->req = NULL;
}

int nfsd_dispatch1(Byz_req *inb, Byz_rep *outb, Byz_buffer *non_det, 
		   int client, bool ro, Request_id req_id, 
		   Rep_info *replies, int my_id,int view, Seqno seqno,int rep_id) {
  // This wrapper solves a problem that happens on linux with optimization.
  // nfsd_dispatch1 has C++ linkage as the replication library expects.
  //if (ro) {
    // Kotla : It should not come here Read only optimization is diabaled
    // th_assert(0," Read only optimization is diabled");
  //}

  // Kotla : Put all the arguments in a structure before passing
  create_thread_arg(&thread_arg[thread_id],thread_id,*inb,*outb,*non_det,client,ro,req_id,replies,my_id,view,seqno,rep_id);
 
  bool status = parallelizer->insert(client, req_id, seqno, &thread_arg[thread_id]);
  
  // if status is true then request is enqueued ; else the request is dropped
  if (status) {
    thread_id = (++thread_id) % max_out;
    // if thread_arg circular buffer is full then yeild for 
    // worker threads to service requests
    while (thread_arg[thread_id].res == 1) {
      fprintf(stderr,"Yielding as the buffer is full.\n");
      sched_yield();
    }
  }
  return status;
}

void non_det_choices(Seqno s, Byz_buffer *ndet) {
  th_assert(ndet->size >= (int)sizeof(struct timeval), "Non-det buffer is too small");
  ndet->size = sizeof(struct timeval);
  gettimeofday((struct timeval *)ndet->contents, 0);
}

bool check_non_det(Byz_buffer *non_det) {
  return TRUE;
  th_assert(non_det->size >= (int)sizeof(struct timeval), "Non-det buffer is too small");
  struct timeval t;
  gettimeofday(&t, 0);


  if ( (((struct timeval *)(non_det->contents))->tv_sec > 
	t.tv_sec + (-LOWER_THRESHOLD)) &&
       (((struct timeval *)(non_det->contents))->tv_sec 
	< t.tv_sec + UPPER_THRESHOLD) )
    return TRUE;

  fprintf(stderr, "\t\t> > > > Bad non-deterministic choices < < < <\n");
  return FALSE;
}

extern int FS_init(char *, char *);

#ifdef STATS
extern int num_get_file;
extern int num_get_dir;
extern int num_get_free;
extern int num_put;
extern int num_put_file;
extern int num_put_dir;
extern int num_put_free;
extern int numcalls[];
#endif

static void dump_profile(int sig) {

#ifdef STATS
  show_stats(); // temporario - retirarXXX RR-TODO!!
  printf("# gets for: files = %d dirs = %d free = %d\n", num_get_file, num_get_dir, num_get_free);
  printf("# puts = %d.  For: files = %d dirs = %d free = %d\n", num_put, num_put_file, num_put_dir, num_put_free);
  for (int i=0; i<NUM_CALLS; i++)
    printf("#of %d calls %5d, ",i, numcalls[i]);
  fprintf(stderr, "\nLookups cached %d. Reads cached %d.\n", lookup_cached, read_cached);
#endif

 profil(0,0,0,0);

 stats.print_stats();

 exit(0);
}

//char hostname[MAXHOSTNAMELEN+1];
char hostname[256];

int main(int argc, char **argv) {
  // Process command line options.
  char config[PATH_MAX];
  char config_priv[PATH_MAX];
  char dirname[PATH_MAX];
  int port = 0;
  hostname[0] = dirname[0] = config[0] = config_priv[0] = 0;


  // Process command line options.
  int opt;
  while ((opt = getopt(argc, argv, "c:p:h:d:r:t:n:")) != EOF) {
    switch (opt) {
    case 'h':
      strncpy(hostname, optarg, MAXHOSTNAMELEN+1);
      break;

    case 'd':
      strncpy(dirname, optarg, PATH_MAX);
      break;

    case 'p':
      strncpy(config_priv, optarg, PATH_MAX);
      break;

    case 'c':
      strncpy(config, optarg, PATH_MAX);
      break;

    case 'r':
      port = atoi(optarg);
      break;

    case 't':
      sleep_time = atoi(optarg);
      break;
 
    case 'n':
      num_threads = atoi(optarg);
      break;      
      
    default:
      fprintf(stderr, "%s -c config_file -p config_priv_file -d dir -h hoste", argv[0]);
      exit(-1);
    }
  }

  if (config[0] == 0) {
    // Try to open default file
    strcpy(config, "./config");
  }

  if (config_priv[0] == 0) {
    // Try to open default file
    char hname[MAXHOSTNAMELEN];
    if (gethostname(hname, MAXHOSTNAMELEN) < 0)
      sysfail("hafs replica: could not find local host name");
    sprintf(config_priv, "config_private/%s", hname);
  }

  if (dirname[0] == 0) {
    // Try to open default dir
    strcpy(dirname, "/tmp/hafs");
  }

  if (hostname[0] == 0) {
    // default hostname
    // Get my host name
    if (gethostname(hostname, MAXHOSTNAMELEN+1) < 0)
      sysfail("hafs: could not find local host name");
  }

  // signal handler to dump profile information and stats.
  struct sigaction act;
  act.sa_handler = dump_profile;
  sigemptyset (&act.sa_mask);
  act.sa_flags = 0;
  sigaction (SIGINT, &act, NULL);
  sigaction (SIGTERM, &act, NULL);


  // Initialize file system
  int npages = FS_init(hostname, dirname);

  if (npages < 0) {
    fprintf(stderr, "Error in FS_init. Check if I've been hit by a cosmic ray\n");
    exit(-1);
  }
  fprintf(stderr, "Init on port %d\n", port);

  // Kotla : Initialize the mutex
  if (pthread_mutex_init(&thread_mutex,NULL)) {
    fprintf(stderr,"Cannot initialize pthread mutex successfully. \n");
    exit(-1);
  }

  // Initialize mutex
  if (pthread_mutex_init(&req_queue_mutex,NULL)) {
    fprintf(stderr,"Cannot initialize pthread mutex successfully. \n");
    exit(-1);
  }
    
  // Initialize condition variable
  if (pthread_cond_init(&req_ready_cond,NULL)) {
    fprintf(stderr,"Cannot initialize pthread cond successfully. \n");
    exit(-1);
  }


  // Initialize the parallelizer
  parallelizer = new Parallelizer(checkpoint_interval,max_out);

  // Register nfs read state in parallelizer
  parallelizer->register_read_state(read_state);
 
  // Initialize replica
  if (Byz_init_replica(config, config_priv, npages,
		       nfsd_dispatch1, non_det_choices, sizeof(struct timeval),
		       check_non_det, get_obj, put_objs,
		       shutdown_state, restart_state, port,parallelizer) < 0) {
    fprintf(stderr, "Error in Byz_init. Check if I've been hit by a cosmic ray\n");
    exit(-1);
  }

  stats.zero_stats();
   
  // Per client locks for fine grained locking the replies structure 
  // to access per client reply strcuture

  // Number of locks = Number of principals 
  int num_locks = replica->num_p();

  // Kotla : If we use malloc to allocate memory for client_mutex
  // somehow locks are getting corrupted. Probably pthread allocation
  // of stacks on heap is corrupting this data. So making 
  // it a static array. 

  //    printf(" Number of principals : %d \n", num_locks);
  //    client_mutex = (pthread_mutex_t *) malloc (num_locks*(sizeof(pthread_mutex_t)));
  //    if ( client_mutex == NULL ) {
  //      fprintf(stderr,"Cannot allocate memory for locks. \n");
  //      exit(-1);
  //    }
   
  for (int i=0;i<num_locks;i++) {
    if (pthread_mutex_init(&client_mutex[i],NULL)) {
      fprintf(stderr,"Cannot initialize pthread mutex successfully. \n");
      exit(-1);
    }   
  }

  // Kotla : For diagnostic messages 
 //  dptr = fopen("/home/kotla/sent.log", "w");
//   if (dptr == NULL) {
//     exit(-1);
//   }

  // Kotla : Create thread specific file attr variables 
  tattr = (Thread_fattr *) malloc(num_threads*sizeof(Thread_fattr));

  // Kotla : Create maximum out standing requests number of request
  // queue entries
  thread_arg = (Thread_arg *) malloc(max_out*sizeof(Thread_arg));

  // Kotla : Spawn of worker threads. Currently we are having fixed 
  // number of threads. Disadvantage : We don't know  required number 
  // of outstanding threads to service requests in steady state so that
  // cpu is not idle. 
  // Optimization :  Modify to have apache like adaptive threaded model 
  // to control the number of threads. 

  for (int i=0;i<num_threads;i++) {
    // Kotla : Create a new worker thread
    printf("Creating thread with id : %d",i);
    pthread_create(&req_thread[i], NULL, start_request, (void*) &i); 
    
    // Set the fs attr variables
    tattr[i].last_inum = tattr[i].last_inum_from = tattr[i].last_inum_to =0;
    tattr[i].inum_clobbered = 0;
    tattr[i].prev_fid = 0;

    // Create rpc clients
    nfs_clnt[i] = clnt_create(hostname, NFS_PROGRAM, NFS_VERSION, "udp");
    if (!nfs_clnt[i]) {
      printf("Couldn't create client\n");
      exit(-1);
    }
    else {
      nfs_clnt[i]->cl_auth = authunix_create_default();
      if (!nfs_clnt[i]->cl_auth) {
	printf("Error in authunix create\n");
	exit(-1);
      }
    }
  }

  // Create nfs rpc client for agreement stage (which is 
  // not shared with worker threads
  nfs_clnt[num_threads] = clnt_create(hostname, NFS_PROGRAM, NFS_VERSION, "udp");
  if (!nfs_clnt[num_threads]) {
    printf("Couldn't create client\n");
    exit(-1);
  }
  else {
    nfs_clnt[num_threads]->cl_auth = authunix_create_default();
    if (!nfs_clnt[num_threads]->cl_auth) {
      printf("Error in authunix create\n");
      exit(-1);
    }
  }
 

  // Initialize fs structure (abstract data) mutex
  if (pthread_mutex_init(&fs_mutex,NULL)) {
       fprintf(stderr,"Cannot initialize pthread mutex successfully. \n");
       exit(-1);
  } 
 
  // Initialize fs structure (abstract data) mutex
  if (pthread_mutex_init(&chkpt_mutex,NULL)) {
       fprintf(stderr,"Cannot initialize pthread mutex successfully. \n");
       exit(-1);
  }
  
  // Loop executing requests.
  Byz_replica_run();
}


  
