#include<stdio.h>
#include<list>
#include<pthread.h>
#include "Lock.h"
#include "Req.h"
#include "types.h"
#include "arg.h"

extern pthread_cond_t req_ready_cond;
extern pthread_mutex_t req_queue_mutex;

class Parallelizer;

// exception class for accessing empty queue
class ReadEmptyQueue : public std::exception {
public:
  virtual const char* what() const throw() {
    return "Reading empty queue";
  }
};

// exception class for accessing empty queue
class InsertFullQueue : public std::exception {
public:
  virtual const char* what() const throw() {
    return "Inserting in full queue";
  }
};

// Request queue of pending requests.  
class RequestQueue {
private :
  std::list<Req *> obj; // List of pending requests
  Lock mutex;  // Lock for mutual exclusion 
  unsigned int max_elem; // Maximum nuber of elements that can be in list 
  Seqno next_chkpt; // Next chkpt 

 
  // number of elements
  int size() const {
    return obj.size();
  }

  // is queue empty?
  bool empty() const {
    return (obj.empty() == 0) ;
  }

  // is queue full?
  bool full() const {
    return (obj.size() == max_elem);
  }
  
  friend class Parallelizer;

public:
  
  // All the queue functions with wrappers
  RequestQueue();
  ~RequestQueue();

  
  RequestQueue(Seqno chkpt, int max_elements) { 
    max_elem = max_elements; next_chkpt = chkpt;
  }

  // insert element into the queue, throw execption
  // if queue is full
  // Callee should catch the InsertFullQueue exception
  void push (Req* elem) {

    mutex.lock();
    if (full()) {
      // Unlock before stack unwinds
      mutex.unlock();
      throw InsertFullQueue();
    }
    obj.push_back(elem);
    mutex.unlock();

  }

  // read element from the queue and return its value.
  // Throw execetion if queue is empty
  // Callee should catch the ReadEmptyQueue execption
  Req* pop () {
    mutex.lock();
    if (empty()) {
      // unlock before stack unwinds
      mutex.unlock();
      throw ReadEmptyQueue();
    }
    Req* elem(obj.front());
    obj.pop_front();
    mutex.unlock();

    return elem;
  }

  // Set next chkpt
  void set_chkpt( Seqno chkpt) {
    mutex.lock();
    next_chkpt = chkpt;
    mutex.unlock();
  }

  // get next chkpt
  Seqno get_chkpt() {
    mutex.lock();
    Seqno chkpt = next_chkpt;
    mutex.unlock();
    return chkpt;
  }

  void clear() {
    mutex.lock();
    obj.clear();
    mutex.unlock();
  }

  // This function is called when a request execution is complete.
  // This is called by the "EXEC" stage.
  void remove (Req * req) {

    // First remove the entry from the list of pending 
    mutex.lock();
    if (obj.empty()) {
      // unlock before stack unwinds
      mutex.unlock();
      throw ReadEmptyQueue();
    }
    //printf("Removing from RequestQueue :Req : %x Seq no : %qu Rid : %qu thread_id %d Olist size %d\n",
    //      req,req->req_num,req->rid,req->req_state->tid,req->olist.size());
      
    // Remove this request from the dependency list(olist) of all the 
    // requests that are dependent on this request.
    if (req->olist.size() != 0) {
      std::list<Req*>::iterator iter;
      for (iter = req->olist.begin(); iter != req->olist.end(); iter++) {
	Req * cur = *iter;
	if (cur == NULL) {continue;}
	//printf("Removing %qu from dlist of %qu \n", req->req_num,
	//   cur->req_num);
	cur->remove_dlist(req);
	// If "cur" have no dependencies make it ready
	if (cur->dlist.size() == 0) {
	  th_assert(cur->state == blocked," Request is not in blocked State");
	  cur->state = ready;
          //printf(" Request ready in remove : %qu \n",cur->req_num);
	  // Signal blocking threads that a request is ready
	  //pthread_mutex_lock(&req_queue_mutex);
          // Broadcast the signal if number of threads are small
	   pthread_cond_broadcast(&req_ready_cond);
	   //pthread_mutex_unlock(&req_queue_mutex);
	   //printf("Signalled and Unlocked the req_queue_mutex \n");
	}	
      }
    }
    
    obj.remove(req);
    mutex.unlock();
  }

  // Returns next ready request in the pending list of requests
  Req* next_request() {
    
    mutex.lock();
    std::list<Req*>::iterator iter;
    for (iter=obj.begin(); iter!=obj.end();iter++) {
      Req* cur = *iter;
      // Request is ready to be executed if there are no 
      // dependent requests (dlist) that are pending.  
      // Request beyond the next_chkpt is not executed until
      // the checkpoint is taken at next_chkpt.
      if ((cur->state != blocked) && (cur->state != running) && (cur->req_num <= next_chkpt)) {
	//fprintf(stderr," Ready Request State : %d id : %qu \n", cur->state,cur->rid);
	cur->state = running;
	mutex.unlock();
	return cur;
      }
    }
    mutex.unlock();

    return NULL;
  }
};
