import java.util.Hashtable;

/*
 *------------------------------------------------------------------
 *
 * LESS Group 
 * Computer Sciences Department 
 * University of Texas at Austin
 *
 * Cache ---
 *          Simple dummy cache. Keeps track of what has
 *          been inserted. Infinite size.
 *
 *          Store the size of each object. If request
 *          indicates "0" or negative size, update the
 *          request to reflect the cached size (some
 *          traces use "0" for hits.)
 *
 *          Synchronized to allow multi-threaded access.
 *
 *          Also keep track of "pending" prefetches.
 *          (Note that we assume that pending demand
 *          requests are immediately inserted. From the
 *          point of view of the simultion, they are hits --
 *          we don't need to wait for the dat to come back.
 *
 * $Date: 2003/01/06 06:16:30 $ $Id: Cache.java,v 1.1.1.1 2003/01/06 06:16:30 ypraveen Exp $
 *
 * Copyright (c) 2002 by Mike Dahlin.
 * 
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for noncommercial use without fee and
 * without written agreement is hereby granted, provided that 
 * the above copyright notice appear in all copies of this software.
 * 
 *------------------------------------------------------------------
 */
public class Cache{
private Hashtable stored;
private Hashtable pendingPref;

public Cache()
  {
    stored = new Hashtable();
  }

/*
 *------------------------------------------------------------------
 *
 * checkMissUpdateSize --
 *
 *          See if the specified request is a miss.
 *
 *          For bookeeping update the cached size if the
 *          trace has a new size for the object.
 *          (Do we really want to do this?)
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized boolean checkMissUpdateSize(RawReq r)
  {
    String key = makeKey(r);
    Integer size = (Integer)stored.get(key);
    if(size == null){
      return true; // It is a miss
    }
    if(size.intValue() == r.getSize()){
      return false; // Not a miss
    }
    if(r.getSize() > 0){
      stored.put(key, new Integer(r.getSize()));
    }
    else{
      r.setSize(size.intValue());
    }
    return false; // Not a miss
  }


/*
 *------------------------------------------------------------------
 *
 * updateSize --
 *
 *          For bookeeping update the cached size if the
 *          trace has a new size for the object.
 *          (Do we really want to do this?)
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void updateSize(RawReq r)
  {
    String key = makeKey(r);
    Integer size = (Integer)stored.get(key);
    // Should only do this for things in cache
    Assert.myAssert(size != null,
		    "Why update size for not in cache? " + key); 
    if(r.getSize() > 0){
      stored.put(key, new Integer(r.getSize()));
    }
  }
    



/*
 *------------------------------------------------------------------
 *
 * isCached --
 *
 *          Return true if the specified item is in the cache.
 *
 * Arguments:
 *      RawReq r
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized boolean isCached(RawReq r)
  {
    String key = makeKey(r);
    Integer size = (Integer)stored.get(key);
    if(size == null){
      return false; // miss
    } 
    return true; // hit
  }

/*
 *------------------------------------------------------------------
 *
 * insertIfCachable --
 *
 *          If the object is cachable, store it in the cache.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void insertIfCachable(RawReq r)
  {
    String key = makeKey(r);

    if(r.isCachable()){
      stored.put(key, new Integer(r.getSize()));
    }
  }


/*
 *------------------------------------------------------------------
 *
 * insertIfCachable --
 *
 *          store object in the cache whether it is cachable or not.
 *          (Used by ServerPopulator).
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
protected synchronized void insertIgnoreCachability(RawReq r)
  {
    String key = makeKey(r);

    stored.put(key, new Integer(r.getSize()));
  }




/*
 *------------------------------------------------------------------
 *
 * makeKey --
 *
 *         Assemble a key (URL) from the server and object ID
 *
 * Arguments:
 *      RawReq r
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */

public static String makeKey(RawReq r)
  {
    return r.getServer() + r.getObjId();
  }

};

