import java.util.LinkedList;
import java.util.Hashtable;

/*
 *------------------------------------------------------------------
 *
 * LESS Group 
 * Computer Sciences Department 
 * University of Texas at Austin
 *
 * DelayedFetchQ ---
 *
 * See design.txt
 * Wait a bit of time after getting a new list of stuff to
 * prefetch before starting to prefetch (this will allow in-line
 * images, etc. to be fetched.)
 *
 * This thing also keeps track of all outstanding prefetch
 * requests to ensure that we don't fetch the same thing multiple
 * times. See design.txt for issues.
 *
 * $Date: 2003/01/06 06:16:30 $ $Id: DelayedFetchQ.java,v 1.1.1.1 2003/01/06 06:16:30 ypraveen Exp $
 *
 * Copyright (c) 2002 by Mike Dahlin.
 * 
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for noncommercial use without fee and
 * without written agreement is hereby granted, provided that 
 * the above copyright notice appear in all copies of this software.
 * 
 *------------------------------------------------------------------
 */
public class DelayedFetchQ{
private LinkedList list;
private long EMBARGO_TIME_MS = 1;
private Hashtable pendingList;
private Cache cache;
private float threshold;

public DelayedFetchQ(Cache c, float thresh, long embargoMS)
{
  cache = c;
  list = new LinkedList();
  pendingList = new Hashtable();
  threshold = thresh;
  EMBARGO_TIME_MS = embargoMS;
}


/*
 *------------------------------------------------------------------
 *
 * putList --
 *
 *          Put a copy of a list of requests on the embargo list
 *          (where it will sit for embargotime until
 *          the DFQThread moves it to the real FetchQ
 *          where worker threads will actually issue the 
 *          request.
 *
 * Arguments:
 *      PredRawReq prefetchList[] -- the list of stuff to prefetch
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      Note: we keep a reference to the prefetchList you pass
 *      in. Dont change it after you pass it in.
 *
 *------------------------------------------------------------------
 */
public synchronized void putList(PredRawReq prefetchList[])
  {

    //    System.out.println("DQ: Put list: " + prefetchList.length + " allLists: " + list.size());

    long now = System.currentTimeMillis();
    DFQReq req = new DFQReq(prefetchList, now + EMBARGO_TIME_MS);
    list.addLast(req);
    notifyAll(); // Multiple prefetchable objects may have arrived
  }


/*
 *------------------------------------------------------------------
 *
 * prefetchDoneUpdateCache --
 *
 *          The prefetch is done.
 *          Remove the request from pendingList.
 *          Update the cache.
 *          Make these two updates atomically.
 *
 * Arguments:
 *      RawReq req
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void prefetchDoneUpdateCache(RawReq r, Cache c,
						 boolean worked)
  {
    String key = Cache.makeKey(r);
    Object o = pendingList.remove(key);
    Assert.myAssert(o != null, "Data structures are not in sync?");
    if(worked){
      c.insertIfCachable(r);
    }
  }

/*
 *------------------------------------------------------------------
 *
 * cancelAll --
 *
 *     Delete all pending requests from the list.
 * 
 * Arguments:
 *      None.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized void cancelAll()
  {

    list.clear();
    //    System.out.println("DQ: cancelAll:  allLists: " + list.size());
  }


/*
 *------------------------------------------------------------------
 *
 * get --
 *
 *       Return the next available request.
 *     
 *
 * Arguments:
 *      None.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized PredRawReq get()
  {
    PredRawReq got = null;
    got = getNextAvailable();
    while(got == null){
      // Wake up either when something inserted or when
      // embargo ends.
      long remaining = checkRemainingMS();
      if(remaining > 0){
	try{
	  //	  System.out.println("DFQ: wait " + remaining);
	  wait(remaining);  
	  //	  System.out.println("DFQ: wakes ");
	}
	catch(InterruptedException e){
	}
      }
      got = getNextAvailable();
    }
    String key = Cache.makeKey(got);
    pendingList.put(key, key);
    return got;
  }




/*
 *------------------------------------------------------------------
 *
 * getNextAvailable --
 *
 *          Return the next avaialable prefetch target
 *          (filtered so that we dont fetch something
 *          below threshold or something already
 *          in progress or something already cached.)
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      Returns a prefetch request or null if (a) no
 *      pending requests or (b) all pending requests
 *      are embargoed.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private synchronized PredRawReq getNextAvailable()
  {
    int skipped = 0;
    PredRawReq got = null;
    while(!headIsEmbargoedOrEmpty() && got == null){
      Assert.myAssert(!headIsEmbargoedOrEmpty());
      DFQReq dfqr = (DFQReq)list.getFirst();
      Assert.myAssert(dfqr != null);
      long now = System.currentTimeMillis();
      got = dfqr.getNext(now);
      if(dfqr.isConsumed()){
	list.removeFirst();
      }
      Assert.myAssert(got != null);
      if(!checkShouldPrefetch(got)){
	got = null; // Go around again and try the next one
	skipped++;
      }
    }
    //    System.out.println("DFQ: Skipped " + skipped + " and found "+ (got == null?"nothing":"something"));
    return got;
  }







/*
 *------------------------------------------------------------------
 *
 * checkShouldPrefetch -- 
 *
 *    Check to see if a specified request is worth prefetching
 *       o request threshold > threshold
 *       o request not already in cache
 *       o request not already being precessed
 *
 * Arguments:
 *      PredRawReq proposal -- object being considered
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private synchronized boolean checkShouldPrefetch(PredRawReq proposal)
  {
    String key;

    key = Cache.makeKey(proposal);
    if(proposal.getProb() >= threshold){
      if(!pendingList.containsKey(key)){
	if(!cache.isCached(proposal)){
	  return true;
	}
      }
    }
    return false;
  }


/*
 *------------------------------------------------------------------
 *
 * headIsEmbargoedOrEmpty --
 *
 *          Return true if there is no work to be had
 *          on the list.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private synchronized boolean headIsEmbargoedOrEmpty()
  {
    if(list.isEmpty()){
      return true;
    }
    long remain = checkRemainingMS();
    Assert.myAssert(remain < Long.MAX_VALUE, "Since list not empty");
    if(remain > 0){
      return true;
    }
    return false;
  }

 /*
  *------------------------------------------------------------------
  *
  * checkRemainingMS --
  *
  *     Return the amount of time remaining on the embargo 
  *     of the top element (or Long.MAX_VALUE if no top element.)
  *
  * Arguments:
  *      None.
  *
  * Results:
  *      None.
  *
  * Side effects:
  *      None.
  *
  *------------------------------------------------------------------
  */
 private synchronized long checkRemainingMS()
   {
     DFQReq head;
     long remainingMS = Long.MAX_VALUE;
     if(list.size() > 0){
       head = (DFQReq)list.getFirst();
       remainingMS = head.releaseTime() - System.currentTimeMillis();
     }
     return remainingMS;
   }





/*
 *------------------------------------------------------------------
 *
 * class DFQReq --
 *
 *     Just a container for embargoed requests.
 *
 *------------------------------------------------------------------
 */
protected class DFQReq{
private PredRawReq list[];
private long wakeupTime;
private int consumed;      // How many have been consumed
protected DFQReq(PredRawReq l[], long w)
  {
    Assert.myAssert(w > (long)1000000000,
	 "Remember -- wakeup time is absolute time, not a relative delay");
    wakeupTime = w;
    list = l;
    consumed = 0;
  }
protected long releaseTime()
  {
    return wakeupTime;
  }
protected PredRawReq getNext(long checkTime)
  {
    Assert.myAssert(checkTime >= wakeupTime, 
		    new String(checkTime + " / " + wakeupTime));
    if(consumed < list.length){
      PredRawReq prr = (PredRawReq)list[consumed];
      consumed++;
      return prr;
    }
    else{
      return null;
    }
  }
protected boolean isConsumed()
  {
    if(consumed == list.length){
      return true;
    }
    return false;
  }
};





public static void main(String argv[])
  {
    String[] testList = {"http://www.cs.utexas.edu/index.html",
			 "http://www.cs.utexas.edu/users/dahlin/index.html",
			 "http://www.dahlins.com"};
    String[] testList1 = {"http://www.cs.utexas.edu/index1.html",
			 "http://www.cs.utexas.edu/users/dahlin/index1.html",
			 "http://www.dahlins1.com"};
    String[] testList2 = {"http://www.cs.utexas.edu/index2.html",
			 "http://www.cs.utexas.edu/users/dahlin/index2.html",
			 "http://www.dahlins2.com"};
    String[] testList3 = {"http://www.cs.utexas.edu/index3.html",
			 "http://www.cs.utexas.edu/users/dahlin/index3.html",
			 "http://www.dahlins3.com"};
    String[] testList4 = {"http://www.cs.utexas.edu/index4.html",
			 "http://www.cs.utexas.edu/users/dahlin/index4.html",
			 "http://www.dahlins4.com"};
    int ii;
    Assert.myAssert(testList.length == testList2.length, "I'm lazy.");

    Cache c = new Cache();
    DelayedFetchQ q = new DelayedFetchQ(c, 0, 1000);
    PredRawReq safe[] = new PredRawReq[testList.length];
    PredRawReq prr[] = new PredRawReq[testList.length];
    PredRawReq prr1[] = new PredRawReq[testList1.length];
    PredRawReq prr2[] = new PredRawReq[testList2.length];
    PredRawReq prr3[] = new PredRawReq[testList3.length];
    PredRawReq prr4[] = new PredRawReq[testList4.length];
    PredRawReq prr7[] = new PredRawReq[testList4.length];
    PredRawReq prr8[] = new PredRawReq[testList4.length];
    long putTime = 0;
    long getTime;
    long startTime;

    try{

      for(ii = 0; ii < testList.length; ii++){
	prr[ii] = new PredRawReq(new Prediction(testList[ii], (float)1.0));
	safe[ii] = new PredRawReq(new Prediction(testList[ii], (float)1.0));
      }
      putTime = System.currentTimeMillis();
      q.putList(prr);
      for(ii = 0; ii < testList.length; ii++){
	PredRawReq got = q.get();
	getTime = System.currentTimeMillis();
	Assert.myAssert(getTime - putTime > q.EMBARGO_TIME_MS);
	Assert.myAssert(getTime - putTime < q.EMBARGO_TIME_MS + 1000,
   "Not quite an assert, but we shouldn't wait much longer than expected.");
	Assert.myAssert(got.equals(safe[ii]));
      }
      Assert.myAssert(q.pendingList.size() == 3);
      Assert.myAssert(q.list.size() == 0);
      System.out.println("Pass test 1.");


      //
      // Since "the worker" has removed the requests
      // they can no longer be cancelled.
      //
      q.cancelAll();
      Assert.myAssert(q.pendingList.size() == 3);
      Assert.myAssert(q.list.size() == 0);
      System.out.println("Pass test 2.");



      Assert.myAssert(q.list.size() == 0);
      startTime = System.currentTimeMillis();
      for(ii = 0; ii < testList.length; ii++){
	prr1[ii] = new PredRawReq(new Prediction(testList1[ii], (float)1.0));
      }
      q.putList(prr1);
      Thread.sleep(q.EMBARGO_TIME_MS/2);
      for(ii = 0; ii < testList.length; ii++){
	prr2[ii] = new PredRawReq(new Prediction(testList2[ii], (float)1.0));
      }
      q.putList(prr2);
      Thread.sleep(q.EMBARGO_TIME_MS/2);
      for(ii = 0; ii < testList.length; ii++){
	prr3[ii] = new PredRawReq(new Prediction(testList3[ii], (float)1.0));
      }
      q.putList(prr3);
      Thread.sleep(q.EMBARGO_TIME_MS/2);
      for(ii = 0; ii < testList.length; ii++){
	prr4[ii] = new PredRawReq(new Prediction(testList4[ii], (float)1.0));
      }
      q.putList(prr4);
      Assert.myAssert(q.pendingList.size() == 3);
      Assert.myAssert(q.list.size() == 4);
      q.cancelAll();
      Assert.myAssert(q.pendingList.size() == 3);
      System.out.println("Pass test 3.");


      System.out.println("Pass test 4.");

      //
      // Now get testList1 out.
      //
      for(ii = 0; ii < testList.length; ii++){
	prr1[ii] = new PredRawReq(new Prediction(testList1[ii], (float)1.0));
      }
      putTime = System.currentTimeMillis();
      q.putList(prr1);
      for(ii = 0; ii < testList1.length; ii++){
	PredRawReq got = q.get();
	getTime = System.currentTimeMillis();
	Assert.myAssert(getTime - putTime > q.EMBARGO_TIME_MS);
	Assert.myAssert(getTime - putTime < q.EMBARGO_TIME_MS + 1000,
   "Not quite an assert, but we shouldn't wait much longer than expected.");
			
	Assert.myAssert(got.equals(prr1[ii]));
      }
      System.out.println("Pass test 5.");


      
      //
      // Now pretend the first set of prefetches are done
      //
      for(ii = 0; ii < testList.length; ii++){
	q.prefetchDoneUpdateCache(prr[ii], c, true);
      }
      Assert.myAssert(q.pendingList.size() == 3, "Size is " + q.pendingList.size());
      System.out.println("Pass test 6.");




      System.exit(0);



    }
    catch(InterruptedException ie){
      System.out.println("Interrupted Exception. Test fails. " + ie);
      System.exit(-1);
    }
    catch(RuntimeException e){
      System.out.println("Runtime exception. Test fails." + e);
      e.printStackTrace();
      System.exit(-1);
    }
  }
};
