import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.FileInputStream;
import java.io.IOException;

/*
 *------------------------------------------------------------------
 *
 * LESS Group 
 * Computer Sciences Department 
 * University of Texas at Austin
 *
 * InputTrace ---
 *          Return a trace of RawReqs from an input stream.
 *
 * $Date: 2003/01/06 06:16:30 $ $Id: InputTrace.java,v 1.1.1.1 2003/01/06 06:16:30 ypraveen Exp $
 *
 * Copyright (c) 2002 by Mike Dahlin
 * 
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for noncommercial use without fee and
 * without written agreement is hereby granted, provided that 
 * the above copyright notice appear in all copies of this software.
 * 
 *------------------------------------------------------------------
 */
public class InputTrace{

private String serverFilter; // All or a serverId
private String clientFilter; // All or a clientId
private BufferedReader in;
private int format;
private float timeFactor;
private long firstRecordTimeMS = UNINITIALIZED_TIME;
private long firstRealTimeMS = UNINITIALIZED_TIME;

private static final long UNINITIALIZED_TIME = -999999;
private static final int FORMAT_BU = 0;
private static final int FORMAT_SQUID = 1;
private static final int FORMAT_OLYMPIC = 2;

/*
 *------------------------------------------------------------------
 *
 * InputTrace --
 *
 *          Constructor.
 *
 * Arguments:
 *      InputStream inStrm - input stream containig trace
 *      String traceFormat - bu or squid
 *      String client  - "ALL" or clientId
 *      String server - "ALL" or serverId
 *      float timeFact - > 1.0 speeds up trace replay v. real time
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
  public InputTrace(InputStream inStrm, String traceFormat, 
		    String client, String server, float timeFact)
  {
    in = new BufferedReader(new InputStreamReader(inStrm));
    Assert.myAssert(traceFormat.equals("bu") || traceFormat.equals("squid") || traceFormat.equals("olympic"));
    if(traceFormat.equals("bu")){
      format = FORMAT_BU;
    }
    else if(traceFormat.equals("squid")){
      format = FORMAT_SQUID;
    }
    else if(traceFormat.equals("olympic")){
      format = FORMAT_OLYMPIC;
    }
    else{
      Assert.myAssert(false, "Bad trace format");
    }
    timeFactor = timeFact;
    serverFilter = server;
    clientFilter = client;
  }

/*
 *------------------------------------------------------------------
 *
 * next --
 *
 *          Return the next trace record *at the appropriate 
 *          scaled wallclock time*. 
 *
 * Arguments:
 *      none.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public RawReq next() 
  throws EndOfInputException, MalformedRecordException
  {
    RawReq r = nextNoWait();
    sleepUntilScaledTime(r.getTimeMS());
    return r;
  }



/*
 *------------------------------------------------------------------
 *
 * nextNoWait() --
 *
 *          Return the next record that meets the client/server/method
 *          filter.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
protected RawReq nextNoWait()
  throws EndOfInputException, MalformedRecordException
  {
    RawReq r;
    do{
      switch(format){
      case FORMAT_BU:
	r = new BURawReq(in, false, false);
	break;
      case FORMAT_SQUID:
	r = new SquidRawReq(in, false);
	break;
      case FORMAT_OLYMPIC:
	  r = new OlympicRawReq(in, false) ;
	  break ;
      default:
	Assert.myAssert(false, "Bad format in InputTrace::nextNoWait");
	r = null;
      }
    }while(!meetsFilter(r));
    return r;
  }


/*
 *------------------------------------------------------------------
 *
 * meetsFilter --
 *
 *          Check to see if the specified request is
 *          one that the program wants to see.
 *
 * Arguments:
 *      RawReq r -- the next record in the trace.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private boolean meetsFilter(RawReq r)
  {
    String rServer = r.getServer();
    String rClient = r.getClient();
    String rMethod = r.getMethod();

    if(!serverFilter.equals("ALL") && !serverFilter.equalsIgnoreCase(rServer)){
      return false;
    }
    if(!clientFilter.equals("ALL") && !clientFilter.equalsIgnoreCase(rClient)){
      return false;
    }
    if(!rMethod.equalsIgnoreCase("GET")){
      return false;
    }
    return true;
  }

/*
 *------------------------------------------------------------------
 *
 * sleepUntilScaledTime --
 *
 *          Sleep until the wallclock time exceeds
 *          (nextRecordTime - firstRecordTime)/timeFactor + firstRealTime
 *
 * Arguments:
 *      long nextRecordTimeMS -- the time from the trace when
 *                               the pending record occurred.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */

private void sleepUntilScaledTime(long nextRecordTimeMS)
  {
    long remainingMS;
    long now;
    long scaled;
    long diff;
    long done;

    if(firstRecordTimeMS == UNINITIALIZED_TIME){
      firstRecordTimeMS = nextRecordTimeMS;
      firstRealTimeMS = System.currentTimeMillis();
    }
    
    Assert.myAssert(firstRealTimeMS != UNINITIALIZED_TIME);
    Assert.myAssert(firstRecordTimeMS != UNINITIALIZED_TIME);
    
    now = System.currentTimeMillis();
    diff = nextRecordTimeMS - firstRecordTimeMS;
    scaled = (long)(((float)diff)/timeFactor);
    done = firstRealTimeMS + scaled;

    remainingMS = done - now;
    while(remainingMS > 0){
      try{
	Assert.myAssert(remainingMS > 0);
	Thread.sleep(remainingMS);
	now = System.currentTimeMillis();
	diff = nextRecordTimeMS - firstRecordTimeMS;
	scaled = (long)(((float)diff)/timeFactor);
	done = firstRealTimeMS + scaled;
	remainingMS = done - now;
      }
      catch(InterruptedException e){
	// Keep sleeping.
      }
    }
  }



/*
 *------------------------------------------------------------------
 *
 * main --
 *
 *          Unit test.
 *
 * Arguments:
 *      None.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public static void main(String argv[])
  {
    int testNum = -1;
    int count = -1;
    long start = 0;
    long stop = 0;
    InputTrace t;
    FileInputStream s;

    try{

      //
      // Test 6
      //
      count = 0;
      testNum = 6;
      s = new FileInputStream("test-squid.dat");
      // NB: Raw trace is 50006 seconds long w 14 records
      //     12 to same server
      //     13 from same client
      //     13 are GET
      t = new InputTrace(s, "squid", "146.110.15.179", "www.voltronforce.com", (float)100000.0);
      try{
	start = System.currentTimeMillis();
	while(true){
	  RawReq r = t.next();
	  count++;
	}
      }
      catch(EndOfInputException e){
	stop = System.currentTimeMillis();
	Assert.myAssert(count == 10, "Read squid server and client filter. " 
			+ count);
	Assert.myAssert(stop - start < 1500, "Should not take more than 1 second or so to run");
	System.out.println("Pass test 6");
      }
      


      //
      // Test 5
      //
      count = 0;
      testNum = 5;
      s = new FileInputStream("test-squid.dat");
      // NB: Raw trace is 50006 seconds long w 14 records
      //     12 to same server
      //     13 from same client
      //     13 are GET
      t = new InputTrace(s, "squid", "146.110.15.179", "ALL", (float)100000.0);
      try{
	start = System.currentTimeMillis();
	while(true){
	  RawReq r = t.next();
	  count++;
	}
      }
      catch(EndOfInputException e){
	stop = System.currentTimeMillis();
	Assert.myAssert(count == 12, "Read squid client. " + count);
	Assert.myAssert(stop - start < 1500, "Should not take more than 1 second or so to run");
	System.out.println("Pass test 5");
      }
      


      //
      // Test 4
      //
      count = 0;
      testNum = 4;
      s = new FileInputStream("test-squid.dat");
      // NB: Raw trace is 50006 seconds long w 14 records
      //     12 to same server
      //     13 from same client
      //     13 are GET
      t = new InputTrace(s, "squid", "ALL", "ALL", (float)100000.0);
      try{
	start = System.currentTimeMillis();
	while(true){
	  RawReq r = t.next();
	  count++;
	}
      }
      catch(EndOfInputException e){
	stop = System.currentTimeMillis();
	Assert.myAssert(count == 13, "Read all squid. " + count);
	Assert.myAssert(stop - start < 1500, "Should not take more than 1 second or so to run");
	System.out.println("Pass test 4");
      }
      



      //
      // Test 3 -- filter client
      //
      count = 0;
      testNum = 3;
      s = new FileInputStream("test-bu.dat");
      // NB: Raw trace is 256 seconds long w 100 records
      t = new InputTrace(s, "bu", "eeyore", "ALL", (float)1000.0);
      try{
	start = System.currentTimeMillis();
	while(true){
	  RawReq r = t.next();
	  count++;
	}
      }
      catch(EndOfInputException e){
	stop = System.currentTimeMillis();
	Assert.myAssert(count == 34, "Read clientFilter bu. " + count);
	Assert.myAssert(stop - start < 1500, "Should not take more than 1 second or so to run");
	System.out.println("Pass test 3");
      }
      

      //
      // Test 2 -- filter server
      //
      count = 0;
      testNum = 2;
      s = new FileInputStream("test-bu.dat");
      // NB: Raw trace is 256 seconds long w 100 records
      t = new InputTrace(s, "bu", "ALL", "cs-www.bu.edu", (float)1000.0);
      try{
	start = System.currentTimeMillis();
	while(true){
	  RawReq r = t.next();
	  count++;
	}
      }
      catch(EndOfInputException e){
	stop = System.currentTimeMillis();
	Assert.myAssert(count == 40, "Read serverfilter bu. " + count);
	Assert.myAssert(stop - start < 1500, "Should not take more than 1 second or so to run");
	System.out.println("Pass test 2");
      }
      





      

      //
      // Test 1 -- slower
      //
      count = 0;
      testNum = 1;
      s = new FileInputStream("test-bu.dat");
      // NB: Raw trace is 256 seconds long w 100 records
      t = new InputTrace(s, "bu", "ALL", "ALL", (float)100.0);
      try{
	start = System.currentTimeMillis();
	while(true){
	  RawReq r = t.next();
	  count++;
	}
      }
      catch(EndOfInputException e){
	stop = System.currentTimeMillis();
	Assert.myAssert(count == 100, "Read all bu.");
	Assert.myAssert(stop - start > 2560, "Should take more than 2.5 seconds to run");
	Assert.myAssert(stop - start < 4000, "Should not take more than 4 seconds to run");
	System.out.println("Pass test 1");
      }
      




      //
      // Test 0
      //
      count = 0;
      testNum = 0;
      s = new FileInputStream("test-bu.dat");
      // NB: Raw trace is 256 seconds long w 100 records
      t = new InputTrace(s, "bu", "ALL", "ALL", (float)1000.0);
      try{
	start = System.currentTimeMillis();
	while(true){
	  RawReq r = t.next();
	  count++;
	}
      }
      catch(EndOfInputException e){
	stop = System.currentTimeMillis();
	Assert.myAssert(count == 100, "Read all bu.");
	Assert.myAssert(stop - start < 1500, "Should not take more than 1 second or so to run");
	System.out.println("Pass test 0");
      }



    }
    catch(IOException i){
      i.printStackTrace();
      Assert.myAssert(false, "IO Exception test " + testNum);
    }
    catch(MalformedRecordException m){
      m.printStackTrace();
      Assert.myAssert(false, "Malformed Record Exception test " + testNum);
    }
  }
};


