import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;
import java.io.*;
import java.util.*;

/*
 *------------------------------------------------------------------
 *
 * LESS Group 
 * Computer Sciences Department 
 * University of Texas at Austin
 *
 * MarkovPrefetchPredictor ---
 *          Given a request, decide what, if anything, should
 *          be prefetched once this request has been fetched.
 *
 *          Uses a variation of prediction by partial matching.
 * 
 * $Date: 2003/02/10 05:58:43 $ $Id: MarkovPrefetchPredictor.java,v 1.2 2003/02/10 05:58:43 ypraveen Exp $
 *
 * Copyright (c) 2002 by Mike Dahlin.
 * 
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for noncommercial use without fee and
 * without written agreement is hereby granted, provided that 
 * the above copyright notice appear in all copies of this software.
 * 
 *------------------------------------------------------------------
 */
public class MarkovPrefetchPredictor implements PrefetchPredictor{

  Hashtable predictionTable; // Contains key --> Prediction[]
  Hashtable clientHistories; // Contains key --> ClientContext

  // assume we predict based on up to MAX_HISTORY
  // html files of context
  // to predict which objects will appear
  // within MAX_WINDOW after the context
  // Both MAX_HISTORY and MAX_WINDOW
  // are measured in terms of .html file hops
private static int MAX_WINDOW = -1;
private static int MAX_HISTORY = -1;

private final static int CHECK_N_CACHES_SIM_SEPARATE = 0;

/*
 *------------------------------------------------------------------
 *
 * MarkovPrefetchPredictor --
 *
 *          Constructor that reads predictionTable from a
 *          serialized file.
 *
 * Arguments:
 *      String stateFileName -- name of file containing predictionTable
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public MarkovPrefetchPredictor(String stateFileName)
  {
    try{
      FileInputStream fis = new FileInputStream(stateFileName);
      ObjectInputStream ois = new ObjectInputStream(fis);

      MAX_HISTORY = ((Integer)ois.readObject()).intValue();
      MAX_WINDOW = ((Integer)ois.readObject()).intValue();
      predictionTable = (Hashtable)ois.readObject();
      clientHistories = new Hashtable();
      fis.close();
    }
    catch(IOException i){
      System.out.println("IO Exception loading prediction state: " 
			 + i.toString());
      i.printStackTrace();
      System.exit(-1);
    }
    catch(ClassNotFoundException cnfe){
      System.out.println("IO Exception loading prediction state: " 
			 + cnfe.toString());
      cnfe.printStackTrace();
      System.exit(-1);
    }
  }


/*
 *------------------------------------------------------------------
 *
 * getList --
 *
 *          The key PrefetchPredictor interface. Get a list of predictions
 *          for the current request.
 *
 *          For safety in this multi-threaded world, we return a copy
 *          of the prediction list. (Things would get real ugly
 *          real fast if the predictions we return were to be changed.
 *          To bad java doesn't have "const" variables.)
 *
 *          New cleverness: if the list is LONG_LIST, then
 *          return the first LONG_LIST part plus LONG_FRAC
 *          of the rest.
 *
 * Arguments:
 *      RawReq r -- the current request.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public synchronized PredRawReq[] getList(RawReq r)
  {
    int ii;
    int histLength;
    PredRawReq retCopy[] = null;
    boolean done = false;
    String key = Cache.makeKey(r);
    ClientContext clCtx = (ClientContext)clientHistories.get(r.getClient());
    if(clCtx == null){
      clCtx = new ClientContext();
      clientHistories.put(r.getClient(), clCtx);
    }
    clCtx.append(key, MAX_HISTORY + MAX_WINDOW + 1);
    
    
    for(histLength = MAX_HISTORY; !done && histLength >= 0; histLength--){
      String history = clCtx.getHistory(0, histLength);
      if(history != null){
	Prediction p[] = (Prediction [])predictionTable.get(history);
	if(p != null){
	  retCopy = makeRetCopy(p);
	  return retCopy;
	}
      }
    }
    // Should we allow this -- returning no prediction
    // Alternative would be to assume that there is always at least a
    // default -- markov-0 -- prediction
    return retCopy;
  }
/*
 *------------------------------------------------------------------
 *
 * makeRetCopy --
 *
 *        Make a copy of the listto be returned (transforming
 *        the Predictions into PredRawReqs).  
 *        
 *          New cleverness: if the list is LONG_LIST, then
 *          return the first LONG_LIST part plus LONG_FRAC
 *          of the rest.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private int lastListSet = 0;
private static final int LONG_LIST = 10000;
private static final int LONG_FRAC = 10;    // Send 1/10th
private boolean printedWarning = false;
private synchronized PredRawReq[] makeRetCopy(Prediction[] p)
  {
    if(!printedWarning){
      System.out.println("WARNING: MarkovPrefetchPredictor returns 1/" + LONG_FRAC + " of list if list is longer than " + LONG_LIST);
      printedWarning = true;
    }

    int length;
    if(p.length > LONG_LIST){
      length = LONG_LIST + (p.length - LONG_LIST)/LONG_FRAC;
    }
    else{
      length = p.length;
    }
    int idest = 0;
    int isrc = 0;
    PredRawReq retCopy[] = new PredRawReq[length];
    while(idest < length){
      retCopy[idest] = new PredRawReq(p[isrc]);
      idest++;
      if(idest == LONG_LIST){
	isrc += lastListSet; // Skip to get different parts
	lastListSet = (lastListSet + 1) % LONG_FRAC;
      }
      if(idest > LONG_LIST){
	isrc += LONG_FRAC;
      }
      else{
	isrc++;
      }
    }
    return retCopy;
  }
/*
 *------------------------------------------------------------------
 *
 * Main --
 *
 *          The standalone program can do two things.
 *          First, it can generate a prefetch list based on a trace.
 *          Second, given a prefetch list (in a file) and a trace
 *          (in stdin) it can report on what hit rates could be
 *          achieved by prefetching.
 *
 * Arguments:
 *      See below.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
public static void main(String argv[])
  {
    PrefetchClient.printArgs(argv);
    int c;

    //
    // Globals set by arguments in standalone mode.
    //
    final float DEFAULT_DUMMY_THRESH = (float)-1.0;
    String client = "UNINITIALIZED";
    String server = "UNINITIALIZED";
    float prefThresh = DEFAULT_DUMMY_THRESH;
    String traceFormat = "UNINITIALIZED";
    String prefetchStateFile = "UNINITIALIZED";
    String mode = "UNINITIALIZED";
    int checkNCaches = 1;        // Default -- 1 shared cache
    int checkMaxListLen = Integer.MAX_VALUE;

    final int MAX_OPT = 10;
    LongOpt[] longopts = new LongOpt[MAX_OPT];
    longopts[0] = new LongOpt("client", LongOpt.REQUIRED_ARGUMENT, 
			      null, 0);
    longopts[1] = new LongOpt("server", LongOpt.REQUIRED_ARGUMENT, 
			      null, 1);
    longopts[2] = new LongOpt("prefThresh", LongOpt.REQUIRED_ARGUMENT, 
			      null, 2);
    longopts[3] = new LongOpt("traceFormat", LongOpt.REQUIRED_ARGUMENT, 
			      null, 3);
    longopts[4] = new LongOpt("prefetchStateFile", LongOpt.REQUIRED_ARGUMENT, 
			      null, 4);
    longopts[5] = new LongOpt("mode", LongOpt.REQUIRED_ARGUMENT, 
			      null, 5);
    longopts[6] = new LongOpt("maxWindow", LongOpt.REQUIRED_ARGUMENT, 
			      null, 6);
    longopts[7] = new LongOpt("maxHistory", LongOpt.REQUIRED_ARGUMENT, 
			      null, 7);
    longopts[8] = new LongOpt("checkNCaches", LongOpt.REQUIRED_ARGUMENT, 
			      null, 8);
    longopts[9] = new LongOpt("checkMaxList", LongOpt.REQUIRED_ARGUMENT, 
			      null, 9);
    Getopt g = new Getopt("MarkovPrefetchPredictor", argv, "", longopts, true);

    while((c = g.getopt()) != -1){
      switch(c){
      case 0:
	client = g.getOptarg();
	break;
      case 1:
	server = g.getOptarg();
	break;
      case 2:
	prefThresh = (new Float(g.getOptarg())).floatValue();
	break;
      case 3:
	traceFormat = g.getOptarg();
	Assert.myAssert(traceFormat.equals("bu")
			|| traceFormat.equals("olympic")
			|| traceFormat.equals("squid"),
		      "-traceFormat must be 'bu' or 'squid'");
	break;
      case 4:
	prefetchStateFile = g.getOptarg();
	break;
      case 5:
	mode = g.getOptarg();
	Assert.myAssert(mode.equals("gen") 
			|| mode.equals("check") 
			|| mode.equals("unit"),
			"-mode must be 'gen' or 'check' or 'unit'");
	break;
      case 6:
	MAX_WINDOW = (new Integer(g.getOptarg())).intValue();
	break;
      case 7:
	MAX_HISTORY = (new Integer(g.getOptarg())).intValue();
	break;
      case 8:
	checkNCaches = (new Integer(g.getOptarg())).intValue();
	Assert.myAssert(checkNCaches == CHECK_N_CACHES_SIM_SEPARATE || checkNCaches > 0,
			"-checkNCaches must be 0 (separate caches) or >0 (n shared caches)");
	break;
      case 9:
	checkMaxListLen = (new Integer(g.getOptarg())).intValue();
	if(checkMaxListLen == -1){
	  checkMaxListLen = Integer.MAX_VALUE;
	}
	Assert.myAssert(checkMaxListLen > 0);
	break;
      case '?':
	System.out.println("The option '" + (char)g.getOptopt() 
			   + "' is not valid.");
	System.exit(-1);
	break;
      default:
	System.out.println("getopt() returned " + c);
	System.exit(-1);
	break;
      }
    }

    if(mode.equals("gen")){
      Assert.myAssert(!client.equals("UNINITIALIZED"), "-client is required");
      Assert.myAssert(!server.equals("UNINITIALIZED"), "-server is required");
      Assert.myAssert(prefThresh != DEFAULT_DUMMY_THRESH, 
		    "You must set -prefThresh when running in mode -gen");
      Assert.myAssert(!traceFormat.equals("UNINITIALIZED"), 
		      "-traceFormat [bu|squid] is required");
      Assert.myAssert(!prefetchStateFile.equals("UNINITIALIZED"),
		      "-prefetchStateFile is required");
      Assert.myAssert(MAX_HISTORY != -1, "-maxHistory is required");
      Assert.myAssert(MAX_WINDOW != -1, "-maxWindow is required");
      InputTrace trace = new InputTrace(System.in, traceFormat, 
					client, server, (float)1000000000);
      preprocessTrace(trace,  prefThresh, prefetchStateFile);
    }
    else if(mode.equals("check")){
      Assert.myAssert(!client.equals("UNINITIALIZED"), "-client is required");
      Assert.myAssert(!server.equals("UNINITIALIZED"), "-server is required");
      //      Assert.myAssert(prefThresh == DEFAULT_DUMMY_THRESH, 
      //	      "You may not set -prefThresh when running in mode -check (it is ignored)");
      Assert.myAssert(!traceFormat.equals("UNINITIALIZED"), 
		      "-traceFormat [bu|squid] is required");
      Assert.myAssert(!prefetchStateFile.equals("UNINITIALIZED"),
		      "-prefetchStateFile is required");
      Assert.myAssert(MAX_HISTORY == -1, "-maxHistory is not allowed -- reading from state file");
      Assert.myAssert(MAX_WINDOW == -1, "-maxWindow is not allowed -- reading from state file");
      InputTrace trace = new InputTrace(System.in, traceFormat, 
					client, server, (float)1000000000);
      MarkovPrefetchPredictor p = new MarkovPrefetchPredictor(prefetchStateFile);
      p.checkTrace(trace, checkNCaches, checkMaxListLen, prefThresh);
      }
    else if(mode.equals("unit")){
      Assert.myAssert(client.equals("UNINITIALIZED"), "-client is not permitted for -mode unit");
      Assert.myAssert(server.equals("UNINITIALIZED"), "-server is not permitted for -mode unit");
      Assert.myAssert(prefThresh == DEFAULT_DUMMY_THRESH, 
		      "You may not set -prefThresh when running in mode -unit (it is ignored)");
      Assert.myAssert(traceFormat.equals("UNINITIALIZED"), 
		      "-traceFormat is not permitted for -mode unit");
      Assert.myAssert(!prefetchStateFile.equals("UNINITIALIZED"),
		      "-prefetchStateFile is not permitted (we always use test-squid-2.pred)");
      Assert.myAssert(MAX_HISTORY == -1, "-maxHistory is not allowed -- reading from state file");
      Assert.myAssert(MAX_WINDOW == -1, "-maxWindow is not allowed -- reading from state file");
      MarkovPrefetchPredictor p = new MarkovPrefetchPredictor(prefetchStateFile);
      p.unit();
    }
    else{
      Assert.myAssert(false);
    }
  }

/*
 *------------------------------------------------------------------
 *
 * unit --
 *
 *        Check some facts about test-squid-2.pred
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private void unit()
  {
    String input[] = new String[22];

    input[1] = "http://www.server1.com/";           //
    input[2] = "http://www.server1.com/file1.html"; //
    input[3] = "http://www.server1.com/file1.gif";
    input[4] = "http://www.server1.com/file2.html"; //
    input[5] = "http://www.server1.com/file2.jpg";
    input[6] = "http://www.server1.com/file2.gif";
    input[7] = "http://www.server1.com/file3.html"; //
    input[8] = "http://www.server1.com/file3.jpg";
    input[9] = "http://www.server1.com/file3.gif";
    input[10] = "http://www.server1.com/file4.html";  //
    input[11] = "http://www.server1.com/file5.html";  //
    input[12] = "http://www.server1.com/file6.cgi";
    input[13] = "http://www.server1.com/file6.jpg";
    input[14] = "http://www.server1.com/file6.gif";
    input[15] = "http://www.server2.com/file1.html"; //
    input[16] = "http://www.server1.com/file99.jpg";
    input[17] = "http://www.server1.com/file99.gif";
    input[18] = "http://www.server1.com/file2.html"; // 4
    input[19] = "http://www.server1.com/file4.html"; // 11
    input[20] = "http://www.server1.com/file99.jpg"; 
    input[21] = "http://www.server1.com/file99.gif"; 

      
    PredRawReq rr = new PredRawReq(new Prediction(input[2], (float)1.0));

    // "file1.html"
    PredRawReq pred[] = getList(rr);
    Assert.myAssert(onList(pred, input[3], (float)1.0));
    Assert.myAssert(onList(pred, input[4], (float)1.0));
    Assert.myAssert(onList(pred, input[5], (float)1.0));
    Assert.myAssert(onList(pred, input[6], (float)1.0));
    Assert.myAssert(onList(pred, input[7], (float)1.0));
    Assert.myAssert(offList(pred, input[8]));
    System.out.println("Pass test 1.");


    // "file1.html", "file2.html" -->
    rr = new PredRawReq(new Prediction(input[4], (float)1.0));
    pred = getList(rr);
    Assert.myAssert(onList(pred, input[5], (float)1.0));
    Assert.myAssert(onList(pred, input[6], (float)1.0));
    Assert.myAssert(onList(pred, input[7], (float)1.0));
    Assert.myAssert(onList(pred, input[8], (float)1.0));
    Assert.myAssert(onList(pred, input[9], (float)1.0));
    Assert.myAssert(onList(pred, input[10], (float)1.0));
    Assert.myAssert(offList(pred, input[11]));
    System.out.println("Pass test 2.");



    // Gif file not change history, right? Same as above.
    rr = new PredRawReq(new Prediction("http://www.foo.com/foo.gif", (float)1.0));
    pred = getList(rr);
    Assert.myAssert(onList(pred, input[5], (float)1.0));
    Assert.myAssert(onList(pred, input[6], (float)1.0));
    Assert.myAssert(onList(pred, input[7], (float)1.0));
    Assert.myAssert(onList(pred, input[8], (float)1.0));
    Assert.myAssert(onList(pred, input[9], (float)1.0));
    Assert.myAssert(onList(pred, input[10], (float)1.0));
    Assert.myAssert(offList(pred, input[11]));
    System.out.println("Pass test 3.");


    rr = new PredRawReq(new Prediction("http://www.foo.com/foo.html", (float)1.0));
    pred = getList(rr);
    // context = NO MATCH", "file2.html" -->
    rr = new PredRawReq(new Prediction(input[4], (float)1.0));
    pred = getList(rr);
    Assert.myAssert(onList(pred, input[5], (float)0.5));
    Assert.myAssert(onList(pred, input[6], (float)0.5));
    Assert.myAssert(onList(pred, input[7], (float)0.5));
    Assert.myAssert(onList(pred, input[8], (float)0.5));
    Assert.myAssert(onList(pred, input[9], (float)0.5));
    Assert.myAssert(onList(pred, input[10], (float)1.0));
    Assert.myAssert(onList(pred, input[20], (float)0.5));
    Assert.myAssert(onList(pred, input[21], (float)0.5));
    Assert.myAssert(offList(pred, input[11]));
    Assert.myAssert(offList(pred, "http://sldkfjs.com"));
    System.out.println("Pass test 4.");


  }


private static boolean onList(PredRawReq p[], String s, float prob)
  {
    int ii;
    for(ii = 0; ii < p.length; ii++){
      if(p[ii].getServer().equals(RawReq.getServerFromURL(s))
	 && p[ii].getObjId().equals(RawReq.getObjIdFromURL(s))){
	Assert.myAssert(p[ii].getProb() ==prob);
	return true;
      }
    }
    return false;
  }

private static boolean offList(PredRawReq p[], String s)
  {
    int ii;
    for(ii = 0; ii < p.length; ii++){
      if(p[ii].getServer().equals(RawReq.getServerFromURL(s))
	 && p[ii].getObjId().equals(RawReq.getObjIdFromURL(s))){
	return false;
      }
    }
    return true;
  }

/*
 *------------------------------------------------------------------
 *
 * checkTrace --
 *
 *          Given a trace of inputs, report what the demand hit rate
 *          would be, as well as the hit rate if all proposed prefetches
 *          were instantaneously accomplished.
 *
 * Arguments:
 *      InputTrace t -- an input trace
 *      int nCachesMode -- simulate a unified shared cache or per-client caches
 *      int maxListLen -- maximum prefetch list length
 *                        that will be sent. Ignore extra entries.
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private void checkTrace(InputTrace t, int nCachesMode,
			int maxListLength, float prefThresh)
  {
    Hashtable demandCaches = new Hashtable();
    Hashtable prefetchCaches = new Hashtable();
    Stats demandStats = new Stats();
    Stats prefetchStats = new Stats();
    int count = 0;
    try{
      while(true){
	count++;
	if(count % 100 == 0){
	  System.out.print(".");
	}

	RawReq req = t.nextNoWait();
	String key = Cache.makeKey(req);
	String cacheId = checkMakeCacheId(req, nCachesMode);
	Cache demandCache;
	Cache prefetchCache;

	//
	// Find the appropriate cache to use
	//
	demandCache = (Cache)demandCaches.get(cacheId);
	prefetchCache = (Cache)prefetchCaches.get(cacheId);
	if(demandCache == null){
	  Assert.myAssert(prefetchCache == null);
	  demandCache = new Cache();
	  prefetchCache = new Cache();
	  demandCaches.put(cacheId, demandCache);
	  prefetchCaches.put(cacheId, prefetchCache);
	  System.out.println("NCaches = " + demandCaches.size());
	}
	

	if(demandCache.isCached(req)){
	  demandStats.incrementHit(req);
	}
	else{
	  demandStats.incrementMiss(req);
	  demandStats.demandFetchDone(req, (long)1);
	  demandCache.insertIfCachable(req);
	}
	if(prefetchCache.isCached(req)){
	  prefetchStats.incrementHit(req);
	}
	else{
	  prefetchStats.incrementMiss(req);
	  prefetchStats.demandFetchDone(req, (long)1);
	  prefetchCache.insertIfCachable(req);
	  PredRawReq p[] = getList(req);
	  if(p != null){
	    int ii;
	    for(ii = 0; ii < p.length && ii < maxListLength && p[ii].getProb() > prefThresh; ii++){
	      RawReq pr = p[ii];
	      Assert.myAssert(pr.isCachable(), 
                 "We should not be prefetching uncachable stuff");
	      if(!prefetchCache.isCached(pr)){
		prefetchCache.insertIfCachable(pr);
		prefetchStats.prefetchDone(req, (long)1);
	      }
	    }
	    if(ii > 110) {
		System.out.println("hehe exceeded 110 --"+ii) ;
	    }
	  }
	}
      }
    }
    catch(EndOfInputException e){
      System.out.println("\nStats for demand cache under this workload:");
      demandStats.done();

      System.out.println("Stats for super-aggressive infinitely fast prefetching under this workload:");
      prefetchStats.done();

      System.out.println("Done.");
    }
    catch(MalformedRecordException m){
      System.out.println("Bad record in input: " + m.toString());
      System.exit(-1);
    }
    catch(RuntimeException z){
      System.out.println("Runtime exception: " + z.toString());
      z.printStackTrace();
      System.exit(-1);
    }

    
  }

/*
 *------------------------------------------------------------------
 *
 * checkMakeCacheId --
 *
 *          Make an effective cache Id for check-mode simulation.
 *
 * Arguments:
 *      RawReq req -- the request
 *      int nCaches -- either CHECK_N_CACHES_SIM_SEPARATE (one cache per
 *                     client) or n -- nuber of shared caches
 *                     across which to randomly spread clients.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static String
checkMakeCacheId(RawReq req, int nCaches)
  {
    Assert.myAssert(nCaches == CHECK_N_CACHES_SIM_SEPARATE 
		    || nCaches > 0);
    if(nCaches == CHECK_N_CACHES_SIM_SEPARATE){
      return req.getClient();
    }
    else if(nCaches == 1){
      return "GLOBAL_SHARED";
    }
    long key = makeHashVal(req.getClient());
    if(key < 0){
      key = -1 * key;
    }
    key = key % nCaches;
    Assert.myAssert(key >= 0 && key < nCaches);
    return "SHARED" + key;
  }

/*
 *------------------------------------------------------------------
 *
 * makeHashVal --
 *
 *          Use md5 to make a hash value of key. Yes, it is overkill.
 *          But you would be surpised how easy it is to come
 *          up with a bad hash.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static MD5 md = new MD5();
private static long makeHashVal(String key)
{
  byte got[];
  long ret;
  md.Init();
  md.Update(key);
  got = md.Final();

  ret = (long)(got[0] ^ (got[1] << 8) ^ (got[2] << 16) ^ (got[3] << 24))
    ^ (((long)((got[4]) ^ (got[5] << 8) ^ (got[6] << 16) ^ (got[7] << 24))) << (long)24);


  return ret;
}


/*
 *------------------------------------------------------------------
 *
 * preprocessTrace --
 *
 *          Given a trace of inputs, generate a prediction file.
 *
 * Arguments:
 *      InputTrace trace -- input trace  
 *      float minThresh -- cull all predictions with lower probability
 *                         than this before saving file
 *	String prefetchStateFile -- name of file in which to save predictions
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static void  preprocessTrace(InputTrace trace,  
				     float minThresh, 
				     String prefetchStateFile)
  {
    Hashtable mkPredictionTable; // Contains key --> MarkovPredictionList
    Hashtable clientHistories; // Contains key --> ClientContext
    Hashtable histAppeared; // Contains history--> count of times history appeared
    int histLength;

    mkPredictionTable = new Hashtable();
    clientHistories = new Hashtable();
    histAppeared = new Hashtable();

    int count = 0;
    try{
      while(true){
	count++;
	if(count % 100 == 0){
	  System.out.print(".");
	}

	RawReq req = trace.nextNoWait();
	//System.out.println("MarkovPrefetchPredictor::preprocessTrace got:" + req.toString());
	String key = Cache.makeKey(req);
	ClientContext clCtx = (ClientContext)clientHistories.get(req.getClient());
	if(clCtx == null){
	  clCtx = new ClientContext();
	  clientHistories.put(req.getClient(), clCtx);
	}
	if(req.isCachable()){
	  globalUpdate(mkPredictionTable, clCtx, key);
	}
	if(ClientContext.isNewContext(key)){
	  clCtx.append(key, MAX_HISTORY + MAX_WINDOW + 1);
	  for(histLength = 0; histLength <= MAX_HISTORY; histLength++){
	    String history = clCtx.getHistory(0, histLength);
	    if(history != null){
	      updateTotalHistoryCount(histAppeared, history);
	    }
	  }
	} 
      }
    }
    catch(EndOfInputException e){
      System.out.println("Serializing.");
      serializeTable(mkPredictionTable, histAppeared, minThresh, 
		     prefetchStateFile);
      System.out.println("Done.");
    }
    catch(MalformedRecordException m){
      System.out.println("Bad record in input: " + m.toString());
      System.exit(-1);
    }
    catch(RuntimeException z){
      System.out.println("Runtime exception: " + z.toString());
      z.printStackTrace();
      System.exit(-1);
    }
  }

/*
 *------------------------------------------------------------------
 *
 * globalUpdate --
 *
 *         Update the prediction table based on this client's
 *         history and what the client has just observed.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static void globalUpdate(Hashtable mkPredictionTable, 
				 ClientContext clCtx, 
				 String objKey)
  {
    int histLength;
    int winEntry;
    for(winEntry = 0; winEntry < MAX_WINDOW; winEntry++){
      for(histLength = 0; histLength <= MAX_HISTORY; histLength++){
	//
	// Update the predictions based on a context of length histLength
	//
	//      <--- past                      | present
	// [histList:     9 8 7 6 5 4 3 2 1 0 ]  objKey
	//                          ^ start at win entry
        //                  |--------| use histLength records as history to pred objKey
	//
	String history = clCtx.getHistory(winEntry, histLength);
	if(history != null){
	  updateCount(mkPredictionTable, history, objKey);
	}
      }
    }
  }



/*
 *------------------------------------------------------------------
 *
 * updateTotalHistoryCount --
 *
 *          Update the count of the total number of times
 *          the specified history has appeared.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static void updateTotalHistoryCount(Hashtable histAppeared, 
					    String history)
  {
    Integer countSoFar = (Integer)histAppeared.get(history);
    if(countSoFar == null){
      countSoFar = new Integer(1);
    }
    else{
      countSoFar = new Integer(countSoFar.intValue() + 1);
    }
    histAppeared.put(history, countSoFar);
  }


/*
 *------------------------------------------------------------------
 *
 * updateCount --
 *
 *          Update the prediction table count
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static void  updateCount(Hashtable mkPredictionTable, 
				 String history, 
				 String objKey)
  {
    MarkovPredictionList current 
      = (MarkovPredictionList)mkPredictionTable.get(history);
    if(current == null){
      //
      // Never seen this history before
      //
      current = new MarkovPredictionList();
      mkPredictionTable.put(history, current);
    }
    current.add(objKey);
  }

/*
 *------------------------------------------------------------------
 *
 * serializeTable --
 *
 *          Prepare to serialize a prediction table to a specified file.
 *          Then do it.
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static void serializeTable(Hashtable mkPredictionTable,
				   Hashtable histAppeared,
				   float threshold,
				   String fileName)
  {
    Hashtable newPredictionTable = new Hashtable();
    Enumeration keys = mkPredictionTable.keys();
    while(keys.hasMoreElements()){
      String history = (String)keys.nextElement();
      MarkovPredictionList list = 
	(MarkovPredictionList)mkPredictionTable.get(history);
      int historyAppearedCount = ((Integer)histAppeared.get(history)).intValue();
      Assert.myAssert(historyAppearedCount > 0);
      Prediction pred[] = list.normalizeCullAndSort(historyAppearedCount, threshold);
      if(pred.length > 0){
	newPredictionTable.put(history, pred);
      }
    }
    writeTable(fileName, newPredictionTable);
  }




/*
 *------------------------------------------------------------------
 *
 * writeTable --
 *
 *          Serialize the table to the file
 *
 * Arguments:
 *      type1 arg1 -- description.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      None.
 *
 *------------------------------------------------------------------
 */
private static void writeTable(String fileName, Hashtable predictionTable)
  {
    try{
      FileOutputStream fos = new FileOutputStream(fileName);
      ObjectOutputStream oos = new ObjectOutputStream(fos);
      oos.writeObject(new Integer(MAX_HISTORY));
      oos.writeObject(new Integer(MAX_WINDOW));
      oos.writeObject(predictionTable);
      oos.flush();
      oos.close();
    }
    catch(IOException i){
      System.out.println("IO Exception: " + i.toString());
      i.printStackTrace();
      System.exit(-1);
    }
    
  }
};
